/*
 * Decompiled with CFR 0.152.
 */
package io.methvin.watcher;

import com.sun.nio.file.ExtendedWatchEventModifier;
import io.methvin.watcher.DirectoryChangeEvent;
import io.methvin.watcher.DirectoryChangeListener;
import io.methvin.watcher.PathUtils;
import io.methvin.watcher.hashing.FileHasher;
import io.methvin.watcher.hashing.HashCode;
import io.methvin.watchservice.MacOSXListeningWatchService;
import io.methvin.watchservice.WatchablePath;
import java.io.IOException;
import java.nio.file.FileSystems;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardWatchEventKinds;
import java.nio.file.WatchEvent;
import java.nio.file.WatchKey;
import java.nio.file.WatchService;
import java.nio.file.Watchable;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executor;
import java.util.concurrent.ForkJoinPool;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class DirectoryWatcher {
    private final Logger logger;
    private final WatchService watchService;
    private final List<Path> paths;
    private final List<Path> files;
    private final boolean isMac;
    private final DirectoryChangeListener listener;
    private final Map<Path, HashCode> pathHashes;
    private final Map<WatchKey, Path> keyRoots;
    private final Set<Path> watchingPathRoots;
    private final Set<Path> nonRecursivePathRoots;
    private Boolean fileTreeSupported = null;
    private FileHasher fileHasher;

    public static Builder builder() {
        return new Builder();
    }

    public DirectoryWatcher(List<Path> list, List<Path> list2, DirectoryChangeListener directoryChangeListener, WatchService watchService, FileHasher fileHasher, Logger logger) throws IOException {
        this.paths = list;
        this.files = list2;
        this.listener = directoryChangeListener;
        this.watchService = watchService;
        this.isMac = watchService instanceof MacOSXListeningWatchService;
        this.pathHashes = PathUtils.createHashCodeMap(list, fileHasher);
        this.keyRoots = PathUtils.createKeyRootsMap();
        this.fileHasher = fileHasher;
        this.watchingPathRoots = PathUtils.createKeyRootsSet();
        this.nonRecursivePathRoots = PathUtils.createKeyRootsSet();
        this.logger = logger;
        for (Path path : list) {
            this.registerAll(path);
        }
        this.registerRootsForFiles(list2);
    }

    public CompletableFuture<Void> watchAsync() {
        return this.watchAsync(ForkJoinPool.commonPool());
    }

    public CompletableFuture<Void> watchAsync(Executor executor) {
        return CompletableFuture.supplyAsync(() -> {
            this.watch();
            return null;
        }, executor);
    }

    public void watch() {
        while (true) {
            WatchKey watchKey;
            if (!this.listener.isWatching()) {
                return;
            }
            try {
                watchKey = this.watchService.take();
            }
            catch (InterruptedException interruptedException) {
                return;
            }
            for (WatchEvent<?> watchEvent : watchKey.pollEvents()) {
                try {
                    WatchEvent.Kind<?> kind = watchEvent.kind();
                    WatchEvent watchEvent2 = PathUtils.cast(watchEvent);
                    final int n = watchEvent2.count();
                    Path path = (Path)watchEvent2.context();
                    if (!this.keyRoots.containsKey(watchKey)) {
                        throw new IllegalStateException("WatchService returned key [" + watchKey + "] but it was not found in keyRoots!");
                    }
                    Path path2 = path == null ? null : this.keyRoots.get(watchKey).resolve(path);
                    this.logger.debug("{} [{}]", kind, (Object)path2);
                    if (kind == StandardWatchEventKinds.OVERFLOW) {
                        this.listener.onEvent(new DirectoryChangeEvent(DirectoryChangeEvent.EventType.OVERFLOW, path2, n));
                        continue;
                    }
                    if (path == null) {
                        throw new IllegalStateException("WatchService returned a null path for " + kind.name());
                    }
                    if (kind == StandardWatchEventKinds.ENTRY_CREATE) {
                        if (Files.isDirectory(path2, LinkOption.NOFOLLOW_LINKS)) {
                            if (!Boolean.TRUE.equals(this.fileTreeSupported)) {
                                if (this.shouldIgnoreDirectoryEvent(path2)) {
                                    this.logger.debug("Ignored {} [{}] because one of its parent is a non-recursive directory", kind, (Object)path2);
                                    continue;
                                }
                                this.registerAll(path2);
                            }
                            if (!this.isMac) {
                                Files.walkFileTree(path2, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                                    @Override
                                    public FileVisitResult preVisitDirectory(Path path, BasicFileAttributes basicFileAttributes) throws IOException {
                                        DirectoryWatcher.this.notifyCreateEvent(path, n);
                                        return FileVisitResult.CONTINUE;
                                    }

                                    @Override
                                    public FileVisitResult visitFile(Path path, BasicFileAttributes basicFileAttributes) throws IOException {
                                        DirectoryWatcher.this.notifyCreateEvent(path, n);
                                        return FileVisitResult.CONTINUE;
                                    }
                                });
                            }
                        }
                        this.notifyCreateEvent(path2, n);
                        continue;
                    }
                    if (kind == StandardWatchEventKinds.ENTRY_MODIFY) {
                        if (this.fileHasher != null || Files.isDirectory(path2, new LinkOption[0])) {
                            HashCode hashCode = this.pathHashes.get(path2);
                            HashCode hashCode2 = PathUtils.hash(this.fileHasher, path2);
                            if (hashCode2 != null && !hashCode2.equals(hashCode)) {
                                this.pathHashes.put(path2, hashCode2);
                                this.listener.onEvent(new DirectoryChangeEvent(DirectoryChangeEvent.EventType.MODIFY, path2, n));
                                continue;
                            }
                            if (hashCode2 != null) continue;
                            this.logger.debug("Failed to hash modified file [{}]. It may have been deleted.", (Object)path2);
                            continue;
                        }
                        this.listener.onEvent(new DirectoryChangeEvent(DirectoryChangeEvent.EventType.MODIFY, path2, n));
                        continue;
                    }
                    if (kind != StandardWatchEventKinds.ENTRY_DELETE) continue;
                    this.pathHashes.remove(path2);
                    this.listener.onEvent(new DirectoryChangeEvent(DirectoryChangeEvent.EventType.DELETE, path2, n));
                }
                catch (Exception exception) {
                    this.logger.debug("DirectoryWatcher got an exception while watching!", (Throwable)exception);
                    this.listener.onException(exception);
                }
            }
            boolean bl = watchKey.reset();
            if (bl) continue;
            this.logger.debug("WatchKey for [{}] no longer valid; removing.", (Object)watchKey.watchable());
            this.keyRoots.remove(watchKey);
            this.watchingPathRoots.remove(watchKey);
            if (this.keyRoots.isEmpty()) break;
        }
        this.logger.debug("No more directories left to watch; terminating watcher.");
    }

    public DirectoryChangeListener getListener() {
        return this.listener;
    }

    public void close() throws IOException {
        this.watchService.close();
    }

    private void registerAll(Path path) throws IOException {
        if (!Boolean.FALSE.equals(this.fileTreeSupported)) {
            try {
                this.register(path, true);
                this.fileTreeSupported = true;
            }
            catch (UnsupportedOperationException unsupportedOperationException) {
                this.logger.debug("Assuming ExtendedWatchEventModifier.FILE_TREE is not supported", (Throwable)unsupportedOperationException);
                this.fileTreeSupported = false;
                this.registerAll(path);
            }
        } else {
            Files.walkFileTree(path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult preVisitDirectory(Path path, BasicFileAttributes basicFileAttributes) throws IOException {
                    DirectoryWatcher.this.register(path, false);
                    return FileVisitResult.CONTINUE;
                }
            });
        }
    }

    private void register(Path path, boolean bl) throws IOException {
        WatchEvent.Modifier[] modifierArray;
        Watchable watchable;
        this.logger.debug("Registering [{}].", (Object)path);
        Watchable watchable2 = watchable = this.isMac ? new WatchablePath(path, bl) : path;
        if (bl) {
            WatchEvent.Modifier[] modifierArray2 = new WatchEvent.Modifier[1];
            modifierArray = modifierArray2;
            modifierArray2[0] = ExtendedWatchEventModifier.FILE_TREE;
        } else {
            modifierArray = new WatchEvent.Modifier[]{};
        }
        WatchEvent.Modifier[] modifierArray3 = modifierArray;
        WatchEvent.Kind[] kindArray = new WatchEvent.Kind[]{StandardWatchEventKinds.ENTRY_CREATE, StandardWatchEventKinds.ENTRY_DELETE, StandardWatchEventKinds.ENTRY_MODIFY};
        WatchKey watchKey = watchable.register(this.watchService, kindArray, modifierArray3);
        this.keyRoots.put(watchKey, path);
        this.watchingPathRoots.add(path);
    }

    private void registerRootsForFiles(List<Path> list) throws IOException {
        for (Path path : list) {
            Path path2 = path.getParent();
            if (!this.watchingPathRoots.contains(path2)) {
                if (this.watchingPathRoots.isEmpty()) {
                    this.nonRecursivePathRoots.add(path2);
                    this.register(path2, false);
                    continue;
                }
                for (Path path3 : this.watchingPathRoots) {
                    if (path2.startsWith(path3)) {
                        this.logger.debug("Parent file " + path2.toString() + " is contained in " + path3);
                        continue;
                    }
                    this.nonRecursivePathRoots.add(path2);
                    this.register(path2, false);
                }
                continue;
            }
            this.logger.debug("Parent file " + path2.toString() + " already is being watched");
        }
    }

    private boolean shouldIgnoreDirectoryEvent(Path path) {
        boolean bl = false;
        Path path2 = path.getParent();
        for (Path path3 : this.nonRecursivePathRoots) {
            if (!path2.equals(path3) || !path.startsWith(path3)) continue;
            bl = true;
            break;
        }
        return bl;
    }

    private void notifyCreateEvent(Path path, int n) throws IOException {
        if (this.fileHasher != null || Files.isDirectory(path, new LinkOption[0])) {
            HashCode hashCode = PathUtils.hash(this.fileHasher, path);
            if (hashCode == null) {
                this.logger.debug("Failed to hash created file [{}]. It may have been deleted.", (Object)path);
                return;
            }
            if (!this.pathHashes.containsKey(path)) {
                this.logger.debug("{} [{}]", (Object)DirectoryChangeEvent.EventType.CREATE, (Object)path);
                this.listener.onEvent(new DirectoryChangeEvent(DirectoryChangeEvent.EventType.CREATE, path, n));
                this.pathHashes.put(path, hashCode);
            }
        } else {
            this.logger.debug("{} [{}]", (Object)DirectoryChangeEvent.EventType.CREATE, (Object)path);
            this.listener.onEvent(new DirectoryChangeEvent(DirectoryChangeEvent.EventType.CREATE, path, n));
        }
    }

    public static final class Builder {
        private List<Path> paths = Collections.emptyList();
        private List<Path> files = Collections.emptyList();
        private DirectoryChangeListener listener = directoryChangeEvent -> {};
        private Logger logger = null;
        private FileHasher fileHasher = FileHasher.DEFAULT_FILE_HASHER;
        private WatchService watchService = null;

        private Builder() {
        }

        public Builder paths(List<Path> list) {
            this.paths = list;
            return this;
        }

        public Builder files(List<Path> list) {
            this.files = list;
            return this;
        }

        public Builder path(Path path) {
            return this.paths(Collections.singletonList(path));
        }

        public Builder listener(DirectoryChangeListener directoryChangeListener) {
            this.listener = directoryChangeListener;
            return this;
        }

        public Builder watchService(WatchService watchService) {
            this.watchService = watchService;
            return this;
        }

        public Builder logger(Logger logger) {
            this.logger = logger;
            return this;
        }

        public Builder fileHashing(boolean bl) {
            this.fileHasher = bl ? FileHasher.DEFAULT_FILE_HASHER : null;
            return this;
        }

        public Builder fileHasher(FileHasher fileHasher) {
            this.fileHasher = fileHasher;
            return this;
        }

        public DirectoryWatcher build() throws IOException {
            if (this.watchService == null) {
                this.osDefaultWatchService();
            }
            if (this.logger == null) {
                this.staticLogger();
            }
            return new DirectoryWatcher(this.paths, this.files, this.listener, this.watchService, this.fileHasher, this.logger);
        }

        private Builder osDefaultWatchService() throws IOException {
            boolean bl = System.getProperty("os.name").toLowerCase().contains("mac");
            return this.watchService(bl ? new MacOSXListeningWatchService() : FileSystems.getDefault().newWatchService());
        }

        private Builder staticLogger() {
            return this.logger(LoggerFactory.getLogger(DirectoryWatcher.class));
        }
    }
}

