

############################################################ Descripcion
# Codigo creado para generar una serie diaria representativa de la cuenca
# a partir de un NetCDF a resolucion diaria, utilizando un archivo .shp
# para su recorte espacial (el archivo .shp debe estar en WGS84 lat-lon)

############################################################
require(raster)       # Trabajo con rasters y shapes
require(ncdf4)        # Trabajo con netcdf
require(zoo)          # Trabajo con fechas
require(lubridate)    # Trabajo con fechas
require(reshape)      # Modificar estructura de dataframes
require(tmap)         # Grafica de rasters y shapes
require(ggplot2)      # Funciones graficas
require(rgeos)        # Trabajo con rasters y shapes
require(dplyr)        # Modificar estructura de dataframes
require(tidyr)        # Modificar estructura de dataframes
require(RColorBrewer) # Colores para graficas
custom_theme = theme_bw()+theme(legend.title = element_blank(),
                                legend.text = element_text(size = 22),
                                axis.text = element_text(color = "black",size = 21.5),
                                axis.title = element_text(size = 22, color = "black"), 
                                plot.title = element_text(size = 22, color = "black", face = "bold",hjust = 0.5), 
                                legend.position = c(.95, .95),
                                legend.justification = c("right", "top"),
                                legend.box.just = "right",
                                legend.margin = margin(6, 6, 6, 6))
theme_set(custom_theme)
############################################################


#### Definicion de directorios e importacion de datos
############################################################
# Carga del shape
directorio_shp = paste0(getwd())
shp_name = "polygon.shp"
shp_bas = shapefile(paste(directorio_shp,shp_name, sep = "/"))
plot(shp_bas)
# Carga del netcdf CR2MET
directorio_netcdf = paste0(getwd())
nc_name = "CR2MET_pr_v2.0_day_1979_2020_005deg.nc"
nc_daily = nc_open(paste(directorio_netcdf,
                         nc_name, 
                         sep = "/")) #lon x lat x time
############################################################


#### Extraccion de informacion y creacion de vectores utiles
############################################################
# Lats y Lons limites de la cuenca
lims_bas = extent(shp_bas)
xlim_west = lims_bas@xmin
xlim_east = lims_bas@xmax
ylim_south = lims_bas@ymin
ylim_north = lims_bas@ymax
# Dimensiones del netcdf
lats_netcdf = nc_daily$dim$lat$vals
lons_netcdf = nc_daily$dim$lon$vals
time_netcdf = nc_daily$dim$time$vals #seconds since 1979-01-01 00:00:00.0
dates_netcdf = as.Date(as.POSIXct(time_netcdf, 
                                    origin = "1979-01-01"))
# Ubicar limites temporales para comparar con cr2met
days_seq = dates_netcdf
idx_ini_time = which(days_seq == as.Date("1979-01-01"))
idx_fin_time = which(days_seq == as.Date("2014-12-31"))
n_time = abs(idx_fin_time - idx_ini_time) + 1
dates_netcdf = dates_netcdf[idx_ini_time:idx_fin_time]
n_day = length(dates_netcdf)
# Vectores diarios, mensuales y anuales
months_seq = seq.Date(dates_netcdf[1], 
                      dates_netcdf[n_day],
                      by = "months") 
n_months = length(months_seq)
years_seq = seq.Date(dates_netcdf[1], 
                     dates_netcdf[n_day],
                     by = "years") 
n_years = length(years_seq)
# Ubicar limites del shape en netcdf
idx_xlim_west = which.min(abs(xlim_west - lons_netcdf))
idx_xlim_east = which.min(abs(xlim_east - lons_netcdf))
idx_ylim_south = which.min(abs(ylim_south - lats_netcdf))
idx_ylim_north = which.min(abs(ylim_north - lats_netcdf))
lons_bas_netcdf = lons_netcdf[idx_xlim_west:idx_xlim_east]
lats_bas_netcdf = lats_netcdf[idx_ylim_south:idx_ylim_north]
n_lats = abs(idx_ylim_north - idx_ylim_south) + 1
n_lons = abs(idx_xlim_east - idx_xlim_west) + 1
# Extraccion de netcdf en dominio de la cuenca
var_day = ncvar_get(nc = nc_daily, 
                    varid = "pr", 
                    start = c(idx_xlim_west, idx_ylim_south, idx_ini_time),
                    count = c(n_lons, n_lats, n_time))
nc_close(nc_daily)
############################################################


#### Convertir la matriz 3D de variable diaria a rasterstack diario
############################################################
# Crear un raster vacio a la resolucion nativa del NetCDF
# esto arregla el problema de "y cell sizes are not regular"
res_netcdf = round(diff(lats_netcdf)[1], 2)
raster_aux = raster(extent(shp_bas), 
                    res = res_netcdf)
crs(raster_aux) = crs(shp_bas)
raster_aux_fine = raster(extent(shp_bas), 
                         res = 0.05)
crs(raster_aux_fine) = crs(shp_bas)
# Generar lista con rasters diarios
list_rasters = list()
for (iday in 1:n_day) {
  iday_netcdf = as.data.frame(var_day[,,iday])
  colnames(iday_netcdf) = lats_bas_netcdf
  iday_netcdf$lons = lons_bas_netcdf
  iday_netcdf <- melt(iday_netcdf, id.var="lons")
  colnames(iday_netcdf) = c("lon","lat","Valor")
  iday_netcdf$lat = as.numeric(levels(iday_netcdf$lat))[iday_netcdf$lat]
  iday_netcdf = as.matrix(iday_netcdf)
  colnames(iday_netcdf) = c("X", "Y", "Z")
  raster_iday_netcdf = rasterize(iday_netcdf[,1:2], 
                                 raster_aux, 
                                 iday_netcdf[,3], 
                                 fun = mean)
  raster_iday_netcdf = resample(raster_iday_netcdf, 
                                raster_aux_fine, 
                                "bilinear")
  list_rasters[[iday]] = raster_iday_netcdf
  print(100 * iday / n_day)
}
# Generar rasterstack 
rasterstack_daily = stack(list_rasters)
############################################################


#### Generar serie diaria a escala de cuenca
############################################################
# Generar raster de fraccion por pixel
raster_frac_bas = rasterize(shp_bas, 
                            rasterstack_daily[[1]], 
                            getCover =  T)
plot(raster_frac_bas)
plot(shp_bas, add = T)
raster_frac_bas = raster_frac_bas / cellStats(raster_frac_bas, sum)
# Generar serie diaria a escala de cuenca
df_daily = data.frame("Fechas" = dates_netcdf,
                      "Valor" = NA)
View(df_daily)
for (iday in 1:n_day) {
  val_bas = raster_frac_bas * rasterstack_daily[[iday]]
  df_daily$Valor[iday] = cellStats(val_bas, sum)
  print(100 * iday / n_day)
}
# Graficar
ggplot(df_daily, aes(Fechas, Valor))+
  geom_line(size = 1)+
  ggtitle("Serie diaria representativa de la cuenca")
############################################################