# Creado por Felipe Gateño, Noviembre 2020

############################################################ Descripcion
# Codigo creado para trabajar con el producto grillado de 
# precipitaciones CR2MET_pr_v2.0_day_1979_2018_005deg.nc
# Mediante la delimitacion dada por un shape en WGS84, calcula
# para cada celda del producto grillado: (1) Ciclo anual (promedio 
# por dia calendario), (2) Serie mensual, (3) Serie anual (por año 
# hidrologico). Se presentan las graficas de la precipitacion 
# promedio anual y anual distribuidas para la cuenca.
# Luego, calcula la fraccion de pixel dentro de la cuenca, y
# con ello calcula las mismas series anteriores, ahora a escala
# de cuenca (ponderacion por fraccion de area). Se genera ademas
# la serie estacional representativa de la cuenca

############################################################
require(raster)       # Trabajo con rasters y shapes
require(ncdf4)        # Trabajo con netcdf
require(zoo)          # Trabajo con fechas
require(lubridate)    # Trabajo con fechas
require(reshape)      # Modificar estructura de dataframes
require(tmap)         # Grafica de rasters y shapes
require(ggplot2)      # Funciones graficas
require(rgeos)        # Trabajo con rasters y shapes
require(dplyr)        # Modificar estructura de dataframes
require(tidyr)        # Modificar estructura de dataframes
require(RColorBrewer) # Colores para graficas
custom_theme = theme_bw()+theme(legend.title = element_blank(),
                                legend.text = element_text(size = 22),
                                axis.text = element_text(color = "black",size = 21.5),
                                axis.title = element_text(size = 22, color = "black"), 
                                plot.title = element_text(size = 22, color = "black", face = "bold",hjust = 0.5), 
                                legend.position = c(.95, .95),
                                legend.justification = c("right", "top"),
                                legend.box.just = "right",
                                legend.margin = margin(6, 6, 6, 6))
theme_set(custom_theme)
############################################################


#### Definicion de directorios e importacion de datos
############################################################
# Carga del shape
directorio_shp = getwd()
shp_name = "gauge_id_8313000.shp"
shp_bas = shapefile(paste(directorio_shp,shp_name, sep = "/"))
# Carga del netcdf CR2MET
directorio_CR2MET = "D:/Estaciones_CR2_DGA/Forzantes_CR2/Resolucion_Diaria"
nc_name = "CR2MET_pr_v2.0_day_1979_2018_005deg.nc"
nc_CR2MET = nc_open(paste(directorio_CR2MET,nc_name, sep = "/"), readunlim = F) #lon x lat x time
############################################################


#### Extraccion de informacion y creacion de vectores utiles
############################################################
# Lats y Lons limites de la cuenca
lims_bas = extent(shp_bas)
xlim_west = lims_bas@xmin
xlim_east = lims_bas@xmax
ylim_south = lims_bas@ymin
ylim_north = lims_bas@ymax
# Dimensiones del netcdf
lats = nc_CR2MET$dim$lat$vals
lons = nc_CR2MET$dim$lon$vals
time = nc_CR2MET$dim$time$vals # dias desde 1978-12-31
# Vectores diarios, mensuales y anuales
days_seq = seq.Date(as.Date("1979-01-01"), 
                    length.out = length(time),
                    by = "days")
n_day = length(days_seq)
months_seq = seq.Date(as.Date("1979-01-01"), 
                      days_seq[n_day],
                      by = "months") 
n_months = length(months_seq)
years_seq = seq.Date(as.Date("1979-01-01"), 
                     days_seq[n_day],
                     by = "years") 
n_years = length(years_seq)
# Ubicar limites del shape en netcdf
idx_xlim_west = which.min(abs(xlim_west - lons))
idx_xlim_east = which.min(abs(xlim_east - lons))
idx_ylim_south = which.min(abs(ylim_south - lats))
idx_ylim_north = which.min(abs(ylim_north - lats))
n_lats = idx_ylim_north - idx_ylim_south + 1
n_lons = idx_xlim_east - idx_xlim_west + 1
# Extraccion de netcdf en dominio de la cuenca
pr_day_CR2MET = ncvar_get(nc_CR2MET, 
                          varid = "pr", 
                          start = c(idx_xlim_west, idx_ylim_south, 1),
                          count = c(n_lons, n_lats, -1))
nc_close(nc_CR2MET)
############################################################


#### Calcular ciclo anual
############################################################
Mean_day_Pixel <- function(x) {
  ciclo_anual = rep(NA,365)
  for (iday in 1:365) {
    idx_day = which(yday(days_seq) == iday)
    ciclo_anual[iday] = mean(x[idx_day], na.rm = T)
  }
  return(ciclo_anual)
}
pr_ciclo_anual_CR2MET = apply(pr_day_CR2MET, c(1,2), Mean_day_Pixel)
pr_ciclo_anual_CR2MET = aperm(pr_ciclo_anual_CR2MET, c(2,3,1))
############################################################


#### Calcular serie mensual
############################################################
idx_aggregate_month = rep(NA, n_day)
k = 1
for (iday in 1:n_day) {
  if (month(days_seq[iday]) == month(days_seq[iday]+1)) {
    idx_aggregate_month[iday] = k
  }
  if (month(days_seq[iday]) != month(days_seq[iday]+1)) {
    idx_aggregate_month[iday] = k
    k = k + 1
  }
  
}
Sum_Month_Pixel <- function(x) {
  pixel.ts <- zoo(x, days_seq)
  out <- as.numeric(aggregate(pixel.ts, idx_aggregate_month, sum, na.rm=F))
  out[is.nan(out)] <- NA     
  return(out)
}
pr_month_CR2MET = apply(pr_day_CR2MET, c(1,2), Sum_Month_Pixel)
pr_month_CR2MET = aperm(pr_month_CR2MET, c(2,3,1))
############################################################


#### Calcular serie anual (por año hidrologico)
############################################################
idx_aggregate_years = rep(0, n_months)
idx_marchs = c(1,which(month(months_seq) == 4))
k = 0
for (i in 1:length(idx_marchs)) {
  if (is.na(idx_marchs[i+1]) ==  T) {
    idx_aggregate_years[idx_marchs[i]:n_months]= k
  }
  if (is.na(idx_marchs[i+1]) ==  F) {
    idx_aggregate_years[idx_marchs[i]:(idx_marchs[i+1]-1)]= k
  }
  k = k + 1
}
Sum_Year_Pixel <- function(x) {
  pixel.ts <- zoo(x, months_seq)
  out <- as.numeric(aggregate(pixel.ts, idx_aggregate_years, sum, na.rm=F))
  out[is.nan(out)] <- NA     
  return(out)
}
pr_year_CR2MET = apply(pr_month_CR2MET, c(1,2), Sum_Year_Pixel)
pr_year_CR2MET = aperm(pr_year_CR2MET, c(2,3,1))
pr_year_CR2MET = pr_year_CR2MET[,,2:dim(pr_year_CR2MET)[3]]
years_seq = years_seq[1:dim(pr_year_CR2MET)[3]]
n_years = length(years_seq)
############################################################


#### Calcular promedio anual y graficar
############################################################
# Calcular promedio
pr_year_mean_CR2MET = apply(pr_year_CR2MET, c(1,2), mean)
## Graficar con tmap
# Preparar info para crear raster
grid_pr = as.data.frame(pr_year_mean_CR2MET)
colnames(grid_pr) = lats[idx_ylim_south:idx_ylim_north]
grid_pr$lons = lons[idx_xlim_west:idx_xlim_east]
grid_pr <- melt(grid_pr, id.var="lons")
colnames(grid_pr) = c("lon","lat","Valor")
# Crear raster 
raster_pr = rasterFromXYZ(grid_pr)
crs(raster_pr) = crs(shp_bas)
# Graficar
tm_shape(raster_pr)+tm_raster(legend.show = T, palette = brewer.pal("RdYlBu", n = 5))+
  tm_shape(shp_bas)+
  tm_borders(lty = "solid")+
  tm_layout(panel.label.size = 2, main.title  = "Pr media anual", main.title.position = "center")
## Graficar con ggplot
# Preparar info para crear dataframe
grid_pr_aux = grid_pr
grid_pr_aux$lat = as.numeric(levels(grid_pr_aux$lat))[grid_pr_aux$lat]
# Transformar shape a dataframe
shp_bas_f <- fortify(shp_bas)
shp_bas$id <- row.names(shp_bas) 
lnd_f <- left_join(shp_bas_f, shp_bas@data) 
# Graficar
ggplot()+
  geom_raster(data = grid_pr_aux, aes(lon,lat,fill = Valor))+
  geom_polygon(data = shp_bas_f, aes(long, lat), fill = "transparent", color = "black")+
  scale_fill_gradientn(colours = brewer.pal("RdYlBu", n = 5))
############################################################


#### Graficar precipitacion anual para todos los años
############################################################
# Preparar info para crear raster
grid_pr_yearly = as.data.frame(pr_year_CR2MET[,,1])
colnames(grid_pr_yearly) = lats[idx_ylim_south:idx_ylim_north]
grid_pr_yearly$lons = lons[idx_xlim_west:idx_xlim_east]
grid_pr_yearly <- melt(grid_pr_yearly, id.var="lons")
colnames(grid_pr_yearly) = c("lon","lat","1979")
for (iyear in 2:n_years) {
  grid_pr_yearly_aux = as.data.frame(pr_year_CR2MET[,,iyear])
  colnames(grid_pr_yearly_aux) = lats[idx_ylim_south:idx_ylim_north]
  grid_pr_yearly_aux$lons = lons[idx_xlim_west:idx_xlim_east]
  grid_pr_yearly_aux <- melt(grid_pr_yearly_aux, id.var="lons")
  colnames(grid_pr_yearly_aux) = c("lon","lat",toString(1979+iyear-1))
  grid_pr_yearly[,iyear+2] = grid_pr_yearly_aux[,3]
  colnames(grid_pr_yearly)[iyear+2] = toString(1979+iyear-1)
}
grid_pr_yearly_tidy <- gather(grid_pr_yearly, year, precipitacion, -lon, -lat)
grid_pr_yearly_tidy$lat = as.numeric(levels(grid_pr_yearly_tidy$lat))[grid_pr_yearly_tidy$lat]
# Graficar
ggplot()+
  geom_raster(data = grid_pr_yearly_tidy, aes(lon,lat,fill = precipitacion))+
  geom_polygon(data = shp_bas_f, aes(long, lat), fill = "transparent", color = "black")+
  scale_fill_gradientn(colours = brewer.pal("RdYlBu", n = 5))+
  facet_wrap(~ year, ncol = 10)+
  theme(axis.ticks = element_blank(),
        axis.text = element_blank(),
        axis.title = element_blank(),
        legend.position = "right",
        strip.background = element_rect(fill = "white", color = "transparent"),
        strip.text = element_text(color = "black", size = 10))
############################################################


#### Crear raster con fraccion de pixel asociada a la cuenca
############################################################
raster_frac_bas = rasterize(shp_bas, raster_pr, getCover =  T)
plot(raster_frac_bas)
plot(shp_bas, add = T)
matrix_frac_bas = as.matrix(raster_frac_bas)
matrix_frac_bas = matrix_frac_bas/sum(matrix_frac_bas)
sum(matrix_frac_bas)
############################################################


#### Calcular ciclo anual representativo de la cuenca
############################################################
pr_ciclo_anual_represent = rep(NA, 365)
for (iday in 1:365) {
  pr_iday_bas = pr_ciclo_anual_CR2MET[,,iday]
  pr_iday_bas = t(pr_iday_bas)
  pr_iday_bas = pr_iday_bas[nrow(pr_iday_bas):1,]
  pr_ciclo_anual_represent[iday] = sum(pr_iday_bas*matrix_frac_bas)
}
df_ciclo_anual = data.frame("Dia" = seq(1,365,1),
                            "Precipitacion" = pr_ciclo_anual_represent)
# Graficar
ggplot(df_ciclo_anual, aes(Dia, Precipitacion))+
  geom_line()+
  scale_y_continuous("Precipitacion [mm/d]")+
  ggtitle("Ciclo anual de precipitacion")
############################################################


#### Calcular serie mensual representativa de la cuenca
############################################################
pr_monthly_represent = rep(NA, n_months)
for (imonth in 1:n_months) {
  pr_imonth_bas = pr_month_CR2MET[,,imonth]
  pr_imonth_bas = t(pr_imonth_bas)
  pr_imonth_bas = pr_imonth_bas[nrow(pr_imonth_bas):1,]
  pr_monthly_represent[imonth] = sum(pr_imonth_bas*matrix_frac_bas)
}
df_serie_mensual = data.frame("Mes" = months_seq,
                              "Precipitacion" = pr_monthly_represent)
# Graficar
ggplot(df_serie_mensual, aes(Mes, Precipitacion))+
  geom_line()+
  scale_y_continuous("Precipitacion [mm/mes]")+
  ggtitle("Serie mensual de precipitacion")
############################################################


#### Calcular serie anual representativa de la cuenca
############################################################
pr_yearly_represent = rep(NA, n_years)
for (iyear in 1:n_years) {
  pr_iyear_bas = pr_year_CR2MET[,,iyear]
  pr_iyear_bas = t(pr_iyear_bas)
  pr_iyear_bas = pr_iyear_bas[nrow(pr_iyear_bas):1,]
  pr_yearly_represent[iyear] = sum(pr_iyear_bas*matrix_frac_bas)
}
df_serie_anual = data.frame("Año" = years_seq,
                            "Precipitacion" = pr_yearly_represent)
# Graficar
ggplot(df_serie_anual, aes(Año, Precipitacion))+
  geom_line()+
  scale_y_continuous("Precipitacion [mm/año]")+
  ggtitle("Serie mensual de precipitacion")

#### Calcular serie estacional representativa de la cuenca
############################################################
df_serie_estacional = Mensuales_a_Estacional(df_serie_mensual)
ggplot(df_serie_estacional, aes(x = Mes_n, y = Precipitacion))+
  geom_line()+
  scale_y_continuous("Precipitacion [mm/mes]")+
  scale_x_discrete("",limit=c(1:12), labels=c("Abr","May","Jun","Jul","Ago","Sep","Oct","Nov","Dic","Ene","Feb","Mar"))+
  ggtitle("Serie estacional de precipitacion")
