package tictactoe.gui;

import javafx.scene.Group;
import javafx.scene.image.Image;
import javafx.scene.image.ImageView;
import tictactoe.Player;

/**
 * Represents the view of a single place on the tictactoe.BoardGame.
 * An instance of PlaceListener forwards mouse clicks to a tictactoe.Player.
 * Here we assume that Players have the mark 'X' or 'O'.
 */
public class Place extends Group {
	private final int col;
	private final int row;
	
	private final String imageDir = "tictactoe/gui/images";
	private Image image;
	private ImageView imageView;
	private final Image oImage;
	private final Image xImage;
	private final Image blankImage;
	
	/**
	 * A Place knows its row and column number.
	 * It also knows how to display either an X or O image
	 * when its state changed.
	 */
	Place(int myCol, int myRow) {
		col = myCol;
		row = myRow;
		this.xImage = getImage(imageDir + "/cross.gif");
		this.oImage = getImage(imageDir + "/not.gif");
		this.blankImage = getImage(imageDir + "/blank.gif");
		
		imageView = new ImageView();
		//imageView.setFitWidth(GameGUI.CELL_SIZE);
		imageView.setFitWidth(40);
		imageView.setPreserveRatio(true);
		this.getChildren().add(imageView);
		this.clearImage();
	}
	
	private void setImage(Image anImage) {
		imageView.setImage(anImage);
	}

	/**
	 * Load an image from a given file name
	 * @param imageFile
	 * @return the loaded image
	 */
	private Image getImage(String imageFile){
		return new Image(imageFile, GameGUI.CELL_SIZE, GameGUI.CELL_SIZE, false, false);
	}

	
	/**
	 * Called by PlaceListener to determine in which location
	 * the mouse has been clicked.
	 * @see PlaceListener
	 */
	public int getRow() { return row; }
	public int getCol() { return col; }
	
	/**
	 * Called when the tictactoe.BoardGame changes state.
	 * Note that we hard-wire the interpretation of the mark
	 * as either 'X' or 'O'.  This would have to be changed
	 * if the application were extended to deal with other
	 * kinds of Players.
	 */
	public void setMove(Player player) {
		if (player.mark() == 'X') {
			image = xImage;
		} else {
			image = oImage;
		}
		this.setImage(image);
	}
	
	public void clearImage() {
		this.setImage(blankImage);
	}
}

