package tictactoe.gui;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import javafx.application.Application;
import javafx.event.ActionEvent;
import javafx.event.EventHandler;
import javafx.scene.Node;
import javafx.scene.Scene;
import javafx.scene.control.Button;
import javafx.scene.layout.BorderPane;
import javafx.scene.layout.GridPane;
import javafx.scene.text.Text;
import javafx.stage.Stage;
import tictactoe.BoardGame;
import tictactoe.Move;
import tictactoe.Player;
import tictactoe.TicTacToe;

/**
 * Main class representing the User Interface of the Game GUI
 *
 * @author alexandrebergel
 */
public class GameGUI extends Application implements PropertyChangeListener {
	private BoardGame game;
	private Text statusbar;
	private Place[][] places;

	/** 
	 *  Size of a cell 
	 */
	public static final int CELL_SIZE = 45;

    /**
     * In case of the application is run from the main method
     *
     * @param args
     */
	public static void main(String[] args) {
		Application.launch(args);
	}

    /**
     * Entry point of the JavaFX applciation
     *
     * @param stage the window
     */
	public void start(Stage stage) {
		stage.setTitle("Tic Tac Toe");
		
		game = makeGame();
		
		// notify GameGui if change of state
		game.addObserver(this);

		BorderPane borderpane = new BorderPane();
		borderpane.setTop(makeControls());
		borderpane.setCenter(makeGrid());
		statusbar = new Text();
		borderpane.setBottom(statusbar);
	
		// Create a JavaFX scene
		Scene scene = new Scene(borderpane, horizontalSize(), verticalSize());		
		stage.setScene(scene);
		//stage.setOpacity(0.7);
		stage.show();
	}

	/**
     * Return the height of the window
     *
	 * @return the height of the window
	 */
	private int verticalSize() {
		return CELL_SIZE * (1 + game.getRows());
	}

	/**
     * Return the width of the window
     *
	 * @return the width of the window
	 */
	private int horizontalSize() {
		return CELL_SIZE * game.getCols();
	}

    /**
     * Return the game. According to the MVC architectural pattern, the
     * UI knows about the model (i.e., the game)
     * The method getGame() is called from the controllers
     *
     * @return the game model
     */
	public BoardGame getGame() { return game; }

    /**
     * Define the button to begin a new game
     *
     * @return the button to be added
     */
	private Node makeControls() {
		Button again = new Button("New game");
		
		again.setPrefSize(horizontalSize(), 20);
		again.setOnAction(new EventHandler<ActionEvent>() {
			@Override
			public void handle(ActionEvent event) {
				showFeedBack("starting new game ...");
				newGame();
			}
		});
		return again;
	}

    /**
     * Create a new game
     *
     * @return the game to be played
     */
	private BoardGame makeGame() {
		Player X = new GUIplayer('X');
		Player O = new GUIplayer('O');
		return new TicTacToe(X, O);
		//return new tictactoe.Gomoku(X, O);
	}
	
	/**
	 * Start a new game by deleting the old one, instantiating
	 * a new one, and clearing the images of the Places.
	 */
	private void newGame() {
		game = makeGame();
		game.addObserver(this);

		int cols = game.getCols();
		int rows = game.getRows();

		for (int col = cols - 1; col >= 0; col--) {
			for (int row = 0; row < rows; row++) {
				places[col][row].clearImage();
			}
		}
		showFeedBack(game.currentPlayer().mark() + " plays");
	}
	
	/**
	 * Make the grid that serves as a view and controller for the game.
	 */
	private Node makeGrid() {
		int cols = game.getCols();
		int rows = game.getRows();

		GridPane grid = new GridPane();
		grid.setHgap(5);
		grid.setVgap(5);

		places = new Place[cols][rows];
		for (int row = rows - 1; row >= 0; row--) {
			for (int col = 0; col < cols; col++) {
				Place p = new Place(col, row);
				p.setOnMouseClicked(new PlaceListener(p, this));
				grid.add(p, col, row);
				places[col][row] = p;
			}
		}
		return grid;
	}
	
	/**
	 * Displays a feedback string in the associated Label component.
	 * @param msg the string to display
	 */
	public void showFeedBack(String msg) {
		statusbar.setText(msg);
	}

    /**
     * Implements the Observer pattern.
     * Called by the tictactoe.BoardGame when its state changes.
     * @param event an instance of PropertyChangeEvent that refers to the tictactoe.BoardGame,
     *              and the instance of tictactoe.Move
     * @see Move
     */
    @Override
    public void propertyChange(PropertyChangeEvent event) {
        Move move = (Move) event.getNewValue();
        showFeedBack("got an update: " + move);
        places[move.col][move.row].setMove(move.player);
    }
}
