#!/usr/bin/env python
# -*- coding: utf-8 -*-
import click
import cv2
import numpy as np

sobel_x = lambda image: cv2.convertScaleAbs(cv2.Sobel(image,
                                                      cv2.CV_16S,
                                                      1,
                                                      0,
                                                      ksize=3,
                                                      scale=1,
                                                      delta=0,
                                                      borderType=cv2.BORDER_DEFAULT))

sobel_y = lambda image: cv2.convertScaleAbs(cv2.Sobel(image,
                                                      cv2.CV_16S,
                                                      0,
                                                      1,
                                                      ksize=3,
                                                      scale=1,
                                                      delta=0,
                                                      borderType=cv2.BORDER_DEFAULT))

border_methods = {
    'sobel_x': sobel_x,
    'sobel_y': sobel_y,
    'sobel': lambda sobel_x_value, sobel_y_value: cv2.addWeighted(sobel_x_value, 0.5, sobel_y_value, 0.5, 0),
    'canny': lambda image, t_inf, t_sup: cv2.Canny(image, t_inf, t_sup)
}


@click.command()
@click.argument('border_method', type=click.Choice(border_methods.keys()))
@click.option('--inferior-thresh', default=100, help='Inferior threshold for Canny')
@click.option('--superior-thresh', default=200, help='Superior threshold for Canny')
def get_borders_from_video(border_method, inferior_thresh, superior_thresh):
    cap = cv2.VideoCapture(1)

    while True:
        # Capture frame-by-frame
        ret, frame = cap.read()
        if border_method == 'sobel':
            frame = cv2.resize(frame, (0, 0), None, .5, .5)
            grey = cv2.cvtColor(frame, cv2.COLOR_BGR2GRAY)
            grey_x = border_methods['sobel_x'](grey)
            grey_3_channel_x = cv2.cvtColor(grey_x, cv2.COLOR_GRAY2BGR)
            grey_y = border_methods['sobel_y'](grey)
            grey_3_channel_y = cv2.cvtColor(grey_y, cv2.COLOR_GRAY2BGR)
            grey_sobel = border_methods['sobel'](grey_x, grey_y)
            grey_3_channel_sobel = cv2.cvtColor(grey_sobel, cv2.COLOR_GRAY2BGR)
            numpy_horizontal_concat = np.concatenate((np.concatenate((frame, grey_3_channel_x),
                                                                     axis=1),
                                                     np.concatenate((grey_3_channel_y, grey_3_channel_sobel),
                                                                    axis=1)),
                                                     axis=0)

        else:
            frame = cv2.resize(frame, (0, 0), None, .75, .75)
            grey = cv2.cvtColor(frame, cv2.COLOR_BGR2GRAY)
            if border_method == 'canny':
                grey = border_methods[border_method](grey, inferior_thresh, superior_thresh)
            else:
                grey = border_methods[border_method](grey)

            # Make the grey scale image have three channels
            grey_3_channel = cv2.cvtColor(grey, cv2.COLOR_GRAY2BGR)
            numpy_horizontal_concat = np.concatenate((frame, grey_3_channel), axis=1)

        cv2.imshow('Borders with {}'.format(border_method), numpy_horizontal_concat)

        # Our operations on the frame come here
        gray = cv2.cvtColor(frame, cv2.COLOR_BGR2GRAY)

        # Display the resulting frame

        if cv2.waitKey(1) & 0xFF == ord('q'):
            break

    # When everything done, release the capture
    cap.release()
    cv2.destroyAllWindows()


if __name__ == '__main__':
    get_borders_from_video()