% function daisypred0D(lower,upper) 
%
% Modified Daisyworld 0-D model that shows life can influence global temperature, based
% on initial work by Watson and Lovelock (1983) and later work by Lovelock (2000).
% By Mark Wittwer, 2005. %
% Arguments: -lower- fraction of current insolation energy at the start of daisyworld.
% -upper- fraction of current insolation energy at the end of daisyworld. %
% Note: There are constants that can also be changed within the m-file, including:
% -Black daisy albedo (black).
% -White daisy albedo (white).
% -Bare planet albedo (bare).
% -Horizontal heat flux (q).
% -Daisy death rate (death).
% -Present day solar luminosity (S).
% -Lower temperature limit for daisy growth (lowgrowthK).
% -Upper temperature limit for daisy growth (highgrowthK).
% -The minimum daisy area to start growth (startarea).
% -Size of each increment determined by (numba).
% -Growth rate of rabbits (grab).
% -Growth rate of foxes (gfox).
% -Severity of plagues (severe).
% -Number of plagues (plagues).
% -Daisy food fixed to rabbit biomass (rfix).
% -Predator food fixed to fox biomass (ffix).
% -Minimum fraction of rabbit and fox population (minpop).
% -Predator death rate (rdeath and fdeath).
% -Accesability to food source (reat and feat).
%
%
% eg. daisypred0D(0.6,1.8) 
%
% Outputs: - Growth of each coloured daisy vs time.
% - Lifeless global temperature and daisyworld temperature vs time.
% - Planterary albedo vs time.


function daisypred0D(lower,upper) 

S = 1000;
q = 15;
black = .25; 
bare  = .5; 
white = .75; 
death = .1; 
lowgrowthK  = 278; 
highgrowthK = 313; 
startarea   = 0.01;  %start daisy area as a fraction of 1
numba       = 100;   %number of time steps relative to insolation increase
grab        = 2.5;   %rabbit growth rate 
gfox        = .8;    %fox growth rate
blacksevere = 0.3;   %severity of each plague (fraction of black area killed)
whitesevere = 0.3;   %severity of each plague (fraction of white area killed) 
plagues = 4;         %number of plagues (+ve integer)
rfix = .3;           %food consumed (daisy) that fixes to rabbit biomass 
ffix = .5;           %food consumed (rabbit) that fixes to fox biomass
rminpop = 0.01;      %minimum population of rabbits 
fminpop = 0.005;     %minimum population of foxes
rdeath = .1;         %death rate of rabbits
fdeath = .1;         %death rate of foxes
reat = .8;           %accessability of daisies to rabbits (0,1) 
feat = .4;           %accesability of rabbits to foxes (0,1)






% Calculate number of time steps (numba per 3 billion years) 
length = (upper-lower)*numba;

if plagues>length
   disp('Error- increase "numba" or decrease "plagues"- (too much death).')
end
if length~=round(length) 
   length=round(length); 
   boof=abs(length);
 else boof=abs(length); 
end

if length>0
  jump=(upper*S-lower*S)/(length-1);
  E(1,1)=lower*S; %initialises the lower limit relative to todays insolation 
  for a=2:length
     E(a,1)=E((a-1),1)+jump; %creates the column matrix of solar Energy 
  end
end

% Allows the insolation to be decreasing throughout daisyworld 
if length<0
  jump=(lower*S-upper*S)/(boof+1);
  E(1,1)=lower*S; %initialises the lower limit relative to todays insolation 
  for a = 2:boof
     E(a,1) = E((a-1),1) - jump; 
  end
end

if length<0 
   length = abs(length);
end

SB = 5.669*10^-8; %sets the Stefan-Boltzman constant

%initialises the daisy area to 1% to allow growth to start 

Ablack=startarea;
Awhite=startarea;

% Initialises the predator populations 
Rpop = rminpop; %rabbit population 
Fpop = fminpop; %fox population

% Creates an array of plague times 
plinc = 0;
if plagues>=1
  plague=round(length/(plagues+1)); 
  for z=1:plagues
      plaguetime(z,1)=plague+plinc;
      plinc=plaguetime(z,1); 
  end
end 
[f,g]=size(plaguetime);

% Begin iterations for each time step 
for a=1:length
    % Sets variables for rabbits and foxes growths later 
    oldAwhite = Awhite;
    oldAblack = Ablack;
    oldRpop = Rpop;
    oldFpop = Fpop;

    % Calculates area of bare ground and planet albedo 
    Abare = 1-Ablack-Awhite; 
    Ap = Abare*bare + Awhite*white + Ablack*black;
    % Calculate emission and local temperatures 
    Temission = ((E(a,1)/SB)*(1-Ap))^0.25; 
    Trock = ((E(a,1)/SB)*(1-bare))^0.25; 
    Tblack = q*(Ap-black) + Temission; 
    Twhite = q*(Ap-white) + Temission;

    % Calculate daisy growth- define growth and add previous daisy cover 
    growthb = Ablack*(beta(lowgrowthK,highgrowthK,Tblack)*Abare-death); 
    growthw = Awhite*(beta(lowgrowthK,highgrowthK,Twhite)*Abare-death); 
    Ablack  = Ablack + growthb;
    Awhite  = Awhite + growthw;
    
    % Ensure that there is no negative/too much area (numerical procedure) 
    if Ablack <= startarea
      Ablack = startarea; 
    end
    if Ablack > 1 
      Ablack = 1;
    end

    if Awhite<=startarea 
      Awhite=startarea;
    end
    if Awhite>1
      Awhite=1; 
    end
    
    %daisy plague statement and deaths 
    for y=1:f
         if plaguetime(y,1) == a 
            Ablack = Ablack*(1-blacksevere); 
            Awhite = Awhite*(1-whitesevere);
         end 
    end
    
    % Rabbit and fox predation
    % Calculate rabbit growth (other trial growth schemes left) 
    %Rpopgrowth=rfix*Rpop*((1-exp(-reat*grab*(Awhite+Ablack)))*(1-Rpop)-rdeath); 
    %Rpopgrowth=oldRpop*((1-exp(-grab*(oldAwhite+oldAblack)))*reat*(oldAwhite+oldAblack)-rdeath); 
    Rpopgrowth = grab*oldRpop*((reat*(oldAwhite+oldAblack))^2-rdeath); 
    Rpop = Rpop + rfix*Rpopgrowth;
    
    % Take away predated daisies if Rpopgrowth>0
    Ablack=Ablack-Ablack*Rpopgrowth;
    Awhite=Awhite-Awhite*Rpopgrowth; end

    % Calculate fox growth (other trial growth schemes left) 
    %Fpopgrowth = ffix*Fpop*((1-exp(-feat*gfox*Rpop))*(1-Fpop)-fdeath); 
    %Fpopgrowth = oldFpop*((1-exp(-gfox*(oldRpop)))*feat*(oldRpop)-fdeath); 
    Fpopgrowth = gfox*oldFpop*((feat*oldRpop)^2-fdeath); 
    Fpop = Fpop + ffix*Fpopgrowth;

    % Take away predated rabbits 
    if Fpopgrowth>0
       Rpop=Rpop-Rpop*Fpopgrowth; 
    end
    
    % Ensure that there is no negative/too much area (numerical procedure) 
    if Ablack<=startarea
       Ablack=startarea; 
    end
    			     
    if Ablack>1 
       Ablack=1;
    end
    if Awhite<=startarea
       Awhite=startarea; 
    end
    if Awhite>1 
       Awhite=1;
    end
    if Rpop<=rminpop
       Rpop=rminpop; 
    end
    
    if Fpop<=fminpop 
      Fpop=fminpop;
    end
    
    % Store results 
    Temissionresults(1,a) = Temission; 
    Ablackresults(1,a) = Ablack; 
    Awhiteresults(1,a) = Awhite; 
    Trockresults(1,a)  = Trock; 
    Apresults(1,a)     = Ap; 
    Rresults(1,a)      = Rpop; 
    Fresults(1,a)      = Fpop;
end

% Plot the results
figure(1)
subplot(2,1,1)
plot(1:length,Ablackresults,'k-')
hold on;
plot(1:length,Awhiteresults,'y-')
hold on; plot(1:length,Awhiteresults+Ablackresults,'g--') 
xlabel('Time')
ylabel('Daisy Area')
legend('Black daisies','White daisies','Total area',2) 
title('Daisy area versus time')
axis tight;
subplot(2,1,2)
plot(1:length,Rresults,'b-')
hold on;
plot(1:length,Fresults,'r-')
xlabel('Time')
ylabel('Relative populations') 
legend('Rabbits','Foxes',2)
title('Predators versus time')
axis tight;

figure(2)
subplot(2,1,1) 
plot(1:length,Temissionresults,'g-')
hold on;
plot(1:length,Trockresults,'r--')
xlabel('Time')
ylabel('Temperature (K)') 
legend('Daisyworld','Lifeless world',2) 
title('Temperature of Daisyworld versus time') 
axis tight;
hold on;
subplot(2,1,2)
plot(1:length,Apresults,'m-')
axis([1 length black white]);
xlabel('Time')
ylabel('Average albedo')
title('Planetary albedo versus time')
