# //========================================================\\ #
# || Sebastian Gonzalez - 2015                               || #
# || . . . . . . . . . . . . . . . . . . . . . . . . . . . . || #
# || Python module developed to help students in homework    || #
# || 2B: Function Ski.                                       || #
# ||                                                         || #
# || Modulo de python desarrollado como ayuda al estudiante  || #
# || en la tarea 2B: Esqui de Funciones.                     || #
# || . . . . . . . . . . . . . . . . . . . . . . . . . . . . || #
# || Universidad de Chile, FCFM, DCC                         || #
# || CC3501 Modelacion y Computacion Grafica para Ingenieros || #
# || Semestre Otono                                          || #
# ||                                                         || #
# || Inorder parsing of functions.                           || #
# || Precedence climbing parser.                             || #
# || Adapted from Eli Benderski's [1]                        || #
# ||                                                         || #
# \\=========================================================// #
# REFERENCES:
# [1] http://eli.thegreenplace.net/2012/08/02/parsing-expressions-by-precedence-climbing
# [2] http://faculty.ycp.edu/~dhovemey/fall2012/cs340/lecture/lecture06.html
# [3] http://en.wikipedia.org/wiki/Operator-precedence_parser#Precedence_climbing_method
# [4] http://www.engr.mun.ca/~theo/Misc/exp_parsing.htm

import re
import math

# GRAMMAR
# <s-expr>   ::= <add-expr>
# <add-expr> ::= <mul-expr> ( ( '+ | '- ) <mul-expr> ) *
# <mul-expr> ::= <pow-expr> ( ( '* | '/ ) <pow-expr> ) *
# <pow-expr> ::= <primary> ( '^ <primary> ) *
# <primary>  ::= '( <s-expr> ') | NUM | SYM | '- <primary> | SYM <primary>

TOKENPATTERN = re.compile("\s*(?:(\d*\.?\d+|\w+)|(.))")#"\s*(?:(\d*\.?\d+)|(.))"
NUMBERPATTERN = re.compile("\d*\.?\d+")

OPINFO = {
	'+' : (3,'LEFT'),
	'-' : (3,'LEFT'),
	'*' : (5,'LEFT'),
	'/' : (5,'LEFT'),
	'^' : (6,'RIGHT')
}
class Token:
	def __init__(self,name,value):
		self.name = name
		self.value = value

class Parser:
	def __init__(self,st):
		self.tokgen = self.genTokens(st)
		self.curToken = None
		self.enviroment = {}

	def genTokens(self,st):
		for number, operator in TOKENPATTERN.findall(st):
			if number:
				if NUMBERPATTERN.match(number):
					yield Token('NUM', number)
				elif number.isalpha():
					yield Token('SYM', number)
			elif operator == '(':
				yield Token('LEFTPAREN', '(')
			elif operator == ')':
				yield Token('RIGHTPAREN', ')')
			else:
				yield Token('BINOP', operator)

	def getNextToken(self):
		try:
			self.curToken = self.tokgen.next()
		except StopIteration:
			self.curToken = None
		return self.curToken

	def computePrimary(self):
		tok = self.curToken
		if tok.name == 'LEFTPAREN':
			self.getNextToken()
			val = self.computeExpr(1)
			if self.curToken.name != 'RIGHTPAREN':
				raise ValueError("Unbalanced Parenthesis: '('.")
			self.getNextToken()
			return val
		elif tok is None:
			raise ValueError("Source ended unexpectedly.")
		elif tok.name == 'BINOP':
			if tok.value == '-':
				self.getNextToken()
				val = self.computePrimary()
				return lambda env: -val(env)
			else:
				raise ValueError("Expected a primary, not an operator '%s'" % tok.value)
		else:
			self.getNextToken()
			if tok.name == 'NUM':
				return lambda env: float(tok.value)
			elif tok.name == 'SYM':
				if re.match(r'cos|sin|tan|exp|sqrt|ln|abs',tok.value):
					val = self.computePrimary()
					return lambda env: env[tok.value](val(env))
				else:
					return lambda env: env[tok.value] if tok.value in env else (_ for _ in ()).throw(ValueError("Symbol '%s' is not an argument." % tok.value))

	def computeExpr(self,minPrec):
		lhs = self.computePrimary()
		while True:
			cur = self.curToken
			if cur is None or cur.name !='BINOP' or OPINFO[cur.value][0] < minPrec:
				break
			assert cur.name == 'BINOP'
			op = cur.value
			prec, assoc = OPINFO[op]
			nextMinPrec = prec + 1 if assoc == 'LEFT' else prec

			self.getNextToken()
			rhs = self.computeExpr(nextMinPrec)

			lhs = computeOp(op,lhs,rhs)
		return lhs

def computeOp(op, lhs, rhs):
    if op == '+':   return lambda env: lhs(env) + rhs(env)
    elif op == '-': return lambda env: lhs(env) - rhs(env)
    elif op == '*': return lambda env: lhs(env) * rhs(env)
    elif op == '/': return lambda env: lhs(env) / rhs(env)
    elif op == '^': return lambda env: lhs(env) ** rhs(env)
    else: raise ValueError("Unknown operator '%s'." % op)

def parse(st):
	p = Parser(re.sub(r'\s','',st))
	p.getNextToken()
	f = p.computeExpr(1)
	builtIn = {
		'cos'  : lambda x: math.cos(x),
		'sin'  : lambda x: math.sin(x),
		'tan'  : lambda x: math.tan(x),
		'exp'  : lambda x: math.exp(x),
		'sqrt' : lambda x: math.sqrt(x),
		'ln'   : lambda x: math.log(x),
		'abs'  : lambda x: abs(x),
		'pi'   : math.pi,
		'e'    : math.e,
		'rtwo' : math.sqrt(2),
		'phi'  : (1.0 + math.sqrt(5))/2
	}
	return lambda *arg,**keywrd: f(dict(builtIn,**keywrd))