# -*- coding: utf-8; -*-
#
# "THE BEER-WARE LICENSE" (Revision 42):
# <nlehmann@dcc.uchile.cl> wrote this file. As long as you retain this notice you
# can do whatever you want with this stuff. If we meet some day, and you think
# this stuff is worth it, you can buy me a beer in return Nicolás Lehmann

import estructura
from lista import *

# cerveza: nombre (string) IBU (int) SRM (int) OG (num) FG (num)
estructura.crear("cerveza", "nombre IBU SRM OG FG")

# nodo: cerveza (cerveza) izq (nodo) der (nodo)
estructura.crear("nodo", "cerveza izq der")

# identificador para nodos vacíos
nodoVacio = None

# crearNodo: cerveza nodo nodo -> nodo
# Crea un nodo de un árbol binario que contiene una cerveza y tiene
# los correspondientes hijos izquierdo y derecho
def crearNodo(cerveza, izq, der):
    return nodo(cerveza, izq, der)

# vacio: nodo -> bool
# Determina si el parametro corresponde a un nodo vacio.
def vacio(nodo):
    return nodo == nodoVacio

# deltaDensidad: cerveza -> num
# Devuelve la diferencia entre la densidad original y final
# ej:
# c1 = cerveza("Animal IPA", 30, 12, 1.11, 1.01)
# deltaDensidad(c1, 0.10)
def deltaDensidad(cerveza):
    return cerveza.OG - cerveza.FG
assert abs(deltaDensidad(cerveza("Animal IPA", 30, 12, 1.11, 1.01)) - 0.1) <= 0.01

# esABB: nodo -> bool
# Determina si un árbol es un ABB ordenado por la diferencia de densidades
# ej:
# c1 = cerveza("Animal IPA", 30, 12, 1.11, 1.01)
# c2 = cerveza("Stout", 25, 30, 1.25, 1.10)
# c3 = cerveza("Golden Ale", 20, 10, 1.12, 1.01)
# n1 = crearNodo(c1, nodoVacio, nodoVacio)
# n2 = crearNodo(c2, nodoVacio, nodoVacio)
# n3 = crearNodo(c3, n1, n2)
# esABB(n3) retorna True
def esABB(nodo):
    if vacio(nodo):
        return True

    delta = deltaDensidad(nodo.cerveza)
    if not vacio(nodo.izq) and delta < maxValue(nodo.izq):
        return False

    if not vacio(nodo.der) and delta > maxValue(nodo.der):
        return False

    if not esABB(nodo.der) or not esABB(nodo.izq):
        return False

    return True

# maxValue: nodo -> num
# Retorna el valor máximo del árbol
# ej:
# c1 = cerveza("Animal IPA", 30, 12, 1.11, 1.01)
# c2 = cerveza("Stout", 25, 30, 1.25, 1.10)
# c3 = cerveza("Golden Ale", 20, 10, 1.12, 1.01)
# n1 = crearNodo(c1, nodoVacio, nodoVacio)
# n2 = crearNodo(c2, nodoVacio, nodoVacio)
# n3 = crearNodo(c3, n1, n2)
# maxValue(n3) retorna aproximadamente 0.1
def maxValue(nodo):
    assert not vacio(nodo)
    if vacio(nodo.izq) and vacio(nodo.der):
        return deltaDensidad(nodo.cerveza)
    elif vacio(nodo.izq):
        return maxValue(nodo.der)
    elif vacio(nodo.der):
        return maxValue(nodo.izq)
    else:
        return max(maxValue(nodo.izq), maxValue(nodo.der))


# filtrarPorDensidad: nodo num num -> lista(cerveza)
# Devuelve una lista con todas las cervezas que cuya diferencia de densidad
# se encuentra en el rango [a,b]
# ej:
# c1 = cerveza("Animal IPA", 30, 12, 1.11, 1.01)
# c2 = cerveza("Stout", 25, 30, 1.25, 1.10)
# c3 = cerveza("Golden Ale", 20, 10, 1.12, 1.01)
# n1 = crearNodo(c1, nodoVacio, nodoVacio)
# n2 = crearNodo(c2, nodoVacio, nodoVacio)
# n3 = crearNodo(c3, n1, n2)
# filtrarPorDensidad(n3, 5, 12) devuelve
# crearLista(c1, crearLista(c3, listaVacia))
def filtrarPorDensidad(nodo, a, b):
    assert esABB(nodo) and a <= b
    if vacio(nodo):
        return listaVacia
    else:
        delta = deltaDensidad(nodo.cerveza)
        lista = listaVacia
        if delta >= a:
            lista = concatenar(lista, filtrarPorDensidad(nodo.izq, a, b))
        if a <= delta <= b:
            lista = concatenar(lista, crearLista(nodo.cerveza, listaVacia))
        if delta <= b:
            lista = concatenar(lista, filtrarPorDensidad(nodo.der, a, b))
        return lista

c1 = cerveza("Animal IPA", 30, 12, 1.11, 1.01)
c2 = cerveza("Stout", 25, 30, 1.25, 1.10)
c3 = cerveza("Golden Ale", 20, 10, 1.12, 1.01)
c4 = cerveza("Golden Ale", 20, 10, 1.22, 1.01)
n1 = crearNodo(c1, nodoVacio, nodoVacio)
n2 = crearNodo(c2, nodoVacio, nodoVacio)
n3 = crearNodo(c3, n1, n2)
l = crearLista(c1, crearLista(c3, listaVacia))
assert abs(maxValue(n3) - 0.15) <= 0.001
assert esABB(n3)
assert not esABB(crearNodo(c4, n1, n2))
assert filtrarPorDensidad(n3, 0.05, 0.12) == l
