/*
 * Decompiled with CFR 0.152.
 */
package oracle.bpm.jdbc;

import java.sql.SQLException;
import java.util.Map;
import java.util.Properties;
import java.util.TreeMap;
import java.util.Vector;
import oracle.bpm.jdbc.PoolableStatement;
import oracle.bpm.jdbc.StatementCacheSizeExceededException;
import oracle.bpm.lib.msg.LibMsg;
import oracle.bpm.log.Log;
import oracle.bpm.util.ExceptionFormatter;
import oracle.bpm.util.StackTrace;

public class LRUStatementCache {
    private Map<String, CacheEntry> cache;
    private boolean clearing;
    private int maxOpenCursors;
    private int openCursors;
    private Vector<String> order;
    private int quota;
    private long timeStamp;
    public static final String QUOTA = "jdbc.statementcache.quota";
    public static final String MAX_OPEN_CURSORS = "jdbc.pool.maxopencursors";
    private static final String DEFAULT_QUOTA = "1";
    private static final String DEFAULT_MAX_OPEN_CURSORS = "100";

    private LRUStatementCache(int quota, int maxOpenCursors) {
        assert (quota <= maxOpenCursors) : "The quota can not be greater than the max opened cursors";
        this.quota = quota;
        this.maxOpenCursors = maxOpenCursors;
        this.order = new Vector();
        this.cache = new TreeMap<String, CacheEntry>();
        this.timeStamp = 0L;
        this.openCursors = 0;
    }

    public static LRUStatementCache create(Properties properties) {
        int quota = Integer.parseInt(properties.getProperty(QUOTA, DEFAULT_QUOTA));
        int maxOpenCursors = Integer.parseInt(properties.getProperty(MAX_OPEN_CURSORS, DEFAULT_MAX_OPEN_CURSORS));
        return new LRUStatementCache(quota, maxOpenCursors);
    }

    public boolean isFull() {
        return this.openCursors >= this.maxOpenCursors;
    }

    public void setTimeStamp(long timeStamp) {
        this.timeStamp = timeStamp;
        this.releaseExceeded();
    }

    public void clear() {
        this.clear(true);
    }

    public void clear(boolean close) {
        this.clearing = true;
        for (CacheEntry cacheEntry : this.cache.values()) {
            --this.openCursors;
            if (cacheEntry.statement == null || !close) continue;
            try {
                cacheEntry.statement.close(this);
            }
            catch (SQLException sqle) {
                Log.logWarning(LibMsg.CLOSE_STATEMENT_ERROR(ExceptionFormatter.fullTechLevel(sqle)));
            }
        }
        this.order.clear();
        this.cache.clear();
        this.clearing = false;
    }

    public PoolableStatement get(String sql) throws SQLException {
        CacheEntry entry = this.cache.get(sql);
        if (entry == null) {
            return null;
        }
        if (entry.statement == null) {
            this.order.remove(sql);
            return null;
        }
        if (entry.locked) {
            return null;
        }
        this.order.remove(sql);
        this.order.add(0, sql);
        ++entry.used;
        entry.timeStamp = System.currentTimeMillis();
        entry.locked = true;
        String stackTrace = "";
        assert (LRUStatementCache.TRUE(stackTrace = StackTrace.getStackTrace()));
        entry.stackTrace = stackTrace;
        entry.restoreValues();
        return entry.statement;
    }

    public void put(String sql, PoolableStatement statement) throws StatementCacheSizeExceededException {
        if (this.order.size() > this.quota - 1) {
            CacheEntry entry;
            String lastkey = this.order.lastElement();
            CacheEntry cacheEntry = entry = lastkey == null ? null : this.cache.get(lastkey);
            if (entry == null) {
                this.order.remove(lastkey);
            } else if (entry.timeStamp < this.timeStamp && !entry.locked) {
                this.order.remove(lastkey);
                this.cache.remove(lastkey);
                --this.openCursors;
                if (entry.statement != null) {
                    try {
                        entry.statement.close(this);
                    }
                    catch (SQLException sqle) {
                        Log.logWarning(LibMsg.CLOSE_STATEMENT_ERROR(ExceptionFormatter.fullTechLevel(sqle)));
                    }
                }
            }
        }
        if (this.order.size() < this.maxOpenCursors) {
            CacheEntry oldEntry = this.cache.get(sql);
            if (oldEntry == null || !oldEntry.locked) {
                CacheEntry newEntry = new CacheEntry(statement, System.currentTimeMillis());
                newEntry.locked = true;
                String stackTrace = "";
                assert (LRUStatementCache.TRUE(stackTrace = StackTrace.getStackTrace()));
                newEntry.stackTrace = stackTrace;
                this.cache.put(sql, newEntry);
                this.order.add(0, sql);
                ++this.openCursors;
            } else {
                statement.setNotCacheable(true);
            }
        } else {
            StringBuffer sb = new StringBuffer();
            for (Map.Entry<String, CacheEntry> entry : this.cache.entrySet()) {
                CacheEntry cacheEntry = entry.getValue();
                if (!cacheEntry.locked) continue;
                sb.append("[locker=\n");
                sb.append(cacheEntry.stackTrace);
                sb.append("\n]");
            }
            throw StatementCacheSizeExceededException.create(sb.toString());
        }
    }

    public void releaseExceeded() {
        int qty = this.order.size();
        if (this.quota <= qty) {
            for (int i = qty - 1; i > this.quota - 2; --i) {
                String key = this.order.get(i);
                if (key == null) continue;
                CacheEntry entry = this.cache.get(key);
                if (entry == null) {
                    this.order.remove(i);
                    continue;
                }
                if (entry.timeStamp >= this.timeStamp || entry.locked) continue;
                this.order.remove(i);
                this.cache.remove(key);
                --this.openCursors;
                if (entry.statement == null) continue;
                try {
                    entry.statement.close(this);
                    continue;
                }
                catch (SQLException sqle) {
                    Log.logWarning(LibMsg.CLOSE_STATEMENT_ERROR(ExceptionFormatter.fullTechLevel(sqle)));
                }
            }
        }
    }

    public void remove(String sql) {
        if (!this.clearing && this.order.contains(sql)) {
            this.order.remove(sql);
            CacheEntry entry = this.cache.remove(sql);
            --this.openCursors;
            if (entry != null && entry.statement != null) {
                try {
                    entry.statement.close(this);
                }
                catch (SQLException sqle) {
                    Log.logWarning(LibMsg.CLOSE_STATEMENT_ERROR(ExceptionFormatter.fullTechLevel(sqle)));
                }
            }
        }
    }

    public void resize(int quota) {
        this.quota = quota;
        this.releaseExceeded();
    }

    public void unlock(String sql) {
        CacheEntry cacheEntry = this.cache.get(sql);
        if (cacheEntry != null) {
            cacheEntry.locked = false;
        }
    }

    public void unlockAll() {
        for (CacheEntry cacheEntry : this.cache.values()) {
            if (cacheEntry == null) continue;
            cacheEntry.locked = false;
        }
    }

    private static boolean TRUE(String dummy) {
        return true;
    }

    static class CacheEntry {
        public volatile boolean locked;
        public String stackTrace;
        public PoolableStatement statement;
        public long timeStamp;
        public int used;
        private int defaultFetchSize;
        private int defaultMaxRows;

        public CacheEntry(PoolableStatement statement, long timeStamp) {
            int fetchSize;
            int maxRows;
            this.statement = statement;
            this.timeStamp = timeStamp;
            this.used = 1;
            this.locked = false;
            try {
                maxRows = statement.getMaxRows();
            }
            catch (SQLException e) {
                maxRows = 0;
            }
            this.defaultMaxRows = maxRows;
            try {
                fetchSize = statement.getFetchSize();
            }
            catch (SQLException e) {
                fetchSize = 0;
            }
            this.defaultFetchSize = fetchSize;
        }

        public String toString() {
            return "locked=" + this.locked;
        }

        public void restoreValues() throws SQLException {
            this.statement.setMaxRows(this.defaultMaxRows);
            this.statement.setFetchSize(this.defaultFetchSize);
        }
    }
}

