/* $Id: GetterSetterManagerImpl.java 18339 2010-04-27 17:56:41Z linus $
 *******************************************************************************
 * Copyright (c) 2009 Contributors - see below
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Bob Tarling - Original implementation
 *******************************************************************************
 */

package org.argouml.core.propertypanels.model;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;
import java.util.StringTokenizer;

import javax.swing.Icon;
import javax.swing.JOptionPane;

import org.apache.log4j.Logger;
import org.argouml.application.helpers.ResourceLoaderWrapper;
import org.argouml.core.propertypanels.meta.IconIdentifiable;
import org.argouml.core.propertypanels.meta.Named;
import org.argouml.i18n.Translator;
import org.argouml.kernel.Command;
import org.argouml.kernel.NonUndoableCommand;
import org.argouml.kernel.Project;
import org.argouml.kernel.ProjectManager;
import org.argouml.model.Model;
import org.argouml.profile.Profile;
import org.argouml.profile.ProfileException;
import org.argouml.ui.targetmanager.TargetManager;
import org.argouml.uml.ui.UMLAddDialog;
import org.argouml.util.ArgoFrame;

/**
 * Property getters and setters for UML1.4
 * @author Bob Tarling
 */
class GetterSetterManagerImpl extends GetterSetterManager {
    
    private static final Logger LOG =
        Logger.getLogger(GetterSetterManagerImpl.class);
    
    /**
     * The constructor
     */
    public GetterSetterManagerImpl(String type) {
        build(type);
    }
    
    /**
     * Create all the getter/setters for this implementation
     */
    private void build(String type) {
        addGetterSetter("isAbstract", new AbstractGetterSetter());
        addGetterSetter("isLeaf", new LeafGetterSetter());
        addGetterSetter("isRoot", new RootGetterSetter());
        addGetterSetter("isActive", new ActiveGetterSetter());
        addGetterSetter("ownerScope", new OwnerScopeGetterSetter());
        addGetterSetter("targetScope", new TargetScopeGetterSetter());
        addGetterSetter("isQuery", new QueryGetterSetter());
        addGetterSetter("isNavigable", new NavigableGetterSetter());
        addGetterSetter("isSynchronous", new AsynchronousGetterSetter());
        addGetterSetter("isSynch", new SynchGetterSetter());
        addGetterSetter("ordering", new OrderingGetterSetter());
        addGetterSetter("navigable", new NavigableGetterSetter());
        addGetterSetter("derived", new DerivedGetterSetter());
        addGetterSetter("visibility", new VisibilityGetterSetter());
        addGetterSetter("aggregation", new AggregationGetterSetter());
        addGetterSetter("kind", new ParameterDirectionGetterSetter());
        addGetterSetter("changeability", new ChangeabilityGetterSetter());
        addGetterSetter("concurrency", new ConcurrencyGetterSetter());
        addGetterSetter("feature", new FeatureGetterSetter());
        addGetterSetter("parameter", new ParameterGetterSetter());
        addGetterSetter("receiver", new ReceiverGetterSetter());
        addGetterSetter("sender", new SenderGetterSetter());
        addGetterSetter("literal", new LiteralGetterSetter());
        addGetterSetter("body", new MethodExpressionGetterSetter());
        addGetterSetter("ownedElement", new OwnedElementGetterSetter());
        addGetterSetter("raisedSignal", new RaisedExceptionGetterSetter());
        addGetterSetter("raisedException", new RaisedExceptionGetterSetter());
        addGetterSetter("method", new MethodGetterSetter());
        addGetterSetter("message", new MessageGetterSetter());
        addGetterSetter("actualArgument", new ArgumentGetterSetter());
        addGetterSetter("extensionPoint", new ExtensionPointGetterSetter());
        addGetterSetter("guard", new GuardGetterSetter());
        addGetterSetter("effect", new EffectGetterSetter());
        addGetterSetter("trigger", new TriggerGetterSetter());
        addGetterSetter("elementImport", new ElementImportGetterSetter());
        addGetterSetter("templateParameter", new TemplateParameterGetterSetter());
        addGetterSetter("reception", new ReceptionGetterSetter());
        addGetterSetter("deferrableEvent", new DeferrableEventGetterSetter());
        addGetterSetter("entry", new EntryActionGetterSetter());
        addGetterSetter("action", new ActionGetterSetter());
        addGetterSetter("subvertex", new SubvertexGetterSetter());
        addGetterSetter("internalTransition", new InternalTransitionGetterSetter());
        addGetterSetter("classifier", new ClassifierGetterSetter());
        addGetterSetter("baseClass", new BaseClassGetterSetter());
        
        // UML2 only
        addGetterSetter("ownedOperation", new FeatureGetterSetter());
    }
    
    /**
     * Helper method for adding a new getter/setter
     * @param propertyName
     * @param bgs
     */
    private void addGetterSetter(String propertyName, BaseGetterSetter bgs) {
        getterSetterByPropertyName.put(propertyName, bgs);
    }
    
    /**
     * Set a UML property by property name
     * @param handle the element to which a property must be set
     * @param value the new property value
     * @param propertyName the property name
     */
    public void set(Object handle, Object value, String propertyName) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs != null) {
            bgs.set(handle, value);
        }
    }
    
    /**
     * Get a UML property by property name
     * @param handle the element from which a property must be return
     * @param value the new property value
     * @param propertyName the property name
     */
    public Object get(Object handle, String propertyName, String type) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs != null) {
            return bgs.get(handle, type);
        }
        
        return null;
    }
    
    public Collection getOptions(
            final Object umlElement,
            final String propertyName,
            final String type) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs instanceof OptionGetterSetter) {
            return ((OptionGetterSetter) bgs).getOptions(umlElement, type);
        }
        
        return null;
    }
    
    public boolean isFullBuildOnly(
            final String propertyName) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs instanceof ListGetterSetter) {
            return ((ListGetterSetter) bgs).isFullBuildOnly();
        }
        
        return false;
    }
    
    
    public Object create(String propertyName, String language, String body) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs instanceof ExpressionGetterSetter) {
            return ((ExpressionGetterSetter) bgs).create(language, body);
        }
        
        return null;
    }
    
    public boolean isValidElement(
            final String propertyName,
            final String type,
            final Object element) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs instanceof ListGetterSetter) {
            return ((ListGetterSetter) bgs).isValidElement(element, type);
        }
        
        return false;
    }
    
    public Object getMetaType(String propertyName) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs instanceof ListGetterSetter) {
            return ((ListGetterSetter) bgs).getMetaType();
        }
        
        return null;
    }
    

	@Override
	public Command getAddCommand(String propertyName, Object umlElement) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs instanceof Addable) {
            return ((Addable) bgs).getAddCommand(umlElement);
        }
		return null;
	}

	@Override
	public List<Command> getAdditionalCommands(String propertyName, Object umlElement) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs instanceof ListGetterSetter) {
            return ((ListGetterSetter) bgs).getAdditionalCommands(umlElement);
        }
		return null;
	}

	@Override
	public Command getRemoveCommand(String propertyName, Object umlElement, Object objectToRemove) {
        BaseGetterSetter bgs = getterSetterByPropertyName.get(propertyName);
        if (bgs instanceof Removeable) {
            return ((Removeable) bgs).getRemoveCommand(umlElement, objectToRemove);
        }
		return null;
	}
	
	private interface Addable {
		Command getAddCommand(Object umlElement);
	}
    
	private interface Removeable {
		Command getRemoveCommand(Object umlElement, Object objectToRemove);
	}
    
    
    /**
     * The getter/setter for the Absrtact property
     * @author Bob Tarling
     */
    private class AbstractGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isAbstract(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setAbstract(modelElement, (Boolean) value);
        }
    }
    
    private class LeafGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isLeaf(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setLeaf(modelElement, (Boolean) value);
        }
    }
    
    private class RootGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isRoot(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setRoot(modelElement, (Boolean) value);
        }
    }
    
    private class ActiveGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isActive(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setActive(modelElement, (Boolean) value);
        }
    }
    
    private class OwnerScopeGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isStatic(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setStatic(modelElement, (Boolean) value);
        }
    }
    
    private class TargetScopeGetterSetter extends BaseGetterSetter {
        // Have we handled UML2 here?
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isStatic(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setStatic(modelElement, (Boolean) value);
        }
    }
    
    private class QueryGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isQuery(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setQuery(modelElement, (Boolean) value);
        }
    }
    
    private class NavigableGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isNavigable(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setNavigable(modelElement, (Boolean) value);
        }
    }
    
    private class AsynchronousGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isAsynchronous(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getCommonBehaviorHelper().setAsynchronous(modelElement, (Boolean) value);
        }
    }
    
    private class SynchGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().isSynch(modelElement);
        }
        public void set(Object modelElement, Object value) {
            Model.getActivityGraphsHelper().setSynch(modelElement, (Boolean) value);
        }
    }
    
    private class OrderingGetterSetter extends BaseGetterSetter {
        public Object get(Object modelElement, String type) {
            return Model.getFacade().getOrdering(modelElement) ==
                Model.getOrderingKind().getOrdered();
        }
        public void set(Object modelElement, Object value) {
            if ((Boolean) value) {
                Model.getCoreHelper().setOrdering(modelElement,
                        Model.getOrderingKind().getOrdered());
            } else {
                Model.getCoreHelper().setOrdering(modelElement,
                        Model.getOrderingKind().getUnordered());
            }
        }
    }
    
    private class DerivedGetterSetter extends BaseGetterSetter {
        
        /**
         * Derived is not a true UML property but is in fact a pseudo property
         * stored in a tag named "derived"
         */
        private static final String TagName = "derived";
        
        public Object get(Object modelElement, String type) {
            Object tv = Model.getFacade().getTaggedValue(modelElement, TagName);
            if (tv != null) {
                String tag = Model.getFacade().getValueOfTag(tv);
                return ("true".equals(tag));
            }
            return false;
        }
        public void set(Object modelElement, Object value) {
            Object taggedValue = Model.getFacade().getTaggedValue(modelElement, TagName);
            if (taggedValue == null) {
                taggedValue =
                        Model.getExtensionMechanismsFactory().buildTaggedValue(
                                TagName, "");
                Model.getExtensionMechanismsHelper().addTaggedValue(
                        modelElement, taggedValue);
            }
            if ((Boolean) value) {
                Model.getCommonBehaviorHelper().setValue(taggedValue, "true");
            } else {
                Model.getCommonBehaviorHelper().setValue(taggedValue, "false");
            }
        }
    }
    
    
    public class VisibilityGetterSetter extends OptionGetterSetter {
        
        /**
         * Identifier for public visibility.
         */
        public static final String PUBLIC = "public";

        /**
         * Identifier for protected visibility.
         */
        public static final String PROTECTED = "protected";

        /**
         * Identifier for private visibility.
         */
        public static final String PRIVATE = "private";

        /**
         * Identifier for package visibility.
         */
        public static final String PACKAGE = "package";
        
        public VisibilityGetterSetter() {
            setOptions(Arrays.asList((new String[] {PUBLIC, PACKAGE, PROTECTED, PRIVATE})));
        }
        
        public Object get(Object modelElement, String type) {
            Object kind = Model.getFacade().getVisibility(modelElement);
            if (kind == null) {
                return null;
            } else if (kind.equals(Model.getVisibilityKind().getPublic())) {
                return PUBLIC;
            } else if (kind.equals(Model.getVisibilityKind().getPackage())) {
                return PACKAGE;
            } else if (kind.equals(Model.getVisibilityKind().getProtected())) {
                return PROTECTED;
            } else if (kind.equals(Model.getVisibilityKind().getPrivate())) {
                return PRIVATE;
            } else {
                return PUBLIC;
            }
        }
        
        public void set(Object modelElement, Object value) {
            Object kind = null;
            if (value.equals(PUBLIC)) {
                kind = Model.getVisibilityKind().getPublic();
            } else if (value.equals(PROTECTED)) {
                kind = Model.getVisibilityKind().getProtected();
            } else if (value.equals(PACKAGE)) {
                kind = Model.getVisibilityKind().getPackage();
            } else {
                kind = Model.getVisibilityKind().getPrivate();
            }
            Model.getCoreHelper().setVisibility(modelElement, kind);
        }
    }
    
    
    private class AggregationGetterSetter extends OptionGetterSetter {
        
        /**
         * Identifier for aggregate aggregation kind.
         */
        public static final String AGGREGATE = "aggregate";

        /**
         * Identifier for composite aggregation kind.
         */
        public static final String COMPOSITE = "composite";

        /**
         * Identifier for no aggregation kind.
         */
        public static final String NONE = "none";
        
        public AggregationGetterSetter() {
            setOptions(Arrays.asList(new String[] {AGGREGATE, COMPOSITE, NONE}));
        }
        
        public Object get(Object modelElement, String type) {
            Object kind = Model.getFacade().getAggregation(modelElement);
            if (kind == null) {
                return null;
            } else if (kind.equals(Model.getAggregationKind().getNone())) {
                return NONE;
            } else if (kind.equals(Model.getAggregationKind().getAggregate())) {
                return AGGREGATE;
            } else if (kind.equals(Model.getAggregationKind().getComposite())) {
                return COMPOSITE;
            } else {
                return NONE;
            }
        }
        
        public void set(Object modelElement, Object value) {
            Object kind = null;
            
            if (value.equals(AGGREGATE)) {
                kind = Model.getAggregationKind().getAggregate();
            } else if (value.equals(COMPOSITE)) {
                kind = Model.getAggregationKind().getComposite();
            } else {
                kind = Model.getAggregationKind().getNone();
            }
            Model.getCoreHelper().setAggregation(modelElement, kind);
            
        }
    }
    
    private class ParameterDirectionGetterSetter extends OptionGetterSetter {
        
        /**
         * Identifier for an "in" parameter.
         */
        public static final String IN = "in";

        /**
         * Identifier for an "out" parameter.
         */
        public static final String OUT = "out";

        /**
         * Identifier for an "in/out" parameter.
         */
        public static final String INOUT = "inout";

        /**
         * Identifier for a "return" parameter.
         */
        public static final String RETURN = "return";
        
        public ParameterDirectionGetterSetter() {
            setOptions(Arrays.asList(new String[] {
                    IN,
                    OUT,
                    INOUT,
                    RETURN}));
        }
        
        public Object get(Object modelElement, String type) {
            Object kind = Model.getFacade().getKind(modelElement);
            if (kind == null) {
                return null;
            } else if (kind.equals(Model.getDirectionKind().getInParameter())) {
                return IN;
            } else if (kind.equals(Model.getDirectionKind().getInOutParameter())) {
                return INOUT;
            } else if (kind.equals(Model.getDirectionKind().getOutParameter())) {
                return OUT;
            } else {
                return RETURN;
            }
        }
        
        public void set(Object modelElement, Object value) {
            Object kind = null;
            if (value == null) {
                kind = null;
            } else if (value.equals(IN)) {
                kind = Model.getDirectionKind().getInParameter();
            } else if (value.equals(OUT)) {
                kind = Model.getDirectionKind().getOutParameter();
            } else if (value.equals(INOUT)) {
                kind = Model.getDirectionKind().getInOutParameter();
            } else if (value.equals(RETURN)) {
                kind = Model.getDirectionKind().getReturnParameter();
            }
            Model.getCoreHelper().setKind(modelElement, kind);
            
        }
    }
    
    
    private class ConcurrencyGetterSetter extends OptionGetterSetter {
        
        /**
         * Identifier for sequential concurrency.
         */
        public static final String SEQUENTIAL= "sequential";

        /**
         * Identifier for guarded concurrency.
         */
        public static final String GUARDED = "guarded";

        /**
         * Identifier for concurrent concurrency.
         */
        public static final String CONCURRENT = "concurrent";

        public ConcurrencyGetterSetter() {
            setOptions(Arrays.asList(new String[] {
                    SEQUENTIAL,
                    GUARDED,
                    CONCURRENT}));
        }
        
        public Object get(Object modelElement, String type) {
            Object kind = Model.getFacade().getConcurrency(modelElement);
            if (kind == null) {
                return null;
            } else if (kind.equals(Model.getConcurrencyKind().getSequential())) {
                return SEQUENTIAL;
            } else if (kind.equals(Model.getConcurrencyKind().getGuarded())) {
                return GUARDED;
            } else if (kind.equals(Model.getConcurrencyKind().getConcurrent())) {
                return CONCURRENT;
            } else {
                return SEQUENTIAL;
            }
        }
        
        public void set(Object modelElement, Object value) {
            Object kind = null;
            if (value.equals(SEQUENTIAL)) {
                kind = Model.getConcurrencyKind().getSequential();
            } else if (value.equals(GUARDED)) {
                kind = Model.getConcurrencyKind().getGuarded();
            } else {
                kind = Model.getConcurrencyKind().getConcurrent();
            }
            Model.getCoreHelper().setConcurrency(modelElement, kind);
        }
    }
    
    
    private class ChangeabilityGetterSetter extends OptionGetterSetter {
        
        /**
         * Identifier for addonly changeability.
         * TODO: Note this should not be ni UML2 version
         */
        public static final String ADDONLY = "addonly";

        /**
         * CHANGEABLE_COMMAND determines a changeability kind.
         */
        public static final String CHANGEABLE = "changeable";

        /**
         * FROZEN_COMMAND determines a changeability kind.
         */
        public static final String FROZEN = "frozen";

        public ChangeabilityGetterSetter() {
            setOptions(Arrays.asList(new String[] {ADDONLY, CHANGEABLE, FROZEN}));
        }
        
        public Object get(Object modelElement, String type) {
            Object kind = Model.getFacade().getChangeability(modelElement);
            if (kind == null) {
                return null;
            } else if (kind.equals(Model.getChangeableKind().getAddOnly())) {
                return ADDONLY;
            } else if (kind.equals(Model.getChangeableKind().getChangeable())) {
                return CHANGEABLE;
            } else if (kind.equals(Model.getChangeableKind().getFrozen())) {
                return FROZEN;
            } else {
                return CHANGEABLE;
            }
        }
        
        public void set(Object modelElement, Object value) {
            if (value.equals(CHANGEABLE)) {
                Model.getCoreHelper().setReadOnly(modelElement, false);
            } else if (value.equals(ADDONLY)) {
                Model.getCoreHelper().setChangeability(
                        modelElement, Model.getChangeableKind().getAddOnly());
            } else {
                Model.getCoreHelper().setReadOnly(modelElement, true);
            }
        }
    }

    private class FeatureGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the features for the model
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type /* TODO: change this to a metatype */ ) {
            
            StringTokenizer st = new StringTokenizer(type, ",");
            try {
                Class metaType = Class.forName(st.nextToken());
                if (Model.getMetaTypes().getAttribute().equals(metaType)) {
                    return Model.getFacade().getAttributes(modelElement);
                } else if (Model.getMetaTypes().getOperation().equals(metaType)) {
                    return Model.getFacade().getOperationsAndReceptions(modelElement);
                } else {
                    return Collections.EMPTY_LIST;
                }
            } catch (ClassNotFoundException e) {
                LOG.error("Exception", e);
                return Collections.EMPTY_LIST;
            }
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getOperation();
        }
    }
    
    

    private class OwnedElementGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the owned elements for the namespace
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            
            return Model.getFacade().getOwnedElements(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getOperation();
        }
    }
    

    private class RaisedExceptionGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the owned elements for the namespace
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            return Model.getFacade().getRaisedExceptions(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getSignal();
        }
    }
    
    private class MethodGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the method for the operation
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            return Model.getFacade().getMethods(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getMethod();
        }
    }
    
    
    private class MessageGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the method for the operation
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            return Model.getFacade().getMessages(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getMessage();
        }
    }
    
    
    private class ArgumentGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the method for the operation
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            return Model.getFacade().getArguments(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getArgument();
        }
    }
    
    
    private class ExtensionPointGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the extension points
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            return Model.getFacade().getExtensionPoints(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getExtensionPoint();
        }
    }
    
    private class GuardGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the guards
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            final ArrayList l = new ArrayList(1);
            l.add(Model.getFacade().getGuard(modelElement));
            return l;
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getGuard();
        }
    }
    
    private class EffectGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the effects
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            final ArrayList l = new ArrayList(1);
            l.add(Model.getFacade().getEffect(modelElement));
            return l;
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getAction();
        }
    }
    
    private class TriggerGetterSetter extends ListGetterSetter {
        
        /**
         * Get all the effects
         * @param modelElement
         * @param type
         * @return
         * @see org.argouml.core.propertypanels.model.GetterSetterManager.OptionGetterSetter#getOptions(java.lang.Object, java.lang.String)
         */
        public Collection getOptions(
                final Object modelElement,
                final String type) {
            final ArrayList l = new ArrayList(1);
            l.add(Model.getFacade().getTrigger(modelElement));
            return l;
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(
                final Object element,
                final String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getEvent();
        }
    }
    
    private class ParameterGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getParameters(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getParameter();
        }
    }
    
    private class EntryActionGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
        	final ArrayList list = new ArrayList(1);
        	list.add(Model.getFacade().getEntry(modelElement));
            return list;
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getAction();
        }
    }
    
    private class ActionGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getActions(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getAction();
        }
    }

    private class SubvertexGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getSubvertices(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getState();
        }
        
        /**
         * Returns additional commands that cannot be deduced from the panel
         * xml or other means. This is currently only used by
         * SubvertexGetterSetter and should be removed as soon as we have some
         * configurable way to replace.
         */
        public List<Command> getAdditionalCommands(Object modelElement) {
        	final List<Command> commands = new ArrayList<Command>(6);
        	commands.add(new NewPseudoStateCommand(
        			modelElement, Model.getPseudostateKind().getFork()));
        	commands.add(new NewPseudoStateCommand(
        			modelElement, Model.getPseudostateKind().getJoin()));
        	commands.add(new NewPseudoStateCommand(
        			modelElement, Model.getPseudostateKind().getChoice()));
        	commands.add(new NewPseudoStateCommand(
        			modelElement, Model.getPseudostateKind().getDeepHistory()));
        	commands.add(new NewPseudoStateCommand(
        			modelElement, Model.getPseudostateKind().getShallowHistory()));
        	commands.add(new NewPseudoStateCommand(
        			modelElement, Model.getPseudostateKind().getInitial()));
        	commands.add(new NewPseudoStateCommand(
        			modelElement, Model.getPseudostateKind().getJunction()));
        	return commands;
        }
        
        private class NewPseudoStateCommand extends NonUndoableCommand implements IconIdentifiable, Named {

        	private final String label;
        	private final Object kind;
        	private final Icon icon;
        	private final Object target;
        	
        	/**
        	 * A Command to create a new pseudostate inside some target model
        	 * element.
        	 * @param target the target model element the pseudo state should
        	 * belong to
        	 * @param kind the required kind of pseudostate
        	 */
        	NewPseudoStateCommand(Object target, Object kind) {
        		this.target = target;
        		this.kind = kind;
                if (kind == Model.getPseudostateKind().getFork()) {
                    label = Translator.localize("label.pseudostate.fork");
                } else if (kind == Model.getPseudostateKind().getJoin()) {
                    label = Translator.localize("label.pseudostate.join");
                } else if (kind == Model.getPseudostateKind().getChoice()) {
                	label = Translator.localize("label.pseudostate.choice");
                } else if (kind == Model.getPseudostateKind().getDeepHistory()) {
                	label = Translator.localize("label.pseudostate.deephistory");
                } else if (kind == Model.getPseudostateKind().getShallowHistory()) {
                	label = Translator.localize("label.pseudostate.shallowhistory");
                } else if (kind == Model.getPseudostateKind().getInitial()) {
                	label = Translator.localize("label.pseudostate.initial");
                } else if (kind == Model.getPseudostateKind().getJunction()) {
                	label = Translator.localize("label.pseudostate.junction");
                } else {
                	throw new IllegalArgumentException(
                			kind + " is not a known PseudostateKind");
                }
                icon = ResourceLoaderWrapper.lookupIcon(
                		Model.getFacade().getName(kind));
            }
        	
			@Override
			public Object execute() {
                final Object ps =
                    Model.getStateMachinesFactory().buildPseudoState(target);
                if (kind != null) {
                    Model.getCoreHelper().setKind(ps, kind);
                }                	
                TargetManager.getInstance().setTarget(ps);
				return null;
			}
        	
			public Icon getIcon() {
				return icon;
			}
        	
			public String getName() {
				return label;
			}
        }
    }
    
    private class TemplateParameterGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getTemplateParameters(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getTemplateParameter();
        }
    }
    
    private class ElementImportGetterSetter extends ListGetterSetter implements Addable, Removeable {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getImportedElements(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
        
        public boolean isFullBuildOnly() {
        	return true;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getModelElement();
        }
        
        public Command getAddCommand(Object modelElement) {
        	return new AddElementImportCommand(modelElement);
        }
        
        public Command getRemoveCommand(Object modelElement, Object objectToRemove) {
        	return new RemoveElementImportCommand(modelElement, objectToRemove);
        }
        
        private class AddElementImportCommand extends AddModelElementCommand {

            /**
             * Constructor for ActionAddPackageImport.
             */
            public AddElementImportCommand(Object target) {
                super(target);
            }


            protected List getChoices() {
                List list = new ArrayList();
                /* TODO: correctly implement next function 
                 * in the model subsystem for 
                 * issue 1942: */
                list.addAll(Model.getModelManagementHelper()
                        .getAllPossibleImports(getTarget()));
                return list;
            }


            protected List getSelected() {
                List list = new ArrayList();
                list.addAll(Model.getFacade().getImportedElements(getTarget()));
                return list;
            }


            protected String getDialogTitle() {
                return Translator.localize("dialog.title.add-imported-elements");
            }


            @Override
            protected void doIt(Collection selected) {
            	if (LOG.isInfoEnabled()) {
                	LOG.info("Setting " + selected.size() + "imported elements");
            	}
                Model.getModelManagementHelper().setImportedElements(getTarget(), selected);
            }
        }
        
        private class RemoveElementImportCommand
    	    extends NonUndoableCommand {
        	
        	private final Object target;
        	private final Object objectToRemove;
        	
    	    /**
    	     * Constructor for ActionRemovePackageImport.
    	     */
    	    public RemoveElementImportCommand(final Object target, final Object objectToRemove) {
    	        this.target = target;
    	        this.objectToRemove = objectToRemove;
    	    }
    	    
    	    /*
    	     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
    	     */
    	    public Object execute() {
    	        Model.getModelManagementHelper()
    	            .removeImportedElement(target, objectToRemove);
    	        return null;
    	    }
    	}
    }
    
    private class DeferrableEventGetterSetter extends ListGetterSetter implements Addable, Removeable {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getDeferrableEvents(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
        
        public boolean isFullBuildOnly() {
        	return false;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getEvent();
        }
        
        public Command getAddCommand(Object modelElement) {
        	return new AddCommand(modelElement);
        }
        
        public Command getRemoveCommand(Object modelElement, Object objectToRemove) {
        	return new RemoveCommand(modelElement, objectToRemove);
        }
        
        private class AddCommand extends AddModelElementCommand {

            /**
             * Constructor for ActionAddPackageImport.
             */
            public AddCommand(Object target) {
                super(target);
            }


            protected List getChoices() {
                List list = new ArrayList();
                list.addAll(Model.getModelManagementHelper().getAllModelElementsOfKind(
                        Model.getFacade().getRoot(getTarget()),
                        Model.getMetaTypes().getEvent()));
                return list;
            }


            protected List getSelected() {
                List list = new ArrayList();
                list.addAll(Model.getFacade().getDeferrableEvents(getTarget()));
                return list;
            }


            protected String getDialogTitle() {
                return Translator.localize("dialog.title.add-events");
            }


            @Override
            protected void doIt(Collection selected) {
                Object state = getTarget();
                assert (Model.getFacade().isAState(state));
                
                Collection oldOnes = new ArrayList(Model.getFacade()
                        .getDeferrableEvents(state));
                Collection toBeRemoved = new ArrayList(oldOnes);
                for (Object o : selected) {
                    if (oldOnes.contains(o)) {
                        toBeRemoved.remove(o);
                    } else {
                        Model.getStateMachinesHelper().addDeferrableEvent(state, o);
                    }
                }
                for (Object o : toBeRemoved) {
                    Model.getStateMachinesHelper().removeDeferrableEvent(state, o);
                }
            }
        }
        
        private class RemoveCommand extends NonUndoableCommand {
        	
        	private final Object target;
        	private final Object objectToRemove;
        	
    	    public RemoveCommand(final Object target, final Object objectToRemove) {
    	        this.target = target;
    	        this.objectToRemove = objectToRemove;
    	    }
    	    
    	    /*
    	     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
    	     */
    	    public Object execute() {
    	        Model.getStateMachinesHelper()
    	            .removeDeferrableEvent(target, objectToRemove);
    	        return null;
    	    }
    	}
    }
    
    
    
    private class ReceptionGetterSetter extends ListGetterSetter implements Addable, Removeable {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getReceptions(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
        
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getReception();
        }
        
        public Command getAddCommand(Object modelElement) {
        	return new AddCommand(modelElement);
        }
        
        public Command getRemoveCommand(Object modelElement, Object objectToRemove) {
        	return new RemoveCommand(modelElement, objectToRemove);
        }
        
        private class AddCommand extends AddModelElementCommand {

            /**
             * Constructor for ActionAddPackageImport.
             */
            public AddCommand(final Object target) {
                super(target);
            }


            protected List getChoices() {
                List list = new ArrayList();
                /* TODO: correctly implement next function 
                 * in the model subsystem for 
                 * issue 1942: */
                Object model =
                    ProjectManager.getManager().getCurrentProject().getModel();
                list.addAll(Model.getModelManagementHelper()
                        .getAllModelElementsOfKind(model, 
                            Model.getMetaTypes().getReception()));
                return list;
            }


            protected List getSelected() {
                List list = new ArrayList();
                list.addAll(Model.getFacade().getReceptions(getTarget()));
                return list;
            }


            protected String getDialogTitle() {
                return Translator.localize("dialog.title.add-imported-elements");
            }


            @Override
            protected void doIt(Collection selected) {
                Model.getCommonBehaviorHelper().setReception(getTarget(), selected);
            }
        }
        
        private class RemoveCommand
    	    extends NonUndoableCommand {
        	
        	private final Object target;
        	private final Object objectToRemove;
        	
    	    /**
    	     * Constructor for ActionRemovePackageImport.
    	     */
    	    public RemoveCommand(final Object target, final Object objectToRemove) {
    	        this.target = target;
    	        this.objectToRemove = objectToRemove;
    	    }
    	    
    	    /*
    	     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
    	     */
    	    public Object execute() {
    	        Model.getCommonBehaviorHelper()
    	            .removeReception(target, objectToRemove);
    	        return null;
    	    }
    	}
    }
    
    
    
    private abstract class AddModelElementCommand extends NonUndoableCommand {

        private Object target;
        private boolean multiSelect = true;
        private boolean exclusive = true;

        /**
         * Construct a command to add a model element to some list.
         */
        protected AddModelElementCommand(final Object target) {
            if (target == null) {
            	throw new IllegalArgumentException("target expected");
            }
        	this.target = target;
        }

        /*
         * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
         */
        public Object execute() {
            UMLAddDialog dialog =
                new UMLAddDialog(getChoices(), getSelected(), getDialogTitle(),
                                 isMultiSelect(),
                                 isExclusive());
            int result = dialog.showDialog(ArgoFrame.getFrame());
            if (result == JOptionPane.OK_OPTION) {
                doIt(dialog.getSelected());
            }
            return null;
        }
        
        /**
         * Returns the choices the user has in the UMLAddDialog. The choices are
         * depicted on the left side of the UMLAddDialog (sorry Arabic users) and
         * can be moved via the buttons on the dialog to the right side. On the
         * right side are the selected modelelements.
         * @return List of choices
         */
        protected abstract List getChoices();

        
        /**
         * The modelelements already selected BEFORE the dialog is shown.
         * @return List of model elements
         */
        protected abstract List getSelected();

        /**
         * The action that has to be done by ArgoUml after the user clicks ok in the
         * UMLAddDialog.
         * @param selected The choices the user has selected in the UMLAddDialog
         */
        protected abstract void doIt(Collection selected);

        /**
         * Returns the UML model target.
         * @return UML ModelElement
         */
        protected Object getTarget() {
            return target;
        }

        /**
         * Sets the UML model target.
         * @param theTarget The target to set
         */
        public void setTarget(Object theTarget) {
            target = theTarget;
        }

        /**
         * Returns the title of the dialog.
         * @return String
         */
        protected abstract String getDialogTitle();

        /**
         * Returns the exclusive.
         * @return boolean
         */
        public boolean isExclusive() {
            return exclusive;
        }

        /**
         * Returns the multiSelect.
         * @return boolean
         */
        public boolean isMultiSelect() {
            return multiSelect;
        }

        /**
         * Sets the exclusive.
         * @param theExclusive The exclusive to set
         */
        public void setExclusive(boolean theExclusive) {
            exclusive = theExclusive;
        }

        /**
         * Sets the multiSelect.
         * @param theMultiSelect The multiSelect to set
         */
        public void setMultiSelect(boolean theMultiSelect) {
            multiSelect = theMultiSelect;
        }

    }
    
    private class MethodExpressionGetterSetter extends ExpressionGetterSetter {
        
        @Override
        public Object get(Object modelElement, String type) {
            return Model.getFacade().getBody(modelElement);
        }
      
        @Override
        public void set(Object modelElement, Object value) {
            Model.getCoreHelper().setBody(modelElement, value);
        }

        @Override
        public Object create(final String language, final String body) {
            return Model.getDataTypesFactory().createProcedureExpression(language, body);
        }
    }
    
    private class SenderGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getSentStimuli(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
          
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getAttribute();
        }
    }
    
    private class LiteralGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getEnumerationLiterals(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
          
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getEnumerationLiteral();
        }
    }
    
    private class ReceiverGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getReceivedStimuli(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
          
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getAttribute();
        }
    }
    private class InternalTransitionGetterSetter extends ListGetterSetter {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getInternalTransitions(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
          
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getTransition();
        }
    }
    

    private class ClassifierGetterSetter extends ListGetterSetter implements Addable, Removeable {
        
        public Collection getOptions(Object modelElement, String type) {
            return Model.getFacade().getClassifiers(modelElement);
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Model.getMetaTypes().getClassifier();
        }

        public Command getAddCommand(Object modelElement) {
            return new AddCommand(modelElement);
        }
    
        public Command getRemoveCommand(Object modelElement, Object objectToRemove) {
    	    return new RemoveCommand(modelElement, objectToRemove);
        }
    
        private class AddCommand extends AddModelElementCommand {

            /**
             * Constructor for ActionAddPackageImport.
             */
            public AddCommand(Object target) {
                super(target);
            }
    
            protected List getChoices() {
                List list = new ArrayList();
                
                // Get all classifiers in our model
                // TODO: We need the property panels to have some reference to the
                // project they belong to instead of using deprecated functionality
                Project p = ProjectManager.getManager().getCurrentProject();
                Object model = p.getRoot();
                list.addAll(Model.getModelManagementHelper()
                        .getAllModelElementsOfKindWithModel(model, Model.getMetaTypes().getClassifier()));
                
                // Get all classifiers in all top level packages of all profiles
                for (Profile profile : p.getProfileConfiguration().getProfiles()) {
            	try {
    		    for (Object topPackage : profile.getProfilePackages()) {
    		            list.addAll(Model.getModelManagementHelper()
    		                    .getAllModelElementsOfKindWithModel(topPackage,
    		                	    Model.getMetaTypes().getClassifier()));
    		    }
    		} catch (ProfileException e) {
    		    // TODO: We need to rethrow this as some other exception
    		    // type but that is too much change for the moment.
    		    LOG.error("Exception", e);
    		}
                }
                return list;
            }
    
            protected List getSelected() {
                List list = new ArrayList();
                list.addAll(Model.getFacade().getClassifiers(getTarget()));
                return list;
            }
    
    
            protected String getDialogTitle() {
                return Translator.localize("dialog.title.add-specifications");
            }
    
    
            @Override
            protected void doIt(Collection selected) {
           	    Model.getCommonBehaviorHelper().setClassifiers(getTarget(), selected);
            }
        }
    
        private class RemoveCommand
	    extends NonUndoableCommand {
    	
            private final Object target;
            private final Object objectToRemove;
    	
	    /**
	     * Constructor for ActionRemovePackageImport.
	     */
	    public RemoveCommand(final Object target, final Object objectToRemove) {
	        this.target = target;
	        this.objectToRemove = objectToRemove;
	    }
	    
	    /*
	     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	     */
	    public Object execute() {
	        Model.getCommonBehaviorHelper()
	            .removeClassifier(target, objectToRemove);
	        return null;
	    }
	}
    }
    
    private class BaseClassGetterSetter extends ListGetterSetter implements Addable, Removeable {
        
        public Collection getOptions(Object modelElement, String type) {
            LinkedList<String> list = new LinkedList<String>(
                    Model.getFacade().getBaseClasses(modelElement));
            Collections.sort(list);
            return list;
        }
      
        public Object get(Object modelElement, String type) {
            // not needed
            return null;
        }
      
        public void set(Object element, Object x) {
            // not needed
        }

        public boolean isValidElement(Object element, String type) {
            return getOptions(element, type).contains(element);
        }
        
        public Object getMetaType() {
            return Class.class;
        }

        public Command getAddCommand(Object modelElement) {
            return new AddCommand(modelElement);
        }
    
        public Command getRemoveCommand(Object modelElement, Object objectToRemove) {
    	    return new RemoveCommand(modelElement, objectToRemove);
        }
    
        private class AddCommand extends AddModelElementCommand {

            final private List<String> metaClasses;
            
            /**
             * Constructor for ActionAddPackageImport.
             */
            public AddCommand(Object target) {
                super(target);
                metaClasses = initMetaClasses();
            }
            
            /**
             * Initialize the meta-classes list. <p>
             * 
             * All this code is necessary to be independent of 
             * model repository implementation, 
             * i.e. to ensure that we have a 
             * sorted list of strings.
             */
            private List<String> initMetaClasses() {
                Collection<String> tmpMetaClasses = 
                    Model.getCoreHelper().getAllMetatypeNames();
                List<String> metaClasses;
                if (tmpMetaClasses instanceof List) {
                    metaClasses = (List<String>) tmpMetaClasses;
                } else {
                    metaClasses = new LinkedList<String>(tmpMetaClasses);
                }
                try {
                    Collections.sort(metaClasses);
                } catch (UnsupportedOperationException e) {
                    // We got passed an unmodifiable List.  Copy it and sort the result
                    metaClasses = new LinkedList<String>(tmpMetaClasses);
                    Collections.sort(metaClasses);
                }
                
                return metaClasses;
            }
            
            protected List getChoices() {
        	return Collections.unmodifiableList(metaClasses);
            }
    
            protected List getSelected() {
                LinkedList<String> list = new LinkedList<String>(
                        Model.getFacade().getBaseClasses(getTarget()));
                Collections.sort(list);
                return list;
            }
    
    
            protected String getDialogTitle() {
                return Translator.localize("dialog.title.add-baseclasses");
            }
    
    
            @Override
            protected void doIt(Collection selected) {
                Object stereo = getTarget();
                Set<Object> oldSet = new HashSet<Object>(getSelected());
                Set toBeRemoved = new HashSet<Object>(oldSet);

                for (Object o : selected) {
                    if (oldSet.contains(o)) {
                        toBeRemoved.remove(o);
                    } else {
                        Model.getExtensionMechanismsHelper()
                                .addBaseClass(stereo, o);
                    }
                }
                for (Object o : toBeRemoved) {
                    Model.getExtensionMechanismsHelper().removeBaseClass(stereo, o);
                }
            }
        }
    
        private class RemoveCommand
	    extends NonUndoableCommand {
    	
            private final Object target;
            private final Object objectToRemove;
    	
	    /**
	     * Constructor for RemoveCommand.
	     */
	    public RemoveCommand(final Object target, final Object objectToRemove) {
	        this.target = target;
	        this.objectToRemove = objectToRemove;
	    }
	    
	    /*
	     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
	     */
	    public Object execute() {
                Model.getExtensionMechanismsHelper().removeBaseClass(target, objectToRemove);
	        return null;
	    }
	}
    }
}
