/* $Id: Facade.java 18181 2010-03-28 14:20:44Z linus $
 *******************************************************************************
 * Copyright (c) 2010 Contributors - see below
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *    Linus Tolke - initial implementation
 *    <see source control change log for other early contributors>
 *    Tom Morris - UML 1.4 and UML 2.x additions
 *    
 *******************************************************************************
 */
// Copyright (c) 2005-2009 The Regents of the University of California. All
// Rights Reserved. Permission to use, copy, modify, and distribute this
// software and its documentation without fee, and without a written
// agreement is hereby granted, provided that the above copyright notice
// and this paragraph appear in all copies. This software program and
// documentation are copyrighted by The Regents of the University of
// California. The software program and documentation are supplied "AS
// IS", without any accompanying services from The Regents. The Regents
// does not warrant that the operation of the program will be
// uninterrupted or error-free. The end-user understands that the program
// was developed for research purposes and is advised not to rely
// exclusively on the program for any reason. IN NO EVENT SHALL THE
// UNIVERSITY OF CALIFORNIA BE LIABLE TO ANY PARTY FOR DIRECT, INDIRECT,
// SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES, INCLUDING LOST PROFITS,
// ARISING OUT OF THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF
// THE UNIVERSITY OF CALIFORNIA HAS BEEN ADVISED OF THE POSSIBILITY OF
// SUCH DAMAGE. THE UNIVERSITY OF CALIFORNIA SPECIFICALLY DISCLAIMS ANY
// WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE
// PROVIDED HEREUNDER IS ON AN "AS IS" BASIS, AND THE UNIVERSITY OF
// CALIFORNIA HAS NO OBLIGATIONS TO PROVIDE MAINTENANCE, SUPPORT,
// UPDATES, ENHANCEMENTS, OR MODIFICATIONS.

package org.argouml.model;

import java.util.Collection;
import java.util.Iterator;
import java.util.List;

/**
 * Facade object for the Model subsystem in ArgoUML.<p>
 *
 * The purpose of this Facade object is to allow for decoupling other
 * subsystems from the insides of the model. For this purpose all of
 * the methods in this class give away and accept handles (of type
 * {@link java.lang.Object}) to the objects within the model.<p>
 *
 * This is just recognizers and getters.<p>
 *
 * To manipulate the objects of the model there is a set of factories
 * and helpers within the Model subsystem.<p>
 *
 * Signatures for all recognizers are:
 * <ul>
 * <li>public boolean isATYPE(Object handle)
 * <li>public boolean isPROPERTY(Object handle)
 * <li>public boolean hasPROPERTY(Object handle)
 * </ul>
 *
 * Signatures for all getters are:
 * <ul>
 * <li>public Object getROLENAME(Object handle) - 1..1
 * <li>public Iterator/Collection getROLENAMEs(Object handle) - 0..*
 * <li>public String getName(Object handle) - Name
 * </ul>
 *
 * @author Linus Tolke
 */
public interface Facade {
    /**
     * This tag is set on elements that are generated by reference when
     * importing.
     * If it is set, then the critics could ignore those objects if they want.
     * 
     * TODO: This is really belongs in a higher level utility class
     * that uses the Facade's tagged value methods.  Also, it needs to
     * be updated to use a UML 1.4 TagDefinition instead of a simple String.
     */
    String GENERATED_TAG = "GeneratedFromImport"; //$NON-NLS-1$
    
    /**
     * Tag which indicates a ModelElement is derived. From UML Standard Elements
     * profile.
     */
    String DERIVED_TAG = "derived"; //$NON-NLS-1$

    ////////////////////////////////////////////////////////////////
    
    /**
     * Return the version of the UML metamodel supported by the underlying
     * implementation. The version string is made up of numerical components
     * separated by the '.' character with the most significant digits on the
     * left.
     * <p>
     * The MDR implementation supports UML 1.4 and the initial eUML
     * implementation supports UML 2.1.1. The earlier NSUML implementation
     * supported UML 1.3.
     * 
     * @return a period ('.') separated string of version numbers. e.g. "1.4" or
     *         "2.1.1"
     */
    String getUmlVersion();
    
    ////////////////////////////////////////////////////////////////
    /**
     * Recognizer for Abstraction.
     *
     * @param handle candidate
     * @return true if handle is an Abstraction.
     */
    boolean isAAbstraction(Object handle);

    /**
     * Recognizer for Action.
     *
     * @param handle candidate
     * @return true if handle is an Action
     */
    boolean isAAction(Object handle);

    /**
     * Recognizer for ActionSequence.
     *
     * @param handle candidate
     * @return true if handle is an action sequence
     */
    boolean isAActionSequence(Object handle);

    /**
     * Recognizer for ActionState.
     *
     * @param handle candidate
     * @return true if handle is an ActionState
     */
    boolean isAActionState(Object handle);

    /**
     * Recognizer for CallState.
     *
     * @param handle candidate
     * @return true if handle is an call state
     */
    boolean isACallState(Object handle);

    /**
     * Recognizer for ObjectFlowState.
     *
     * @param handle candidate
     * @return true if handle is an objectflow state
     */
    boolean isAObjectFlowState(Object handle);

    /**
     * Recognizer for SubactivityState.
     *
     * @param handle candidate
     * @return true if handle is an subactivity state
     */
    boolean isASubactivityState(Object handle);

    /**
     * Recognizer for Actor.
     *
     * @param handle candidate
     * @return true if handle is an Actor
     */
    boolean isAActor(Object handle);

    /**
     * Recognizer for AggregationKind.
     *
     * @param handle candidate
     * @return true if handle is an AggregationKind
     */
    boolean isAAggregationKind(Object handle);

    /**
     * Recognizer for Artifact.
     *
     * @param handle candidate
     * @return true if handle is an Artifact
     */
    boolean isAArtifact(Object handle);
    
    /**
     * Recognizer for Association.
     *
     * @param handle candidate
     * @return true if handle is an Association
     */
    boolean isAAssociation(Object handle);

    /**
     * Recognizer for AssociationEnd.
     *
     * @param handle candidate
     * @return true if handle is an AssociationEnd
     */
    boolean isAAssociationEnd(Object handle);

    /**
     * Recognizer for AssociationRole.
     *
     * @param handle candidate
     * @return true if handle is an AssociationRole
     */
    boolean isAAssociationRole(Object handle);

    /**
     * Recognizer for AssociationEndRole.
     *
     * @param handle candidate
     * @return true if handle is an AssociationEndRole
     */
    boolean isAAssociationEndRole(Object handle);

    /**
     * Recognizer for Attribute.
     *
     * @param handle candidate
     * @return true if handle is an Attribute
     */
    boolean isAAttribute(Object handle);

    /**
     * Recognizer for AttributeLink.
     *
     * @param handle candidate
     * @return true if handle is an AttributeLink
     */
    boolean isAAttributeLink(Object handle);
    
    /**
     * Recognizer for asynchronisity of an action.
     *
     * @param handle candidate
     * @return true if the argument is asynchronous
     */
    boolean isAsynchronous(Object handle);

    /**
     * Recognizer for abstract classes and operations. True if it has an
     * <code>isAbstract</code> attribute with a value of <code>true</code>.
     * If there's no such attribute (ie it isn't a type which has that attribute
     * in the UML metamodel), it will return false without throwing an error.
     * 
     * @param handle
     *            candidate
     * @return true if handle is abstract.
     */
    boolean isAbstract(Object handle);

    /**
     * Recognizer for ActivityGraph.
     *
     * @param handle candidate
     * @return true if handle is ActivityGraph.
     */
    boolean isAActivityGraph(Object handle);

    /**
     * Recognizer for BehavioralFeature.
     *
     * @param handle candidate
     * @return true if handle is a behavioral feature
     */
    boolean isABehavioralFeature(Object handle);

    /**
     * Recognizer for Binding.
     *
     * @param handle candidate
     * @return true if handle is a Binding
     */
    boolean isABinding(Object handle);
    
    /**
     * Recognizer for CallAction.
     *
     * @param handle candidate
     * @return true if handle is a CallAction
     */
    boolean isACallAction(Object handle);

    /**
     * Recognizer for CallEvent.
     *
     * @param handle candidate
     * @return true if handle is a CallEvent
     */
    boolean isACallEvent(Object handle);

    /**
     * Recognizer for ChangeEvent.
     *
     * @param handle candidate
     * @return true if handle is a ChangeEvent
     */
    boolean isAChangeEvent(Object handle);

    /**
     * Recognizer for UML Class.
     *
     * @param handle candidate
     * @return true if handle is a UML Class
     */
    boolean isAClass(Object handle);

    /**
     * Recognizer for AssociationClass.
     *
     * @param handle candidate
     * @return true if handle is an AssociationClass
     */
    boolean isAAssociationClass(Object handle);

    /**
     * Recognizer for Classifier.
     *
     * @param handle candidate
     * @return true if handle is a Classifier
     */
    boolean isAClassifier(Object handle);

    /**
     * Recognizer for ClassifierInState.
     *
     * @param handle candidate
     * @return true if handle is a ClassifierInState
     */
    boolean isAClassifierInState(Object handle);

    /**
     * Recognizer for ClassifierRole.
     *
     * @param handle candidate
     * @return true if handle is a ClassifierRole
     */
    boolean isAClassifierRole(Object handle);

    /**
     * Recognizer for Comment.
     *
     * @param handle candidate
     * @return true if handle is a Comment
     */
    boolean isAComment(Object handle);

    /**
     * Recognizer for Collaboration.
     *
     * @param handle candidate
     * @return true if handle is a Collaboration
     */
    boolean isACollaboration(Object handle);

    /**
     * Recognizer for CollaborationInstanceSet.
     *
     * @param handle candidate
     * @return true if handle is a CollaborationInstanceSet
     */
    boolean isACollaborationInstanceSet(Object handle);
    
    /**
     * Recognizer for Component.
     *
     * @param handle candidate
     * @return true if handle is a Component
     */
    boolean isAComponent(Object handle);

    /**
     * Recognizer for ComponentInstance.
     *
     * @param handle candidate
     * @return true if handle is a ComponentInstance
     */
    boolean isAComponentInstance(Object handle);

    /**
     * Recognizer for Constraint.
     *
     * @param handle candidate
     * @return true if handle is a Constraint
     */
    boolean isAConstraint(Object handle);

    /**
     * Recognizer for CreateAction.
     *
     * @param handle candidate
     * @return true if handle is a CreateAction
     */
    boolean isACreateAction(Object handle);

    /**
     * Recognizer for DataType.
     *
     * @param handle candidate
     * @return true if handle is a DataType
     */
    boolean isADataType(Object handle);

    /**
     * Recognizer for DataValue.
     *
     * @param handle candidate
     * @return true if handle is a DataValue
     */
    boolean isADataValue(Object handle);

    /**
     * Recognizer for Dependency.
     *
     * @param handle candidate
     * @return true if handle is a Dependency
     */
    boolean isADependency(Object handle);

    
    /**
     * @param handle element to test
     * @return true if Element is a DirectedRelationship
     */
    public boolean isADirectedRelationship(Object handle);
    
    /**
     * Recognizer for DestroyAction.
     *
     * @param handle candidate
     * @return true if handle is a DestroyAction
     */
    boolean isADestroyAction(Object handle);

    /**
     * Recognizer for CompositeState.
     *
     * @param handle candidate
     * @return true if handle is a CompositeState
     */
    boolean isACompositeState(Object handle);

    /**
     * Recognizer for Element.
     * <p>
     * <em>NOTE:</em> In the UML 1.4 specification there are a number of types
     * that one might expect to be subtypes of Element, but which are not. The
     * spec isn't clear on whether this is intentional or not but if you want
     * all types in the spec, including things which are not subtypes of Element
     * such as Expression, Multiplicity, MultiplicityRange TemplateArgument,
     * TemplateParameter, ElementResidence, ElementImport use the method
     * {@link #isAUMLElement(Object)}.
     * 
     * @param handle
     *            candidate
     * @return true if handle is an Element
     * @see #isAUMLElement(Object)
     */
    boolean isAElement(Object handle);

    /**
     * Recognizer for ElementImport.
     *
     * @param handle candidate
     * @return true if handle is an ElementImport
     */
    boolean isAElementImport(Object handle);

    /**
     * Recognizer for ElementResidence.
     *
     * @param handle candidate
     * @return true if handle is an ElementResidence
     */
    boolean isAElementResidence(Object handle);

    /**
     * Recognizer for UML Enumeration.
     *
     * @param handle candidate
     * @return true if handle is an UML Enumeration
     */
    boolean isAEnumeration(Object handle);

    /**
     * Recognizer for UML EnumerationLiteral
     * 
     * @param handle the candidate
     * @return true if handle is an UML EnumerationLiteral
     */
    boolean isAEnumerationLiteral(Object handle);
    
    /**
     * Recognizer for Event.
     *
     * @param handle candidate
     * @return true if handle is an Event
     */
    boolean isAEvent(Object handle);

    /**
     * Recognizer for UML Exception.
     *
     * @param handle candidate
     * @return true if handle is a UML Exception
     */
    boolean isAException(Object handle);

    /**
     * Recognizer for Expression.
     *
     * @param handle candidate
     * @return true if handle is an Expression
     */
    boolean isAExpression(Object handle);


    /**
     * Recognizer for Extend.
     *
     * @param handle candidate
     * @return true if handle is an Extend
     */
    boolean isAExtend(Object handle);

    /**
     * Recognizer for Extension. For UML2 only.
     *
     * @param handle candidate
     * @return true if handle is an ExtensionPoint
     */
    boolean isAExtension(Object handle);

    /**
     * Recognizer for ExtensionPoint.
     *
     * @param handle candidate
     * @return true if handle is an ExtensionPoint
     */
    boolean isAExtensionPoint(Object handle);

    /**
     * Recognizer for Feature.
     *
     * @param handle candidate
     * @return true if handle is a Feature
     */
    boolean isAFeature(Object handle);

    /**
     * Recognizer for FinalState.
     *
     * @param handle candidate
     * @return true if handle is a FinalState
     */
    boolean isAFinalState(Object handle);

    /**
     * Recognizer for Flow.
     *
     * @param handle candidate
     * @return true if handle is a Flow
     */
    boolean isAFlow(Object handle);

    /**
     * Recognizer for Guard.
     *
     * @param handle candidate
     * @return true if handle is a Guard
     */
    boolean isAGuard(Object handle);

    /**
     * Recognizer for GeneralizableElement.
     *
     * @param handle candidate
     * @return true if handle is a GeneralizableElement
     */
    boolean isAGeneralizableElement(Object handle);

    /**
     * Recognizer for Generalization.
     *
     * @param handle candidate
     * @return true if handle is a Generalization
     */
    boolean isAGeneralization(Object handle);

    /**
     * Recognizer for Include.
     *
     * @param handle candidate
     * @return true if handle is an Include
     */
    boolean isAInclude(Object handle);

    /**
     * Recognizer for Instance.
     *
     * @param handle candidate
     * @return true if handle is a Instance
     */
    boolean isAInstance(Object handle);

    /**
     * Recognizer for Interaction.
     *
     * @param handle candidate
     * @return true if handle is a Interaction
     */
    boolean isAInteraction(Object handle);


    /**
     * Recognizer for InteractionInstanceSet.
     *
     * @param handle candidate
     * @return true if handle is a InteractionInstanceSet
     */
    boolean isAInteractionInstanceSet(Object handle);
    
    /**
     * Recognizer for Interface.
     *
     * @param handle candidate
     * @return true if handle is a Interface
     */
    boolean isAInterface(Object handle);

    /**
     * Recognizer for Link.
     *
     * @param handle candidate
     * @return true if handle is a Link
     */
    boolean isALink(Object handle);

    /**
     * Recognizer for LinkEnd.
     *
     * @param handle candidate
     * @return true if handle is a LinkEnd
     */
    boolean isALinkEnd(Object handle);

    /**
     * Recognizer for LinkObject.
     *
     * @param handle candidate
     * @return true if handle is a LinkObject
     */
    boolean isALinkObject(Object handle);
    
    /**
     * Recognizer for Message.
     *
     * @param handle candidate
     * @return true if handle is a Method
     */
    boolean isAMessage(Object handle);

    /**
     * Recognizer for Method.
     *
     * @param handle candidate
     * @return true if handle is a Method
     */
    boolean isAMethod(Object handle);

    /**
     * Recognizer for Model.
     *
     * @param handle candidate
     * @return true if handle is a Model
     */
    boolean isAModel(Object handle);

    /**
     * Recognizer for ModelElement.  <em>NOTE:</em> Consider using {@link
     * Facade#isAUMLElement(Object)} instead since it is more likely to do what
     * you want.  This is strictly for subtypes of ModelElement which excludes
     * many UML types such as Multiplicities, TemplateParameters, etc.  It also
     * will fail to work for UML 2 types which are subtypes of Element, but
     * not ModelElement.
     *
     * @param handle candidate
     * @return true if handle is a ModelElement
     */
    boolean isAModelElement(Object handle);
    
    /**
     * Recognizer for a UML element which can have a name. In UML 1.x,
     * this is just a ModelElement, but in UML 2.x it's a separate
     * metatype. Use this method if you're principally interested in
     * whether something can be named.
     * 
     * @param handle
     *            candidate
     * @return true if handle is a ModelElement
     */
    boolean isANamedElement(Object handle);

    /**
     * Recognizer for Multiplicity.
     *
     * @param handle candidate
     * @return true if handle is a Multiplicity
     */
    boolean isAMultiplicity(Object handle);

    /**
     * Recognizer for MultiplicityRange.
     *
     * @param handle candidate
     * @return true if handle is a MultiplicityRange
     */
    boolean isAMultiplicityRange(Object handle);

    /**
     * Recognizer for Namespace.
     *
     * @param handle candidate
     * @return true if handle is a Namespace
     */
    boolean isANamespace(Object handle);

    /**
     * Recognizer for N-ary Association.
     *
     * TODO: This is not a recognizer for some type. Rename to
     * isNaryAssociation?
     *
     * @param handle candidate
     * @return true if handle is an Association
     */
    boolean isANaryAssociation(Object handle);

    /**
     * Recognizer for a Node.
     *
     * @param handle candidate
     * @return true if handle is a Node
     */
    boolean isANode(Object handle);

    /**
     * Recognizer for a NodeInstance.
     *
     * @param handle candidate
     * @return true if handle is a NodeInstance
     */
    boolean isANodeInstance(Object handle);

    /**
     * Recognizer for Operation.
     *
     * @param handle candidate
     * @return true if handle is an Operation
     */
    boolean isAOperation(Object handle);

    /**
     * Recognizer for Object.
     *
     * @param handle candidate
     * @return true if handle is an Object
     */
    boolean isAObject(Object handle);

    /**
     * Recognizer for Parameter.
     *
     * @param handle candidate
     * @return true if handle is a Parameter
     */
    boolean isAParameter(Object handle);

    /**
     * Recognizer for Partition.
     *
     * @param handle candidate
     * @return true if handle is a Partition
     */
    boolean isAPartition(Object handle);

    /**
     * Recognizer for PackageImport (was Permission in UML 1.x).
     * 
     * @param handle
     *                candidate
     * @return true if handle is an PackageImport (or Permission in UML 1.4)
     * @since UML 2.0
     */
    boolean isAPackageImport(Object handle);
    
    /**
     * Recognizer for Package.
     *
     * @param handle candidate
     * @return true if handle is a Package
     */
    boolean isAPackage(Object handle);

    /**
     * Recognizer for PrimitiveType.
     *
     * @param handle candidate
     * @return true if handle is a Primitive
     */
    boolean isAPrimitiveType(Object handle);
    
    /**
     * Recognizer for Profile. In UML 2, this is no longer a model, but a
     * separate meta package instance. In UML 1.4, ArgoUML assumes it to be a
     * model instance with a <<profile>> stereotype.
     *
     * @param handle candidate
     * @return true if handle is a Profile
     */
    boolean isAProfile(Object handle);

    /**
     * Recognizer for ProfileApplication. In UML 2 only.
     *
     * @param handle candidate
     * @return true if handle is a ProfileApplication
     */
    boolean isAProfileApplication(Object handle);

    /**
     * Recognizer for Property. UML 2 only.
     *
     * @param handle candidate
     * @return true if handle is a Property
     */
    boolean isAProperty(Object handle);

    /**
     * Recognizer for Pseudostate.
     *
     * @param handle candidate
     * @return true if handle is a Pseudostate
     */
    boolean isAPseudostate(Object handle);

    /**
     * Recognizer for PseudostateKind.
     *
     * @param handle candidate
     * @return true if handle is a PseudostateKind
     */
    boolean isAPseudostateKind(Object handle);

    /**
     * Return the Kind of a Pseudostate or Parameter.
     *
     * @param handle the Pseudostate or Parameter
     * @return the Kind
     */
    Object getKind(Object handle);

    /**
     * Return the receiver Instance for a Stimulus or the receiver
     * ClassifierRole for a Message.
     * 
     * @param handle
     *            candidate
     * @return Instance or ClassifierRole
     */
    Object getReceiver(Object handle);

    /**
     * Return the Link belonging to the given LinkEnd.
     *
     * @param handle the LinkEnd
     * @return the Link
     */
    Object getLink(Object handle);

    /**
     * Check whether two pseudostatekinds are equal/of the same type.
     *
     * @return true if the are the same type
     * @param ps1 one kind
     * @param ps2 one kind
     */
    boolean equalsPseudostateKind(Object ps1, Object ps2);

    /**
     * Recognizer for Reception.
     *
     * @param handle candidate
     * @return true if handle is a Reception
     */
    boolean isAReception(Object handle);

    /**
     * Recognizer for ReturnAction.
     *
     * @param handle candidate
     * @return true if handle is a ReturnAction
     */
    boolean isAReturnAction(Object handle);

    /**
     * Recognizer for Relationship.
     *
     * @param handle candidate
     * @return true if handle is a Relationship
     */
    boolean isARelationship(Object handle);

    /**
     * Recognizer for SendAction.
     *
     * @param handle candidate
     * @return true if handle is a SendAction
     */
    boolean isASendAction(Object handle);

    /**
     * Recognizer for Signal.
     *
     * @param handle candidate
     * @return true if handle is a Signal
     */
    boolean isASignal(Object handle);

    /**
     * Recognizer for SignalEvent.
     *
     * @param handle candidate
     * @return true if handle is a SignalEvent
     */
    boolean isASignalEvent(Object handle);

    /**
     * Recognizer for SimpleState.
     *
     * @param handle candidate
     * @return true if handle is a StateMachine
     */
    boolean isASimpleState(Object handle);

    /**
     * Recognizer for StateMachine.
     *
     * @param handle candidate
     * @return true if handle is a StateMachine
     */
    boolean isAStateMachine(Object handle);

    /**
     * Recognizer for Stimulus.
     *
     * @param handle candidate
     * @return true if handle is a Stimulus
     */
    boolean isAStimulus(Object handle);

    /**
     * Recognizer for StateVertex.
     *
     * @param handle candidate
     * @return true if handle is a StateVertex
     */
    boolean isAStateVertex(Object handle);

    /**
     * Recognizer for Stereotype.
     *
     * @param handle candidate
     * @return true if handle is a Stereotype
     */
    boolean isAStereotype(Object handle);

    /**
     * Recognizer for StructuralFeature.
     *
     * @param handle candidate
     * @return true if handle is a StructuralFeature
     */
    boolean isAStructuralFeature(Object handle);

    /**
     * Recognizer for State.
     *
     * @param handle candidate
     * @return true if handle is a State
     */
    boolean isAState(Object handle);

    /**
     * Recognizer for StubState.
     *
     * @param handle candidate
     * @return true if handle is a StubState
     */
    boolean isAStubState(Object handle);

    /**
     * Recognizer for SubmachineState.
     *
     * @param handle candidate
     * @return true if handle is a SubmachineState
     */
    boolean isASubmachineState(Object handle);

    /**
     * Recognizer for Subsystem.
     *
     * @param handle candidate
     * @return true if handle is a Subsystem
     */
    boolean isASubsystem(Object handle);

    /**
     * Recognizer for SubsystemInstance.
     *
     * @param handle candidate
     * @return true if handle is a SubsystemInstance
     */
    boolean isASubsystemInstance(Object handle);
    
    /**
     * Recognizer for SynchState.
     *
     * @param handle candidate
     * @return true if handle is a SynchState
     */
    boolean isASynchState(Object handle);

    /**
     * Recognizer for TaggedValue.
     *
     * @param handle candidate
     * @return true if handle is a TaggedValue
     */
    boolean isATaggedValue(Object handle);

    /**
     * Recognizer for TemplateArgument.
     *
     * @param handle candidate
     * @return true if handle is a TemplateArgument
     */
    boolean isATemplateArgument(Object handle);


    /**
     * Recognizer for TemplateParameter.
     *
     * @param handle candidate
     * @return true if handle is a TemplateParameter
     */
    boolean isATemplateParameter(Object handle);
    
    /**
     * Recognizer for TerminateAction.
     *
     * @param handle candidate
     * @return true if handle is a TerminateAction
     */
    boolean isATerminateAction(Object handle);

    /**
     * Recognizer for Transition.
     *
     * @param handle candidate
     * @return true if handle is a Transition
     */
    boolean isATransition(Object handle);

    /**
     * Recognizer for TimeEvent.
     *
     * @param handle candidate
     * @return true if handle is a TimeEvent
     */
    boolean isATimeEvent(Object handle);

    /**
     * Recognizer for any UML type defined in the UML specification,
     * including those which are not subtypes of Element. <p>
     * 
     * In the UML 1.4 specification, not all defined types are subtypes
     * of Element, although this has been fixed/changed in UML 2.
     *  
     * @param handle the UML object
     * @return true if the given object is a UML object
     * @see #isAElement(Object)
     */
    public boolean isAUMLElement(Object handle);

    /**
     * Recognizer for UninterpretedAction.
     *
     * @param handle candidate
     * @return true if handle is a UninterpretedAction
     */
    boolean isAUninterpretedAction(Object handle);

    /**
     * Recognizer for Usage.
     *
     * @param handle candidate
     * @return true if handle is a Usage
     */
    boolean isAUsage(Object handle);

    /**
     * Recognizer for a UseCase.
     *
     * @param handle candidate
     * @return true if handle is a UseCase
     */
    boolean isAUseCase(Object handle);

    
    /**
     * Recognizer for VisibilityKind.
     *
     * @param handle candidate
     * @return true if handle is a VisibilityKind
     */
    boolean isAVisibilityKind(Object handle);

    /**
     * Recognizer for UML Classes that are Active.
     *
     * @param handle candidate
     * @return true if Class is Active
     */
    boolean isActive(Object handle);


    /**
     * Recognizer for concurrent CompositeState.
     *
     * @param handle composite state
     * @return true if concurent.
     */
    boolean isConcurrent(Object handle);

    /**
     * Recognizer for ConcurrentRegion.  
     * A concurrent region is a CompositeState which has
     * an owner that is concurrent.
     *
     * @param handle candidate
     * @return true if handle is a ConcurrentRegion
     */
    boolean isAConcurrentRegion(Object handle);

    /**
     * Recognizer for constructor.
     *
     * A constructor is an operation that has a stereotype named create or
     * a method that is a specification of an operator that is a constructor.
     *
     * @param handle candidate
     * @return true if handle is a constructor.
     */
    boolean isConstructor(Object handle);

    /**
     * Returns true if the given element is Frozen.
     *
     * @param handle candidate
     * @return boolean true if Frozen
     */
    boolean isFrozen(Object handle);

    /**
     * Returns true if a given AssociationEnd has an aggretation kind of
     * Composite.
     * 
     * @param handle candidate
     * @return boolean
     */
    boolean isComposite(Object handle);

    /**
     * Returns true if a given AssociationEnd has an aggretation kind of
     * Aggregate.
     * 
     * @param handle candidate
     * @return boolean
     */
    boolean isAggregate(Object handle);

    /**
     * Recognizer for Attributes that are initialized. An Attribute is 
     * considered initalized if it has an initialValue expression
     * with a non-empty body.
     *
     * @param handle candidate
     * @return true if the attribute is initialized.
     */
    boolean isInitialized(Object handle);

    /**
     * Recognizer for internal transitions.  An internal Transition has
     * its owning State as both the source State and target State.
     *
     * @author mvw
     * @param handle candidate
     * @return true if handle is an internal transition.
     */
    boolean isInternal(Object handle);

    /**
     * Recognizer for leaf element. True if it has an <code>isLeaf</code>
     * attribute with a value of <code>true</code>. If there's no such
     * attribute (ie it isn't a type which has that attribute in the UML
     * metamodel), it will return false without throwing an error.
     * 
     * @param handle
     *            candidate GeneralizableElement, Operation, or Reception
     * @return true if handle is a leaf
     */
    boolean isLeaf(Object handle);

    /**
     * Recognizer for roots.  True if it has an <code>isRoot</code>
     * attribute with a value of <code>true</code>.  If there's no such
     * attribute (ie it isn't a type which has that attribute in the UML
     * metamodel), it will return false without throwing an error.
     *
     * @param handle candidate GeneralizableElement, Operation, or Reception
     * @return true if handle is a root
     */
    boolean isRoot(Object handle);

    /**
     * Return value of isSpecification attribute for a ModelElement
     * or ElemetImport.
     *
     * @param handle candidate ModelElement or ElemetImport
     * @return boolean value contained by the isSpecification attribute
     */
    boolean isSpecification(Object handle);

    /**
     * Return navigability of an AssociationEnd.
     *
     * @param handle candidate AssociationEnd
     * @return true if handle is navigable
     */
    boolean isNavigable(Object handle);

    /**
     * Recognizer for primary objects. A primary object is an object that is
     * created by the parser or by a user. Object that are created as a side
     * effect of importing some other object are not.
     * <p>
     * TODO: This doesn't belong in the Facade. It's a higher level function
     * used specifically for certain reverse engineering operations. - tfmorris
     * 
     * @param handle
     *            candidate
     * @return true if primary object.
     */
    boolean isPrimaryObject(Object handle);

    /**
     * Recognizer for ModelElements with package visibility.
     *
     * TODO: This method name was chosen for compatibility with
     * the existing naming scheme, but has the potential of 
     * clashing with some future use.
     * 
     * @param handle candidate
     * @return true if handle has package visibility
     */
    boolean isPackage(Object handle);
    
    /**
     * Recognizer for ModelElements with private visibility.
     *
     * @param handle candidate
     * @return true if handle has private
     */
    boolean isPrivate(Object handle);

    /**
     * Recognizer for ModelElements with public visibility.
     *
     * @param handle candidate
     * @return true if handle has public
     */
    boolean isPublic(Object handle);

    /**
     * Recognizer for BehaviouralFeature's that are queries.
     *
     * @param handle candidate
     * @return true if it is a query
     */
    boolean isQuery(Object handle);

    /**
     * Recognizer for ModelElements with protected visibility.
     *
     * @param handle candidate
     * @return true if handle has protected
     */
    boolean isProtected(Object handle);

    /**
     * Recognizer for ModelElements which have the <<realize>> Stereotype 
     * applied.
     *
     * @param handle candidate
     * @return true if handle has a realize stereotype
     */
    boolean isRealize(Object handle);

    /**
     * Recognizer for Parameter's with a DirectionKind of Return.
     * TODO: This is a duplicate of {@link #hasReturnParameterDirectionKind(Object)}
     * @param handle candidate parameter
     * @return true if handle is a return parameter.
     */
    boolean isReturn(Object handle);

    /**
     * Recognizer for ModelElements which have the <<singleton>> Stereotype
     * applied.
     * <p>
     * TODO: Moved this out of the Facade to someplace elese since it's not
     * Model subsystem specific - tfm 20070607
     * 
     * @param handle
     *            candidate
     * @return true if handle is a singleton.
     */
    boolean isSingleton(Object handle);

    /**
     * Recognizer for ModelElements with a given stereotype.
     *
     * @param handle candidate model element
     * @param stereotypename a string that is the stereotype name.
     * @return true if handle is an object that has the given stereotype.
     */
    boolean isStereotype(Object handle, String stereotypename);

    /**
     * Returns true if the given CompositeState is the top state.  The top
     * state has no containing StateMachine.
     *
     * @param handle CompositeState
     * @return boolean true if top state
     */
    boolean isTop(Object handle);

    /**
     * Recognizer for ModelElements with the <<type>> Stereotype applied.
     *
     * @param handle candidate
     * @return true if handle is a type.
     */
    boolean isType(Object handle);

    /**
     * Recognizer for ModelElements which have the <<utility>> Stereotype 
     * applied.
     *
     * @param handle candidate
     * @return true if handle is a utility.
     */
    boolean isUtility(Object handle);

    /**
     * Returns the Association connected to an AssociationEnd or
     * belonging to the given Link.
     *
     * @param handle the AssociationEnd or Link
     * @return association end
     */
    Object getAssociation(Object handle);

    /**
     * Return the AssociationEnd connecting the given Classifier 
     * and Association.
     *
     * @param classifier the classifier
     * @param association is the association
     * @return association end
     */
    Object getAssociationEnd(Object classifier, Object association);

    /**
     * Return the collection of AssociationEnds for a Classifier
     *
     * @param handle the Classifier for which to get the AssociationEnds.
     * @return Collection with association ends.
     */
    Collection getAssociationEnds(Object handle);

    /**
     * Return the collection of AssociationRoles for an Association.
     *
     * @param handle the Association that we get the association roles from.
     * @return Collection of association roles.
     */
    Collection getAssociationRoles(Object handle);

    /**
     * Return the List of Attributes for a Classifier.
     * 
     * @param handle classifier to examine.
     * @return Collection with attributes.
     */
    List getAttributes(Object handle);

    /**
     * Return the Collection of the names of baseClasses for a Stereotype.
     * Note that the returned values are Names, not Elements.
     *
     * @param handle the stereotype
     * @return collection containing all applicable baseclasses
     */
    Collection<String> getBaseClasses(Object handle);

    /**
     * Return the base of a ModelElement. The element may be one of the
     * following types: AssociationEndRole, AssociationRole, Extend, Include, or
     * ClassifierRoles. <em>NOTE:</em> in the case of a ClassifierRole, the
     * return value is a Collection of elements, not a single element.  If you
     * know you are dealing with a ClassifierRole, it is prefeable to use
     * {@link #getBases(Object)}.
     * <p>
     * For UML 2.x with a target of an Include element, use getIncludingCase().
     * 
     * @param handle
     *            the model element
     * @return the base
     */
    Object getBase(Object handle);
    
    /**
     * @param handle Include element
     * @return the UseCase into which the included UseCase will be added
     */
//    Object getIncludingCase(Object handle);
    
    /**
     * Get the bases of a ClassifierRole.
     *
     * @param handle ClassifierRole.
     * @return the bases.
     */
    Collection getBases(Object handle);

    /**
     * Get the StateMachines which define the behaviors of a Modelelement.
     * 
     * @param handle
     *            modelelement to examine.
     * @return collection of StateMachines.
     */
    Collection getBehaviors(Object handle);

    /**
     * Get the BehavioralFeature of a Parameter.
     *
     * @param handle the Parameter
     * @return the behavioral feature.
     */
    Object getBehavioralFeature(Object handle);

    /**
     * Get the body of a Method, Constraint, Expression, or Comment.<p>
     *
     * If the argument is an Expression, the body returned is a
     * {@link String} (or <code>null</code>). If you want a type-safe
     * version of this version of the call,
     * use {@link DataTypesHelper#getBody(Object)}.
     *
     * @param handle expression.
     * @return the body.
     */
    Object getBody(Object handle);

    /**
     *  Return the SynchState's bound.
     *  
     *  @param handle the synch State
     *  @return bound
     */
    int getBound(Object handle);

    /**
     * Return Changeability of a StructuralFeature or a AssociationEnd.
     *
     * @param handle the StructuralFeature or AssociationEnd
     * @return the Changeability
     * @deprecated for 0.25.4 by tfmorris. Use {@link #isReadOnly(Object)}.
     */
    @Deprecated
    Object getChangeability(Object handle);


    /**
     * Get the specific end of a generalization. This was getChild() in UML 1.x.
     * 
     * @param handle
     *            generalization.
     * @return the specific classifier.
     * @since 0.25.4
     */
    Object getSpecific(Object handle);
    
    
    /**
     * Get the children of a GeneralizableElement.
     *
     * @param handle the GeneralizableElement.
     * @return a collection with all children.
     */
    Collection getChildren(Object handle);

    /**
     * Get the ClassifierRoles of a Classifier or Feature.
     *
     * @param handle the Classifer or Feature.
     * @return the classifiers roles of the instance
     */
    Collection getClassifierRoles(Object handle);

    /**
     * Get the classifier (participant) of an AssociationEnd.
     *
     * TODO: Rename this getParticipant to align with UML spec - tfm - 20070331
     * @param handle The AssociationEnd to get from.
     * @return The classifier of the AssociationEnd.
     */
    Object getClassifier(Object handle);

    /**
     * Get the classifiers of an Instance.
     *
     * @param handle the Instance
     * @return the classifiers of the Instance
     */
    Collection getClassifiers(Object handle);

    /**
     * Get the classifiers in state of a Classifier or State.
     *
     * @param handle the model element
     * @return the classifierss in state
     */
    Collection getClassifiersInState(Object handle);

    /**
     * Get the clients of a Dependency.
     *
     * @param handle the dependency
     * @return the clients of the dependency
     */
    Collection getClients(Object handle);

    /**
     * Get the client dependencies of a ModelElement (i.e. those Dependencies
     * in which this ModelElement participates as a client).
     *
     * @param handle the ModelElement.
     * @return a collection of all client dependencies.
     */
    Collection getClientDependencies(Object handle);

    /**
     * Get the condition of an Extend.
     *
     * @param handle The Extend.
     * @return the condition
     */
    Object getCondition(Object handle);

    /**
     * Get the concurrency of an Operation.
     *
     * @param handle The Operation.
     * @return the concurrency.
     */
    Object getConcurrency(Object handle);

    /**
     * Return the connections to an Association or a Link.
     *
     * @param handle the Association or Link
     * @return a Collection with all connections 
     * of type AssociationEnd or LinkEnd.
     */
    Collection getConnections(Object handle);

    /**
     * Return the effect of a Transition.
     *
     * @param handle the Transition
     * @return the effect
     */
    Object getEffect(Object handle);

    /**
     * Get the ElementResidences of a ModelElement.
     *
     * @param handle the ModelElement that we are getting the residences of
     * @return the collection with ElementResidences
     */
    Collection getElementResidences(Object handle);

    /**
     * Return the ElementImports which have this ModelElement as their
     * importedElement. For the inverse relationship (ElementImports which
     * import into a Package) see {@link #getElementImports(Object)}.
     * 
     * @param handle
     *            the ModelElement
     * @return the collection of ElementImports
     * @see #getElementImports(Object)
     */
    Collection getElementImports2(Object handle);

    /**
     * Return the ElementImports which import into this Package. For the inverse
     * relationship (ElementImports which have a particular ModelElement as
     * their importedElement), see {@link #getElementImports2(Object)}.
     * 
     * @param handle
     *            the package
     * @return the collection of ElementImports
     * @see #getElementImports2(Object)
     */
    Collection getElementImports(Object handle);

    /**
     * Return the entry Action for a State.
     *
     * @param handle the State
     * @return the entry
     */
    Object getEntry(Object handle);

    /**
     * Return the Enumeration which owns a given EnumerationLiteral.
     * 
     * @param handle the EnumerationLiteral
     * @return the owning Enumeration
     */
    Object getEnumeration(Object handle);
    
    /**
     * Return the List of enumeration literals for a UML Enumeration.
     *
     * @param handle the Enumeration
     * @return List of the Enumeration literals.
     */
    List getEnumerationLiterals(Object handle);

    /**
     * Return the exit action to a State.
     *
     * @param handle the state
     * @return the exit action
     */
    Object getExit(Object handle);

    /**
     * Get the Expression belonging to a Argument, Guard, ChangeEvent or
     * TimeEvent.
     * 
     * @param handle
     *            the Object to get the Expression from
     * @return Object the Expression
     */
    Object getExpression(Object handle);

    /**
     * Return ModelElements which have a given Stereotype applied. 
     * (ie. the Stereotype's extendedElements collection)
     * 
     * @param handle stereotype for which to return extended elements
     * @return collection of model elements extended by given stereotype
     */
    public Collection getExtendedElements(Object handle);
    
    /**
     * Return all Extends of a UseCase or ExtensionPoint.
     *
     * @param handle the UseCase or the ExtensionPoint
     * @return the extends
     */
    Collection getExtends(Object handle);

    /**
     * Return all extenders of a UseCase.
     *
     * @param handle  the UseCase
     * @return the extends
     */
    Collection getExtenders(Object handle);

    /**
     * Get the UseCase which is the extension of an Extend.
     *
     * @param handle the extend
     * @return The extension
     */
    Object getExtension(Object handle);

    /**
     * Return the ExtensionPoint at given index-number.
     *
     * @param handle Extend
     * @param index int
     * @return ExtensionPoint
     */
    Object getExtensionPoint(Object handle, int index);

    /**
     * Return all ExtensionPoints of a UseCase or Extend.
     * 
     * @param handle
     *            the UseCase or the Extend
     * @return the Collection of ExtensionPoints. If the argument was an Extend,
     *         the Collection is ordered, ie a List.
     */
    Collection getExtensionPoints(Object handle);

    /**
     * Get the List of Features owned by a Classifier.
     *
     * @param handle Classifier to retrieve from.
     * @return List of Features
     */
    List getFeatures(Object handle);

    /**
     * Get the Generalization connecting two GeneralizableElements.
     * Returns null if there is none.
     *
     * @param handle the child
     * @param parent is the parent
     * @return The generalization or null if none found
     */
    Object getGeneralization(Object handle, Object parent);

    /**
     * Return collection of Generalizations that the given GeneralizableElement
     * participates in as a child.  Use {@link #getSpecializations(Object)} to
     * find the Generalizations that it particpates in as a parent.
     *
     * @param handle GeneralizableElement to retrieve from.
     * @return Generalizations
     */
    Collection getGeneralizations(Object handle);

    /**
     * Get the guard for a Transition.
     *
     * @param handle the Transition
     * @return Object
     */
    Object getGuard(Object handle);

    /**
     * Return the Icon of a Stereotype.
     *
     * @param handle the Stereotype to get the Icon from
     * @return the Icon
     */
    Object getIcon(Object handle);

    /**
     * Return the includes for a UseCase.<p>
     *
     * @param handle the UseCase
     * @return a Collection containing the Includes
     */
    Collection getIncludes(Object handle);

    /**
     * Return Includes which have the given UseCase as an addition.
     * <em>NOTE:</em> The association is not navigable in this direction,
     * so this isn't guaranteed to work in the general case, but it will
     * return all appropriate elements which can be found in the current
     * model repository.
     *
     * @param handle the UseCase
     * @return the includes as a Collection
     */
    Collection getIncluders(Object handle);

    /**
     * Return the incoming Transitions for some StateVertex, Transition, Action,
     * or Guard.  For a Transition, the incomings of its source StateVertex are
     * used.  For an Action or Guard, the owning Transition is found and then
     * the algorithm for Transitions is applied.
     * 
     * @param handle
     *            a StateVertex, Transition, Action, or Guard
     * @return Collection of Transitions
     */
    Collection getIncomings(Object handle);

    /**
     * Return the initial value of an Attribute.
     *
     * @param handle the Attribute
     * @return an Expression representing the initial value
     */
    Object getInitialValue(Object handle);

    /**
     * Return the instance of an AttributeLink or LinkEnd.
     *
     * @param handle the attribute link or link end
     * @return initial value
     */
    Object getInstance(Object handle);

    /**
     * Return the Instances associated with a Classifier.  
     *
     * @param handle the classifier
     * @return Collection
     */
    Collection getInstances(Object handle);

    /**
     * Return the collection of States for a ClassifierInState.
     *
     * @param handle the classifierInState
     * @return Collection
     */
    Collection getInStates(Object handle);

    /**
     * Return the Interaction for a Message.
     *
     * @param handle the Message
     * @return the interaction
     */
    Object getInteraction(Object handle);

    /**
     * Return the Interactions belonging to a Collaboration.
     *
     * @param handle the Collaboration
     * @return Collection
     */
    Collection getInteractions(Object handle);

    /**
     * Return the internalTransitions belonging to a State.
     *
     * @param handle the State
     * @return Collection
     */
    Collection getInternalTransitions(Object handle);

    /**
     * Return the messages belonging to an Interaction, AssociationRole, or
     * Action.
     * 
     * @param handle
     *            the candidate element
     * @return Collection of Messages
     */
    Collection getMessages(Object handle);

    /**
     * Return the Messages which are successors to the given Message.
     *
     * @param handle the message
     * @return Collection
     */
    Collection getSuccessors(Object handle);

    /**
     * Return the Messages that are activated by the given Message.
     *
     * @param handle Message
     * @return the Collection of Messages
     */
    Collection getActivatedMessages(Object handle);

    /**
     * Return the Messages received by the given ClassifierRole.
     *
     * @param handle the ClassifierRole
     * @return Collection
     */
    Collection getReceivedMessages(Object handle);

    /**
     * Return the Messages sent by the given ClassifierRole.
     *
     * @param handle the ClassifierRole
     * @return Collection
     */
    Collection getSentMessages(Object handle);

    /**
     * Get the innermost containing Model of a ModelElement.
     * <p>
     * If no containing Model can be found, the outermost containing
     * ModelElement is returned (which could be the original element itself if
     * it has no owner).
     * <p>
     * This method is intended to return the innermost
     * containing Model, but there are circumstances where one may
     * really expect the outermost containing Model or root package to
     * be returned (e.g. where
     * it's being used to test if two elements are from disjoint namespace
     * hierarchies). Hence, there are two separate methods
     * with different semantics. The method {@link #getRoot(Object)}
     * returns the outermost containing object. 
     * <p>
     * TODO: Note also that some code uses
     * org.argouml.kernel.Project#getRoot() to navigate down from the top
     * instead of up from the bottom. We need to use a consistent algorithm for
     * this. - tfm 20070724
     * 
     * @param handle
     *                to the model element.
     * @return model for the model element.
     */
    Object getInnerContainingModel(Object handle);

    /**
     * Get the top level containing element, i.e. the element which has no
     * owner.  For well-formed models, this will typically be a UML Package,
     * often a Model.
     * 
     * @param handle
     *                to the model element.
     * @return the top level element.
     */
    Object getRoot(Object handle);
    
    /**
     * Return a collection of all Elements which have no owner (ie top level
     * elements).
     * <p>
     * For ArgoUML created models this will typically just return a single 
     * top level Model model element, but models created by other tools may
     * have other top level elements.
     */
    Collection getRootElements();

    /**
     * Get the ModelElement which is one of:
     * <ul>
     * <li>the importElement of an ElementImport </li>
     * <li>the modelElement of a TaggedValue</li>
     * <li>the modelElement of a TemplateParameter</li>
     * <ul>
     * 
     * @param handle
     *            an Element Import or TaggedValue.
     * @return the model element
     */
    Object getModelElement(Object handle);

    /**
     * Get the Multiplicity of an AssociationEnd, ClassifierRole,
     * StructuralFeature, or TagDefinition.
     * 
     * @param handle
     *            model element to retrieve from.
     * @return multiplicity
     */
    Object getMultiplicity(Object handle);

    /**
     * Get the Comments which annotate a ModelElement.
     *
     * @param handle the ModelElement that we are getting the comments of
     * @return the Collection of Comments
     */
    Collection getComments(Object handle);

    /**
     * Get the ModelElements (annotatedElements) which are annotated by a
     * Comment.
     * 
     * @param handle
     *            the Comment that we are getting the model elements of
     * @return a Collection of ModelElements
     */
    Collection getAnnotatedElements(Object handle);

    /**
     * Get the communicationConnection of a Message.
     * 
     * @param handle
     *            the Message for which to find the communicationConnection
     * @return Message which is the communicationConnection or null
     */
    Object getCommunicationConnection(Object handle);

    /**
     * Get the communicationLink of a Stimulus.
     * 
     * @param handle
     *            the Stimulus for which to find the communicationLink
     * @return Link which is the communicationLink or null if none.
     */
    Object getCommunicationLink(Object handle);

    /**
     * Get the collaborations for a Classifier or Operation. This returns the
     * Collaborations which have the given Classifier or Operation as their
     * representedClassifier or representedOperation.
     * 
     * @param handle
     *            the model element that we are getting the collaborations of.
     * @return Collection of Collaborations
     */
    Collection getCollaborations(Object handle);

    /**
     * Get the componentInstance of an Instance.
     *
     * @param handle the Instance
     * @return the ComponentInstance
     */
    Object getComponentInstance(Object handle);

    /**
     * Return the constrainingElements of a Collaboration.
     *
     * @param handle the Collaboration
     * @return Collection of ModelElements
     */
    Collection getConstrainingElements(Object handle);

    /**
     * Return the constrainedElements of a Constraint.
     *
     * @param handle the Constraint
     * @return List of ModelElements
     */
    List getConstrainedElements(Object handle);

    /**
     * Return the Constraints for the given ModelElement.
     *
     * @param handle the ModelElement
     * @return the collection of all Constraints
     */
    Collection getConstraints(Object handle);

    /**
     * Return the container for the given modelelement. If you know the type of
     * ModelElement you are working with, you should use the appropriate more
     * specific function (e.g. getAssociation for an AssociationEnd).<p>
     * 
     * This is principally intended as a convenience function for generic code
     * that deals with a variety of element types that just wants a parent of
     * some type without really caring what it is.<p>
     * 
     * The container is the owner of the modelelement. It will be null for
     * elements that don't have an owner. All elements except for the root
     * element in a project should have an owner. The root element is always a
     * model.<p>
     * 
     * In the future, this function could return the container of Figs too.
     * 
     * @param handle
     *            is the base
     * @return Object
     */
    Object getModelElementContainer(Object handle);
    
    /**
     * Return all composite components of given ModelElement.
     * This is the inverse of getModelElementContainer and
     * that method will return handle for all elements in the
     * Collection.<p>
     * 
     * @param handle ModelElement for which to fetch components
     * @return collection of components contained by ModelElement
     */
    List getModelElementContents(Object handle);

    /**
     * Return all ModelElements associated with the given ModelElement
     * by a MOF level association (not an just an association in the user 
     * model, although it includes those too).<p>
     * 
     * This is useful for traversing the model to find elements which
     * are related in ANY way to the current element.
     * 
     * @param handle ModelElement for which to find associated elements
     * @return collection of elements associated with ModelElement
     */
    List getModelElementAssociated(Object handle);

    /**
     * Return the CompositeState or Component that is the container of the
     * given StateVertex or ElementResidence, respectively.<p>
     * 
     * ElementResidence support is new in UML 1.4 (was
     * getImplementationLocation)
     * 
     * @param handle
     *            the StateVertex or ElementResidence
     * @return the CompositeState or Component that is the container
     */
    Object getContainer(Object handle);

    /**
     * Return all ModelElements contained in a Partition.
     *
     * @param handle the Partition
     * @return Collection of ModelElements
     */
    Collection getContents(Object handle);

    /**
     * Return the context of a StateMachine or Interaction.
     * 
     * @param handle
     *            the StateMachine or the Interaction
     * @return the ModelElement representing the context, or null if no context
     *         set.
     */
    Object getContext(Object handle);

    /**
     * Return the BehavioralFeatures which are contexts for a given Signal (ie
     * have this Signal as a raisedSignal).
     * 
     * @param handle
     *            the Signal
     * @return Collection of BehavioralFeatures
     */
    Collection getContexts(Object handle);

    /**
     * Return the CreateActions that create/instantiate the given Classifier.
     * 
     * @param handle
     *            the Classifier
     * @return a collection of CreateActions
     */
    Collection getCreateActions(Object handle);

    /**
     * Get the defaultValue of a Parameter.
     *
     * @param handle the Parameter that we are getting the defaultvalue from
     * @return an Expression representing the default value
     */
    Object getDefaultValue(Object handle);

    /**
     * Get deferrable events of a State.
     *
     * @param handle the state that we are getting the deferrable event from
     * @return Collection of Events
     */
    Collection getDeferrableEvents(Object handle);

    /**
     * Return all Components that are deployed inside the given Node.
     * 
     * @param handle the given Node
     * @return Collection of Components
     */
    Collection getDeployedComponents(Object handle);

    /**
     * Return all Nodes in which the given Component is deployed.
     * 
     * @param handle the Componet
     * @return Collection of Nodes
     */
    Collection getDeploymentLocations(Object handle);

    /**
     * Get the discriminator for a Generalization.
     * 
     * @param handle
     *            the Generalization
     * @return the discriminator a String
     * @deprecated for 0.25.4 by tfmorris. Discriminators have been removed from
     *             UML 2.x.
     */
    @Deprecated
    Object getDiscriminator(Object handle);

    /**
     * Get the dispatchAction of a Stimulus.
     *
     * @param handle the stimulus that we are getting the dispatchaction of
     * @return the dispatchaction (or null)
     */
    Object getDispatchAction(Object handle);

    /**
     * Return the Action which is the doActivity for the given State.
     *
     * @param handle the state
     * @return the do activity
     */
    Object getDoActivity(Object handle);

    /**
     * The collection of imported elements of the given package.
     *
     * @param pack package to retrieve from.
     * @return A Collection with the imported elements.
     */
    Collection getImportedElements(Object pack);

    /**
     * Return the imported element from a ElementImport.
     * 
     * @param elementImport the given ElementImport
     * @return the ModelElement that was imported
     */
    Object getImportedElement(Object elementImport);

    /**
     * Return the Links of a given Association.
     *
     * @param handle the Association
     * @return the collection of Links
     */
    Collection getLinks(Object handle);

    /**
     * Return the LinkEnds of a given Instance or AssociationEnd.
     *
     * @param handle the candidate
     * @return the collection of LinkEnds
     */
    Collection getLinkEnds(Object handle);

    /**
     * Return the location of an ExtensionPoint.
     *
     * @param handle ExtensionPoint
     * @return the location
     */
    String getLocation(Object handle);

    /**
     * Return the Methods of an Operation.
     *
     * @param handle the operation that we are getting the methods of
     * @return Collection of Methods
     */
    Collection getMethods(Object handle);

    /**
     * Get the namespace of a ModelElement.  Note: the namespace/ownedElement
     * association is a composition, so ModelElements which are owned by other
     * elements will have a null namespace attribute.
     *
     * @param handle the model element that we are getting the namespace of
     * @return the namespace (or null)
     */
    Object getNamespace(Object handle);

    /**
     * Get the node instance of a ComponentInstance.
     *
     * @param handle the model element that we are getting the node instance of
     * @return the node instance
     */
    Object getNodeInstance(Object handle);

    /**
     * Get ObjectFlowStates which have this Classifier as their type.
     *
     * @param handle the classifier
     * @return collection of ObjectFlowStates
     */
    Collection getObjectFlowStates(Object handle);

    /**
     * Get the Operation of a CallAction or CallEvent.
     *
     * @param handle the model element that we are getting the operation of
     * @return the Operation
     */
    Object getOperation(Object handle);

    /**
     * Get the CallEvents which are occurrences of an Operation.
     * 
     * @param handle the Opration
     * @return Collection of CallEvents
     */
    Collection getOccurrences(Object handle);

    /**
     * Get the list of Operations owned by a Classifier.
     *
     * @param handle classifier to examine.
     * @return list of operations.
     */
    List getOperations(Object handle);

    /**
     * Get the list of operations and receptions owned by a Classifier.
     *
     * @param handle classifier to examine.
     * @return list of operations.
     */
    List getOperationsAndReceptions(Object handle);

    
    //////// Javadoc reviewed above this point - tfm /////////////

    

    /**
     * Return the next AssociationEnd in the owning Association. If this is the
     * last AssociationEnd, the search will wrap and return the first end so
     * that all AssociationEnds may be visited by repeatedly calling this
     * method.
     * 
     * @param handle
     *            the AssociationEnd
     * @return the next AssociationEnd in sequence
     */
    Object getNextEnd(Object handle);
    
    /**
     * Get ordering of an association end.
     *
     * @param handle association end to retrieve from
     * @return ordering
     */
    Object getOrdering(Object handle);

    /**
     * Return the collection of Transitions outgoing from the given 
     * stateVertex.
     *
     * @param handle statevertex
     * @return Collection
     */
    Collection getOutgoings(Object handle);

    /**
     * Get the collection of Associations Ends connected to this association 
     * end.
     *
     * @param handle association end to start from
     * @return A Collection with all connected association ends.
     */
    Collection getOtherAssociationEnds(Object handle);

    /**
     * Get the collection of Link Ends connected to this link end.
     *
     * @param handle link end to start from
     * @return A Collection with all connected link ends.
     */
    Collection getOtherLinkEnds(Object handle);

    /**
     * The collection of owned elements of the the namespace.
     *
     * @param handle namespace to retrieve from.
     * @return A Collection with the owned elements.
     */
    Collection getOwnedElements(Object handle);

    /**
     * Get the powertype of a generalization.
     *
     * @param handle generalization
     * @return powertype
     */
    Object getPowertype(Object handle);

    /**
     * Get the powertype ranges of a classifier.
     *
     * @param handle classifier to retrieve from
     * @return collection of poertype ranges
     */
    Collection getPowertypeRanges(Object handle);

    /**
     * Get the predecessors of a message.
     *
     * @param handle message to retrieve from
     * @return collection of predecessors
     */
    Collection getPredecessors(Object handle);

    /**
     * Get the qualified attributes of an association end.
     *
     * @param handle association end to retrieve from
     * @return List of qualifiers.
     */
    List getQualifiers(Object handle);

    /**
     * Determine if the passed parameter has a RETURN direction kind.
     * 
     * TODO: This is a duplicate of {@link #isReturn(Object)}
     * 
     * @return true if it is a return direction kind
     * @param handle the parameter
     */
    boolean hasReturnParameterDirectionKind(Object handle);

    /**
     * Return the Package that is connected by the given ElementImport.
     *
     * @param handle the ElementImport
     * @return the Package
     */
    Object getPackage(Object handle);

    /**
     * Get the parameter of a TemplateParameter.
     * 
     * @param handle
     *            the TemplateParameter
     * @return parameter the ModelElement which provides the type and other info
     *         for the given parameter.
     */
    Object getParameter(Object handle);
    
    /**
     * Get a specific parameter of a Behavioral Feature or Event.
     *
     * @param handle behavioral feature to retrieve from
     * @param n parameter number
     * @return parameter.
     */
    Object getParameter(Object handle, int n);

    /**
     * Get the parameters of a Behavioral Feature, Event, or Object Flow State,
     * or all the parameters which have a given classifier as their type. The
     * collection returned for the first two parameter types (Behavioral Feature
     * and Event) is an ordered List, while the latter two return an unordered
     * collection.
     * 
     * @param handle
     *            operation to retrieve from
     * @return A Collection with the parameters.
     */
    Collection getParameters(Object handle);

    /**
     * Get an ordered list of parameters of a Behavioral Feature
     * or Event.
     * 
     * @param handle
     *            operation to retrieve from
     * @return List of parameters.
     */
    List getParametersList(Object handle);


    /**
     * Get the general end of a generalization. This was getParent in UML 1.x.
     * 
     * @param handle
     *            generalization.
     * @return the general end.
     * @since 0.25.4
     */
    Object getGeneral(Object handle);
    
    /**
     * Return the raised signals of an operation.
     *
     * @param handle the operation
     * @return raised signals
     * @deprecated by Bob Tarling in 0.30.1 use the UML2 replacement
     * of getRaisedExceptions
     */
    Collection getRaisedSignals(Object handle);
    
    /**
     *  Return the raised exceptions of an operation (UML 2 only).
     *  
     *  @param handle the operation
     *  @return raised exceptions
     */
    Collection getRaisedExceptions(Object handle);

    /**
     * Return the receptions of a signal.
     *
     * @param handle the signal
     * @return receptions
     */
    Collection getReceptions(Object handle);

    /**
     * Return the recurrence iteration expression of an action.
     *
     * @param handle the action.
     * @return the recurrence
     */
    Object getRecurrence(Object handle);

    /**
     * Return the represented classifier of a collaboration.
     *
     * @param handle the collaboration
     * @return represented classifier
     */
    Object getRepresentedClassifier(Object handle);

    /**
     * Return the represented operation of a collaboration.
     *
     * @param handle the collaboration
     * @return represented operation
     */
    Object getRepresentedOperation(Object handle);

    /**
     * Return the script belonging to a given action.
     *
     * @param handle the action
     * @return the script
     */
    Object getScript(Object handle);

    /**
     * Return the sender object of a stimulus or a message.
     *
     * @param handle the stimulus or message
     * @return the sender
     */
    Object getSender(Object handle);

    /**
     * Return the Signal of a SendAction, SignalEvent or Reception.
     *
     * @param handle the object
     * @return the signal
     */
    Object getSignal(Object handle);

    /**
     * Get the resident element.
     *
     * @param handle the ElementResidence
     * @return the resident ModelElement
     */
    Object getResident(Object handle);

    /**
     * Return the collection of ElementResidences for a given Component.
     *
     * @param handle the Component
     * @return the Collection of ElementResidence
     */
    Collection getResidentElements(Object handle);

    /**
     * Return a collection with all residents belonging to the given
     * Component, ComponentInstance or NodeInstance.
     *
     * @param handle the Component, NodeInstance, ComponentInstance
     * @return the Collection of resident: <ul><li> 
     *          ComponentInstances (in case of a NodeInstance)<li>
     *          Instances (in case of a ComponentInstance)<li>
     *          ModelElements (in case of a Component)</ul>
     */
    Collection getResidents(Object handle);

    /**
     * Return the source for a given transition.
     *
     * @param handle the transition
     * @return Object (MStateVertex)
     */
    Object getSource(Object handle);

    /**
     * Return the source for some given flow or a UML2 DirectedRelationship.
     *
     * @param handle the flow or DirectedRelationship
     * @return Collection
     */
    Collection getSources(Object handle);

    /**
     * Return the sourceflows of a model element.
     *
     * @param handle the model element
     * @return a collection of sourceflows
     */
    Collection getSourceFlows(Object handle);

    /**
     * The collection of Specializations from a GeneralizableElement.
     *
     * @param handle GeneralizableElement to retrieve from.
     * @return Collection of Specializations.
     */
    Collection getSpecializations(Object handle);

    /**
     * Return the state machine belonging to some given state or transition
     * If you need to find the StateMachine for an internal transition,
     * or for ANY state,
     * use StateMachinesHelper.getStateMachine() instead.
     *
     * @param handle the state or transition
     * @return Object
     */
    Object getStateMachine(Object handle);

    /**
     * Return the state belonging to some given transition.
     *
     * @param handle the transition
     * @return Object
     */
    Object getState(Object handle);

    /**
     * Return the states from a deferable event.
     *
     * @param handle the event
     * @return Object
     */
    Collection getStates(Object handle);

    /**
     * Return the stereotypes belonging to some given model element.<p>
     *
     * Note! For UML version 1.3 there can only be one stereotype
     * per model element. This means that the returned Collection will
     * only have zero or one elements. Don't write any code that rely
     * on this! Consider it to be a Collection with zero or more
     * elements as it will be in later UML versions.
     *
     * @param handle The model element.
     * @return a Collection with all stereotypes or empty if none.
     */
    Collection getStereotypes(Object handle);

    /**
     * Return the stimuli belonging to some given link.
     *
     * @param handle the link
     * @return Object
     */
    Collection getStimuli(Object handle);

    /**
     * Return the Stimuli that are received by the given Instance.
     *
     * @param handle the Instance
     * @return the collection of stimuli
     */
    Collection getReceivedStimuli(Object handle);

    /**
     * Return the Stimuli that are send by the given Instance.
     *
     * @param handle the Instance
     * @return the collection of stimuli
     */
    Collection getSentStimuli(Object handle);

    /**
     * Return all direct subvertices belonging to the given
     * CompositeState.
     *
     * @param handle the composite state
     * @return Collection
     */
    Collection getSubvertices(Object handle);

    /**
     * Return the submachine of a submachine state.
     *
     * @param handle the submachine state
     * @return submachine
     */
    Object getSubmachine(Object handle);

    /**
     * Return the submachine of a submachine state.
     *
     * @param handle the submachine state
     * @return submachine
     */
    Collection getSubmachineStates(Object handle);

    /**
     * The collection of SupplierDependencies from a ModelElement.
     *
     * @param handle model element.
     * @return A Collection with the supplier dependencies.
     */
    Collection getSupplierDependencies(Object handle);

    /**
     * The top of a state machine.
     *
     * @param handle the state machine
     * @return the top
     */
    Object getTop(Object handle);

    /**
     * Get the transition of a guard or action.
     *
     * @param handle the guard or action
     * @return the transition
     */
    Object getTransition(Object handle);

    /**
     * Get the trigger of a transition.
     *
     * @param handle the transition
     * @return the trigger
     */
    Object getTrigger(Object handle);

    /**
     * The type of a StructuralFeature, AssociationEnd, Parameter,
     * ObjectFlowState, TagDefinition, TaggedValue or ClassifierInState.
     *
     * @param handle the StructuralFeature, AssociationEnd, Parameter,
     *  ObjectFlowState, TagDefinition, TaggedValue or ClassifierInState
     * @return the type
     */
    Object getType(Object handle);

    /**
     * Return all TaggedValues typed by a TagDefinition
     *
     * @param handle the TagDefinition
     * @return collection of TaggedValues
     */
    Collection getTypedValues(Object handle);

    /**
     * Return the target of a Transition.
     *
     * @param handle the transition
     * @return Object
     */
    Object getTarget(Object handle);

    /**
     * Return the targets of a UML 2.x DirectedRelationship.
     *
     * @param handle the relationship
     * @return Object
     */
    Collection getTargets(Object handle);
    
    /**
     * Return the target scope of a StructuralFeature or AssociationEnd.
     * 
     * @param handle the model element
     * @return Object
     * @deprecated for 0.25.4 by tmorris. If being used with an AssociationEnd
     *             as an argument, {@link #isStatic(Object)} can be used. For
     *             StructuralFeatures, this has been removed from UML 2.x and
     *             should no longer be used.
     */
    @Deprecated
    Object getTargetScope(Object handle);

    /**
     * Return the targetflows of a model element.
     *
     * @param handle the model element
     * @return a collection of targetflows
     */
    Collection getTargetFlows(Object handle);
    
    /**
     * Get the list of TemplateParameters for a ModelElement that is
     * parameterized.
     * 
     * @param handle
     *            the ModelElement
     * @return a list of TemplateParameters
     */
    List getTemplateParameters(Object handle);
    
    /**
     * Return the defaultElement for a TemplateParameter.
     * @param handle the TemplateParameter
     * @return the default ModelElement or null
     */
    Object getDefaultElement(Object handle);
    
    /**
     * Return the ModelElement that owns a TemplateParameter.
     * @param handle the TemplateParameter
     * @return the containing ModelElement or null if none
     */
    Object getTemplate(Object handle);
    
    /**
     * Return the Binding associated with a TemplateArgument.
     * @param handle the TemplateArgument
     * @return the Binding
     */
    Object getBinding(Object handle);

    /**
     * Get the list of TemplateArguments associated with a Binding.
     * 
     * @param handle the binding
     * @return the list of TemplateArguments
     */
    List getArguments(Object handle);

    /**
     * Return the upper bound of the multiplicity of the given element
     * (AssociationEnd, Multiplicity or MultiplicityRange). If the argument is
     * an AssociationEnd and it has a null multiplicity, the value 0 will be
     * return.
     * 
     * @param handle a model element
     * @return int upper bound. A value of -1 corresponds to the special UML
     *         value 'unlimited' represented graphically by '*'.
     */
    int getUpper(Object handle);

    /**
     * Return the UseCase of an ExtensionPoint.
     *
     * @param handle the ExtensionPoint
     * @return a UseCase
     */
    Object getUseCase(Object handle);

    /**
     * Return the lower bound of the multiplicity of the given element
     * (AssociationEnd, Multiplicity or MultiplicityRange).
     *
     * @param handle the model element
     * @return int
     */
    int getLower(Object handle);

    /**
     * Return the transitions belonging to the given handle. The handle can be
     * a StateMachine, CompositeState, or Event. For a StateMachine, the
     * Transitions transitions belonging to StateMachine are returned. For a
     * CompositeState, its internalTransitions are returned. For an Event, all
     * transitions triggered by this event will be given back.
     * 
     * @param handle
     *            a StateMachine, CompositeState, or Event
     * @return Collection of Transitions
     */
    Collection getTransitions(Object handle);

    /**
     * Return all StructuralFeatures of a given Classifier.
     *
     * @param handle the classifier to query.
     * @return Collection of StructuralFeatures
     */
    List getStructuralFeatures(Object handle);

    /**
     * Return the Specification of a given Reception or Operation.
     * If you need the Specification of a Method, use the method
     * in CoreHelper which returns an object instead of a string.
     * @see CoreHelper#getSpecification(Object)
     *
     * @param handle the Reception or Operation
     * @return String the Specification
     */
    String getSpecification(Object handle);

    /**
     * Return specifications for an AssociationEnd or for a Classifier it
     * returns all Interfaces of which this class is a realization.<p>
     * 
     * When returning Interfaces it follows all Abstraction depencies
     * which have the "realize" Stereotype and returns the Interfaces
     * at the far end.
     * 
     * @param handle
     *            the class you want to have the interfaces for
     * @return a collection of the Interfaces
     */
    Collection getSpecifications(Object handle);

    /**
     * Return the suppliers of a Dependency.
     *
     * @param handle the dependency
     * @return a collection of the suppliers
     */
    Collection getSuppliers(Object handle);

    /**
     * Returns the action belonging to some message or argument.
     *
     * @param handle the message or argument.
     * @return the action
     */
    Object getAction(Object handle);


    /**
     * Get the list of Actions in an ActionSequence.
     *
     * @param handle the ActionSequence.
     * @return the List of Actions
     */
    List getActions(Object handle);


    /**
     * Get the ActionSequence, if any, that owns this Action.
     *
     * @param handle the Action.
     * @return the ActionSequence or null
     */
    Object getActionSequence(Object handle);

    /**
     * Return the activator belonging to some message.
     *
     * @param handle the message
     * @return the activator
     */
    Object getActivator(Object handle);
    
    /**
     * Return the activityGraph of the given partition.
     * 
     * @param handle the partition
     * @return the activityGraph
     */
    Object getActivityGraph(Object handle);

    /**
     * Return the actual arguments for a given action.
     *
     * @param handle the action
     * @return the actual arguments
     */
    List getActualArguments(Object handle);

    /**
     * Return an addition for a given include.
     *
     * @param handle the include
     * @return the addition
     */
    Object getAddition(Object handle);

    /**
     * Return the AggregationKind of a given AssociationEnd.
     *
     * @param handle the AssociationEnd
     * @return the AggregationKind
     */
    Object getAggregation(Object handle);

    /**
     * Return the alias of an ElementImport.
     * 
     * @param handle the ElementImport
     * @return the alias Name
     */
    String getAlias(Object handle);

    /**
     * Return all associated classes (classes connected by an
     * Association) for some given classifier.
     * Returns an empty collection if the given argument handle is not
     * a classifier.  The given parameter is included in the returned
     * collection if it has a self-referencing association.
     *
     * @param handle the classifier
     * @return Collection
     */
    Collection getAssociatedClasses(Object handle);

    ////////////////////////////////////////////////////////////////

    /**
     * The name of a model element or some diagram part.
     *
     * @param handle that points out the object.
     * @return the name
     */
    String getName(Object handle);

    /**
     * Return the owner of a Feature, TagDefinition, Attribute, or
     * TemplateParameter. For a qualifier Attribute of an AssociationEnd, it
     * returns the owning AssociationEnd. For a TemplateParameter ot returns the
     * owning template ModelElement.
     * 
     * @param handle
     *            the Feature, TagDefinition, Attribute, or TemplateParameter
     * @return the owner
     */
    Object getOwner(Object handle);

    /**
     * Return the tag of a tagged value, i.e. the name 
     * of the TagDefinition which is the type of the given TaggedValue.
     *
     * @param handle the given tagged value
     * @return The found tag as a String.
     */
    String getTag(Object handle);

    /**
     * Return the tagged values iterator of a model element.
     *
     * @param handle The tagged values belong to this.
     * @return The tagged values iterator
     */
    Iterator getTaggedValues(Object handle);

    /**
     * Return the TaggedValues of a ModelElement.
     *
     * @param handle the ModelElement
     * @return the Collection of TaggedValues
     */
    Collection getTaggedValuesCollection(Object handle);

    /**
     * Return the tagged value with a specific tag.
     *
     * @param handle The model element the tagged value belongs to.
     * @param name The tag name.
     * @return The found tag, null if not found
     */
    Object getTaggedValue(Object handle, String name);

    /**
     * Return the value of a tagged value with a specific tag.
     *
     * @param handle The model element that the tagged value belongs to.
     * @param name The tag name.
     * @return The value of the found tag. "" if not found.
     */
    String getTaggedValueValue(Object handle, String name);

    /**
     * Return the key (tag) of some tagged value.
     *
     * TODO: This does exactly the same as getTag(Object). Remove one of them.
     *
     * @param handle The tagged value.
     * @return The found value as String.
     */
    String getTagOfTag(Object handle);

    /**
     * Return the value of some UML Object.
     *
     * @param handle Object
     * @return Object the exact type depends on the handle type
     * (String, Expression, Instance, TaggedValue...)
     */
    Object getValue(Object handle);

    /**
     * Return the value of some tagged value.
     *
     * @param handle The tagged value.
     * @return The found value as String.
     */
    String getValueOfTag(Object handle);

    /**
     * Return the referenceValue collection for a taggedValue.
     * 
     * @param taggedValue the tagged value to read from
     * @return collection of referenceValues for the given TaggedValue
     */
    Collection getReferenceValue(Object taggedValue);

    /**
     * Return the dataValue collection for a taggedValue.
     * 
     * @param taggedValue taggedValue the tagged value to read from
     * @return a collection of dataValues for the given TaggedValue
     */
    Collection getDataValue(Object taggedValue);
    
    /**
     * Return the ID of this element.
     * 
     * This is guaranteed to always return a value, but the
     * uniqueness and stability of the ID may vary by implementation.
     * The original implementation guaranteed universally unique
     * (across time and space) identifiers which were also stable
     * across reloads of an XMI file.
     *
     * @param element ModelElement to retrieve ID for.
     * @return UUID
     */
    String getUUID(Object element);

    /**
     * Return the visibility of this element.
     *
     * @param handle a model element
     * @return the visibility in the form of a VisibilitKind object
     */
    Object getVisibility(Object handle);

    /**
     * Get the partitions from a container.
     *
     * @param container The container.
     * @return The partitions (a collection).
     */
    Collection getPartitions(Object container);

    /**
     *  Return the StubState's referenceState.
     *  @param o Stub State
     *  @return referenced state
     */
    String getReferenceState(Object o);

    /**
     * Return a named object in the given object by calling it's lookup method.
     *
     * @param handle the object that we search through
     * @param name of the model element
     * @return found object, null otherwise
     */
    Object lookupIn(Object handle, String name);

    /**
     * Return the name of the UML Model class.<p>
     *
     * @param handle The object to check.
     * @return classname of modelelement
     */
    String getUMLClassName(Object handle);

    /**
     * Recognizer for Arguments.
     *
     * @param modelElement candidate
     * @return true if an argument.
     */
    boolean isAArgument(Object modelElement);

    /**
     * Return a tooltip that should be shown for the given model element.<p>
     *
     * @param modelElement The model element for which to gerneate a tip
     * @return the tip
     */
    String getTipString(Object modelElement);

    /**
     * Return a textual representation of the given model element.<p>
     * All implementations should return the same value as would be expected
     * from the toString() method of the NSUML class MModelElement.
     *
     * @param modelElement The model element for which to convert to text
     * @return the textual representation of the model element
     */
    String toString(Object modelElement);

    /**
     * @param target The TimeEvent.
     * @return TimeExpression
     */
    Object getWhen(Object target);

    /**
     * @param target The ChangeEvent.
     * @return BooleanExpression
     */
    Object getChangeExpression(Object target);

    /**
     * Recognizer for TagDefinition. Implemented only in UML 1.4+ metamodels.
     *
     * @param handle candidate
     * @return true if handle is a TagDefinition
     */
    boolean isATagDefinition(Object handle);

    /**
     * Return the TagDefinitions owned by the given Stereotype.
     *
     * @param handle Stereotype to fetch TagDefinitions for
     * @return collection of TagDefinitions
     */
    Collection getTagDefinitions(Object handle);

    /**
     * Return the TagDefinition for a TaggedValue.
     *
     * @param handle TaggedValue to return TagDefinition for
     * @return TagDefinition for given element
     */
    Object getTagDefinition(Object handle);
    
    /**
     * Return the value of the isSynch attribute on an ObjectFlowState
     * @param handle the ObjectFlowState
     * @return true if this ObjectFlowState is a synch state.
     */
    boolean isSynch(Object handle);

    /**
     * Return the value of the isStatic attribute. This replaces the ScopeKind
     * enumeration of UML 1.x and is equivalent to a ScopeKind of CLASSIFIER.
     * <p>
     * For UML 1.4 it fetches the ownerScope for StructuralFeatures and the
     * targetScope for AssociationEnds. It returns <code>true</code> if the
     * value is <code>ScopeKind.SK_CLASSIFIER</code>.
     * 
     * @param handle the Feature or AssociationEnd
     * @return true if the element is static
     * @since 0.25.4
     */
    boolean isStatic(Object handle);
    
    /**
     * Return the value of the isReadOnly attribute. This replaces the
     * ChangeableKind enumeration of UML 1.x and is equivalent to a
     * ChangeableKind of FROZEN. The UML 1.x enum of ADD_ONLY has been removed
     * from UML 2.x.
     * 
     * @param handle
     *            the StructuralFeature
     * @return true if the element is readonly
     * @since 0.25.4
     */
    boolean isReadOnly(Object handle);
    
    
    /**
     * Get the names of all valid metatypes.
     * 
     * @return the list of valid metatype names.
     * @since 0.28
     */
    String[] getMetatypeNames();
    
    /**
     * Test whether the provided element is an instance of the given metatype.
     * 
     * @param metatypeName the name of the metatype given as a string. Use the
     *            name as given in the UML spec, not any specific
     *            implementation's renaming (ie Class, not UmlClass which has
     *            been renamed to avoid a Java conflict)
     * @param element the UML element to test
     * @return true if the element is an instance of the given metatype
     */
    boolean isA(String metatypeName, Object element);
    
    /**
     * @param element a PackageImport or ElementImport
     * @return the Namepace into which the PackageImport is pulling things
     */
//    Object getImportingNamespace(Object element);
    
    /**
     * @param element a PackageImport
     * @return the Package being imported
     */
//    Object getImportedPackage(Object element);
    
}
