/*  Ejemplos de OpenGL
 *	Alfredo Cofr E.
 *	acofre@dcc.uchile.cl
 */


/*
 * Los detalles importantes se encuentran dentro de las funciones DrawGLScene (dibujar),
 * InitGL (configuracin de OpenGL) y WinMain (sensores de teclado)
 * Este programa est basado en los progranas de Nehe Productions
 * _____________________________________________________________________________________
 *
 * Teclas:
 * Z: desplazamiento en -Z
 * A: desplazamiento en Z
 * Flecha Izq: desplazamiento en X
 * Flecha Der: desplazamiento en -X
 * Flecha Arriba: desplazamiento en -Y
 * Flecha Abajo: desplazamiento en Y
 * R: Reset
 * L: Enciende/Apaga la luz
 * I: Enciende/Apaga la simulacin de glLoadIdentity()
 * 1: Rotacin en X
 * 2: Rotacin en -X
 * 3: Rotacin en Y
 * 4: Rotacin en -Y
 * 5: Rotacin en Z
 * 6: Rotacin en -Z
 * Recomiendo partir apretando Z. Asi se desplaza el sistema de coordenadas
 * a lo largo del eje Z adonde lo podamos ver, puesto que inicialmente la 
 * camara se encuentra en (0,0,0). Mas explicaciones, en clase auxiliar.
 */

#include <windows.h>
#include <stdio.h>		
#include <gl\gl.h>			
#include <gl\glu.h>			
#include <gl\glaux.h>		
#include <math.h>
#include <stdarg.h>

HDC			hDC=NULL;		
HGLRC		hRC=NULL;		
HWND		hWnd=NULL;		
HINSTANCE	hInstance;		

bool	keys[256];			
bool	active=TRUE;		
bool	fullscreen=TRUE;	

GLfloat	xdesp = 0.0f;				
GLfloat	ydesp = 0.0f;				
GLfloat	zdesp = 0.0f;				
GLfloat	xdesp_ac = 0.0f;				
GLfloat	ydesp_ac = 0.0f;				
GLfloat	zdesp_ac = 0.0f;				
GLfloat xrot = 0.0f;
GLfloat yrot = 0.0f;
GLfloat zrot = 0.0f;
bool sw = false;
GLUquadricObj *quadratic; 
GLuint base;
bool light = true;
bool id_sw = true;
GLfloat LightAmbient[]= { 0.5f, 0.5f, 0.5f, 1.0f };
GLfloat LightDiffuse[]= { 1.0f, 1.0f, 1.0f, 1.0f };
GLfloat LightPosition[]= { 2.0f, 2.0f, 2.0f, 1.0f };

LRESULT	CALLBACK WndProc(HWND, UINT, WPARAM, LPARAM);	

GLvoid BuildFont(GLvoid)
{
	HFONT font;
	HFONT oldfont;
	base = glGenLists(96);
	font = CreateFont( -15,0,0,0,FW_BOLD,FALSE,FALSE,FALSE,
		ANSI_CHARSET,OUT_TT_PRECIS,CLIP_DEFAULT_PRECIS,
		ANTIALIASED_QUALITY,FF_DONTCARE|DEFAULT_PITCH,
		"Courier New");
	oldfont = (HFONT)SelectObject(hDC, font); 
	wglUseFontBitmaps(hDC, 32, 96, base); 
	SelectObject(hDC, oldfont); 
	DeleteObject(font); 
}

GLvoid KillFont(GLvoid){
	glDeleteLists(base, 96); 
}

GLvoid glPrint(const char *fmt, ...){
	char text[256];
	va_list ap;
	if (fmt==NULL)
		return;
	va_start(ap,fmt);
	vsprintf(text, fmt, ap);
	va_end(ap);
	glPushAttrib(GL_LIST_BIT);
	glListBase(base - 32);
	glCallLists(strlen(text), GL_UNSIGNED_BYTE, text);
	glPopAttrib();
}


GLvoid ReSizeGLScene(GLsizei width, GLsizei height)		
{
	if (height==0)										
	{
		height=1;										
	}

	glViewport(0,0,width,height);						

	glMatrixMode(GL_PROJECTION);						
	glLoadIdentity();									

	
	gluPerspective(45.0f,(GLfloat)width/(GLfloat)height,0.1f,100.0f);

	glMatrixMode(GL_MODELVIEW);							
	glLoadIdentity();									
}

//>>>>>>> Ver aqui

int InitGL(GLvoid)							// Funcion de configuracion de OpenGL
{
	glShadeModel(GL_SMOOTH);                // Enable smooth shading
	glClearColor(0.0f, 0.0f, 0.0f, 0.5f);   // Black background
	glClearDepth(1.0f);                     // Depth buffer setup
	glEnable(GL_DEPTH_TEST);                // Enables depth testing
	glDepthFunc(GL_LEQUAL);                 // The type of depth testing to do
	glHint(GL_PERSPECTIVE_CORRECTION_HINT, GL_NICEST);      // Really nice perspective calculations
	quadratic=gluNewQuadric(); // Create A Pointer To The Quadric Object
	gluQuadricNormals(quadratic, GLU_SMOOTH); // Create Smooth Normals
	gluQuadricTexture(quadratic, GL_TRUE); // Create Texture Coords
	BuildFont();
	glLightfv(GL_LIGHT1, GL_AMBIENT, LightAmbient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, LightDiffuse);
	glLightfv(GL_LIGHT1, GL_POSITION,LightPosition);
	glEnable(GL_LIGHT1);

	return true;                            // Initialization went OK
}

//>>>>>>> Ver aqui

int DrawGLScene(GLvoid)									
{
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);	
	if (light)
		glEnable(GL_LIGHTING);
	else
		glDisable(GL_LIGHTING);

	glLoadIdentity();
		
	// Texto. Traslado el origen 1 unidad hacia el interior de la pantalla, 
	// para que lo podamos ver, dibujo, y vuelvo el
	// orgen a su posicin original. Este texto 2D siempre dibuja en el 
	// plano (X,Y) = (0,0). No se ve sin luz, porque es un texto de bitmap.
	
	glTranslatef(0.0f,0.0f,-1.0f);
	glColor3f(0.0f,0.0f,0.0f);
	glRasterPos2f(0.2f, 0.38f);
	glPrint("Desp x - %7.2f",xdesp);
	glRasterPos2f(0.2f, 0.36f);
	glPrint("Desp y - %7.2f",ydesp);
	glRasterPos2f(0.2f, 0.34f);
	glPrint("Desp z - %7.2f",zdesp);
	glRasterPos2f(0.214f, 0.32f);
	glPrint("Rot x - %7.2f",xrot);
	glRasterPos2f(0.214f, 0.30f);
	glPrint("Rot y - %7.2f",yrot);
	glRasterPos2f(0.214f, 0.28f);
	glPrint("Rot z - %7.2f",zrot);
	glRasterPos2f(0.214f, 0.26f);
	if (light)
		glPrint("Luz z - True");
	else
		glPrint("Luz z - False");
	glRasterPos2f(0.012f, 0.24f);
	if (id_sw)
		glPrint("glLoadIdentity() z - True");
	else
		glPrint("glLoadIdentity() z - False");
	glRasterPos2f(-0.1f, 0.22f);
	glPrint("Instrucciones de uso en el codigo fuente");
	glTranslatef(0.0f,0.0f,1.0f);
	
	// Fin texto
	
	// Reset. Se activa con la tecla R

	if(sw){
		glLoadIdentity();
		xdesp = ydesp = zdesp = 0.0f;
		xrot = yrot = zrot = 0.0f;
		sw = !sw;
	}

	//Fin reset

	//Rotaciones de los ejes

	glTranslatef(xdesp,ydesp,zdesp);
	glRotatef(xrot,1.0f,0.0f,0.0f);
	glRotatef(yrot,0.0f,1.0f,0.0f);
	glRotatef(zrot,0.0f,0.0f,1.0f);
	glTranslatef(-xdesp,-ydesp,-zdesp);
	

	// SIMULADOR de glLoadIdentity(). Es es lo que pasa cuando NO SE PONE.
	// es solo una simulacion, porque si quito glLoadIdentity() no puedo
	// escribir correctamente los textos.

	glTranslatef(xdesp,ydesp,zdesp);
	if (!id_sw){
		xdesp-=0.01f;
		ydesp-=0.01f;
		zdesp-=0.01f;
	}
	glTranslatef(xdesp,ydesp,zdesp);
	
	
	// Dibujamos los ejes.

		// Eje X

	glBegin(GL_LINES);
		glColor3f(1.0f,0.0f,0.0f);					// Rojo
		glVertex3f(0.0f,0.0f,0.0f);					
		glVertex3f(1.0f,0.0f,0.0f);					
	glEnd();

		// Dibujamos el cono de la punta del eje

	glRotatef(90.0f,0.0f,1.0f,0.0f);				// Orientamos el cono
	glTranslatef(0.0f,0.0f,1.0f);					// Nos movemos al centro del cono
	gluCylinder(quadratic,0.02f,0.0f,0.02f,32,32);	// Cono: un cilindro con radio basal 0.02, 
													// radio final 0.0, altura 0.02 y 32 caras
	glTranslatef(0.0f,0.0f,-1.0f);					// Deshacemos las rotaciones y traslaciones
	glRotatef(-90.0f,0.0f,1.0f,0.0f);
	
		//Eje Y

	glBegin(GL_LINES);								//Verde
		glColor3f(0.0f,1.0f,0.0f);						
		glVertex3f(0.0f,0.0f,0.0f);					
		glVertex3f(0.0f,1.0f,0.0f);					
	glEnd();
	
		// El cono de la punta del eje

	glRotatef(90.0f,0.0f,0.0f,1.0f);
	glRotatef(90.0f,0.0f,1.0f,0.0f);
	glTranslatef(0.0f,0.0f,1.0f);					
	gluCylinder(quadratic,0.02f,0.0f,0.02f,32,32);	
	glTranslatef(0.0f,0.0f,-1.0f);					
	glRotatef(-90.0f,0.0f,1.0f,0.0f);
	glRotatef(-90.0f,0.0f,0.0f,1.0f);
	
		//Eje Z

	glBegin(GL_LINES);
		glColor3f(0.0f,0.0f,1.0f);					//Azul					
		glVertex3f(0.0f,0.0f,0.0f);					
		glVertex3f(0.0f,0.0f,1.0f);					
	glEnd();

		//El cono de la punta del eje	

	glTranslatef(0.0f,0.0f,1.0f);					
	gluCylinder(quadratic,0.02f,0.0f,0.02f,32,32);	
	glTranslatef(0.0f,0.0f,-1.0f);					
	
	return TRUE;									// Fin de la diversion
}

GLvoid KillGLWindow(GLvoid)								
{
	if (fullscreen)										
	{
		ChangeDisplaySettings(NULL,0);					
		ShowCursor(TRUE);								
	}

	if (hRC)											
	{
		if (!wglMakeCurrent(NULL,NULL))					
		{
			MessageBox(NULL,"Release Of DC And RC Failed.","SHUTDOWN ERROR",MB_OK | MB_ICONINFORMATION);
		}

		if (!wglDeleteContext(hRC))						
		{
			MessageBox(NULL,"Release Rendering Context Failed.","SHUTDOWN ERROR",MB_OK | MB_ICONINFORMATION);
		}
		hRC=NULL;										
	}

	if (hDC && !ReleaseDC(hWnd,hDC))					
	{
		MessageBox(NULL,"Release Device Context Failed.","SHUTDOWN ERROR",MB_OK | MB_ICONINFORMATION);
		hDC=NULL;										
	}

	if (hWnd && !DestroyWindow(hWnd))					
	{
		MessageBox(NULL,"Could Not Release hWnd.","SHUTDOWN ERROR",MB_OK | MB_ICONINFORMATION);
		hWnd=NULL;										
	}

	if (!UnregisterClass("OpenGL",hInstance))			
	{
		MessageBox(NULL,"Could Not Unregister Class.","SHUTDOWN ERROR",MB_OK | MB_ICONINFORMATION);
		hInstance=NULL;									
	}
	KillFont();
}

BOOL CreateGLWindow(char* title, int width, int height, int bits, bool fullscreenflag)
{
	GLuint		PixelFormat;			
	WNDCLASS	wc;						
	DWORD		dwExStyle;				
	DWORD		dwStyle;				
	RECT		WindowRect;				
	WindowRect.left=(long)0;			
	WindowRect.right=(long)width;		
	WindowRect.top=(long)0;				
	WindowRect.bottom=(long)height;		

	fullscreen=fullscreenflag;			

	hInstance			= GetModuleHandle(NULL);				
	wc.style			= CS_HREDRAW | CS_VREDRAW | CS_OWNDC;	
	wc.lpfnWndProc		= (WNDPROC) WndProc;					
	wc.cbClsExtra		= 0;									
	wc.cbWndExtra		= 0;									
	wc.hInstance		= hInstance;							
	wc.hIcon			= LoadIcon(NULL, IDI_WINLOGO);			
	wc.hCursor			= LoadCursor(NULL, IDC_ARROW);			
	wc.hbrBackground	= NULL;									
	wc.lpszMenuName		= NULL;									
	wc.lpszClassName	= "OpenGL";								

	if (!RegisterClass(&wc))									
	{
		MessageBox(NULL,"Failed To Register The Window Class.","ERROR",MB_OK|MB_ICONEXCLAMATION);
		return FALSE;											
	}
	
	if (fullscreen){
		DEVMODE dmScreenSettings;								
		memset(&dmScreenSettings,0,sizeof(dmScreenSettings));	
		dmScreenSettings.dmSize=sizeof(dmScreenSettings);		
		dmScreenSettings.dmPelsWidth	= width;				
		dmScreenSettings.dmPelsHeight	= height;				
		dmScreenSettings.dmBitsPerPel	= bits;					
		dmScreenSettings.dmFields=DM_BITSPERPEL|DM_PELSWIDTH|DM_PELSHEIGHT;

		
		if (ChangeDisplaySettings(&dmScreenSettings,CDS_FULLSCREEN)!=DISP_CHANGE_SUCCESSFUL)
		{
			
			if (MessageBox(NULL,"The Requested Fullscreen Mode Is Not Supported By\nYour Video Card. Use Windowed Mode Instead?","NeHe GL",MB_YESNO|MB_ICONEXCLAMATION)==IDYES)
			{
				fullscreen=FALSE;		
			}
			else
			{
				
				MessageBox(NULL,"Program Will Now Close.","ERROR",MB_OK|MB_ICONSTOP);
				return FALSE;									
			}
		}
	}

	if (fullscreen)												
	{
		dwExStyle=WS_EX_APPWINDOW;								
		dwStyle=WS_POPUP;										
		ShowCursor(FALSE);										
	}
	else
	{
		dwExStyle=WS_EX_APPWINDOW | WS_EX_WINDOWEDGE;			
		dwStyle=WS_OVERLAPPEDWINDOW;							
	}

	AdjustWindowRectEx(&WindowRect, dwStyle, FALSE, dwExStyle);		

	
	if (!(hWnd=CreateWindowEx(	dwExStyle,							
								"OpenGL",							
								title,								
								dwStyle |							
								WS_CLIPSIBLINGS |					
								WS_CLIPCHILDREN,					
								0, 0,								
								WindowRect.right-WindowRect.left,	
								WindowRect.bottom-WindowRect.top,	
								NULL,								
								NULL,								
								hInstance,							
								NULL)))								
	{
		KillGLWindow();								
		MessageBox(NULL,"Window Creation Error.","ERROR",MB_OK|MB_ICONEXCLAMATION);
		return FALSE;								
	}

	static	PIXELFORMATDESCRIPTOR pfd=				
	{
		sizeof(PIXELFORMATDESCRIPTOR),				
		1,											
		PFD_DRAW_TO_WINDOW |						
		PFD_SUPPORT_OPENGL |						
		PFD_DOUBLEBUFFER,							
		PFD_TYPE_RGBA,								
		bits,										
		0, 0, 0, 0, 0, 0,							
		0,											
		0,											
		0,											
		0, 0, 0, 0,									
		16,											
		0,											
		0,											
		PFD_MAIN_PLANE,								
		0,											
		0, 0, 0										
	};
	
	if (!(hDC=GetDC(hWnd)))							
	{
		KillGLWindow();								
		MessageBox(NULL,"Can't Create A GL Device Context.","ERROR",MB_OK|MB_ICONEXCLAMATION);
		return FALSE;								
	}

	if (!(PixelFormat=ChoosePixelFormat(hDC,&pfd)))	
	{
		KillGLWindow();								
		MessageBox(NULL,"Can't Find A Suitable PixelFormat.","ERROR",MB_OK|MB_ICONEXCLAMATION);
		return FALSE;								
	}

	if(!SetPixelFormat(hDC,PixelFormat,&pfd))		
	{
		KillGLWindow();								
		MessageBox(NULL,"Can't Set The PixelFormat.","ERROR",MB_OK|MB_ICONEXCLAMATION);
		return FALSE;								
	}

	if (!(hRC=wglCreateContext(hDC)))				
	{
		KillGLWindow();								
		MessageBox(NULL,"Can't Create A GL Rendering Context.","ERROR",MB_OK|MB_ICONEXCLAMATION);
		return FALSE;								
	}

	if(!wglMakeCurrent(hDC,hRC))					
	{
		KillGLWindow();								
		MessageBox(NULL,"Can't Activate The GL Rendering Context.","ERROR",MB_OK|MB_ICONEXCLAMATION);
		return FALSE;								
	}

	ShowWindow(hWnd,SW_SHOW);						
	SetForegroundWindow(hWnd);						
	SetFocus(hWnd);									
	ReSizeGLScene(width, height);					

	if (!InitGL())									
	{
		KillGLWindow();								
		MessageBox(NULL,"Initialization Failed.","ERROR",MB_OK|MB_ICONEXCLAMATION);
		return FALSE;								
	}

	return TRUE;									
}

LRESULT CALLBACK WndProc(	HWND	hWnd,			
							UINT	uMsg,			
							WPARAM	wParam,			
							LPARAM	lParam)			
{
	switch (uMsg)									
	{
		case WM_ACTIVATE:							
		{
			if ((LOWORD(wParam) != WA_INACTIVE) && !((BOOL)HIWORD(wParam)))
				active=TRUE;						
			else
				active=FALSE;						

			return 0;								
		}

		case WM_SYSCOMMAND:							
		{
			switch (wParam)							
			{
				case SC_SCREENSAVE:					
				case SC_MONITORPOWER:				
				return 0;							
			}
			break;									
		}

		case WM_CLOSE:								
		{
			PostQuitMessage(0);						
			return 0;								
		}

		case WM_KEYDOWN:							
		{
			keys[wParam] = TRUE;					
			return 0;								
		}

		case WM_KEYUP:								
		{
			keys[wParam] = FALSE;					
			return 0;								
		}

		case WM_SIZE:								
		{
			ReSizeGLScene(LOWORD(lParam),HIWORD(lParam));  
			return 0;								
		}
	}

	
	return DefWindowProc(hWnd,uMsg,wParam,lParam);
}

//>>>>>>> Ver Aqui

int WINAPI WinMain(	HINSTANCE	hInstance,			
					HINSTANCE	hPrevInstance,		
					LPSTR		lpCmdLine,			
					int			nCmdShow)			
{
	MSG		msg;									
	BOOL	done=FALSE;								

	
	if (MessageBox(NULL,"Would You Like To Run In Fullscreen Mode?", "Start FullScreen?",MB_YESNO|MB_ICONQUESTION)==IDNO)
	{
		fullscreen=FALSE;							
	}

	
	if (!CreateGLWindow("OpenGL - Alfredo Cofr",640,480,16,fullscreen))
	{
		return 0;									
	}

	while(!done)									
	{
		if (PeekMessage(&msg,NULL,0,0,PM_REMOVE))	
		{
			if (msg.message==WM_QUIT)				
			{
				done=TRUE;							
			}
			else									
			{
				TranslateMessage(&msg);				
				DispatchMessage(&msg);				
			}
		}
		else										
		{
			
			if ((active && !DrawGLScene()) || keys[VK_ESCAPE])	
			{
				done=TRUE;							
			}
			else									
			{
				SwapBuffers(hDC);					
			}
			
			// Lo interesante esta de aqui para abajo. Con keys[] defino las teclas
			// que quiero que el programa use. Aca tambien se puede colocar un 
			// sensor para mouse.

			if (keys[VK_F1])						
			{
				keys[VK_F1]=FALSE;					
				KillGLWindow();						
				fullscreen=!fullscreen;				
				
				if (!CreateGLWindow("OpenGL - Alfredo Cofr",640,480,16,fullscreen))
				{
					return 0;						
				}
			}
			if (keys[VK_LEFT])						
			{
				xdesp+=0.01f;
			}
			if (keys[VK_RIGHT])						
			{
				xdesp-=0.01f;
			}
			if (keys[VK_UP])						
			{
				ydesp-=0.01f;
			}
			if (keys[VK_DOWN])						
			{
				ydesp+=0.01f;
			}
			if (keys['A'])						
			{
				zdesp+=0.01f;
			}
			if (keys['Z'])						
			{
				zdesp-=0.01f;
			}
			if (keys['R']){
				sw = true;
				light = true;
				id_sw = true;
			}
			if (keys['L']){
				keys['L'] = false;
				light = !light;
			}
			if (keys['I']){
				keys['I'] = false;
				id_sw = !id_sw;
			}
			if (keys['1']){
				xrot+=0.2f;
			}
			if (keys['2']){
				xrot-=0.2f;
			}
			if (keys['3']){
				yrot+=0.2f;
			}
			if (keys['4']){
				yrot-=0.2f;
			}
			if (keys['5']){
				zrot+=0.2f;
			}
			if (keys['6']){
				zrot-=0.2f;
			}
		}
	}

	
	KillGLWindow();	
	return (msg.wParam);							
}

// Todo esto es posible gracias a San Nehe, nehe.gamedev.net. Demos gracias.