#include <stdio.h>

unsigned char set_parity(unsigned char c) {
  /* En el bit más significativo de parity 
   * iremos guardando la paridad parcial
   * de c, q inicialmente es uno 
   * (0x80 = 1000 0000). */
  int parity = 0x80, i;
  for(i = 0; i < 7; i++)
    /* Ahora leemos cada bit de c, desde el
     * menos al más significativo sin incluir el 8º. 
     * Para ello aplicamos la máscara que tiene un 1 en
     * la posición i-ésima, que sería 0x01<<i.
     * Si leemos un 1 el bit + significativo de
     * parity debe cambiar (si era 0 pasa a 1 y si 
     * era 1 pasa a 0), si leemos un 0 dejamos
     * todo tal cual. Para no distinguir entre
     * casos anteriores hacemos uso del ^ (o 
     * exclusivo), pues posee la propiedad de que 
     * algo^0 = algo y algo^1=~algo. Pero, antes 
     * de hacer eso necesitamos que el bit leído de
     * c quede en la misma posición que el bit de 
     * paridad de parity. Como el bit leído esta
     * en la posición i-ésima, necesitamos correrlo
     * hacia la derecha en 8-i posiciones. Asi por
     * ejemplo si parity era 0x00, i es 1 y c es 0x07
     * obtenemos el 2º bit de c 0000 0111&0x01<<2 
     * que da 0000 0010, lo dejamos lo + a la derecha
     * posible con 0000 0010 << 6 = 1000 0000 y 
     * finalmente actualizamos parity con
     * 0000 0000 ^ 1000 0000 = 1000 0000 pues llevamos
     * dos 1's leídos */
    parity ^= (c&0x01<<i)<<(7-i);
  /* Finalmente tenemos que parity = x000 0000 con x
   * 1 si c tenía cant par de 1's, y 0 de lo contrario.
   * Ahora necesitamos setear los 7 bits de la derecha
   * de parity con los 7 bits de la derecha de c. Para
   * es tenemos que hacer parity|algo, con algo un 
   * char que empieza en 0 y le siguen los 7 bits de la
   * derecha de c. Para obtener algo hacemos c&0111 1111.
   * En la aux puse c|parity que, como les dije, funciona
   * solo si en el bit + significativo de c hay un 0 o
   * hay un 1 y corresponde a la paridad de c.*/
  return parity|(c&0x7f);
}

int check_parity(unsigned char c) {
  int i, parity = 0x80, check;
  for(i = 0; i < 7; i++)
    parity ^= (c&0x01<<i)<<(7-i);
  /* Calculamos parity = que en el caso anterior. y lo com-
   * paramos el bit de paridad de c seguido de puros 0's (check).
   * Para compararlos no usaremos el ==, sino que usaremos que la
   * negación de un ^ pues tiene la misma tabla de verdad.
   * En un principio usé a==b <=> ~(a^b), pero esto es cierto para 
   * números de un solo bit. Me explico, en C 0 es falso y cualquier
   * cosa != de 0 es verdadero, por lo tanto basta que que a y b 
   * sean ='s en 1 solo bit para que el resultado sea algo != de 0 
   * y por lo tanto verdadero. La solución de esto es notar que si
   * uno hace !(algo_!=_0) da 0, pues de lo contrario
   * no funcionarían bien los if's. Por lo tanto el operador ! mapea
   * todos los número que tienen un bit != de 0 a 0, y 0 a algun número
   * != de 0, lo que necesitabamos. Entonces podemos decir que
   * a==b <=> !(a^b). */
  check = c & 0x80;
  printf("c = 0x%x, check = 0x%x, parity = 0x%x y parity^check = 0x%x\n", c, check, parity, check^parity);
  return  !(parity ^ check);
}

main() {
  char c2 = 0x77;
  printf("set_parity(0x%x)=0x%x\n", c2, set_parity(c2));
  printf("check_parity(set_parity(0x%x))=0x%x\n", set_parity(c2), check_parity(set_parity(c2)));
  if(check_parity(set_parity(c2))) printf("par c2 ok\n");
}
