/*
 * WebSPHINX web crawling toolkit
 * Copyright (C) 1998,1999 Carnegie Mellon University 
 * 
 * This library is free software; you can redistribute it
 * and/or modify it under the terms of the GNU Library
 * General Public License as published by the Free Software 
 * Foundation, version 2.
 *
 * WebSPHINX homepage: http://www.cs.cmu.edu/~rcm/websphinx/
 */
package websphinx;

import java.net.URL;
import java.net.MalformedURLException;

/**
 * Button element in an HTML form -- for example, &lt;INPUT TYPE=submit&gt; or &lt;INPUT TYPE=image&gt;.
 * 
 * @author Rob Miller
 * @see Page
 * @see Link
 */
public class FormButton extends Link {

    Form form;

    /**
     * Make a LinkElement from a start tag and end tag and its containing form.
     * The tags and form must be on the same page.
     * @param startTag Start tag of button
     * @param endTag End tag of button (or null if none)
     * @param form Form containing this button
     */
    public FormButton (Tag startTag, Tag endTag, Form form) throws MalformedURLException {
        super (startTag, endTag, null);
        this.form = form;
        if (form == null)
            throw new MalformedURLException ();
    }

    /**
     * Get the URL.
     * @return the URL of the link
     */ 
    public URL getURL () {
        if (url == null)
            try {
                url = urlFromHref (getStartTag (), null);
            } catch (MalformedURLException e) {
                url = null;
            }

        return url;
    }

    /**
     * Get the form.
     * @return the form containing this button
     */ 
    public Form getForm () {
        return form;
    }

    /**
     * Get the method used to access this link.
     * @return GET or POST.
     */ 
    public int getMethod () {
        return form.getMethod ();
    }

    /**
     * Construct the URL for this button, from its start tag and a base URL (for relative references).
     * @param tag Start tag of button, such as &lt;INPUT TYPE=submit&gt;.
     * @param base Base URL used for relative references
     * @return URL to which the button points
     */
    protected URL urlFromHref (Tag tag, URL base) throws MalformedURLException {
        if (parent == null || form == null)
            // can't figure out URL until we're linked into an HTML element tree
            // containing our complete form
            return null;
        return form.makeQuery (this);
    }

}
