/*
 * WebSPHINX web crawling toolkit
 * Copyright (C) 1998,1999 Carnegie Mellon University 
 * 
 * This library is free software; you can redistribute it
 * and/or modify it under the terms of the GNU Library
 * General Public License as published by the Free Software 
 * Foundation, version 2.
 *
 * WebSPHINX homepage: http://www.cs.cmu.edu/~rcm/websphinx/
 */
package websphinx;

import websphinx.util.Timer;

/**
 * Run a crawler periodically.
 */
public class Chronicle extends Timer implements Runnable {
    Crawler crawler;
    int interval;
    boolean running = false;
    boolean triggered = false;

    /**
     * Make a Chronicle.
     * @param crawler Crawler to run periodically
     * @param interval Invocation interval, in seconds. Crawler is invoked
     * every interval seconds.  If the crawler is still running
     * when interval seconds have elapsed, it is aborted.
     *
     */
    public Chronicle (Crawler crawler, int interval) {
        this.crawler = crawler;
        this.interval = interval;
    }

    /**
     * Start chronicling.  Starts a background thread which
     * starts the crawler immediately, then re-runs the crawler
     * every interval seconds from now until stop() is called.
     */
    public void start () {
        if (running)
            return;

        running = true;
        set (interval * 1000, true);
        Thread thread = new Thread (this, crawler.getName ());
        thread.start ();
    }

    /**
     * Stop chronicling.  Also stops the crawler, if it's currently running.
     */
    public synchronized void stop () {
        if (!running)
            return;

        running = false;
        crawler.stop ();
        notify ();
        cancel ();
    }

    /**
     * Background thread that runs the crawler.  Clients shouldn't
     * call this.
     */
    public synchronized void run () {
        try {
            while (running) {
                crawler.run ();
                while (!triggered)
                    wait ();
                triggered = false;
            }
        } catch (InterruptedException e) {}
    }

    protected synchronized void alarm () {
        crawler.stop ();
        triggered = true;
        notify ();
    }

//#ifdef JDK1.1
    // FIX: allow crawler class name (starting up Workbench to configure it)
  public static void main (String[] args) throws Exception {
    java.io.ObjectInputStream in =
      new java.io.ObjectInputStream (new java.io.FileInputStream (args[0]));
    Crawler loadedCrawler = (Crawler)in.readObject ();
    in.close ();

    EventLog.monitor (loadedCrawler);

    Chronicle track = new Chronicle (loadedCrawler, Integer.parseInt (args[1]));
    track.start ();
  }
//#endif JDK1.1
}
