package symantec.itools.awt;

import java.awt.Image;
import java.awt.Color;

//	02/10/97	RKM	Added data object
//	02/27/97	RKM	Merged in accessors for getParent, getChild, & getSibling

/**
 * This is a single node in the TreeView panel.
 * It displays text and optionally one of two images depending on its state,
 * collapsed or expanded.
 * It also may have an object associated with it that doesn't get displayed.
 * @see TreeView
 */
public class TreeNode2
{
	TreeNode2	sibling;
	TreeNode2	child;
	TreeNode2	parent;
	String		text;
	Color       color;
	Image		collapsedImage;
	Image		expandedImage;
	int			numberOfChildren;
	Object		dataObject;

	int			depth = -1;
	boolean	isExpanded = false;

	//constructors

    /**
     * Constructs a TreeNode2 with the given text label.
     * @param text the text to display for this node
     */
	public TreeNode2(String text)
	{
	    this(text, null, null);
	}

    /**
     * Constructs a TreeNode2 with the given text label, and collapsed and
     * expanded images.
     * @param text the text to display for this node
     * @param collapsedImage the image to use when this node is collapsed, hiding
     * all of its child nodes
     * @param expandedImage the image to use when this node is expanded, showing
     * all of its child nodes
     */
	public TreeNode2(String text, Image collapsedImage, Image expandedImage)
	{
	    this.text				= text;
	    this.color              = null;
	    this.sibling			= null;
	    this.child				= null;
	    this.collapsedImage 	= collapsedImage;
	    this.expandedImage  	= expandedImage;
	    this.numberOfChildren	= 0;
	    this.dataObject			= null;
	}

    /**
     * Notes the current depth of this node.
     * @param depth
     * @see #getDepth
     */
	void setDepth(int depth)
	{
	    this.depth = depth;
	}

    /**
     * Gets the depth of this node as previously noted.
     * @return the depth of this node
     */
	public int getDepth()
	{
	    return depth;
	}

    /**
     * Notes the current color of this node.
     * @param color
     * @see #getColor
     */
	public void setColor (Color color)
	{
	    this.color = color;
	}

    /**
     * Gets the color of this node as previously noted.
     * @return the color of this node
     */
	public Color getColor()
	{
	    return color;
	}

    /**
     * Determines whether this node is expanded.
     * A node is expanded if its child nodes are visible.
     * @return true if the node is expanded, false if it is collapsed
     */
	public boolean isExpanded()
	{
	    return isExpanded;
	}

    /**
     * Determines whether this node is expandable.
     * A node is expandable if it has one or more child nodes.
     * @return true if the node is expandable, false if not
     */
	public boolean isExpandable()
	{
	    return (child!=null);
	}

	/**
	 * Sets a flag indicating that this node is expanded, if it is expandable.
	 */
	public void expand()
	{
	    if (isExpandable())
	    {
	        isExpanded=true;
	    }
	}

	/**
	 * Sets a flag indicating that this node is not expanded.
	 */
	public void collapse()
	{
	    isExpanded = false;
	}

	/**
	 * Toggles the node state between collapsed and expanded, if the node
	 * is expandable.
	 */
	public void toggle()
	{
	    if (isExpanded)
	    {
	        collapse();
	    }
	    else if (isExpandable())
	    {
	        expand();
	    }
	}

	/**
	 * Gets the proper image for this node in its current state, expanded or collapsed.
	 * @return the current image for this node in its current state
	 */
	public Image getImage()
	{
	    return ((isExpanded && (expandedImage != null))
	            ? expandedImage
	            : collapsedImage);
	}

	/**
	 * Sets the image to use for this node when it is expanded.
	 * @param image the image to use when this node is expanded
	 * @see #setCollapsedImage
	 * @see #getImage
	 */
	public void setExpandedImage(Image image)
	{
	    expandedImage = image;
	}

	/**
	 * Sets the image to use for this node when it is not expanded.
	 * @param image the image to use when this node is collapsed
	 * @see #setExpandedImage
	 * @see #getImage
	 */
	public void setCollapsedImage(Image image)
	{
	    collapsedImage = image;
	}

	/**
	 * Gets the current text label for this node.
	 * @return the current text label for this node
	 * @see #setText
	 */
	public String getText()
	{
	    return text;
	}

	/**
	 * Sets a new text label for this node.
	 * @param s the new text label for this node
	 * @see #getText
	 */
	public void setText(String s)
	{
	    text = new String(s);
	}

	/**
	 * Gets the object associated with this node.
	 * This object does not get displayed.
	 * @return the object associated with this node
	 * @see #setDataObject
	 */
	public Object getDataObject()
	{
	    return dataObject;
	}

	/**
	 * Sets an object to associate with this node.
	 * This object does not get displayed.
	 * @param theObject an object to associate with this node
	 * @see #getDataObject
	 */
	public void setDataObject(Object theObject)
	{
	    dataObject = theObject;
	}

	/**
	 * Gets the parent of this node.
	 * @return this node's parent node
	 * @see #getChild
	 * @see #getSibling
	 */
	public TreeNode2 getParent()
    {
        return parent;
    }

	/**
	 * Gets the child of this node.
	 * @return this node's child node
	 * @see #getParent
	 * @see #getSibling
	 */
    public TreeNode2 getChild()
    {
        return child;
    }

	/**
	 * Gets the next sibling of this node.
	 * @return this node's next sibling node
	 * @see #getChild
	 * @see #getParent
	 */
    public TreeNode2 getSibling()
    {
        return sibling;
    }
}
