/*
 * WebSPHINX web crawling toolkit
 * Copyright (C) 1998,1999 Carnegie Mellon University 
 * 
 * This library is free software; you can redistribute it
 * and/or modify it under the terms of the GNU Library
 * General Public License as published by the Free Software 
 * Foundation, version 2.
 *
 * WebSPHINX homepage: http://www.cs.cmu.edu/~rcm/websphinx/
 */
package websphinx.searchengine;

import websphinx.*;

/**
 * Result returned by a search engine query, identifying a Web page that matches the query.
 */
public class SearchEngineResult extends Region {
    /**
     * Relevancy rank of page in search engine's ordering.  In other words, rank=1 
     * is the first result the search engine returned.  If search engine
     * results are not explicitly numbered, then rank may be 0.
     */
    public int rank = 0;

    /**
     * Relevancy score of page, by search engine's scale.  If search engine
     * does not provide a score, the score defaults to 0.0. 
     * 
     */
    public double score = 0.0;

    /**
     * Title of page as reported by search engine, or null if not provided
     */
    public String title;

    /**
     * Short description of page as reported by search engine.  Typically the first few words
     * of the page.  If not provided, description is null.
     */
    public String description;

    /**
     * Link to the actual page.
     */
    public Link link;

    /**
     * Search engine that produced this hit.
     */
    public SearchEngine searchengine;

    /**
     * Make a SearchEngineResult.
     * @param result Region of a search engine's results page.  Should be annotated with rank, title,
     * description, and link fields.
     */
    public SearchEngineResult (Region result) {
        super (result);
        rank = result.getNumericLabel ("rank", new Integer(0)).intValue();
        score = result.getNumericLabel ("score", new Double(0)).doubleValue();
        title = result.getLabel ("title");
        description = result.getLabel ("description");
        
        try {
            link = (Link)result.getField ("link");
        } catch (ClassCastException e) {}
        searchengine = (SearchEngine)result.getSource().getObjectLabel ("searchengine.source");
    }

    public String toString () {
        return rank + ". " + title + " [" + (link!=null ? link.getURL ().toString() : "(null)") + "]" + " " + score + "\n"
               + "    " + description;
    }
} 
