%function used to calculated radiation,transport and weathering terms in the 
%four box steady state model

function [W,Ha,Ho,F,fiout,q,v,Tsw,Tsc,TLNI,THNI,Sw,Sc]=...
    SilWeafunc(pCO2,Q,Tsw,Tsc,Sw,Sc)

global Soc Ofr HHA fd Sm cp rho sy Tf Lv Va Q10 del Tref

Vw=(1-Ofr*2*del)/(2*(1-Ofr));
Vc=1-Vw;
pCO2=pCO2*1e6;

% Albedo, function of latitude (Hartmann 1994)

al0  =  0.7;                           % st. val 0.7
al2  = -0.175;                         %st. val -0.175

%al0  =  0.7+0.1*(0.45-del);                   % st. val 0.7
%al2  = 0.7-0.175-al0;                         %st. val -0.175

ali  =  .62; % Albedo of ice covered area, st. val. 0.62
Ti   = -10;  % Fixed ice line temperature, st. val. -10

Q2   = -0.482;     %constant for calculating incoming solar radiation as a 
                   %function of latitude, see Hartman 1994

chi  =  0.71;      % Atmospheric exchange coefficient for(sensible)heat      
                   % Note: atmospheric heat transport is now divided into 
                   % sensible and latent heat;standard value 0.62 [W/m2/K]
                    
cfw =3.4e6;        %Atmospheric exchange coefficient for water vapour, st. val 2.95e6,
                    %scaled to give F = 0.7 Sv for standard case

%Ocean model parameters                    
                    
alfa     = 0.2e-3;                %Thermal expansion coeficient
beta     = 0.8e-3;                %Salinity contraction coefficient
Sm       = 35.0;                  %Ocean mean salinity
k        = 2.5e-4;               %THC flow parameter, scaled to give about 15 
                                  %Sv flow in THC for standard case
kwdc     = 2*4/26.216;              %WDC flow parameter, scaled to give 5 Sv 
                                  %for standard case
cw=1.382e-9;                      %weathering constant, st. val 1.382e-9 for 
                                  %modern day climate
             
r=pCO2/300;   %pCO2=r*300; note that reference pCO2 in CK is 300 ppm
lr=log(r);

A = -321.21 + 9.161*lr - 3.164*lr*lr + 0.5468*lr*lr*lr; %first constant adjusted from CK's
                                                  %value of -326.4 to get
                                                  %mean T of 15
B = 1.953 - 0.04866*lr + 0.01309*lr*lr - 0.002577*lr*lr*lr;

A = A + B*273;                     %to convert back to reference temperature of zero


%Iterations
 
n=5000;
for i=1:n           %iteration to find stable steady states
    
delT=Tsw-Tsc;

% Establish the atmospheric temperature profile, 2.order legendre pol. in 
% sine of lat, Ta(lat) = PTa(1) + .5*PTa(2) * ( 3*sin(lat)^2 -1 )
% Coefficients are calculated so that the area weighted mean of the profile
% matches the surface ocean temperatures in each sector.

CTa(1,:) = [ 1 .5*(sin(fd)^2-1)                   ];
CTa(2,:) = [ 1 .5*(sin(fd)-sin(fd)^3)/(1-sin(fd)) ];
RTa      = [ Tsw Tsc]';
PTa      = CTa\RTa;

 %Calculate the latitude fi where Ta(fi)=Ti
fimax=pi/2-0.001; 
fimin=0.0001;
Tzero=PTa(1) - .5*PTa(2);           %temperature at equator
if Tzero<=Ti
    fi=fimin;
    fideg=0;
else    
fi  = min(fimax , asin((2/(3*PTa(2))*(Ti-PTa(1)+PTa(2)/2))^(.5)) );
fideg=fi*180/pi;
end
if fi ==fimax
    fideg=90;
end   
fiout=fi;
%Calculation of atmospheric moisture transport (uses Clausius-Claperon)

PTad  = PTa(1) + PTa(2)*.5*(3*sin(fd).^2-1);  % Temperature at latitude = fd
F   =  cfw*exp(-5420/(PTad+273))*delT;        %atmosphere moisture transport, in Sv  

%Calculation of thermohaline and "wind-driven" circulation

fac=(fi-fd)/(fi-fd+0.01);               %so that the "wind-driven" circulation v 
v=kwdc*delT*fac;                        %approaches zero as fi approaches fd 

if fi>fd 
    THNI  =  1/(sin(fi)-sin(fd))*...        % mean, ice-free, cold box temperature
         [(PTa(1)-PTa(2)*.5)*(sin(fi)-sin(fd))+...
         0.5*PTa(2)*(sin(fi)^3-sin(fd)^3 )];        
    TLNI  =  Tsw;
    if THNI < Tf   
      THNI = Tf;                        %freezing temperature lower bound on THNI
    end
    Alc=2*(1-del)*Ofr*HHA*(sin(fi)-sin(fd));
    Alw=2*del*Ofr*HHA*sin(fd);
    else 
    THNI = Tf;
    TLNI  =  1/(sin(fi))*...           %mean, ice-free, warm box temperature
         [(PTa(1)-PTa(2)*.5)*(sin(fi))+...
         0.5*PTa(2)*(sin(fi)^3)];
    v=0;                              %no wind driven circulation when cold box ice covered
    if TLNI<Tf 
       TLNI=Tf;                        %freezing temperature lower bound on THNI 
    end 
    Alc=0;
    Alw=2*del*Ofr*HHA*sin(fi);
end 

delT2 = TLNI-THNI;
delS  = Sw-Sc;

q  =  (alfa*delT2-beta*delS)/k;            %Thermohaline circulation
Scn=(abs(q)+v)*Sm/(Vw*F+(abs(q)+v));
Swn=(Sm-Vc*Scn)/Vw;
Sw=Sw-(Sw-Swn)/200;
Sc=(Sm-Vw*Sw)/Vc;
Err1 = ((Sw-Swn)^2)^0.5;

% Integrated incoming shortwave radiation for the low latitude sector,
% south of fd, including ice cover if fi<fd

if fi<fd

Rw1= 1/sin(fi)*Q/4 * ( ...
         (al0-.5*al2)*(1-.5*Q2)*sin(fi) + ...
         .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fi)^3 + ...
         9/20*al2*Q2*sin(fi)^5 );
     
Rw2=  1/(sin(fd)-sin(fi))*Q/4 * ( ...
         (1-ali)*(1-.5*Q2)*(sin(fd)-sin(fi)) + ...
         .5*(Q2*(1-ali))*(sin(fd)-sin(fi))^3); 
Rw = (Rw1*sin(fi)+Rw2*(sin(fd)-sin(fi)))/sin(fd);   %mean adsorbed solar
                                                    %radiation in warm box 
                                                  
else
Rw   = 1/sin(fd)*Q/4 * ( ...
         (al0-.5*al2)*(1-.5*Q2)*sin(fd) + ...
         .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fd)^3 + ...
         9/20*al2*Q2*sin(fd)^5 );                                   %[W/m2]
end

% Integrate incoming radiation for the high latitude sector, north of fd, 
% including icecover north of fi of fixed albedo ali. 

if fi<fd
    fi=fd;
end
Rc1 = (al0-.5*al2)*(1-.5*Q2) - (1-ali)*(1-.5*Q2);
Rc2 = .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2)) - (1-ali)*Q2*.5;
Rc3 = 9/20*al2*Q2;
Rc4 = -(al0-.5*al2)*(1-.5*Q2)*sin(fd) ...
      -.5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fd)^3 ...
      - 9/20*al2*Q2*sin(fd)^5 ...
      +(1-ali)*(1-.5*Q2) + (1-ali)*Q2*.5; 
Rc= 1/(1-sin(fd))*Q/4*(sin(fi)*Rc1 + sin(fi)^3*Rc2 + sin(fi)^5*Rc3 + Rc4);
%mean absorbed solar radiation in high latitude box

% Atmospheric energy transport
Hsh =  chi*delT;                               %sensible heat flux,[W/m2]
eps = Lv*cfw*exp(-5420/(PTad+273))*1e6/HHA;
Hlh  = eps*delT;                               %latent heat flux, [W/m2] 
Ha=(Hsh+Hlh)*HHA/1e15;

%Ocean energy transport
gamma = rho*cp*1e6/HHA;               
Hq =  gamma*abs(q)*delT2;                      %THC heat flux,[W/m2]
Hv =  gamma*v*delT2;                           %WDC heat flux,[W/m2] 
Ho=(Hq+Hv)*HHA/1e15;

delT=((Rw-Rc)-2*(Hq+Hv))/(B+2*(chi+eps));
Tm = (Rw+Rc-2*A)/(2*B);
Tswn=Tm+delT/2;
Tscn=Tm-delT/2;
Tsw=Tsw -(Tsw-Tswn)/100;
Tsc=Tsc - (Tsc-Tscn)/100;
Err2 = ((Tsc-Tscn)^2 + (Tsw-Tswn)^2)^0.5;
if Err1<1e-4 & Err2 <1e-3 break; end
end

Ha=(Hsh+Hlh)*HHA/1e15;
Ho=(Hq+Hv)*HHA/1e15;

W=cw*(exp(0.1*log(Q10)*(TLNI-Tref))*Alw+...
    exp(0.1*log(Q10)*(THNI-Tref))*Alc);