% Two box, coupled atmosphere-ocean- sea-ice model including (specified)
%greenhouse gas forcing, sensible and latent heat transports in the 
%atmosphere and thermohaline and wind driven circulation (THC and WDC)  
%heat transports in the ocean.
% Change parameters p and s for simulating effects of changed
% atmospheric CH4 concentration and solar constant,respectively.
%Stable steady state solutions calculated for varying atmosphere pCO2.

clear all

load standardcaseCClim2007p

fd   = pi/6;          %dividing latitude: 30 degrees
So   = 1365;          %solar constant, present day value is 1367 W/m2
HHA  = 1.27e14;       % Half hemisphere area
cp   = 4.0e3;         %specific heat capacity of water, J/kg*C
rho  = 1.028e3;       %mean density of seawater, kg/m3
Lv   = 2.25e6*1e3;    %Latent heat of vaporization [J/1000kg] = [J/m^3]   
Tf = -1.8;            %freezing point of ocean water

% atmosphere parameters

% Albedo, function of latitude (Hartmann 1994)
al0  =  0.7;
al2  = -0.175;

ali  =  .62; % Albedo of ice covered area, st. val. 0.62
Ti   = -10;  % Fixed ice line temperature, st. val. -10

pCO2o =280;  %(ppmv)           %Pre-anthropogenic(PA) pCO2
pCH4o =700;  % (ppbv)           %Pre-anthropogenic(PA) pCH4 

p=1;                    %multiple of PA pCH4 
s=1;                    %multiple of present day solar constant

pCH4=p*pCH4o;
So=s*So;

Q2   = -0.482;     %constant for calculating incoming solar radiation as a 
                   %function of latitude, see Hartman 1994

Ao   =  212.53;    % Zero degree background radiation, st. val. 212.53[W/m2]

B    =  1.9;       % Sensitivity of LW radiation to temp, st. val 1.9  [W/m2/K]]

chi  =  0.62;      % Atmospheric exchange coefficient for(sensible)heat      
                   % Note: atmospheric heat transport is now divided into 
                   % sensible and latent heat;standard value 0.62 [W/m2/K]
                    
cfw =2.95e6;        %Atmospheric exchange coefficient for water vapour, st. val 2.95e6,
                    %scaled to give F = 0.7 Sv for standard case

%Ocean model parameters                    
                    
alfa     = 0.2e-3;                %Thermal expansion coeficient
beta     = 0.8e-3;                %Salinity contraction coefficient
Sm       = 35.0;                  %Ocean mean salinity
k        = 0.244e-3;        %THC flow parameter, scaled to give about 15 Sv
                            %THC flow in st. case for v = 5, st. val. 0.244e-3

kwdc     = 0.1907;          %WDC flow parameter, scaled to give 5 Sv 
                            %for standard case, st. val 0.1907
%Iterations

Tsw=28;                 %inital guesses for Tsw and Tsc
Tsc=2;

delS=0;                %initial guess for salinity difference,
                       %Ssw-Ssc                                                                  
                                                                             
delr=0.02;           %r is multiple of PA pCO2, st. val 0.02
j=600;               %total stable steady state solution number, st. val 600 
n = 5000;            %maximum number of interations
rmin=0.5;            %minimum r, Last Glacial Maximum r ~ 0.6

for m = 1:j          %stepping over different values of r 

if m<=j/2
    
%runs through increasing and then decreasing pCO2

pCO2   = pCO2o*(rmin+delr*(m-1));    
else
pCO2   =  pCO2o*(rmin+delr*j-delr*(m-1)); 

%uncomment and comment the three lines above to run through decreasing
%and then increasing pCO2

%pCO2   = pCO2o*(rmin+delr*j/2-delr*(m-1)); 
%else
%pCO2   = pCO2o*(rmin-delr*j/2+delr*(m-1));    

end

for i=1:n           %iteration to find stable steady states
    
delT=Tsw-Tsc;

% Change in outgoing longwave radiation as functions of pCO2 and pCH4
% Reference:  Myhre, G.et al., Geophys.Res.Let.25, 2715-2718 (1998). 

ACO2  = 5.35*log(pCO2/280);
ACH4  = 0.036*(pCH4^(.5)-700^(.5))-...
 [  0.47*log(1+2.01e-5*(pCH4*270)^(.75)+5.31e-15*pCH4*(pCH4*270)^(1.52))...
  - 0.47*log(1+2.01e-5*( 700*270)^(.75)+5.31e-15* 700*( 700*270)^(1.52)) ];
    
A = Ao - ACO2 - ACH4;

% Establish the atmospheric temperature profile, 2.order legendre pol. in 
% sine of lat, Ta(lat) = PTa(1) + .5*PTa(2) * ( 3*sin(lat)^2 -1 )
% Coefficients are calculated so that the area weighted mean of the profile
% matches the surface ocean temperatures in each sector.

CTa(1,:) = [ 1 .5*(sin(fd)^2-1)                   ];
CTa(2,:) = [ 1 .5*(sin(fd)-sin(fd)^3)/(1-sin(fd)) ];
RTa      = [ Tsw Tsc]';
PTa      = CTa\RTa;

 %Calculate the latitude fi where Ta(fi)=Ti
fimax=pi/2-0.001; 
fimin=0.0001;
Tzero=PTa(1) - .5*PTa(2);           %temperature at equator
if Tzero<=Ti
    fi=fimin;
    fideg=0;
else    
fi  = min(fimax , asin((2/(3*PTa(2))*(Ti-PTa(1)+PTa(2)/2))^(.5)) );
fideg=fi*180/pi;
end
if fi ==fimax
    fideg=90;
end   

%fideg=63.625;fi=fideg*pi/180;       %Sets fixed ice edge when uncommented

%Calculation of atmospheric moisture transport (uses Clausius-Claperon)

PTad  = PTa(1) + PTa(2)*.5*(3*sin(fd).^2-1);  % Temperature at latitude = fd
F   =  cfw*exp(-5420/(PTad+273))*delT;        %atmosphere moisture transport, in Sv  


%Calculation of thermohaline and "wind-driven" circulation

fac=(fi-fd)/(fi-fd+0.01);               %so that the "wind-driven" circulation v 
v=kwdc*delT*fac;                        %approaches zero as fi approaches fd 

if fi>fd 
    THNI  =  1/(sin(fi)-sin(fd))*...        % mean, ice-free, cold box temperature
         [(PTa(1)-PTa(2)*.5)*(sin(fi)-sin(fd))+...
         0.5*PTa(2)*(sin(fi)^3-sin(fd)^3 )];        
    TLNI  =  Tsw;
    if THNI < Tf   
      THNI = Tf;                        %freezing temperature lower bound on THNI
    end
else 
    THNI = Tf;
    TLNI  =  1/(sin(fi))*...           %mean, ice-free, warm box temperature
         [(PTa(1)-PTa(2)*.5)*(sin(fi))+...
         0.5*PTa(2)*(sin(fi)^3)];
    v=0;                              %no wind driven circulation when cold box ice covered
    if TLNI<Tf 
       TLNI=Tf;                        %freezing temperature lower bound on THNI 
    end    
end 

delT2 = TLNI-THNI;

q  =  (alfa*delT2-beta*delS)/k;            %Thermohaline circulation
%q=13.952;                                 %sets fixed THC when uncommented             
delSn=F*Sm/(abs(q)+v);
delS=delS-(delS-delSn)/200;
Err1 = ((delS-delSn)^2)^0.5;

% Integrated incoming shortwave radiation for the low latitude sector,
% south of fd, including ice cover if fi<fd

if fi<fd

Rw1= 1/sin(fi)*So/4 * ( ...
         (al0-.5*al2)*(1-.5*Q2)*sin(fi) + ...
         .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fi)^3 + ...
         9/20*al2*Q2*sin(fi)^5 );
     
Rw2=  1/(sin(fd)-sin(fi))*So/4 * ( ...
         (1-ali)*(1-.5*Q2)*(sin(fd)-sin(fi)) + ...
         .5*(Q2*(1-ali))*(sin(fd)-sin(fi))^3); 
Rw = (Rw1*sin(fi)+Rw2*(sin(fd)-sin(fi)))/sin(fd);   %mean adsorbed solar
                                                    %radiation in warm box                                                                                                        %box
else
Rw   = 1/sin(fd)*So/4 * ( ...
         (al0-.5*al2)*(1-.5*Q2)*sin(fd) + ...
         .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fd)^3 + ...
         9/20*al2*Q2*sin(fd)^5 );                                   %[W/m2]
end

% Integrate incoming radiation for the high latitude sector, north of fd, 
% including icecover north of fi of fixed albedo ali. 

if fi<fd
    fi=fd;
end
Rc1 = (al0-.5*al2)*(1-.5*Q2) - (1-ali)*(1-.5*Q2);
Rc2 = .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2)) - (1-ali)*Q2*.5;
Rc3 = 9/20*al2*Q2;
Rc4 = -(al0-.5*al2)*(1-.5*Q2)*sin(fd) ...
      -.5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fd)^3 ...
      - 9/20*al2*Q2*sin(fd)^5 ...
      +(1-ali)*(1-.5*Q2) + (1-ali)*Q2*.5; 
Rc= 1/(1-sin(fd))*So/4*(sin(fi)*Rc1 + sin(fi)^3*Rc2 + sin(fi)^5*Rc3 + Rc4);
    %mean absorbed solar radiation in high latitude box

% Atmospheric energy transport
Hsh =  chi*delT;                               %sensible heat flux,[W/m2]
eps = Lv*cfw*exp(-5420/(PTad+273))*1e6/HHA;
Hlh  = eps*delT;                               %latent heat flux, [W/m2] 

%Ocean energy transport
gamma = rho*cp*1e6/HHA;               
Hq =  gamma*abs(q)*delT2;                      %THC heat flux,[W/m2]
Hv =  gamma*v*delT2;                           %WDC heat flux,[W/m2] 


delT=((Rw-Rc)-2*(Hq+Hv))/(B+2*(chi+eps));
Tm = (Rw+Rc-2*A)/(2*B);
Tswn=Tm+delT/2;
Tscn=Tm-delT/2;
Tsw=Tsw -(Tsw-Tswn)/10;
Tsc=Tsc - (Tsc-Tscn)/10;
Err2 = ((Tsc-Tscn)^2 + (Tsw-Tswn)^2)^0.5;

if Err1<1e-3 & Err2 <1e-3 break; end
end

if pCO2==pCO2o           %print mean global temperature,ice edge latitude 
    Tm                   %atmosphere temperature difference and ocean 
    fideg                %temperature difference for prescribed pCO2
    q
    Tsw-Tsc
    TLNI-THNI
 end    

spCO2(m)=pCO2;
sF(m) =F;
sq(m)= q;
sfi(m)=fideg;
tsc(m)=Tsc;
tsw(m)=Tsw;
tm(m)=Tm;
hsh(m)=Hsh*HHA/1e15;
hlh(m)=Hlh*HHA/1e15;
hq(m)=Hq*HHA/1e15;
hv(m)=Hv*HHA/1e15;

%uncomment to create standard case files

%spCO2s(m)=pCO2;sFs(m)=F;sqs(m)=q;sfis(m)=fideg;tscs(m)=Tsc;tms(m)=Tm;
%hshs(m)=Hsh*HHA/1e15;hlhs(m)=Hlh*HHA/1e15;hqs(m)=Hq*HHA/1e15;hvs(m)=Hv*HHA/1e15;

end

%uncomment to save standard case files 

%save('standardcaseCClim2007p','spCO2s','sFs', 'sqs','sfis','tscs','tms','hshs','hlhs','hqs','hvs','-v6') 

figure (1); hold on      %atmospheric moisture transportsteady state, mean surface temperature steady state ice edge latitude 
%as a function of pCO2, steady state Thermohaline circulationocean heat transports atmosphere heat transports 
    
subplot(3,2,1)
plot(spCO2(:),sF(:),'r','LineWidth',2); hold on
plot(spCO2s(:),sFs(:),'b','LineWidth',2); 
plot(pCO2o,0.55:0.001:0.8, 'g');
plot(0.5*pCO2o,0.55:0.001:0.8, 'g');
plot(2*pCO2o,0.55:0.001:0.8, 'g');
xlabel('pCO2 (ppm)');
ylabel('Atmospheric moisture transport (Sv)')

subplot(3,2,2)                                             
plot(spCO2(:),tm(:),'r','LineWidth',2); hold on
plot(spCO2s(:),tms(:),'b','LineWidth',2);
plot(pCO2o,11:0.01:20, 'g');
plot(0.5*pCO2o,11:0.01:20, 'g');
plot(2*pCO2o,11:0.01:20, 'g');
xlabel('pCO2 (ppm)');
ylabel('Mean surface temperature')

subplot(3,2,3)  
plot(spCO2(:),sfi(:),'r','LineWidth',2); hold on
plot(spCO2s(:),sfis(:),'b','LineWidth',2); 
plot(pCO2o,55:0.01:80, 'g');
plot(0.5*pCO2o,55:0.01:80, 'g');
plot(2*pCO2o,55:0.01:80, 'g');
xlabel('pCO2 (ppm)');
ylabel('Ice edge latitude')

subplot(3,2,4)                      
plot(spCO2(:),sq(:),'r','LineWidth',2); hold on
plot(spCO2s(:),sqs(:),'b','LineWidth',2);
plot(pCO2o,-6:0.01:18, 'g');
plot(0.5*pCO2o,-6:0.01:18, 'g');
plot(2*pCO2o,-6:0.01:18, 'g');
plot(0:1:pCO2o*(rmin+delr*j/2),0,'g');
xlabel('pCO2 (ppm)');
ylabel('Thermohaline circulation (Sv)')

subplot(3,2,5)                      
plot(spCO2(:),hq(:),'r:','LineWidth',2); hold on
plot(spCO2(:),hv(:),'r--','LineWidth',2);
plot(spCO2(:),hq(:)+hv(:),'r','LineWidth',2); 
plot(spCO2s(:),hqs(:),'b:','LineWidth',2);
plot(spCO2s(:),hvs(:),'b--','LineWidth',2);
plot(spCO2s(:),hqs(:)+hvs(:),'b','LineWidth',2); 
plot(pCO2o,0:0.01:5, 'g');
plot(0.5*pCO2o,0:0.01:5, 'g');
plot(2*pCO2o,0:0.01:5, 'g');
xlabel('pCO2 (ppm)');
ylabel('THC(..), WDC(--) and total ocean heat trans.')

subplot(3,2,6)
plot(spCO2(:),hsh(:),'r--','LineWidth',2);hold on
plot(spCO2(:),hlh(:),'r:','LineWidth',2);
plot(spCO2(:),hsh(:)+hlh(:),'r','LineWidth',2); 
plot(spCO2s(:),hshs(:),'b--','LineWidth',2);
plot(spCO2s(:),hlhs(:),'b:','LineWidth',2); 
plot(spCO2s(:),hshs(:)+hlhs(:),'b','LineWidth',2); 
plot(pCO2o,0:0.01:5, 'g');
plot(0.5*pCO2o,0:0.01:5, 'g');
plot(2*pCO2o,0:0.01:5, 'g');
xlabel('pCO2 (ppm)');
ylabel('Sens.(--), lat.(..) and total atmos. heat trans.')


%-------------