%Two box, energy balance model with albedo feedback
%and greenhouse gas forcing;
%Change parameters r and q for
%simulating effects of changed atmospheric concentrations of C02 and CH4, 
%respectively<;
%Change parameter s for choosing different calculations for
%CO2 radiative forcing (see below)

%revised September 2006, GS

clear all

load standardcase
fd=pi/6;         %dividing latitude: 30 degrees
So=1365;         %present day solar constant, W/m2
HHA  = 1.27e14;      % Half hemisphere area

% Albedo, function of latitude (Hartmann 1994)
al0  =  0.7;
al2  = -0.175;

ali  =  .62; % Albedo of ice covered area, st. val. 0.62
Ti   = -10;  % Fixed ice line temperature, st. val. -10

pCO2o =280e-6;  %(ppmb)         %Pre-anthropogenic(PA) pCO2
pCH4o =700;  % (ppbv)           %Pre-anthropogenic(PA) pCH4 

Tsw=90;                 %inital guesses for Tsw and Tsc
Tsc=Tsw-10;

delF=0.005;             %F is the solar constant divided by So, st. val 0.005
j=400;                  %total stable steady state solution number, st. val 400 
n = 5000;               %maximum number of interations
Fmax=1.5;               %maximum F, st. val 1.5

r=1;                    %multiple of PA pCO2     
q=1;                    %multiple of PA pCH4 

s=1;                    %s=1 gives CO2 radiative forcing from Myhre .et al. (M),
                        %Geophys.Res.Let.25, 2715-2718 (1998) for "near present day" pCO2;
                        %s=2 gives CO2 radiative forcing from Caldera and Kasting (CK),Nature,
                        %360, 721-723 (1992) for "very high" pCO2
 
pCO2=r*pCO2o;
pCH4=q*pCH4o;

for m = 1:j       %stepping over different values of F, the scaled solar constant 

if m<j/2
 
Q   =  So*(Fmax-delF*(m-1));                    %solar constant, i.e F*So
else
Q   =  So*(Fmax-delF*j+delF*(m-1)); 
end

Q2   = -0.482;     %constant for calculating incoming solar radiation as a function
                   %of latitude, see Hartman 1994


ACH4  = 0.036*(pCH4^(.5)-700^(.5))-...
       [  0.47*log(1+2.01e-5*(pCH4*270)^(.75)+5.31e-15*pCH4*(pCH4*270)^(1.52))...
        - 0.47*log(1+2.01e-5*( 700*270)^(.75)+5.31e-15* 700*( 700*270)^(1.52)) ];
% Change in outgoing longwave radiation as a function of pCH4 from M


if s==1
    
Ao   =  212.53;         %Zero degree background radiation, st. val. 212.53[W/m2]
ACO2  = 5.35*log(pCO2/280e-6);   %From M
A = Ao - ACO2 - ACH4;
B    =  1.9;           % Sensitivity of LW radiation to temp, st. val 1.9  [W/m2/K]]

else
    
lr=log(r*280/300);                   %note that reference pCO2 in CK is 300 ppm
A = -321.62 + 9.161*lr - 3.164*lr*lr + 0.5468*lr*lr*lr; %first constant adjusted from CK's
                                                        %value of -326.4 to get mean T of 15
B = 1.953 - 0.04866*lr + 0.01309*lr*lr - 0.002577*lr*lr*lr;
A = A + B*273-ACH4;           %to convert back to reference temperature of zero

end

chi  =  1.6;        % "Atmospheric" exchange coefficient for heat      [W/m2/K]
                    % standard value, 1.6, in this simple model chi
                    % actually represents the combined atmosphere and ocean
                    % heat transport
                    
for i=1:n           %iteration to find stable steady states  
        
% Establish the atmospheric temperature profile, 2.order legendre pol. in 
% sine of lat, Ta(lat) = PTa(1) + .5*PTa(2) * ( 3*sin(lat)^2 -1 )
% Coefficients are calculated so that the area weighted mean of the profile
% matches the surface ocean temperatures in each sector.

CTa(1,:) = [ 1 .5*(sin(fd)^2-1)                   ];
CTa(2,:) = [ 1 .5*(sin(fd)-sin(fd)^3)/(1-sin(fd)) ];
RTa      = [ Tsw Tsc]';
PTa      = CTa\RTa;

 %Calculate the latitude fi where Ta(fi)=Ti
fimax=pi/2-0.001; 
fimin=0.0001;
Tzero=PTa(1) - .5*PTa(2);           %temperature at equator
if Tzero<=Ti
    fi=fimin;
    fideg=0;
else    
fi  = min(fimax , asin((2/(3*PTa(2))*(Ti-PTa(1)+PTa(2)/2))^(.5)) );
fideg=fi*180/pi;
end
if fi ==fimax
    fideg=90;
end   

%fideg=63.676;fi=fideg*pi/180;       %Sets fixed ice edge when uncommented

% Integrated incoming shortwave radiation for the low latitude sector,
% south of fd, including ice cover if fi<fd

if fi<fd

Rw1= 1/sin(fi)*Q/4 * ( ...
         (al0-.5*al2)*(1-.5*Q2)*sin(fi) + ...
         .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fi)^3 + ...
         9/20*al2*Q2*sin(fi)^5 );
     
Rw2=  1/(sin(fd)-sin(fi))*Q/4 * ( ...
         (1-ali)*(1-.5*Q2)*(sin(fd)-sin(fi)) + ...
         .5*(Q2*(1-ali))*(sin(fd)-sin(fi))^3); 
Rw = (Rw1*sin(fi)+Rw2*(sin(fd)-sin(fi)))/sin(fd);                                                                                                        %box
else
Rw   = 1/sin(fd)*Q/4 * ( ...
         (al0-.5*al2)*(1-.5*Q2)*sin(fd) + ...
         .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fd)^3 + ...
         9/20*al2*Q2*sin(fd)^5 );                                   %[W/m2]
end

%Rw is mean adsorbed solar radiation in the warm box   


% Integrate incoming radiation for the high latitude sector, north of fd, 
% including icecover north of fi of fixed albedo ali. 

if fi<fd
    fi=fd;
end
Rc1 = (al0-.5*al2)*(1-.5*Q2) - (1-ali)*(1-.5*Q2);
Rc2 = .5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2)) - (1-ali)*Q2*.5;
Rc3 = 9/20*al2*Q2;
Rc4 = -(al0-.5*al2)*(1-.5*Q2)*sin(fd) ...
      -.5*(Q2*(al0-.5*al2)+al2*(1-.5*Q2))*sin(fd)^3 ...
      - 9/20*al2*Q2*sin(fd)^5 ...
      +(1-ali)*(1-.5*Q2) + (1-ali)*Q2*.5; 
Rc  = 1/(1-sin(fd))*Q/4*( sin(fi)*Rc1 + sin(fi)^3*Rc2 + sin(fi)^5*Rc3 + Rc4);
    
%Rc is mean absorbed solar radiation in the cold box:

% Atmospheric energy transport
Ha =  chi*(Tsw-Tsc);                                %[W/m2] 

delT=(Rw-Rc)/(B+2*chi);
Tm = (Rw+Rc-2*A)/(2*B);
Tswn=Tm+delT/2;
Tscn=Tm-delT/2;
Tsw=Tsw -(Tsw-Tswn)/10;
Tsc=Tsc - (Tsc-Tscn)/10;
Err = ((Tsc-Tscn)^2 + (Tsw-Tswn)^2)^0.5;
if Err<1e-3 break; end
end

if Q/So==1                  %print mean global temperature, ice edge 
    Tm                      %latitude and meridional heat transport  
    fideg                   %for prescribed, scaled solar constant
    HeatTrans=Ha*HHA/1e15
end    

sQ(m) =Q;                 %creates vectors for plotting
sfi(m)=fideg;
tm(m)=Tm;
h(m)=Ha*HHA/1e15;
td(m)=delT;

%sQs(m) =Q;sfis(m)=fideg;tms(m)=Tm; tds(m)=delT;      %run to create standcase files


if j<=20
figure (1)  %plots stable steady states, latitude dependent temperature
            %profiles and marks ice edge latitudes 

lat= 0 : (pi/2)/100 : (pi/2); % Latitude array 
Ta =  PTa(1) + .5*PTa(2) * ( 3*sin(lat).^2 -1 );
p=plot(lat*180/pi,Ta);              set(p,'color','k','linewidth',2), hold on
l=line([0  fd  ]*180/pi,[Tsw Tsw]); set(l,'color','r','linewidth',1)
l=line([fd pi/2]*180/pi,[Tsc Tsc]); set(l,'color','b','linewidth',1)
l=line([fideg fideg],[-20 40]   ); set(l,'color','k','linewidth',1)
plot([fideg],[Ti],'ko')
%t=text([fideg+1],[Ti],strcat('\phi_i =',num2str(fideg),'^o') );
%t=text([fideg],[30]  ,strcat('Tm =',num2str(Tm),'^oC'    ) ); 
xlabel('\phi'); ylabel('T (^oC)')
end

end 

%save('standardcase', 'sQs', 'sfis', 'tms','tds','-v6') 

if j>20
figure (2); hold on      %Plots stable steady state ice edge latitudes
                         %as functions of the scaled solar constant
subplot(3,1,1)
plot(sQ(1:j)/So,sfi(1:j),'r','LineWidth',2); hold on
plot(sQs(1:j)/So,sfis(1:j),'b','LineWidth',2); 
plot(1,0:0.5:90,'g:');
plot(0.95,0:1:90,'g:');
plot(0.75,0:1:90,'g:');
xlabel('Scaled solar constant');
ylabel('Ice edge latitude')

%figure (3); hold on %Plots stable steady state, mean surface temperatures
                         %as functions of the scaled solar constant
subplot(3,1,2)
plot(sQ(1:j)/So,tm(1:j),'r','LineWidth',2); hold on
plot(sQs(1:j)/So,tms(1:j),'b','LineWidth',2);
plot(1,-70:1:70,'g:');
plot(0.95,-70:2.5:70,'g:');
plot(0.75,-70:2.5:70,'g:');
plot(0.5:0.005:1.5,15,'g:');
xlabel('Scaled solar constant');
ylabel('Mean temperature')

subplot(3,1,3)
plot(sQ(1:j)/So,td(1:j),'r','LineWidth',2); hold on
plot(sQ(1:j)/So,tds(1:j),'b','LineWidth',2); hold on
plot(1,0:0.1:9,'g:');
plot(0.95,0:0.1:9,'g:');
plot(0.75,0:0.1:9,'g:');
xlabel('Scaled solar constant');
%ylabel('Heat transport')
ylabel('Temperature difference')
end


%-------------