      program rcm
************************************************************************
*		     OGI 1DRCM Version 1.0			       *
************************************************************************
*     This program is an 18 layer one-dimensional time-dependent
*     radiative convective model of the Earth-atmosphere system.
*     It was developed by Robert M. MacKay and M.A.K. Khalil at
*     the Oregon Graduate Institute of Science and Technology, Center
*     of Atmospheric Studies, Beaverton, Oregon 97006-1999 USA.
*     Disk copies of this program may be obtained from the authors at
*     any time.
*
*     Copyright (C) R.M. MacKay and M.A.K. Khalil 1991
************************************************************************
*     *****Description of variables*****
************************************************************************
*     t, z, p, b, uo, and u1 are the temperature, height, pressure,
*     planck function, first ozone profile, optional ozone profile
*     associated with the top and bottom of each atmospheric layer
*     respectively.  sg is the sigma variable used for identifying
*     the pressure of each layer.
***********************************************************************
*     at, pa, za, uoa, cp are the temp., press., height, ozone amt,
*     and total heat capacity for each layer. c is the ratio of the
*     total heat capacity, moist + dry, to the dry heat capacity of
*     each layer.
***********************************************************************
*     ta is the average temperature for a path between layer i and j
*     ut is the transfered from subroutines n2o5cool and n2o6cool.
***********************************************************************
*     gam is the critical lapse rate for convective adjustment at each
*     layer.  ce+ch the net convective heating at each layer in K/step
***********************************************************************
*     tr2 is the H2O CO2 overlap transmission
*     tco2n2 is the n2o CO2 overlap transmission
*     tnc4 is the N2O CH4 overlap transmission
*     tr5 is the H2O N2O (1285 cm-1) overlap transmission
*     tr3 is the H2O Ozone overlap transmission
*     tr4 is the H2O CH4 overlap transmission
*     tr6 is the H2O N2O (580 cm-1) overlap transmission
***********************************************************************
*     dg is the total solar heating of each layer in K/day
*     sho,shw,shoc,shwc,shco2,sho2 are the solar heating terms
*     due to ozone, water, ozone cloudy portion, water cloudy
*     portin, CO2, and oxygen respectively in K/day.  Note:
*     these calculations assume that the layer is dry an
*     hence a term (1+c) is use in the actual calculations
*     of temperature change in subroutine tempchg.
*     g(19) is the total flux of solar radiation absorbed
*     by the surface and g(1) is the solar energy into the
*     top of the atmosphere.  gw,go,gwc, and goc are the above
*     for the subroutines for water and ozone clear and cloudy skies.
*     xk, and pk are k and probability values for k distribution
*     method for calculation of solar absorption by water vapor
*     in cloudy skies.
***********************************************************************
*     df is the total ir cooling (assuming a dry layer) in K/day
*     dc,dw,do,d4,d5,d6 are the corresponding cooling rates for
*     CO2, water, ozone methane, nitrous oxide 1285 cm-1, and
*     nitrous oxide 580 cm-1 respectively. As with solar absorption
*     the effective heat capacity due to moisture changes is accounted
*     for in subroutine tempchg
*     f(19) is the total downward flux of ir radiation at the
*     earth's surface, f(1) is the total upward flux of ir radiation at
*     the top of the atmosphere, and f(nt) is the total downward flux of
*     ir radiation at the tropopause.  fw,fc,fo,f4,f5, and f6 are the
*     corresponding values returned from the subroutines water, co2,
*     ch4cool, o3cool, n2o5cool, and n2o6cool
***********************************************************************
*
      real t(20),z(20),p(20),sg(20),b(20),uo(20)
      real at(20),pa(20),za(20),uoa(20),cp(20),c(20)
      real ta(20,20),ut(20,20),ca(20,20),ca1(20,20)
      real gam(20),ch(20),ce(20)
      real tr2(20,20),tco2n2(20,20),tnc4(20,20),tr5(20,20)
      real tr3(20,20),tr4(20,20),tr6(20,20)
      real dg(20),sho(20),shw(20),shoc(20),shwc(20),shco2(20),sho2(20)
      real g(20),gw(20),go(20),gwc(20),goc(20)
      real xk(10),pk(10),at1(20)
      real qc(4),dq(4,14),ec(14),vc(14),sc(14)
      real dc(20),dw(20),do(20),d4(20),d5(20),ds(20)
      real df(20),f(20),fo(20),fw(20),fc(20),f4(20),f5(20),fs(20)
************************************************************************
*     s0-solar constant / 2.0
*     theta- average solar Zenith angle
*     step- time step between calculations (days)
*     albedo - reflectivity of the top atmospheric layer( usually=0.0)
*     s1- sum of abs(delta T) for each layer during each step
*     low- if s1 is less than low then equilibrium has been reached
*     tme- time (days)
*     day- if tme if greater than day then program execution stops
*     sout- the outward flux of solar radiation from the top of atm
*     lap-lapse rate for fixed lapse rate adjustment (usually 6.5)
*     cp(19)- is input as ocean mixed layer depth then converted
*     to mixed layer heat capacity
*     sun- s0*cos(theta)
*     abw-total solar absorption by water vapor (clear skies)
*     abwc-total solar absorption by water vapor (cloudy skies)
*     abo-total solar absorption by ozone (clear skies)
*     aboc-total solar absorption by ozone (cloudy skies)
*     abo2-total solar absorption by oxygen
*     abco2-total solar absorption by carbon dioxide
*     abtot-total solar absorption by all gases
*     c20-initial concentration of carbon dioxide
*     rate- growth rate (1/yr) of CO2
*     c40-initial concentration of methane
*     n20-initial concentration of nitrous oxide
*     c21- concentration of CO2 after time = tm1
*     c41-concentration of CH4 after time = tm2
*     N21-concentration of N2O after time = tm4
*     cf11- conentration of f11 in pptv
*     cf12- concentration of f12 in pptv
*     tm3-time after wich ozone profile changes
*     con2-time dependent concentration of CO2
*     depth- optical depth of the cloud
*     rg- surface albedo
*     ac- horizontal cloud extent
*     pout- output sent to files every (step*pout days)
*     dlast-for multiple runs if dlast is greater than day
*    then then program resets with input from cdat2 and reruns
*    automatically
*     kap- cloud layer
*     nt- tropopause layer
*     out1,out2,out3- controlls output of data see below
*      *
************************************************************************
      real s0,theta,step,albedo,s1,low,tme,day,sout,lap
      real sun,abw,abo,abwc,aboc,abo2,abco2,abtot,cf11,cf12
      real c20,rate,c40,n20,c21,c41,n21,tm1,tm2,tm3,tm4
      real con2,depth,rg,ac,pout,dwat,aoz,boz,coz,a1z,b1z,c1z
      integer kap,nt,out1,out2,out3,qp,nr
************************************************************************
      data xk(1),xk(2),xk(3),xk(4),xk(5),xk(6),xk(7),xk(8)
     > /.00004, .002, .035, .377, 1.95, 9.40, 44.6, 190./
      data pk(1),pk(2),pk(3),pk(4),pk(5),pk(6),pk(7),pk(8)
     > /.647, .0698, .1443, .0584, .0335, .0225, .0158, .0087/
************************************************************************
      data qc(1),qc(2),qc(3),qc(4) /1.0,.0112,.00408,.000742/
      data vc(1),vc(2),vc(3),vc(4),vc(5),vc(6),vc(7),vc(8),
     x vc(9),vc(10),vc(11),vc(12),vc(13),vc(14) /667.38,
     x 618.029,720.805,667.751,647.063,791.446,597.34,
     x 741.72,668.11,544.29,668.67,652.52,720.29,615.89/
      data sc(1),sc(2),sc(3),sc(4),sc(5),sc(6),sc(7),sc(8),
     x sc(9),sc(10),sc(11),sc(12),sc(13),sc(14) /194.0,4.27
     x,5.0,15.0,.7,.022,.14,.144,.85,.01,.3,.045,.005,.015/
      data ec(1),ec(2),ec(3),ec(4),ec(5),ec(6),ec(7),ec(8),
     x ec(9),ec(10),ec(11),ec(12),ec(13),ec(14)/0.00,667.38,
     x 667.38,667.38,1285.41,1285.41,1335.13,1335.13,1335.13,
     x 1388.185,1388.185,1932.473,2076.855,1932.473/
      do 2 i=1,4
        do 1 j=1,14
        if (i.ge.3) then
         dq(i,j)=0.78
          else
           if(j.eq.4.or.j.eq.7.or.j.eq.8.or.j.eq.9.or.j.eq.12) then
             dq(i,j)=.78
            else
             dq(i,j)=1.56
            endif
           endif
1       continue
2      continue
*     open input/output files
      open(unit=10,file='datrcm',status='old')
      qp=0
       nr=0
3	read(10,*) out1,out2,out3
*      if(out1.gt.1.or.out2.gt.1.or.out3.gt.1)then
*      print*,'bad value of out1,2,or 3 in datrcm'
*      goto 100
*      end if
      if(qp.eq.0)then
      if(out1.eq.1) then
      open(unit=20,file='out1' )
      end if
      if (out3.eq.1) then
      open(unit=25,file='out3' )
      end if
      if(out2.eq.1) then
      open(unit=30,file='out2' )
      end if
      end if
      qp=1.0
************************************************************************
*      read input data
	 read(10,1034) (at(i),i=1,19)
	 read(10,*) aoz,boz,coz
	 read(10,*) a1z,b1z,c1z
*	 if (aoz.lt.0.or.boz.lt.0.or.coz.lt.0.or.a1z.lt.0.or.b1z.lt.0
*     > .or.c1z.lt.0) then
*	 print*,'bad ozone parameters in datrcm'
*	 goto 100
*	 end if
         read(10,*) tm3
	 read(10,*) s0,theta,step,low,day
	 if(step.gt.12.or.step.lt.0) then
	 print*,'step size is out of acceptable range'
	 goto 100
	 endif
         read(10,*) lap,cp(19)
	 read(10,*) c20,c21,tm4
*	 if(c20.lt.0.or.c21.lt.0)then
*	 print*,'check CO2 concentration in datrcm'
*	 goto 100
*	 end if
	 read(10,*) c40,c41,tm2
*	 if(c40.lt.0.or.c41.lt.0)then
*	 print*,'check CH4 concentration in datrcm'
*	 goto 100
*	 end if
	 read(10,*) n20,n21,tm1
*	 if(n20.lt.0.or.n21.lt.0)then
*	 print*,'check N2O concentration in datrcm'
*	 goto 100
*	 end if
	 read(10,*) cf11
	 read(10,*) cf12
	 if(cf12.lt.0)then
	 print*,'check F-12 concentration in datrcm'
	 goto 100
	 end if
	 read(10,*) ac,kap
*	 if(ac.gt.1.0.or.ac.lt.0)then
*	 print*,'check cloud fraction in datrcm'
*	 goto 100
*	 endif
*	 if(kap.gt.18.or.kap.lt.1)then
*	 print*,'check cloud height in datrcm'
*	 goto 100
*	 endif
	 read (10,*) rg,depth
*	 if(rg.gt.1.0.or.rg.lt.0.0)then
*	 print*,'check surface albedo in datrcm'
*	 goto 100
*	 endif
*	 if(depth.gt.100.or.depth.lt.0.0)then
*	 print*,'check cloud optical depth in datrcm'
*	 goto 100
*	 endif
	 read(10,*)tme
         read(10,*)pout
	 read(10,*)dwat
       do 5 i=1,19
        ce(i)=0.0
	ch(i)=0.0
5      continue
	 albedo=0.0
	 rate=0.0
*  use heat capacity units of W-Day/(m^2 K)
	 cp(19)=cp(19)*4.186e6/86400
         nt=6.0
         sun=s0
        s0=s0*(1.0-albedo)
        theta=theta*3.1415926/180.0
        sun=sun*cos(theta)
         g(1)=s0*cos(theta)
*
*       set initial pressures,temperature,height, and reference lapse
*       rate of each layer.
        call presset(sg,p,pa)
	call tempset(t,at,p)
	call height(z,za,p,pa,at)
      do 9 i=1,19
       uo(i)=(aoz+aoz*exp(-boz/coz))/(1+exp((z(i)-boz)/coz))
	 uoa(i)=(aoz+aoz*exp(-boz/coz))/(1+exp((za(i)-boz)/coz))
9     continue
       call lap1(gam,lap)
      print*,' '
      print*,' '
      print*,' '
      print*,' '
      print*,' '
      print*,' '
      print*,' '
      print*,'********************************************************'
      print*,'*        THE OGI 1DRCM  VERSION 1.0   Jan 1991         *'
      print*,'********************************************************'
      print*,'This program is an 18 layer 1-dimensional time-dependent'
      print*,'radiative convective model of the Earth-atmos. system.'
      print*,'                It was developed by'
      print*,'            R.M. MacKay and M.A.K. Khalil '
      print*,'                        at'
      print*,'The Oregon Graduate Institute of Science and Technology'
      print*,'           Center for Atmospheric Studies'
      print*,'               19600 NW Von Neumann Dr'
      print*,'          Beaverton, Oregon 97006-1999 USA.'
      print*,'                  (503) 690-1093'
      print*,'********************************************************'
      print*,' '
      print*,'    Copyright (C) R.M. MacKay and M.A.K. Khalil 1991    '
      print*,' '
      print*,' '
      print*,' '
*
*      main iterative part of the program
*      calculate pressure weighted average temperature btwn layer i and j
10      call tempave (at,sg,ta)
        if(tme.lt.dwat)then
        do 11 i=1,19
         at1(i)=at(i)
11      continue
	end if
*     calculate the n2o (1285 cm-1) ch4 overlap
         call n2och4 (tnc4,p,pa,at,c40)
*      moist adiabatic lapse rate calculation (used instead of lap1 above)
	if(lap.lt.0)then
	call lap2 (gam,at,pa)
	end if
*        recalc temps at top and bottom of each layer
	call tempset (t,at,p)
*       optional used to change the concentrations of trace gases
*        for extra long runs
*       call conchg(tme,n20,c20,c40)
*       check tme with tm1,tm2,tm3,tm4 to see is there is a step-wise
*       trace gas concentration change
      if ((tme.gt.tm1.and.tme.le.tm1+step).or.(tme.gt.tm2.and.tme.le
     x.tm2+step).or.(tme.gt.tm3.and.tme.le.tm3+step).or.
     x(tme.gt.tm4.and.tme.le.tm4+step))then
      call conchk(tme,n20,c20,c40,n21,c21,c41
     x,tm1,tm2,tm3,tm4,uo,uoa,step,out1,out2,z,za,a1z,b1z,c1z)
        endif
*
*     Calculate h2o IR absortion and transmissions
       call water (pa,at,p,t,ta,dw,fw,tr2,ac,kap,tr3,
     x tr4,tr5,tr6,nt,at1)
*
*       update co2 concentration and calculate IR absorption transmission
        con2=c20*exp(rate*tme/365.)
        call co2 (t,p,pa,dc,fc,ta,tr2,tr3,con2,ac,kap,tco2n2,nt,
     x qc,dq,ec,vc,sc,ca1,ca,nr)
*
*       Calculate clear sky solar absorption due to water vapor
        call h2osolar(s0,theta,pa,p,at,shw,gw,abw,ac,rg,kap,at1)
*
*      Calculate cloudy sky solar absorption due to water vapor
       call solclod (p,pa,at,xk,pk,depth,theta,s0,rg,kap,ac,shwc
     x ,abwc,gwc,at1)
*
*       Methane IR absorption
        call ch4cool (ta,p,pa,t,d4,f4,tr4,c40,ac,kap,nt)
*
*       N2O (1285 cm-1) IR absorption
        call n2o5cool (ta,p,pa,t,d5,f5,tr5,tnc4,n20,ac,kap,nt,ut)
*
*       N2O (590 cm-1) IR absorption
        call n2o6cool (p,pa,ta,t,d5,f5,tr6,tco2n2,n20,ac,kap,nt,ut)
*
	do 15 i=1,19
         ds(i)=0.0
         fs(i)=0.0
15	 continue
	call small (ta,p,pa,t,ds,fs,cf11,ac,kap,nt,850.0,1828.0)
*
	call small (ta,p,pa,t,ds,fs,cf11,ac,kap,nt,1075.0,679.0)
*
	call small (ta,p,pa,t,ds,fs,cf12,ac,kap,nt,912.0,1446.0)
*
	call small (ta,p,pa,t,ds,fs,cf12,ac,kap,nt,1090.0,1140.0)
*
	call small (ta,p,pa,t,ds,fs,cf12,ac,kap,nt,1150.0,767.0)
*
*       call small (ta,p,pa,t,ds,fs,.00028,ac,kap,nt,1150.0,767.0)
*
*       call small (ta,p,pa,t,ds,fs,.00028,ac,kap,nt,1150.0,767.0)
*
*       Ozone planck function and IR absorption
        call bpl(b,t,1042.)
*
	 call o3cool (b,p,pa,uo,uoa,do,fo,ac,kap,nt,tr3)
*
*       ozone clear and cloudy skies solar absorption
        call solo3 (s0,theta,uo,p,sho,go,abo,ac,rg)
*        
      call sco3 (s0,theta,uo,p,shoc,goc,aboc,ac,rg,kap,depth)
*
*     solar absorption for CO2
      call sco2 (p,pa,shco2,abco2,c20,s0,theta,ac,kap)
*
*     solar absoprption for molecular oxygen
      call soxy (p,sho2,abo2,s0,theta)
*
*     combine IR and Solar heating rates and fluxes
        call combine(dw,dc,do,d4,d5,ds,shw,shwc,shoc
     x  ,sho2,shco2,sho,dg,df)
          f(19)=fw(19)+fc(19)+fo(19)+f4(19)+f5(19)+fs(19)
          f(18)=fw(18)+fc(18)+fo(18)+f4(18)+f5(18)+fs(18)
         f(nt)=fw(nt)+fc(nt)+fo(nt)+f4(nt)+f5(nt)+fs(nt)
        g(19)=goc(19)+gwc(19)+go(19)+gw(19)-
     x(1-rg)*(abo2+abco2)
        abtot=abw+abo+aboc+abwc+rg*(abo2+abco2)
        f(1)=fw(1)+fc(1)+fo(1)+f4(1)+f5(1)+fs(1)
        sout=albedo*sun+(g(1)-abtot)
*      change the average temperature of each layer
      call tempchng(pa,p,cp,at,dg,df,g,step,s1,f,ch,ce,kap,ft,c,at1)
*     recalculate the height of each layer
        call height(z,za,p,pa,at)
*     perform the lapse rate adjustment
        call lapadj (at,gam,za,cp,ch,nt,ce)
*
*      update time
          tme=tme+step
*      update printer count
         prt=prt+1.
          if (prt.gt.pout.or.tme.lt.02*step) then
	 prt=0.0
	 if(tme.lt.2*step) then
	 print*,'The respective screen outputs are:'
      print*,'sum of all DT per step(K), time(Days), t19(K), t18(K)'
	 print*,'abs solar (W/m^2), IR out (W/m^2)'
	 print*,' '
	 end if
*    output to screen
	 print*,s1-low,tme,t(19),t(18),sun-sout,f(1)
            goto 19
          end if
*
         if (tme.gt.day) then
         goto 19
         end if
*
         if (s1.gt.low) then
          goto 10
          end if
*********************************************************************
*      output of IR fluxes and heating/cooling rates(which are optional
*      in this output section
19     if(out1.eq.1) then
       if (tme.le.step.or.tme.gt.day) then
*      optional remove * at the beginning of each line to obtain output
      write(20,*)
      write (20,1019)
      write (20,1010)
      do 120 i=1,19
	 write(20,1000) p(i),z(i),t(i),do(i),dw(i),dc(i),d4(i),
     1 d5(i),ds(i),df(i)/(1.0+c(i))
120	continue
	write(20,*)
25	write(20,1018)
      write(20,1009)
      do 30 i=1,18
	write(20,1005) (ce(i)+ch(i))/step,dg(i)/(1+c(i))
     x,shoc(i)+sho(i),shwc(i)+shw(i),sho2(i),shco2(i)
30    continue
       write(20,*)
*
*
       write (20,1050)
       write (20,1041)
       write (20,1042) c20,c40,n20
       write(20,1051)
       write (20,1043) ac
       write (20,1048) rg
       write (20,1049) theta
       write(20,1052) z(nt)
       write (20,1050)
       write (20,1051)
       write(20,1020)
       end if
       write(20,1022) tme,f(19),g(19),f(1),f(nt),abtot,step,
     x (sun-abtot)/sun,s1
       end if
*********************************************************************
*     output of vertical temperature profile and convective adjustment
*     in K/step
      if(out2.eq.1) then
         write (30,1011)
          do 45 i=1,19
	  write(30,1035)  t(i),ch(i),ce(i)
45      continue
        write(30,1036) tme
      end if
**********************************************************************
*   output of surface time, temperature, ir flux leaving the
*    top of the atmosphere and total absorbed solar energy
       if (out3.eq.1) then
       if (tme.le.step) then
        write(25,1060)
       end if
	   write(25,1038)  tme,t(19),f(1),abtot
       end if
**********************************************************************
         if (tme.gt.day) then
         goto 100
         end if
*
         if (s1.gt.low) then
          goto 10
          end if
*
100	print*,'done'
*********************************************************************
1000  format(2(f6.3,2x),f6.2,2x,7(f6.3,2x))
1005  format (2x,6(f7.3,1x))
1009  format  (2x,'Convect',1x,'tot het',1x,
     2'het o3',2x,'het wat',1x,'het O2',2x,'het CO2')
1010    format('press',3x,'hght',4x,'temp',4x,'do',6x,'dw',6x,'dc',
     1 6x,'d4',6x,'d5',6x,'ds',6x,'dtot')

1011    format ('Temp (K)',6x,'Con ch',8x,'Con ce')
1019  format ('cooling rates in K/day')
1018  format ('heating rates in K/day')

1020   format('Time',9x,'F19',6x,'g19',5x,'F1',6x,'FNT',6x,'abtot',4x,
     x 'step',5x,'albedo',3x,'S1',6x)
1022   format(9(f7.2,2x))
1034  format(f10.5)
1035  format (3(f10.5,4x))
1036  format ('Time=',f8.2,'days')
1038  format (4(f10.5,4x))

1042  format (25x,3(f8.4,2x))
1041  format (1x,'concentration of (ppmv)    CO2       CH4      N2O')
1043  format (2x,'cloud fraction',1x,f10.5)
1048  format (2x,'ground albedo',1x,f10.5)
1049  format (2x,'average solar zenith (radians)',1x,f10.5)
1050  format (80('*'))
1051  format (' ')
1052  format(2x,'tropopause height(km)',1x,f10.5)
1060  format(1x,'time',10x,'Surface T',5x,'Fout',10x,'abtot')
*********************************************************************
       end
*********************************************************************
        subroutine conchg(tme,n20,c20,c40)
        real tme,n20,c40,c20
        if (tme.gt.1860.and.tme.lt.1960) then
        c20=285+.317*(tme-1860)
        c40=.8+.005*(tme-1860)
        end if
        if (tme.gt.1960.and.tme.lt.1970) then
        c20=316.7+.810*(tme-1960)
        c40=1.3+.01*(tme-1960)
        end if
        if (tme.gt.1970.and.tme.lt.1980) then
        c20=324.8+1.28*(tme-1970)
        c40=1.4+.015*(tme-1970)
        end if
        if (tme.gt.1980) then
        c20=337.6+1.59*(tme-1980)
        c40=1.55+.0175*(tme-1980)
        end if
        if (tme.gt.1940) then
        n20=.285+.0005*(tme-1940)
        end if
        return
        end
*****************************************************************
        subroutine conchk(tme,n20,c20,c40,n21,c21,c41
     x,tm1,tm2,tm3,tm4,uo,uoa,step,out1,out2,z,za,a1z,b1z,c1z)
	real uo(20),uoa(20),z(20),za(20)
        real c20,c21,n20,n21,c40,c41,tme,tm1,tm2,tm3,tm4
	real step,a1z,b1z,c1z
        integer out1,out2
        if (tme.gt.tm1.and.tme.le.tm1+step) then
	 n20=n21
        end if
        if (tme.gt.tm2.and.tme.le.tm2+step) then
	 c40=c41
        end if
        if (tme.gt.tm3.and.tme.le.tm3+step) then
	do 201 i=1,19
	  uo(i)=(a1z+a1z*exp(-b1z/c1z))/(1+exp((z(i)-b1z)/c1z))
	  uoa(i)=(a1z+a1z*exp(-b1z/c1z))/(1+exp((za(i)-b1z)/c1z))
201      continue
        end if
        if (tme.gt.tm4.and.tme.le.tm4+step) then
         c20=c21
*       c20=c21
        end if
          if(out1.eq.1) then
          write(20,1041)
       write (20,1042) c20,c40,n20,uo(19)
          write(20,1045)
         write (20,1030) tme
         end if
         if (out2.eq.1) then
         write(30,1041)
       write (30,1042) c20,c40,n20,uo(19)
         write(30,1045)
        write (30,1030) tme
        end if
1030  format (f10.5)
1045  format ('time in days')
1042  format (25x,4(f8.4,2x))
1041  format ('concentration of (ppmv)    CO2       CH4       N2O
     x O3tot')
        return
        end
*****************************************************************
      subroutine tempchng(pa,p,cp,at,dg,df,g,step,s1,f,ch,ce,
     xkap,abt,c,at1)
      real pa(20),at(20),dg(20),df(20),f(20),g(20),ch(19)
      real p(20),cp(20),ce(20),c(20),at1(20)
      real dt,s1,dr,step,e1,e,q,l
      integer kap
      abt=0.0
      et=0.0
      s1=0.0
      do 3400 j=1,18
*       if (j.eq.kap) then
*        h=1.0
*       end if
       e1=esat(at1(j)+1)
       e=esat(at1(j))
*     calculate modified heat capacity Manabe & Wetherald 1967?
      dr=rwat(pa(j),e1,j,kap)-rwat(pa(j),e,j,kap)
      l=2510-2.38*(at(j)-273)
       c(j)=.622*l*l*rwat(pa(j),e,j,kap)/(1.005*.287*at1(j)**2)
      cp(j)=(1.0+c(j))*1.038165e7*(p(j+1)-p(j))/86400
      dt=step*(dg(j)+((ce(j))*(1.0+c(j))/step)-df(j))
     x/(1.0+c(j))
       et=et+cp(j)*(ce(j))
       s1=s1+abs(dt+ch(j))
       at(j)=at(j)+dt
3400  continue
       et=et/step
        ce(19)=-et*step/cp(19)
      q=(g(19)-et+f(19)-5.67e-8*(at(19))**4)
       q=step*q/(cp(19))
       at(19)=at(19)+q
       return
        end
*******************************************************************
      subroutine combine (dw,dc,do,d4,d5,d6,shw,shwc,shoc,
     x sho2,shco2,sho,dg,df)
      real dw(20),dc(20),do(20),shw(20),sho(20),shwc(20),shoc(20)
      real dg(20),df(20),d4(20),d5(20),d6(20),sho2(20),shco2(20)
       do 3500 i=1,18
        df(i)=dw(i)+do(i)+dc(i)+d4(i)+d5(i)+d6(i)
        dg(i)=shw(i)+sho(i)+shwc(i)+shoc(i)+shco2(i)+sho2(i)
3500  continue
      return
      end
*
********************************************************************
      subroutine water  (pa,at,p,t,ta,dw,fw,tr2,ac,kap,tr3,tr4,
     x tr5,tr6,nt,at1)
      dimension pa(20), p(20), dw(20), t(20), fw(20),at(20),try(20,20)
      dimension  ta(20,20),em(20,20), am(20,20), tr1(20,20), tr2(20,20)
      dimension fu(20),fd(20),ftot(20),tr3(20,20)
      real tr4(20,20),tr5(20,20),tr6(20,20),trc(20,20),at1(20)
      real sig,cf, em1,u1,e,ac
      integer kap,nt
      sig=5.67e-8
	call h20trans (pa,p,tr1,tr2,tr3,tr4,tr5,try,tr6,trc,kap,at1)
*
      do 740  i=2,19
         do 739 j=1,i-1
              e=esat(at1(j))
              r=rwat(pa(j),e,j,kap)
              if (j.eq.1) then
              u1=((p(j+1))/2)*sqrt(273/at(j))*r*1033*(p(j+1))
              else
	  u1=((pa(j)+p(j+1))/2)*sqrt(273/at(j))*r*1033*(p(j+1)-pa(j))
              end if
             do 735 k=j+1,i-1
              e=esat(at1(k))
              r=rwat(pa(k),e,k,kap)
              u1=u1+pa(k)*sqrt(273/at(k))*r*1033*(p(k+1)-p(k))
735         continue
           em1=(1-.5*((1/(1+19*sqrt(u1)))+(1/(1+3.5*sqrt(u1)))))
           em1=em1*.59*((273/ta(j,i))**.25)
           em(j,i)=em1+trc(j,i)+try(j,i)+tr1(j,i)
           am(j,i)=.847*(u1**.022)*em(j,i)
739       continue
740    continue
      do 760  i=1,19
         do 759 j=1,i
              e=esat(at1(i))
              r=rwat(pa(i),e,i,kap)
      u1=((pa(i)+p(i))/2)*sqrt(273/at(i))*r*1033*(pa(i)-p(i))
            do 755 k=j,i-1
             e=esat(at1(k)) 
              r=rwat(pa(k),e,k,kap)
              u1=u1+pa(k)*sqrt(273/at(k))*r*1033*(p(k+1)-p(k))
755         continue
           em1=(1-.5*((1/(1+19*sqrt(u1)))+(1/(1+3.5*sqrt(u1)))))
           em1=em1*.59*((273/ta(j,i))**.25)
           em(i,j)=em1+trc(i,j)+try(i,j)+tr1(i,j)
           am(i,j)=.847*(u1**.022)*em(i,j)
759       continue
760    continue
        fd(1)=0.0
        do 780 i=2,19
          if (i.gt.kap) then
            cf=1.0-ac
           else
           cf=1.0
          end if
         fd(i)=cf*sig*em(1,i)*t(1)**4
	   do 779 j=1,i-1
           fd(i)=fd(i)+cf*am(j,i)*sig*(t(j+1)**4-t(j)**4)
779    continue
780   continue
	  fd(2)=em(1,2)*sig*t(1)**4
          do 785 i=kap+1,19
            fd(i)=fd(i)+ac*sig*t(kap+1)**4
           do 784 j=kap+1,i-1
             fd(i)=fd(i)+ac*am(j,i)*sig*(t(j+1)**4-t(j)**4)
784   continue
785   continue
          fu(19)=sig*t(19)**4
         do 792 i=1,18
            if (i.lt.kap+1) then
             cf=1.0-ac
             else
             cf=1.0
            end if
         fu(i)=cf*sig*t(19)**4+sig*(at(19)**4-t(19)**4)*(1-em(19,1))
          do 791 j=i,18
          fu(i)=fu(i)-cf*am(j,i)*sig*(t(j+1)**4-t(j)**4)
791    continue
792    continue
        do 796 i=1,kap
          fu(i)=fu(i)+ac*sig*t(kap)**4
          do 795 j=i,kap-1
           fu(i)=fu(i)-ac*am(j,i)*sig*(t(j+1)**4-t(j)**4)
795    continue
796    continue
        do 798 i=1,19
         ftot(i)=fd(i)-fu(i)
798     continue
        fw(19)=fd(19)
        fw(18)=ftot(19)
        fw(nt)=ftot(nt)
        fw(1)=fu(1)
       do 799 i=1,18
        dw(i)=(ftot(i+1)-ftot(i))/(p(i+1)-p(i))
         dw(i)=dw(i)*.0083224
799     continue
*         write(20,1050) ((try(i,j),j=1,19),i=1,19)
*         write(20,1051)
*         write(20,1050) ((trc(i,j),j=1,19),i=1,19)
*           write(20,1052)
*         write(20,1050) ((tr1(i,j),j=1,19),i=1,19)
1050   format (19(4x,f8.3))
1051    format('try above', 'trc below')
1052    format('tr1 below')
*
       return
       end
*
***********************************************************************
      subroutine h20trans (pa,p,tr1,tr2,tr3,tr4,tr5,
     x try,tr6,trc,kap,at1)
*     calculate h2O transmission
*     tr1 is trans for 660 - 800 cm-1 h2O
*     tr2 is co2 h2o overlap
*     tr4 is ch4 h20 overlap
*     tr5 is 1200-1350 n20 h20 overlap
*      tr6 is 520-660 n2o h2o overlap
*     trc is the emissivity of the 800-1200 cm-1 h2O cont.
*     try is the emissivity of h20 cont 400-660 x tr rot
      real pa(20),p(20),at1(20),r3(20)
      real tr1(20,20),tr2(20,20),tr4(20,20),tr5(20,20),tr6(20,20)
      real trc(20,20),tr3(20,20),try(20,20),rc(20),ry(20),r2(20)
      real th4,tr,tch,u4,sig,trx,u5
      integer kap
               sig=5.67e-8
             do 840 i=1,18
                e=esat(at1(i))
                r=rwat(pa(i),e,i,kap)
                e=e*.77*pa(i)
                 u5=r*1033*(p(i+1)-p(i))
                 call trcont (800.,1200.,u5,at1(i),e,rc(i))
                 call trcont(480.,800.,u5,at1(i),e,ry(i))
                 call trcont(580.,740.,u5,at1(i),e,r2(i))
                 call trc3(u5,at1(i),e,r3(i))
840           continue   
               do 860 i=2,19
                do 859 j=1,i-1
              e=esat(at1(j))
              r=rwat(pa(j),e,j,kap)
              if (j.eq.1) then
              u4=r*1033*(p(j+1))
              else
              u4=r*1033*(p(j+1)-pa(j))
              end if
              e=e*.77*pa(j)
               call trcont(800.,1200.,u4,at1(j),e,tr)
               trc(j,i)=tr
               call trcont(480.,800.,u4,at1(j),e,tr)
               try(j,i)=tr
               call trcont(580.,740.,u4,at1(j),e,tr)
               tr2(j,i)=tr
               call trc3(u4,at1(j),e,tr)
               tr3(j,i)=tr
              tave=at1(j)*(p(j+1)-pa(j))*r*1033
              pave=pa(j)*(p(j+1)-pa(j))*r*1033
          do 857 k=j+1,i-1
              e=esat(at1(k))
              r=rwat(pa(k),e,k,kap)
              u4=u4+r*1033*(p(k+1)-p(k))
               trc(j,i)=trc(j,i)*rc(k)
               try(j,i)=try(j,i)*ry(k)
               tr2(j,i)=tr2(j,i)*r2(k)
               tr3(j,i)=tr3(j,i)*r3(k)
              tave=tave+at1(k)*r*1033*(p(k+1)-p(k))
              pave=pave+pa(k)*r*1033*(p(k+1)-p(k))
857       continue
              pave=pave/u4
              tave=tave/u4
              e=esat(tave)*.77*(pave-.02)/.98
              if(e.lt.(4.8e-6*pave))then
               e=(4.8e-6*pave)
              end if
              call overlap (tave,pave,u4,tch,tr,trx)
              bl=3.742e-16*(73000.**3)/((exp(1.438*730./tave))-1)
              tr1(j,i)=14000.*bl*(1-tr)/(sig*tave**4)
              tr2(j,i)=tr2(j,i)*tch
              bl=3.742e-16*(64000.**3)/((exp(1.438*640./tave))-1)
              try(j,i)=32000.*bl*trx*(1-try(j,i))/(sig*tave**4)
              call ch4ovl(pave,u4,tave,th4,e)
              tr4(j,i)=th4
              call trrn (pave,u4,tr)
              tr5(j,i)=tr
              call trr6 (tave,pave,u4,tr)
              tr6(j,i)=tr
              bl=3.742e-16*(100000.**3)/((exp(1.438*1000./tave))-1)
              trc(j,i)=40000.*bl*(1-trc(j,i))/(sig*tave**4)
859     continue
860     continue
       do 870 i=1,19
        do 869 j=1,i
              if (j.eq.19) then
              goto 869
              end if
              e=esat(at1(i))
              r=rwat(pa(i),e,i,kap)
              u4=r*1033*(pa(i)-p(i))
              tave=at1(i)*(pa(i)-p(i))*r*1033
              pave=pa(i)*(pa(i)-p(i))*r*1033
              e=e*.77*pa(i)
              call trcont(800.,1200.,u4,at1(i),e,tr)
               trc(i,j)=tr
               call trcont(480.,800.,u4,at1(i),e,tr)
               try(i,j)=tr
               call trcont(580.,740.,u4,at1(i),e,tr)
               tr2(i,j)=tr
               call trc3(u4,at1(i),e,tr)
               tr3(i,j)=tr
          do 867 k=j,i-1
              e=esat(at1(k))
              r=rwat(pa(k),e,k,kap)
              u4=u4+r*1033*(p(k+1)-p(k))
              tave=tave+at1(k)*r*1033*(p(k+1)-p(k))
              pave=pave+pa(k)*r*1033*(p(k+1)-p(k))
              trc(i,j)=trc(i,j)*rc(k)
              try(i,j)=try(i,j)*ry(k)
              tr2(i,j)=tr2(i,j)*r2(k)
              tr3(i,j)=tr3(i,j)*r3(k)
867      continue
              pave=pave/u4
              tave=tave/u4
              e=esat(tave)*.77*(pave-.02)/.98
              if(e.lt.(4.8e-6*pave))then
               e=(4.8e-6*pave)
              end if
              call overlap (tave,pave,u4,tch,tr,trx)
              bl=3.742e-16*(73000.**3)/((exp(1.438*730./tave))-1)
              tr1(i,j)=14000.*bl*(1-tr)/(sig*tave**4)
	      tr2(i,j)=tch*tr2(i,j)
              bl=3.742e-16*(64000.**3)/((exp(1.438*640./tave))-1)
              try(i,j)=32000.*bl*trx*(1-try(i,j))/(sig*tave**4)
              call ch4ovl(pave,u4,tave,th4,e)
              tr4(i,j)=th4
              call trrn (pave,u4,tr)
              tr5(i,j)=tr
              call trr6 (tave,pave,u4,tr)
              tr6(i,j)=tr
              bl=3.742e-16*(100000.**3)/((exp(1.438*1000./tave))-1)
              trc(i,j)=40000*bl*(1-trc(i,j))/(sig*tave**4)
869      continue
870     continue
               tr1(19,19)=0.0
               tr2(19,19)=1.0
               tr3(19,19)=1.0
               tr4(19,19)=1.0
               tr5(19,19)=1.0
               tr6(19,19)=1.0
               trc(19,19)=0.0
               try(19,19)=0.0
*       write (20,9011) ((tr2(i,j),j=1,19),i=1,19)
9011    format (19(f6.4,1x))
               return
               end

*******************************************************************
*
      subroutine tempave(at, sg, ta)
      dimension at(20),sg(20)
      dimension ta(20,20)
      real s
*
*     calculate the average temperaure to be used for each path
*     using a pressure weighted average Dp=6sigma(1-sigma)
*
      do 600 i=1,19
        do 599 j=i,19
         ta(i,j)=0.0
         s=0.0
          do 590 k=i,j
           ta(i,j)=ta(i,j)+6.0*sg(k)*(1-sg(k))*at(k)
           s=s+6.0*sg(k)*(1-sg(k))
590       continue
           if(s-0.0) 595,595,596
595         ta(i,j)=at(i)
            goto 597
596         ta(i,j)=ta(i,j)/s
597         ta(j,i)=ta(i,j)
599     continue
600   continue
      return
      end
**********************************************************************
      subroutine kdist (tau,omeg,tx,rx)
      real tau,omeg,g,u,t,tx,rx
      g=.85
      t=sqrt(3*(1-omeg)*(1-g*omeg))*tau
      u=sqrt((1-g*omeg)/(1-omeg))
      bot=(u+1)**2-exp(-2*t)*(u-1)**2
      rx=(u+1)*(u-1)*(1.0-exp(-2*t))/bot
      tx=4*u*exp(-t)/bot
      return
      end
**********************************************************************
*     solar heating for cloudy portion of the sky
*     via Lacis and Hansen 1974
      subroutine solclod (p,pa,at,xk,pk,depth,theta,s0,rg,kap,ac,shc
     x,abs,gw,at1)
      real gw(19),p(20),pa(20),at(20),xk(10),pk(10),shc(20)
      real up(25,10),d(25,10),ab(25,10),clh(25,10),at1(20)
      real r1(25,10),r1s(25,10),r19(25,10),t1(25,10),rx(25,10)
      real tx(25,10)
      real tt,s0,s,depth,abs,u1,mu0,e,r,m,theta,tau,omeg,ref
      mu0=cos(theta)
      m=35.0/sqrt(1224.0*mu0**2+1)
       s=s0*ac
       do 4500 j=1,19
         gw(j)=0.0
         shc(j)=0.0
4500   continue
        gw(1)=s 
        abs=0.0
        ref=0.0
        do 4700 k=2,8
        do 4510 j=1,kap-1
          rx(j,k)=0.0
          e=esat(at1(j))
          r=rwat(pa(j),e,j,kap)
          u1=pa(j)*sqrt(273/at(j))*r*1033*(p(j+1)-p(j))*m
          tau=xk(k)*u1
          tx(j,k)=exp(-tau)
4510    continue
          e=esat(at1(kap))
          r=rwat(pa(kap),e,kap,kap)
          u1=pa(kap)*sqrt(273/at(kap))*r*1033*(p(kap+1)-p(kap))*5/3
          tau=depth+xk(k)*u1
          omeg=depth/tau
           call kdist(tau,omeg,tx(kap,k),rx(kap,k)) 
        do 4520 j=kap+1,18
          rx(j,k)=0.0
          e=esat(at1(j))
          r=rwat(pa(j),e,j,kap)
          u1=pa(j)*sqrt(273/at(j))*r*1033*(p(j+1)-p(j))*5/3
          tau= xk(k)*u1
          tx(j,k)=exp(-tau)
4520    continue
       t1(1,k)=tx(1,k)
       r1(1,k)=0.0
       rx(19,k)=rg
       r19(19,k)=rg
       tx(19,k)=0.0
       r1s(1,k)=0.0
        do 4530 j=2,19
          t1(j,k)=t1(j-1,k)*tx(j,k) 
          r1(j,k)=r1(j-1,k)+rx(j,k)*t1(j-1,k)**2
          r1s(j,k)=rx(j,k)+r1s(j-1,k)*tx(j,k)**2
*   t1s(j,k) would need to be calculated if there were 2 adjacent
*   cloud layers
4530    continue
        do 4540 j=18,1,-1
          tt=tx(j,k)
       r19(j,k)=rx(j,k)+r19(j+1,k)*tt**2/(1-rx(j,k)*r19(j+1,k))
4540    continue
        do 4550 j=1,18
          up(j,k)=t1(j,k)*r19(j+1,k)/(1-r1s(j,k)*r19(j+1,k))
          d(j,k)=t1(j,k)/(1-r1s(j,k)*r19(j+1,k))
           gw(j+1)=gw(j+1)+s*pk(k)*d(j,k)
          ab(j,k)=pk(k)*(1-r1(19,k)+up(j,k)-d(j,k))
4550    continue
        abs=abs+ab(18,k)*s*mu0
        ref=ref+r19(1,k)*s*mu0*pk(k)
      clh(1,k)=ab(1,k)
          shc(1)=shc(1)+clh(1,k)*(.0083224/(p(2)-p(1)))*s*mu0 
        do 4560 j=2,18
          clh(j,k)=ab(j,k)-ab(j-1,k)
          shc(j)=shc(j)+clh(j,k)*(.0083224/(p(j+1)-p(j)))*s*mu0 
4560    continue       
         ab(19,k)=t1(18,k)*(1-rg)*pk(k)
         shc(19)=shc(19)+ab(19,k)
4599    format (7(f6.4,3x))
4600    format (9(f6.4,3x))
4700    continue
       gw(19)=shc(19)*mu0*s
        abs=abs+gw(19)
      return
      end
**************************************************************
      subroutine h2osolar (s0,theta,pa,p,at,shw,gw,abs,ac,rg,kap,at1)
*      Clear sky calculations
      dimension at(20),p(20),pa(20),shw(20),at1(20)
      dimension yl(20),ylb(20),sawd(20),sawu(20),sawt(20),gw(20)
      real m,mu0,theta,e,s,s0,abs
      integer kap
*     cos zenith angle mu0
      mu0=cos(theta)
      m=35./sqrt(1224.0*(mu0)**2+1)
      s=s0*(1-ac)
      cp=1005
      yl(1)=0.0
      sawd(1)=0.0
      do 900 k=1,18
            e=esat(at1(k))
            r=rwat(pa(k),e,k,kap)
          u1= pa(k)*sqrt(273/at(k))*r*1033*(p(k+1)-p(k))*m
          yl(k+1)=yl(k)+u1
          sawd(k+1)=2.9*yl(k+1)/((1+141.5*yl(k+1))**.635+5.925*yl(k+1))
900   continue
       ylb(19)=yl(19)
       sawu(19)=sawd(19)
      do 910 k=18,1,-1
            e=esat(at1(k))
            r=rwat(pa(k),e,k,kap)
          u1=pa(k)*sqrt(273/at(k))*r*1033*(p(k+1)-p(k))*(5.0/3.0)
          ylb(k)=ylb(k+1)+u1
          sawu(k)=2.9*ylb(k)/((1+141.5*ylb(k))**.635+5.925*ylb(k))
910   continue
       gw(1)=mu0*s0
        abs=0.0
       do 920 i=1,18
       gw(i+1)=mu0*s*(1-sawd(i+1))
       sawt(i)=mu0*s*(sawd(i+1)-sawd(i)+rg*(sawu(i)-sawu(i+1)))
       shw(i)=.0083224*(sawt(i))/(p(i+1)-p(i))
       abs=abs+sawt(i)
920   continue
       gw(19)=mu0*(1-rg)*s*(.353-sawd(19))
       abs=abs+gw(19)
      return
      end
*****************************************************************
*
       function rwat(p,e,j,kap)
         real rwat,h,e,p
         integer j,kap
         if(j.eq.kap) then
          h=.77*(p-.02)/.98
          else
          h=.77*(p-.02)/.98
         end if
         rwat=.622*h*e/(p-h*e)
           if (rwat.lt.3.0e-6) then
            rwat=3.0e-6
            end if
            return
          end
*****************************************************************
*
      function esat(t)
      real esat,l,r,t
      l=2510.-2.38*(t-273)
      r=.287
      esat=(6.11/1012.34)*exp((.622*l/r)*(t-273)/(t*273))
      return
      end
******************************************************************
*
*   Calculates the transmission of water vapor continuum in 8-12
*    micro meter region
      subroutine trc2 (u4,t,e,trc)
      real trc,t,e,k,u4
        k=(4.2+5588*exp(-7.87))*exp(1800*(296-t)/(296*t))*e
        trc=(exp(-k*1.66*u4))
       return
       end

******************************************************************
*
*   Calculates the transmission of water vapor continuum in 1042 cm-1
*   region for ozone overlap
      subroutine trc3 (u4,t,e,trc)
      real trc,t,e,k,u4
        k=(4.2+5588*exp(-8.20))*exp(1800*(296-t)/(296*t))*e
        trc=(exp(-k*1.66*u4))
       return
       end

******************************************************************

*
*
      subroutine trcont (nu1,nu2,u4,t,e,trch)
      real trch,nu1,nu2,nu,t,e,k
       integer n1,n2
       n1=int(nu1)
       n2=int(nu2)
       trch=0.0
        do 3000 i=n1,n2-20,20
        nu=i+10
        k=(4.2+5588*exp(-.00787*nu))*exp(1800*(296-t)/(296*t))
     x*e
        trch=trch+(exp(-k*1.66*u4))*20/(nu2-nu1)
3000  continue
       return
       end
**************************************************************
*
       subroutine bpl(b,t,nu)
       real b(20),t(20)
       real nu
       do 5500 i=1,19
        b(i)=3.742e-16*(nu*100)**3/((exp(1.438*nu/t(i)))-1)
5500   continue
       return
       end
*
*************************************************************
*
      subroutine tempset (t,at,p)
      real t(20),at(20),p(20)
       t(1)=at(1)
       t(19)=at(19)
       do 4200 i=2,18
        t(i)=((p(i+1)-p(i))*at(i)+(p(i)-p(i-1))*at(i-1))/
     x(p(i+1)-p(i-1))
4200  continue
        return
       end
*
***************************************************************8
*
      subroutine presset(sg,p,pa)
      real pa(20),sg(20),p(20)
      sg(2)=1.0/36.0
      pa(1)= (sg(2)**2)*(3.0-2.0*sg(2))
      p(1)=0.0
      p(19)=1.0
       pa(19)=1.0
      sg(1)=0.0
      do 4300 i=2,18
        p(i)=p(i-1)+6.0*sg(i)*(1-sg(i))/18.0
        sg(i+1)=sg(i)+(1.00/18.00)
        pa(i)=(sg(i+1)**2)*(3.0-2.0*sg(i+1))
4300    continue
       return
       end
*
*********************************************************
*
      subroutine acool(a,b,p,dc,f,ac,kap,nt)
      real a(20,20)
      real fd(20),fu(20),ftot(20),b(20),dc(20),f(20),p(20)
      real ac,cf
      integer kap,nt
         fd(1)=0.0
        do 4024 i=2,19
           if (i.gt.kap) then
             cf=1.0-ac
            else 
             cf=1.0
           end if
          fd(i)=cf*b(1)*a(1,i)
	  do 4022 j=1,i-1
            fd(i)=fd(i)+cf*a(j,i)*(b(j+1)-b(j))
4022    continue
4024	continue
	 fd(2)=a(1,2)*b(1)
         do 4030 i=kap+1,19
           do 4028 j=kap+1,i-1
            fd(i)=fd(i)+ac*a(j,i)*(b(j+1)-b(j))
4028       continue
4030     continue
        fu(19)=0.0
       do 4035 i=1,18
           if(i.lt.kap+1) then
             cf=1.0-ac
            else
             cf=1.0
           end if
         fu(i)=0.0
         do 4033 j=i,18
          fu(i)=fu(i)-cf*a(j,i)*(b(j+1)-b(j))
4033     continue
4035    continue
        do 4040 i=1,kap
          do 4038 j=i,kap-1
            fu(i)=fu(i)-ac*a(j,i)*(b(j+1)-b(j))
4038      continue
4040    continue
       do 4042 i=1,19
        ftot(i)=fd(i)-fu(i)
4042   continue
       do 4046 i=1,18
         dc(i)=.0083224*(ftot(i+1)-ftot(i))/(p(i+1)-p(i))
4046   continue
         f(1)=fu(1)
         f(18)=ftot(19)
         f(nt)=ftot(nt)
         f(19)=fd(19)
*       do 4049 i=1,19
*       write(20,4050) i,fu(i),fd(i)
*4049   continue
4050   format (i3,2(4x,f8.3))
*
      return
      end
***********************************************************
*
      subroutine tcool(dnu,tr,b,p,dc,f,ac,kap,nt)
       real a(20,20), tr(20,20)
       real b(20), dc(20), f(20), p(20)
       real dnu,ac
       integer kap,nt
        do 4110 i=1,19
         do 4105 j=1,i
          a(j,i)=(1-tr(j,i))*dnu*100.
          a(i,j)=(1-tr(i,j))*dnu*100.
4105   continue
4110   continue
         call acool (a,b,p,dc,f,ac,kap,nt)
      return
      end
*********************************************************
      subroutine trr1(t,p,u4,tr)
*     transmission of rotation band from 660-800 cm-1 using
*     statistical model Goody 1964, Rodgers & Walshaw 1966
      real kd,kpa,a,a1,b,b1,phi,t,psi,phb,m,mb
      m=1.66*u4
      kd=.911
      kpa=15.84
      a=.0204
      a1=.0209
      b=-4.89e-5
      b1=-6.87e-5
      phi=exp(a*(t-260)+b*(t-260)**2)
      psi=exp(a1*(t-260)+b1*(t-260)**2)
      mb=phi*m
      phb=psi*p*m/mb
      tr=exp(-(kd*mb)/sqrt(1+kpa*mb/phb))
       return
       end
****************************************************************
      subroutine expon (y,e1)
      real gamma, e1,y
      gamma=.5772157
      e1=0
        if (y.lt.1.) then
          e1=-log(y)-gamma
          e1=e1+.9999919*y-.2499106*y*y+.0551997*y**3-.0097600*y**4
     x +.0010786*y**5
        else
        e1=(y**4)+8.5733287*y**3+18.0590170*y*y+8.6347609*y+.26777373
        e1=e1/(y**4+9.5733223*y**3+25.6329561*y*y+21.0996531*y
     x    +3.9584969)
        e1=(e1*exp(-y))/y
        endif
        return
        end
******************************************************************
        subroutine bcalc (t,bt1,bt2,bt6,v1,v2)
*       subroutine for co2 overlap calculation using Kuo 1977
        real bt1,bt2,bt6,t,ts,ts2,v1,v2
        ts=t/100.-2.6
        ts2=ts*ts
        bt1=0.
        bt2=0.
        bt6=0.
        bt=0.
        bt=bt+1.6*.8457*(1-.2569*ts+.1191*ts2)
        bt=bt+.60*.4643*(1-.6739*ts+.36*ts2)
        bt=bt+.60*1.464*(1-.2605*ts+.1307*ts2)
        bt1=bt+(.9+v1-5.)*.927*(1-.1641*ts+.0255*ts2)
        bt6=(v2-v1)*.927*(1-.1641*ts+.0255*ts2)
        bt2=bt1+bt6
        return
        end
***************************
       subroutine overlap (t,p,u4,tch,tr,trx)
       real e1,e2,bt1,bt2,bt6,m
       real t,p,u4,tr,tr1,tch,trx
       m=1.66*u4*p
	 call bcalc (t,bt1,bt2,bt6,5.0,8.0)
         y1=51.8845*sqrt(m)
         y5=y1*exp(-bt1)
         y6=y5*exp(-bt6)
         call expon (y5,e1)
         call expon (y6,e2)
         tch=(e2-e1)/bt6
	  call trr1(t,p,u4,tr1)
	  tch=tr1
          tr=tr1
         call bcalc (t,bt1,bt2,bt6,4.8,8.0)
         y1=51.8845*sqrt(m)
         y5=y1*exp(-bt1)
         y6=y5*exp(-bt6)
         call expon (y5,e1)
         call expon (y6,e2)
         trx=(e2-e1)/bt6
           return
           end
**********************************************************
*
      subroutine co2 (t,p,pa,dc,fc,ta,tr2,tr3,con2,ac,kap,
     xtco2n2,nt,qc,dq,ec,vc,sc,a1,a,nr)
      real ta(20,20), a(20,20),tr2(20,20),tco2n2(20,20)
      real tr3(20,20)
      real qc(4),dq(4,14),ec(14),vc(14),sc(14),a1(20,20)
      real b(20),p(20),dc1(20),dc(20),fc(20),pa(20),t(20)
      real fc1(20)
      real a0,con2,ac
      integer kap,nt
      call bpl(b,t,667.0)
      if (nr.eq.0.or.nr.gt.10) then
      do 320 i=1,19
        do 318 j=1,i
           a0=(22.18)*sqrt(ta(j,i)/296.0)
            if (j.eq.1)then
              if(i.eq.1)then
              u=1
              else
              u=1.66*.8*con2*p(i)
              w=.03*u/a0
              pave=p(i)/2.0
           call uco2(qc,dq,ec,vc,sc,ta(j,i),pave,u,a0,ab)
           a(j,i)=ab*100
           a1(j,i)=200*a0*
     xlog(1+w/(4+w*(1+1/(pave*.1084*(298/ta(j,i))**.56)))**.5)
              end if
              u=1.66*.8*con2*pa(i)
              w=.03*u/a0
              pave=(p(i+1)+pa(i))/2.0
           call uco2(qc,dq,ec,vc,sc,ta(i,j),pave,u,a0,ab)
             a(i,j)=100*ab
           a1(i,j)=200*a0*
     xlog(1+w/(4+w*(1+1/(pave*.1084*(298/ta(i,j))**.56)))**.5)
             else
           pave=(pa(j)+p(i))/2.0
           u=1.66*.8*con2*abs(p(i)-pa(j))
              w=.03*u/a0
           call uco2(qc,dq,ec,vc,sc,ta(j,i),pave,u,a0,ab)
           a(j,i)=ab*100
           a1(j,i)=200*a0*
     xlog(1+w/(4+w*(1+1/(pave*.1084*(298/ta(j,i))**.56)))**.5)
           u=1.66*.80*con2*abs(pa(i)-p(j))
              w=.03*u/a0
           call uco2(qc,dq,ec,vc,sc,ta(i,j),pave,u,a0,ab)
           a(i,j)=ab*100
           a1(i,j)=200*a0*
     xlog(1+w/(4+w*(1+1/(pave*.1084*(298/ta(i,j))**.56)))**.5)
          end if
           tco2n2(j,i)=1-a(j,i)/30000.
           tco2n2(i,j)=1-a(j,i)/30000.
                a(j,i)=a(j,i)*tr2(j,i)
                a(i,j)=a(i,j)*tr2(i,j)
                a1(j,i)=a1(j,i)*tr3(j,i)
                a1(i,j)=a1(i,j)*tr3(i,j)
318      continue
320   continue
           nr=0
           end if
            nr=nr+1
*
*      write (20,399) ((a1(i,j)/10000,j=1,19),i=1,19)
399     format (19(f5.3,2x))
      call acool(a,b,p,dc,fc,ac,kap,nt)
      call bpl(b,t,1020.0)
       call acool(a1,b,p,dc1,fc1,ac,kap,nt)
       do 400 i=1,19
        dc(i)=dc(i)+dc1(i)
        fc(i)=fc(i)+fc1(i)
400     continue
*
      return
      end
*
*******************************************************
       subroutine uco2(qc,dq,ec,vc,sc,t,p,u,a0,ab)
       real qc(4),vc(14),dq(4,14),sc(14),ec(14)
       real sum(8),f(4,14),s(14)
       real t1,t2,t3,u,a0,uc,t,p
       if (u.lt.0.0) then
       goto 460
       end if
       do 430 i=1,7
         sum(i)=0.0
430     continue
       do 450 j=1,14
       s(j)=sc(j)*(300/t)*(1-exp(-1.439*vc(j)/t))**3/
     x  (1-exp(-1.439*vc(j)/300))**3*exp(1.439*ec(j)*(t-300)/(t*300))
       do 440 i=1,4
       if (j.eq.1.and.i.eq.1) then
         bet=(4/dq(i,j))*p*.067*(300/t)**.667
         uc=qc(i)*s(j)*u*exp((-53.5/a0))/a0
         t1=1/(1+uc/(4+uc*(1+1/bet))**.5)
         uc=qc(i)*s(j)*u*exp((-72./a0))/a0
         t2=1/(1+uc/(4+uc*(1+1/bet))**.5)
         uc=qc(i)*s(j)*u*exp((-124./a0))/a0
         t3=1/(1+uc/(4+uc*(1+1/bet))**.5)
        end if
       uc=qc(i)*u*s(j)/a0
       bet=(4/dq(i,j))*p*.067*(300/t)**.667
       f(i,j)=uc/(4+uc*(1+1/bet))**.5
440    continue
450    continue
        do 455 i=1,4
        sum(1)=sum(1)+f(i,1)+f(i,4)+f(i,5)+f(i,9)+f(i,11)+f(i,12)
        sum(2)=sum(2)+f(i,2)+f(i,14)
        sum(3)=sum(3)+f(i,3)+f(i,13)
        sum(4)=sum(4)+f(i,7)
        sum(5)=sum(5)+f(i,8)
        sum(6)=sum(6)+f(i,10)
        sum(7)=sum(7)+f(i,6)
455     continue
        ab=2*a0*(log(1+sum(1))+t1*(log(1+sum(2))+log(1+sum(3)))+
     x t2*(log(1+sum(4))+log(1+sum(5)))+t3*(log(1+sum(6))+
     x log(1+sum(7))))
460     return
       end


*******************************************************
        function alpha(u,p)
        real alpha,u,p,au,al,beta
        au=(4.1*u)/(1+9.5*u)
        al=.8467*u*(1.9-u)/(1+2.0*u)
        if (p.ge..015) then
        beta=sqrt((p-.015)/.235)
        end if
        if (p.le..015) then
         alpha=(1.085-.085*p)*au
          else
           if((p.ge..015).and.(p.le..25)) then
            alpha=(au**(1-beta))*(al**beta)
            else
             alpha=.6667*(1.75-p)*al
           endif
         endif
         return
         end
******************************************************
*
      subroutine o3cool (bo,p,pa,uo,uoa,doc,fo,ac,kap,nt,tr3)
      dimension  abs(20,20),tr3(20,20)
      dimension  bo(20), p(20), doc(20), fo(20), pa(20),uo(50),uoa(50)
      dimension  ua(20),us(20),up(20),a(20)
      real u,ut,e1,e2,y,alpha
      integer kap,nt
*
      do 610 i=1,18
        u=(uo(i+1)-uo(i))
        a(i)=alpha(u,pa(i))
        ua(i)=u*1.66*pa(i)**a(i)
        if(i.eq.1)then
          us(i)=uo(i+1)*1.66*pa(i)**a(i)
         else
          us(i)=(uo(i+1)-uoa(i))*1.66*pa(i)**a(i)
         end if
          up(i)=(uoa(i)-uo(i))*1.66*pa(i)**a(i)
610   continue
      do 640 i=2,19
        do 639 j=1,i-1
         ut=us(j)
630        do 635 k=j+1,i-1
            ut=ut+ua(k)
635        continue
        y=.5138*ut/sqrt(1+3.7145*ut)
        call expon (y,e1)
        call expon (17.778*y,e2)
        trn=.3476*(e1-e2)
	abs(j,i)=13700*(1-trn)*tr3(j,i)
639     continue
640     continue
       do 660 i=1,19
        do 659 j=1,i
           ut=up(i)
           do 655 k=j,i-1
           ut=ut+ua(k)
655        continue
	y=.5138*ut/sqrt(1+3.7145*ut)
	if(y.eq.0) then
	trn=1.0
	else
        call expon (y,e1)
        call expon (17.778*y,e2)
	trn=.3476*(e1-e2)
	endif
	abs(i,j)=13700*(1-trn)*tr3(i,j)
659     continue
660     continue
*
*	write (20,399) ((abs(i,j)/10000,j=1,19),i=1,19)
399     format (19(f5.3,2x))

      call acool (abs,bo,p,doc,fo,ac,kap,nt)
*
      return
      end
*
******************************************************
*
      subroutine height (z,za,p,pa,at)
      dimension z(20),za(20),p(20),pa(20),at(20)
      real ao,bo,co,dpr
      ao=.40
      bo=20
      co=5
      z(19)=0.0
      do 900 i=19,2,-1
      dpr=p(i)-p(i-1)
      z(i-1)=z(i)+(dpr*at(i-1)*.02925)/pa(i-1)
      dpr=p(i)-pa(i-1)
      za(i-1)=z(i)+(2*dpr*at(i-1)*.02925)/(pa(i-1)+p(i))
900    continue
      return
      end
*
************************************************************************
*
      subroutine solo3 (s0,theta,uo,p,ho,fso,abo,ac,rg)
*     Calculate ozone solar heating according to Lacis and Hansen 1974
*     Clear Skies
      real a(20),x(20),xu(20),au(20),ab(20)
      real uo(20), ho(20), fso(20),p(20)
      real mu0, mbar, m, rg, ra2, ra1, theta, ra
      real abo,rrm,rrs
      s=s0*(1-ac)
      mu0=cos(theta)
      mbar=1.9
      ra2=.144
      ra1=.219/(1+.816*mu0)
      ra=ra1+(1-ra1)*(1-ra2)*rg/(1-ra2*rg)
      m=35./sqrt(1224*mu0*mu0+1)
        do 5000 i=1,19
         x(i)=m*uo(i)
         a(i)=ao3(x(i))
5000   continue
        do 5010 i=1,19
         xu(i)=x(19)+mbar*(x(19)-x(i))/m
         au(i)=ao3(xu(i))
5010   continue
         fso(1)=s*mu0
         abo=0.0
         do 5020 i=1,18
         ab(i)=s*mu0*((a(i+1)-a(i))+ra*(au(i)-au(i+1)))
         ho(i)=.0083224*ab(i)/(p(i+1)-p(i))
         fso(i+1)=(s-s*a(i+1))*mu0
         abo=abo+ab(i)
5020    continue
        rrm=.28/(1+6.43*mu0)
        rrs=.0685
       fso(19)=s*(1-rg)*mu0*(.647-rrm-a(19))/(1-rrs*rg)
       abo=abo+fso(19)
       return
       end
***************************************************************
************************************************************************
*
      subroutine sco3 (s0,theta,uo,p,ho,fso,abo,ac,rg,kap,depth)
*     Calculate ozone solar heating according to lacis and hansen 1974
*     Cloudy Skies
      real a(20),x(20),xu(20),au(20),ab(20)
      real uo(20), ho(20), fso(20),p(20)
      real mu0, mbar, m, rg, ra2, ra1, theta, ra
      real ac,depth,abo,x1,rrm,rrs
      integer kap
      s=s0*ac
      mu0=cos(theta)
      x1=.85
      mbar=1.9
      ra2=(1-x1)*depth*sqrt(3.0)/(2+(1-x1)*depth*sqrt(3.0))
      ra1=ra2
      ra=ra1+(1-ra1)*(1-ra2)*rg/(1-ra2*rg)
      m=35./sqrt(1224*mu0*mu0+1)
        do 5000 i=1,19
         x(i)=m*uo(i)
         a(i)=ao3(x(i))
5000   continue
        do 5010 i=1,19
         xu(i)=x(kap+1)+mbar*(x(kap+1)-x(i))/m
         au(i)=ao3(xu(i))
5010   continue
         fso(1)=s*mu0
         abo=0.0
         do 5020 i=1,18
         ab(i)=s*mu0*((a(i+1)-a(i))+ra*(au(i)-au(i+1)))
         ho(i)=.0083224*ab(i)/(p(i+1)-p(i))
         fso(i+1)=(s-s*a(i+1))*mu0
         abo=abo+ab(i)
5020    continue
        rrm=.28/(1+6.43*mu0)
        rrs=.0685
       fso(19)=s*(1-rg)*(1-ra1)*mu0*(.647-a(19))/(1-ra2*rg)
       abo=abo+fso(19)
       return
       end
***************************************************************
       function ao3(x)
       a1=.02118*x/(1+.042*x+.000323*x*x)
       a2=(1.082*x/(1+138.6*x)**.805)+.0658*x/(1+(103.6*x)**3)
       ao3=a1+a2
       return
       end
*****************************************************************
*
      subroutine lapadj (at,gam,za,cp,ch,nt,ce)
      real at(20),gam(20),za(20),cp(20),ch(20),ce(20)
      real s2,d1,d2,d3,db,dt
      integer nt
      nt=6
      do 7998 i=1,19
       ch(i)=0.0
7998   continue
7999   s2=0.0
      za(19)=0.0
      do 8002 i=18,2,-1
      d1=at(i+1)-at(i)
      d2=gam(i)*(za(i)-za(i+1))
       if (d1.gt.d2) then
        d3=1.0*(d1-d2)
         db=cp(i)*d3/(cp(i)+cp(i+1))
         dt=cp(i+1)*d3/(cp(i)+cp(i+1))
         at(i)=at(i)+dt
         at(i+1)=at(i+1)-db
         ch(i)=ch(i)+dt
         ch(i+1)=ch(i+1)-db
         s2=s2+d3
         nt=i
        end if
8002  continue
       if (s2.gt..005)then
         goto 7999
       endif
      do 8005 i=2,18
        if (at(19).lt.(at(18)+.04))then
        ce(i)=ce(i)*.95
      end if
        ce(i)=ce(i)+ch(i)/8.0
      if(ch(i).lt..00001) then
      ce(i)=ce(i)*.99
      end if
8005    continue
       return
       end
*
********************************************************************
*
      subroutine lap1 (gam,lap)
      real gam(20)
      real lap
      do 8600 i=1,18
       gam(i)=lap
8600   continue
       return
       end
**********************************************************************
*
      subroutine lap2 (gam,at,pa)
*     moist adiabatic lapse rate from Stone & Carlson 1979
      real gam(20),at(20),pa(20)
      real l,r,de,e
      r=.287
      do 8700 i=1,18
      l=2510.-2.38*(at(i)-273)
       e=esat(at(i))
      de=.622*l*e/(r*at(i)**2)
      gam(i)=9.8*(1+.622*l*e/(pa(i)*r*at(i)))/
     1(1+(.622*l*de)/(1.005*pa(i)))
8700  continue
       return
       end
**********************************************************************
****************************************************************
*      Calculate H2O overlap in methane region1200-1650 vib
*      rotation & 950-1200 continuum
       subroutine ch4ovl (pa,u4,at,th4,e)
       real pa,u4,th4,e,tr1,tr2,at
        call trr4 (pa,u4,tr1)
        call trcont(920.,1200.,u4,at,e,tr2)
        th4=(280*tr2+450*tr1)/750
       return
       end
****************************************************************
*     Calculates Vib-rot H20 in CH4 overlap 1200-1650
*     Rodgers & Walshaw 1966
      subroutine trr4(p,u4,tr)
      real p,u4,tr,m,kd,kpa
      m=1.66*u4
      kd=248
      kpa=1276
      tr=exp(-kd*m/sqrt(1+kpa*m/p))
      return
      end
****************************************************************
*       Calculate methane transmission in N20 1200-1350 cm-1
*       region using Green's 1964 method
      function tn2c4(w)
      real dn(10),we(10)
      real w,tn2c4
      dn(1)=35.6
      dn(2)=12.0
      dn(3)=18.5
      dn(4)=13.1
      dn(5)=72.0
      we(1)=18.4
      we(2)=9.08
      we(3)=2.60
      we(4)=6.47
      we(5)=14.95
      tn2c4=0.0
      do 9400 i=1,5
      tn2c4=tn2c4+exp(-((w/we(i))**.46))*dn(i)/151.2
9400  continue
      return
      end
*****************************************************************
*     Calculate N2O CH4 overlap
      subroutine n2och4 (tnc,p,pa,at,c40)
      real tnc(20,20)
      real p(20),pa(20),at(20)
      real f,w,tn2c4,c40
*     f is ch4 mixing ratio 1.6 ppmv
      f=c40
      do 9360 i=2,19
        do 9359 j=1,i-1
         w=f*(1.29/1.6)*pa(j)*(p(j+1)-pa(j))*sqrt(300/at(j))
          do 9355 k=j+1,i-1
        w=w+f*(1.29/1.6)*pa(k)*(p(k+1)-p(k))*sqrt(300/at(k))
9355      continue
        tnc(j,i)=tn2c4(w)
9359    continue
9360   continue
       do 9370 i=1,19
        do 9369 j=1,i
         if(j.eq.19)then
         goto 9369
         endif
         w=f*(1.29/1.6)*pa(i)*(pa(i)-p(i))*sqrt(300/at(i))
          do 9365 k=j,i-1
        w=w+f*(1.29/1.6)*pa(k)*(p(k+1)-p(k))*sqrt(300/at(k))
9365      continue
        tnc(i,j)=tn2c4(w)
9369    continue
9370   continue
         tnc(19,19)=1.0
        return
        end
********************************************************************
****************************************************************
*     Calculates Vib-rot H20 in CH4 overlap 1200-1350
*     Rodgers & Walshaw 1966
      subroutine trrn(p,u4,tr)
      real p,u4,tr,m,kd,kpa
      m=1.66*u4
      kd=12.65
      kpa=142.13
      tr=exp(-kd*m/sqrt(1+kpa*m/p))
      return
      end
****************************************************************
*********************************************************
      subroutine trr6(t,p,u4,tr)
*     transmission of rotation band from 660-800 cm-1 using
*     statistical model Goody 1964, Rodgers & Walshaw 1966
      real kd,kpa,a,a1,b,b1,phi,t,psi,phb,m,mb
      m=1.66*u4
      kd=9.706
      kpa=162.6
      a=.0168
      a1=.0172
      b=-3.63e-5
      b1=-4.86e-5
      phi=exp(a*(t-260)+b*(t-260)**2)
      psi=exp(a1*(t-260)+b1*(t-260)**2)
      mb=phi*m
      phb=psi*p*m/mb
      tr=exp(-(kd*mb)/sqrt(1+kpa*mb/phb))
       return
       end
****************************************************************
*     Calculate CH4 cooling via Ramanathan 1980
      subroutine ch4cool (ta,p,pa,t,d4,f4,tr4,c40,ac,kap,nt)
      real p(20),pa(20),d4(20),f4(20),b(20),t(20)
      real tr4(20,20),a4(20,20),ta(20,20),ut(20,20)
      real pe,bet,a0,u,c40,c,ac,a,x,d
      integer kap,nt
      a=3.03
      x=0.104
      d=1.012
      call bpl(b,t,1306.0)
      c=1.66*134*c40*(1.28/1.6)
*     134 (Goody 1989) is the band strength S 1.28*(c40/1.6) *delta p
*     is the absorber amount in atm-cm c40 is the concemtration
*     of CH4 in ppmv
      call ucalc(p,pa,ut,5,a,x,d,1.6)
      do 9010 i=1,19
        do 9005 j=1,i
         pe=(pa(j)+p(i))/2
          bet=pe*.211*(300./ta(j,i))
*     bet0=.17 from Ramanathan 1980
        a0=68.2*(ta(j,i)/300)**.858
            u=(c/a0)*ut(j,i)
            a4(j,i)=200*a0*log(1+(u/(.106+sqrt(3.59+u*(1+1/bet)))))
            u=(c/a0)*ut(i,j)
            a4(i,j)=200*a0*log(1+(u/(.106+sqrt(3.59+u*(1+1/bet)))))
         a4(j,i)=a4(j,i)*tr4(j,i)
         a4(i,j)=a4(i,j)*tr4(i,j)
9005    continue
9010   continue
*       write (20,9011) ((a4(i,j)/10000,j=1,19),i=1,19)
9011    format (19(f6.4,1x))
        call acool(a4,b,p,d4,f4,ac,kap,nt)
        return
        end
****************************************************************
*     Calculate N2O 1200-1350 cooling via Ramanathan 1980
      subroutine n2o5cool (ta,p,pa,t,d5,f5,tr5,tnc4,n20,ac,kap,nt,ut)
      real p(20),pa(20),d5(20),f5(20),b(20),t(20)
      real tr5(20,20),a5(20,20),ut(20,20),ta(20,20),tnc4(20,20)
      real pe,bet,a0,u,n20,c,ac,a,x,d
      integer kap,nt
      a=.559
      x=.2
      d=.096
      call bpl(b,t,1285.0)
      c=1.66*264*n20*(.239/.30)
*     264 (Goody 1989) is the band strength S .239*(n20/.30) *delta p
*     is the absorber amount in atm-cm n20 is the concemtration
*     of n2o in ppmv
      call ucalc(p,pa,ut,5,a,x,d,.30)
      do 9010 i=1,19
        do 9005 j=1,i
         pe=(pa(j)+p(i))/2
          bet=pe*1.12*(300./ta(j,i))**.5
*     bet0=1.12 from Ramanathan 1980
        a0=20.4*(ta(j,i)/300)**.5
            u=(c/a0)*ut(j,i)
            a5(j,i)=200*a0*log(1+(u/sqrt(4+u*(1+1/bet))))
            u=(c/a0)*ut(i,j)
            a5(i,j)=200*a0*log(1+(u/sqrt(4+u*(1+1/bet))))
         a5(j,i)=a5(j,i)*tr5(j,i)*tnc4(j,i)
         a5(i,j)=a5(i,j)*tr5(i,j)*tnc4(i,j)
9005    continue
9010   continue
*       write (20,9999) ((a5(i,j)/10000,j=1,19),i=1,19)
9999    format (19(f6.4,1x))
        call acool(a5,b,p,d5,f5,ac,kap,nt)
        return
        end
****************************************************************
*     Calculate N2O  520-660  cooling via Ramanathan 1980
      subroutine n2o6cool (p,pa,ta,t,d5,f5,tr6,tco2n2,n20,ac,kap,nt,ut)
      real p(20),pa(20),d6(20),f6(20),b(20),t(20),d5(20),f5(20)
      real tr6(20,20),a6(20,20),tco2n2(20,20),ta(20,20),ut(20,20)
      real pe,bet,a0,u,n20,c,ac
      integer kap,nt
      call bpl(b,t,589.0)
      c=1.66*24*n20*(.239/.30)
*     24 (Ramanathan 1985) is the band strength S .239*(n20/.30) *delta p
*     is the absorber amount in atm-cm n20 is the concemtration
*     of n2o in ppmv
      do 9010 i=1,19
        do 9005 j=1,i
         pe=(pa(j)+p(i))/2
          bet=pe*1.08*(300./ta(j,i))**.5
*     bet0=1.12 from Ramanathan 1980
        a0=23.0*(ta(j,i)/300)**.5
            u=(c/a0)*ut(j,i)
            a6(j,i)=200*a0*log(1+(u/sqrt(4+u*(1+1/bet))))
            u=(c/a0)*ut(i,j)
            a6(i,j)=200*a0*log(1+(u/sqrt(4+u*(1+1/bet))))
                a6(j,i)=a6(j,i)*tr6(j,i)*tco2n2(j,i)
                a6(i,j)=a6(i,j)*tr6(i,j)*tco2n2(i,j)
9005    continue
9010   continue
*       write (20,9990) ((a6(i,j)/10000,j=1,19),i=1,19)
9990    format (19(f6.4,1x))
        call acool(a6,b,p,d6,f6,ac,kap,nt)
        do 9991 i=1,19
         d5(i)=d5(i)+d6(i)
         f5(i)=f5(i)+f6(I)
9991   continue
        return
        end
****************************************************************
      subroutine ucalc (p,pa,ut,nt,a,x,d,c0)
      real u(20),c(20),p(20),pa(20)
      real ut(20,20)
      integer nt
      do 9520 i=1,19
        if(i.le.nt)then
          c(i)=(a*(pa(i)**x)-d)/c0
         else
          c(i)=1.0
        end if
       u(i)=c(i)*(p(i+1)-p(i))
9520   continue
       do 9540 i=1,19
        do 9535 j=1,i
        ut(i,j)=0.0
         ut(j,i)=0.0
      if (j.eq.1) then
        ut(j,i)=u(j)
        else
        ut(j,i)=c(j)*(p(j+1)-pa(j))
      end if
       if (i.eq.j) then
        ut(i,j)=c(i)*(pa(i)-p(i))
       else
        ut(i,j)=u(j)+c(i)*(pa(i)-p(i))
      end if
      do 9530 k=j+1,i-1
       ut(i,j)=ut(i,j)+u(k)
       ut(j,i)=ut(j,i)+u(k)
9530   continue
9535   continue
9540   continue
      return
      end
************************************************************************
*     Calculates solar heatin due to CO2 following the paramaterization
*     given by Sasamori 1972
      subroutine sco2(p,pa,shco2,abco2,c20,s0,theta,ac,kap)
      real p(20),pa(20),shco2(20),a2(20)
      real u2,abco2,c20,theta,s0,mu0,ac
	integer kap
        mu0=cos(theta)
        abco2=0.0
        u2=0.0
        a2(1)=0.0
      do 9600 i=1,18
        u2=u2+c20*.8*(p(i+1)-p(i))*pa(i)
        a2(i+1)=(2.35e-3)*((u2+.0129)**.26)-7.5e-4
9600  continue
      do 9610 i=1,18
	if (i.gt.kap-1) then
	shco2(i)=(1-ac)*s0*mu0*(a2(i+1)-a2(i))
	 else
        shco2(i)=s0*mu0*(a2(i+1)-a2(i))
	 end if
        abco2=abco2+shco2(i)
        shco2(i)=shco2(i)*.0083224/(p(i+1)-p(i))
9610  continue
      return
      end
**************************************************************************
*      Calculates solar absorbtion due to molecular Oxygen following
*      Sasamori 1972
       Subroutine soxy(p,sho2,abo2,s0,theta)
       real p(20),sho2(20),a4(20)
       real abo2,theta,mu0,s0
       mu0=cos(theta)
       abo2=0.0
       a4(1)=0.0
       do 9700 i=1,18
         a4(i+1)=7.5e-3*(p(i+1)/mu0)**.875
9700   continue
       do 9710 i=1,18
*       if (i.gt.kap-1) then
*       sho2(i)=(1-ac)*s0*mu0*(a4(i+1)-a4(i))
*       else
       sho2(i)=s0*mu0*(a4(i+1)-a4(i))
*       end if
       abo2=abo2+sho2(i)
       sho2(i)=sho2(i)*.0083224/(p(i+1)-p(i))
9710   continue
       return
       end
***********************************************************************
       subroutine small(ta,p,pa,t,ds,fs,cs0,ac,
     x kap,nt,v0,strength)
       real ta(20,20),p(20),pa(20),t(20),b(20),ds(20),fs(20),as(20,20)
       real dst(20),fst(20)
       real cs0,ac,v0,strength
       integer kap,nt
       call bpl(b,t,v0)
       c=1.66*strength*.8*cs0/1e6
       do 9808 i=1,19
        do 9806 j=1,i
         if(j.eq.1) then
          as(j,i)=100*c*ta(j,i)*p(i)/300
          as(i,j)=100*c*(ta(i,j)/300)*pa(i)
           else
          as(j,i)=100*c*(ta(j,i)/300)*abs(p(i)-pa(j))
          as(i,j)=100*c*(ta(i,j)/300)*abs(pa(i)-p(j))
         end if
9806     continue
9808     continue
         call acool(as,b,p,dst,fst,ac,kap,nt)
         do 9810 i=1,19
         ds(i)=ds(i)+dst(i)
         fs(i)=fs(i)+fst(i)
9810     continue
         return
         end
***********************************************************************
