;; Clase 20091020
;; Autor: Victor Ramiro
;; (i) Clausuras, Estado y Objetos
;; (ii) Continuaciones en Scheme
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Clausuras
;; esta lambda se "cierra" sobre las varibles definidas en su scope lexico (el env en la creación de closureV)
(define counter
  (let ((x 0))
    (lambda()
      (set! x (+ 1 x))
      x)))

(counter) ; evalua a 1
(counter) ; que obtenemos?

;; creamos un "constructor" de contadores
;; cada uno guarda su propio valor de inicio
(define make-counter
  (lambda(x)
    (lambda()
      (set! x (+ 1 x))
      x)))

(define c1 (make-counter 0))
(c1)
(c1)
(define c2 (make-counter 10))
(c2)
(c1)

;; un "objeto" pair
;; tenemos un constructor de un pair [pair == (cons car cdr)]
;; la lambda interna recibe la llamada a sus "metodos"
;; los métodos tienen acceso a las "variables de instancia" [variables del scope léxico de definición]
(define make-pair
  (lambda(car cdr)
    (lambda(method)
      (cond
        ((eq? method 'car) car)
        ((eq? method 'cdr) cdr)
        (else "error")))))

(define mypair (make-pair 1 2))
(mypair 'car)
(mypair 'cdr)

;; dummy para nil
(define nil #f)

;; un mutable-pair
;; los métodos reciben 1 argumento [para car y cdr pasamos nil]
(define make-mutable-pair
  (lambda(car cdr)
    (lambda(method arg)
      (cond
        ((eq? method 'car) car)
        ((eq? method 'cdr) cdr)
        ((eq? method 'set-car!) (set! car arg))
        ((eq? method 'set-cdr!) (set! cdr arg))
        (else "error")))))

(define my-mutable-pair (make-mutable-pair 1 2))
(my-mutable-pair 'car nil)
(my-mutable-pair 'cdr nil)
(my-mutable-pair 'set-car! 10)
(my-mutable-pair 'set-cdr! 11)
(my-mutable-pair 'car nil)
(my-mutable-pair 'cdr nil)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; Continuaciones en Scheme
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; el nombre real de call/cc es call-with-current-continuation
; creamos un alias para hacer más fácil el código
(define call/cc call-with-current-continuation)

;; println
(define (print x)
  (display x)
  (newline))

;; creamos una variable para guardar la continuación de la próxima invocación
(define captured-continuation nil)

;; sumamos 1 + (call/cc ...)
;; la continuación kont hace referencia al stack que aún falta por evaluar
;; En este caso, es como tener kont = (lambda(x)(+ 1 x))
(+ 1 (call/cc
      (lambda(kont)
        (set! captured-continuation kont)
        2)))

(captured-continuation 3)

;; otra forma equivalente:
;; nos aprovechamos de que kont es una función que recibe un parámetro
;; que es el valor de retorno desde el call/cc
(+ 1 (call/cc
      (lambda(kont)
        (set! captured-continuation kont)
        (kont 2))))

(captured-continuation 3)

;; y otra forma equivalente:
;; una vez invocada la continuación, se vuelve al stack original
;; Nunca se ejecutará el código que queda en (call/cc ...)
(+ 1 (call/cc
      (lambda(kont)
        (set! captured-continuation kont)
        (kont 2)
        24
        (display "nunca me imprimire :-("))))

(captured-continuation 3)


;; Usando continuaciones para manejar excepciones
(define (div n m)
  (/ n 
     (call/cc
      (lambda(raise)
        (if (= m 0) (raise 1) m)))))

(div 4 2)
(div 4 0)



;; el código de (full-search sym tree)
(define (full-search sym tree)
  (call/cc
   (lambda(kont)
     (define (search tree)
       (cond 
         ((eq? sym tree) (kont #t))
         ((pair? tree) (search (car tree)) (search (cdr tree)))))              
     (search tree) 
     #f)))


(full-search 'x '(x y))

;;Una versión más conocida... [con return]
(define (full-search sym tree)
  (call/cc
   (lambda(return)
     (define (search tree)
       (cond 
         ((eq? sym tree) (return #t))
         ((pair? tree) (search (car tree)) (search (cdr tree)))))              
     (search tree) 
     #f)))

(full-search 'x '(x y))

;;

(define (f return)
  (return 2)
  3)

(print (f (lambda(x) x)))

(print (call/cc f))

;;;;;;
(define the-kont #f)

(define (set-kont)
  (let ((i 0))
    (call/cc 
     (lambda(kont)
       (set! the-kont kont)))
    (set! i (+ 1 i))
    i))

(print "set-kont")
(set-kont)
(the-kont)
(the-kont)

(define saved-kont the-kont)
(set-kont)
(saved-kont)
(set-kont)

;; Codigo de Luis Mateu (sacado de sus tesis de Doctorado)
;; Recorrido de un arbol hoja por hoja
;; Ejemplo de uso: (define walker (make-walker '((a . b) . c) 'eot)
;; (walker) -> a, (walker) -> b, etc
(define (make-walker tree eot)
    (define (walk tree)
        (cond
	  ((not (pair? tree))
	    (dispatch tree)) ; retorna una hoja
	  (else
	    (walk (car tree))
	    (walk (cdr tree)))))

    (define current-walk-k 'void)
    (define current-client-k 'void)

    (define (dispatch leaf)
      (call/cc (lambda(walk-k) (set! current-walk-k walk-k) ; (3)
	                       (current-client-k leaf) )))

    (define (sequencer)
      (call/cc (lambda(client-k)
			  (set! current-client-k client-k) ; (4)
			  (current-walk-k 'void) ))) ; esto retorna en (2)
						     ; luego en (3)

    (set! current-walk-k
      (call/cc (lambda(init-k)	; (1)
	(call/cc
          (lambda(start-walk-k) (init-k start-walk-k))) ;(2) esto retorna en (1)
	(walk tree)
	(dispatch eot))))

    sequencer )

(define walker (make-walker '((a . b) . c) 'eot))
(walker)
(walker)
(walker)
(walker)
