function [K] = rigidez_2D(Datos)
% [K] = rigidez_2D_1(Datos)
% Autor: Rodrigo Carreo V.
% Funcin que genera la matriz de rigidez para una estructura 2D sin cachos
% rgidos.
% ========================================================================%
% Datos     :   String con el nombre del archivo excel (sin escribir la
%               extensin .xls) donde estn los datos de las barras de la
%               estructura.
% Datos a escribir en archivo excel base (ver 1ra fila archivo excel):
% Barra N  :   Describe la numeracin de la barra (solo til para ordenar
%               bien los datos).
% Tipo Apoyo:   Describe la condicin de apoyo de la barra en cuestin,
%               hay 4 posibilidades:
%                                       1: Soldado-Soldado
%                                       2: Soldado-Rotulado
%                                       3: Rotulado-Soldado
%                                       4: Rotulado-Rotulado
% Longitud  :   Longitud de la barra.
% EI        :   Producto entre mdulo de elasticidad del material (E) de la
%               barra y la inercia de la seccin de esta (I).
% GA        :   Producto entre el mdulo de corte (G) del material de la 
%               barra y el rea de esta (A). Nota: Este valor solo es til
%               si se consideran deformaciones por corte de la barra, si no
%               poner cualquier valor NO NULO.
% AE        :   Producto entre el rea de la seccin de la barra (A) y el
%               mdulo de elasticidad del material de esta (E). Si barra
%               axialmente indeformable, conviene numerar el GDL asociado a
%               la deformacin axial de la barra al final, y luego quitar
%               ltimas filas y columnas de la matriz K.
% eta(H)    :   Factor de forma de la seccin de la barra para 
%               deformaciones de corte  (poner =0 para no considerar 
%               deformaciones de corte)
% gamma     :   ngulo entre la barra y un eje horizontal fijo (en grados
%               sexagesimales), definido para todas las barras de la 
%               estructura. 
% ua,va,phia:   Correspondencia entre los GDL en el vertice inicial de la 
%               barra y los GDL definidos para la estructura. 
% ub,vb,phib:   Correspondencia entre los GDL en el vertice final de la 
%               barra y los GDL definidos para la estructura.
% ========================================================================%


Datos=xlsread([Datos,'.xls']);
[filas,columnas]=size(Datos); %#ok<NASGU>
N_GDL=max(max(Datos(:,9:14)));
K=zeros(N_GDL,N_GDL);

for i=1:filas
    beta=6*Datos(i,4)*Datos(i,7)/(Datos(i,5)*Datos(i,3)^2);
    k11=2*Datos(i,4)*(2+beta)/(Datos(i,3)*(1+2*beta));
    k22=k11;
    k12=2*Datos(i,4)*(1-beta)/(Datos(i,3)*(1+2*beta));
    k21=k12;
    k33=Datos(i,6)/Datos(i,3);
    k11p=k11-k12^2/k22;
    k22p=k22-k21^2/k11;
    C=cos(Datos(i,8)*pi/180);
    S=sin(Datos(i,8)*pi/180);
    C_L=C/Datos(i,3);
    S_L=S/Datos(i,3);
    ua=Datos(i,9);
    va=Datos(i,10);
    phia=Datos(i,11);
    ub=Datos(i,12);
    vb=Datos(i,13);
    phib=Datos(i,14);
    switch Datos(i,2)
        case 1 % Soldado-Soldado
            kp=[k11,k12,0;k21,k22,0;0,0,k33];
            ap=[-S_L,C_L,1,S_L,-C_L,0;-S_L,C_L,0,S_L,-C_L,1;C,S,0,-C,-S,0];
            k_barra=ap'*kp*ap;
            K_aux=[k_barra,zeros(6,N_GDL-6);zeros(N_GDL-6,N_GDL)];
            v= vec_pos([ua,va,phia,ub,vb,phib],N_GDL);
            K_aux=K_aux(v,v);
            K=K+K_aux;

        case 2 % Soldado-Rotulado
            kp=[k11p,0;0,k33];
            ap=[-S_L,C_L,1,S_L,-C_L;C,S,0,-C,-S];
            k_barra=ap'*kp*ap;
            K_aux=[k_barra,zeros(5,N_GDL-5);zeros(N_GDL-5,N_GDL)];
            v= vec_pos([ua,va,phia,ub,vb],N_GDL);
            K_aux=K_aux(v,v);
            K=K+K_aux;
        
        case 3 % Rotulado-Soldado
            kp=[k22p,0;0,k33];
            ap=[-S_L,C_L,S_L,-C_L,1;C,S,-C,-S,0];
            k_barra=ap'*kp*ap;
            K_aux=[k_barra,zeros(5,N_GDL-5);zeros(N_GDL-5,N_GDL)];
            v= vec_pos([ua,va,ub,vb,phib],N_GDL);
            K_aux=K_aux(v,v);
            K=K+K_aux;

        case 4  % Rotulado-Rotulado
            kp=k33;
            ap=[C,S,-C,-S];
            k_barra=ap'*kp*ap;
            K_aux=[k_barra,zeros(4,N_GDL-4);zeros(N_GDL-4,N_GDL)];
            v= vec_pos([ua,va,ub,vb],N_GDL);
            K_aux=K_aux(v,v);
            K=K+K_aux;
        otherwise , disp(['Condicin de apoyo para barra N ',...
                num2str(Datos(i,1)),' escrito erroneamente'])
    end
    
end
    function v= vec_pos(posiciones,NGDL)
        v=[];
        v_empty=(length(posiciones)+1:NGDL);
        for j=1:NGDL
            pos_aux=find(posiciones==j);
            if not(isempty(pos_aux))
                v=[v,pos_aux]; %#ok<AGROW>
                continue
            end
            v=[v,v_empty(1)]; %#ok<AGROW>
            if length(v_empty)>1
                v_empty=v_empty(2:end);
            else 
                v_empty=[];
            end     
        end
    end
end