
;; INTERPRETER FOR BASE

(define run
  (lambda (text)
    (eval-program (parse-program text))))

(define eval-program
  (lambda (pgm)
    (set! *classes* (init-classes))
    (elaborate-class-decls! (program-decls pgm))
    (eval
      (parse-exp `(send (new Main) main))
      (make-empty-env))))

;; runtime structures
(define-struct class  (name super fields methods))
(define-struct method (class name params body))
(define-struct field  (class type name))

(define *classes* '())

;; build object, int and string classes
(define init-classes
  (lambda () 
    (let ((class-object (make-class 'Object #f '() (list (make-method 'Object 'init '()
                                                           (parse-body '(1))))))
          (class-int    (make-class 'int    #f '() '()))
          (class-string (make-class 'string #f '() '())))
      (list class-object class-int class-string))))


(define lookup-class 
  (lambda (cname)
    (or (find-if (lambda (class) (eqv? (class-name class) cname)) *classes*)
        (error 'lookup-class "No class named ~s." cname))))

(define elaborate-class-decls!
  (lambda (decls)
    (for-each (lambda (decl)
		(let* ((cname (class-decl-cname decl))
		       (super (lookup-class (class-decl-sname decl)))
		       (fdecls (collect-if  field-decl? (class-decl-decls decl)))
		       (mdecls (collect-if method-decl? (class-decl-decls decl)))
                       (class (make-class cname super '() '()))
		       (fields  (get-fields class fdecls))
		       (methods (make-methods class mdecls)))
                  (set-class-fields! class fields)
                  (set-class-methods! class methods)
		  (set! *classes* (append *classes* (list class)))
		  class))
	      decls)))

(define get-fields
  (lambda (class fdecls)
    (let ((super-fields (class-fields (class-super class)))		
	  (fields (make-fields class fdecls)))
      (append super-fields fields))))

(define make-fields
  (lambda (class fdecls)
    (map (lambda (fdecl) (make-field class
                                     (field-decl-type fdecl)
                                     (field-decl-fname fdecl)))
         fdecls)))

(define make-methods
  (lambda (class mdecls)
    (map (lambda (mdecl) (make-method class 
                                      (method-decl-mname mdecl)
                                      (method-decl-params mdecl)
                                      (method-decl-body mdecl)))
         mdecls)))

;;;;;;;;;;;;;;;; objects ;;;;;;;;;;;;;;;;;;;;;

(define-struct object (class vals))

(define instanceof?
  (lambda (obj class)
    (subclass? (object-class obj) class)))

(define subclass?
  (lambda (sub sup)
    (cond ((eqv? (class-name sub) (class-name sup)) #t)
          ((not sub) #f)
          (else (subclass? (class-super sub) sup)))))


;;;;;;;;;;;;;;;; the interpreter ;;;;;;;;;;;;;;;;

(define eval
  (lambda (exp env)
    (cond ((lit-exp? exp) (lit-exp-value exp))
          
          ;; primitives are absorbed from scheme
          ((prim-app-exp? exp)
           (let ((args (eval-args (prim-app-exp-args exp) env)))
             (apply (prim-app-exp-prim exp) args)))

          ((if-exp? exp)
           (if (eval (if-exp-test exp) env)
               (eval (if-exp-then exp) env)
               (eval (if-exp-else exp) env)))
          
          ((and-exp? exp)
           (let loop ((args (and-exp-args exp)))
             (if (null? args)
                 #t
                 (and (eval (car args) env)
                      (loop (cdr args))))))
     
          ((or-exp? exp)
           (let loop ((args (or-exp-args exp)))
             (if (null? args)
                 #f
                 (or (eval (car args) env)
                     (loop (cdr args))))))
     
          ((let-exp? exp)
           (let ((pnames (pnames (let-exp-params exp)))
                 (args   (eval-args (let-exp-args exp) env)))
             (eval-body (let-exp-body exp) (extend-env pnames args env))))
     
          ((vget? exp)
           (let ((name  (vget-name exp)))
             (env-get name env)))
     
          ((vset? exp)
           (let ((name (vset-name exp))
                 (nval  (eval (vset-val exp) env)))
             (env-set! nval name env)
             'undefined))
 
          ((call? exp)
           (let* ((sig (call-sig exp))
                  (obj (eval (call-target exp) env))
                  (args (eval-args (call-args exp) env)))
             (call-method sig obj args))) 
                  
          ((scall? exp)
           (let* ((sig  (scall-sig exp))
                  (args  (scall-args exp))
                  (class (env-get '%host env))
                  (obj    (env-get 'this env))
                  (args   (eval-args args env)))
             (call-super sig (class-super class) obj args)))
          
          ((new? exp)
           (new-object (lookup-class (new-class exp))
                       (eval-args (new-args exp) env)))

          ((fget? exp)
            (let* ((obj  (env-get 'this env))
                   (name (fget-name exp)))
             ;(get-field-value name obj)))
              (get-field-value name (env-get '%host env) (object-vals obj))))
            
          ((fset? exp)
           (let* ((obj  (env-get 'this env))
                  (name (fset-name exp))
                  (nval (eval (fset-val exp) env)))
             ;(set-field-value! name obj nval)
             (set-field-value! name (env-get '%host env) (object-vals obj) nval)
             'undefined))

          ((instanceof-exp? exp)
           (instanceof? (eval (instanceof-target exp) env)
                        (lookup-class (instanceof-class exp))))
     
          (else (error 'eval "not an expression -- ~s" exp)))))

(define eval-body
  (lambda (body env)
    (let loop ((exps (body-exps body)) (val #f))
      (if (null? exps)
          val
          (loop (cdr exps) (eval (car exps) env))))))

(define eval-args
  (lambda (args env)
    (map (lambda (x) (eval x env)) args)))

(define call-method
  (lambda (sig obj args)
    (execute (lookup-method sig (object-class obj)) obj args)))

(define call-super
  (lambda (sig scls obj args)
    (execute (lookup-method sig scls) obj args)))

(define lookup-method
  (lambda (sig class)
    (if (not class) ;; reached top
        (error 'eval "No method ~s in ~s." sig (class-name class))
        (let ((method (find-if (lambda (method) (eqv? (method-name method) sig))
                               (class-methods class))))
          (if (method? method)
              method
              (lookup-method sig (class-super class)))))))

(define execute
  (lambda (method this args)
    (eval-body (method-body method)
               (new-env (append '(this %host) (pnames (method-params method)))
                        (append (list this (method-class method)) args)))));)

(define new-object
  (lambda (class args)
    (let* ((fields (class-fields class))
           (vals   (make-vector (length fields)))
           (obj    (make-object class vals)))  ;; instantiation
      (execute (lookup-method 'init class) obj args) ;; constructor
      obj)))
    
(define get-field-value
  (lambda (fname class vals)
    (let ((pos (lookup-field-pos fname class)))
      (vector-ref vals pos))))
    
(define set-field-value! 
  (lambda (fname class vals nval)
    (let ((pos (lookup-field-pos fname class)))
      (vector-set! vals pos nval))))

(define lookup-field-pos
  (lambda (fname class)
    (let ((pos (find-last-position fname (fnames (class-fields class)))))
      (if (number? pos) 
          pos
          (error 'lookup-field
                 "~s does not have a field ~s." (class-name class) fname)))))

(define fnames (lambda (fields) (map field-name fields)))



;;;;;;;;;;;;;;;; environments ;;;;;;;;;;;;;;;;

(define-struct environment ())
(define-struct (empty-env    environment) ())
(define-struct (extended-env environment) (ids vals env))

;; new environment with new mappings
(define new-env
  (lambda (ids vals)
    (extend-env ids vals (make-empty-env)))) 

;; extend environment with new mappings
(define extend-env
  (lambda (ids vals env)
    (make-extended-env ids (list->vector vals) env)))

;; lookup id in environment, apply action if found
(define env-lookup
  (lambda (action id env)
    (cond ((empty-env? env) (error 'env-lookup "No binding for ~s" id))
          ((extended-env? env)
           (let ((pos (find-position id (extended-env-ids env))))
             (if (number? pos)
                 (action pos env) ;; read or write
                 (env-lookup action id (extended-env-env env)))))
          (else (error 'env-lookup "not an environment -- ~s" env)))))

(define env-get
  (lambda (id env)
    (env-lookup (lambda (pos env) 
                  (vector-ref (extended-env-vals env) pos))
                id env)))

(define env-set!
  (lambda (id env nval)
    (env-lookup (lambda (pos env) 
                  (vector-set! (extended-env-vals env) pos nval))
                id env)))