;
; CC71P - 26/08/2007
; Autor: Victor Ramiro - vramiro@dcc.uchile.cl
; Obs: Estos ejemplos funcionan en Scheme R5RS
;

; Tema:
; Abstracción de funcionalidad (funciones como parámetros)

; algunas funciones necesarias
(define average (lambda(x y) (/ (+ x y) 2)))
(define square (lambda(x) (* x x)))

; Esta funcion calcula la raiz de x
; es super ineficiente, pero es un ejemplo :-)
; http://en.wikipedia.org/wiki/Methods_of_computing_square_roots#Babylonian_method

(define (sqrt x)
  (define (improve g) (average g (/ x g)))
  (define (good? g) (< (abs (- x (square g))) .001))
  (define (try g)
    (if (good? g)
        g
        (try (improve g))))
  (try 1))

(sqrt 2.0)

; Tema: Funciones como parametros

; Idea: Extraccion de patrones comunes, la sumatoria
; s = \sum_{i=a, i=i+v}^{b} f(i)
; Esta funcion recibe como parametro las funciones term
; term => f(i)
; next => i = i + v

(define (sum term a next b)
  (if (> a b)
      0
      (+ (term a)
         (sum term
              (next a)
              next
              b))))

; La suma de enteros
; en este caso, definimos internamente identity y next
(define (sum-int a b)
  (define (identity a) a)
  (define (next a) (+ 1 a))
  (sum identity a next b))

; La suma de los cuadrados
; en este caso, recordamos la notacion lambda
(define (sum-sq a b)
  (define next (lambda(x)(+ 1 x)))
  (sum square a next b))

; La suma de pi/8 (converge muuuy lento)
; en este caso, hacemos explicito el uso de la notacion lambda anonima
(define (pi-sum a b)
  (sum (lambda(i)(/ 1.0 (* i (+ i 2))))
       a
       (lambda(i)(+ 4 i))
       b))

; ejemplos de uso de las sumatorias
(sum-int 1 10)
(sum-sq 1 10)
(pi-sum 1 100)

; Funciones como resultado
; primera version de 'objetos'
; Encapsulacion de estado y un dispatcher
(define (point x y)
  (lambda (fun)
    (cond 
      ((eq? fun 'getX) x)
      ((eq? fun 'getY) y)
      ((eq? fun 'dist) (sqrt (+ (* x x) (* y y))))
      ))
  )

(define p (point 1.0 2.0))
(p'getX)
(p 'getY)
(p 'dist)
