;; Grammar
(define-type BCFAE
  [num (n number?)]
  [add (l BCFAE?) (r BCFAE?)]
  [id (v symbol?)]
  [fun (arg symbol?) (body BCFAE?)]
  [app (fun-expr BCFAE?) (arg-expr BCFAE?)]
  [if0 (test BCFAE?) (truth BCFAE?) (falsity BCFAE?)]
  [newbox (value BCFAE?)]
  [setbox (boxed-value BCFAE?) (new-value BCFAE?)]
  [openbox (boxed-value BCFAE?)]
  [seqn (e BCFAE?) (f BCFAE?)]
  )

(define-type ValuexStore
  [vxs (value BCFAE-Value?) (store Store?)])

;; Language Value
(define-type BCFAE-Value
  [numV (n number?)]
  [closureV (param symbol?) (body BCFAE?) (env Env?)]
  [boxV (location number?)])

(define-type Env
  [mtSub]
  [aSub (name symbol?) (location number?) (env Env?)])

(define-type Store
  [mtSto]
  [aSto (location number?) (value BCFAE-Value?) (store Store?)])

;; env-lookup : symbol Env -> location!
(define (env-lookup name env)
  (type-case Env env
             [mtSub () (error 'env-lookup (format "no binding for identifier: ~v" name))]
             [aSub (bound-name bound-location rest-env)
                   (if (symbol=? bound-name name)
                       bound-location
                       (env-lookup name rest-env))]))

;; store-lookup : location Store -> BCFAE-Value
(define (store-lookup loc-index sto)
  (type-case Store sto
             (mtSto () (error 'store-lookup (format "no value at location: ~v" loc-index)))
             [aSto (location value rest-store)
                   (if (= location loc-index)
                       value
                       (store-lookup loc-index rest-store))]))

;; parse : sexp -> BCFAE
(define (parse sexp)
  (cond
    [(number? sexp) (num sexp)]
    [(symbol? sexp) (id sexp)]
    [(list? sexp)
     (case (first sexp)
       [(+) (add (parse (second sexp)) (parse (third sexp)))]
       [(with) (app (fun (first (second sexp)) (parse (third sexp))) (parse (second (second sexp))))]       
       [(fun) (fun (first (second sexp)) (parse (third sexp)))]
       [(if0) (if0 (parse (second sexp)) (parse (third sexp)) (parse (fourth sexp)))]       
       [(newbox) (newbox (parse (second sexp)))]
       [(setbox) (setbox (parse (second sexp)) (parse (third sexp)))]
       [(openbox) (openbox (parse (second sexp)))]
       [(seqn) (seqn (parse (second sexp)) (parse (third sexp)))]
       [else (app (parse (first sexp)) (parse (second sexp)))]
       )]))
  
;; next-location (for creating new values at store)
(define next-location
  (local ([define last-loc (box -1)])
    (lambda (store)
      (begin
        (set-box! last-loc (+ 1 (unbox last-loc)))
        (unbox last-loc)
        ))))

(define (num+ n m)
  (numV (+ (numV-n n) (numV-n m))))

(define (num-zero? n)
  (zero? (numV-n n)))

;; interp : BCFAE Env Store -> ValuexStore
(define (interp expr env store)
  (type-case BCFAE expr
             [num (n) (vxs (numV n) store)]
             [add (l r)
                  (type-case ValuexStore (interp l env store)
                             [vxs (l-value l-store)
                                  (type-case ValuexStore (interp l env store)
                                             [vxs (l-value l-store)
                                                  (type-case ValuexStore (interp r env l-store)
                                                             [vxs (r-value r-store)
                                                                  (vxs (num+ l-value r-value) r-store)])])])]
             [id (v) (vxs (store-lookup (env-lookup v env) store) store)]
             [fun (bound-id bound-body)
                  (vxs (closureV bound-id bound-body env) store)]
             [app (fun-expr arg-expr)
                  (type-case ValuexStore (interp fun-expr env store)
                             [vxs (fun-value fun-store)
                                  (type-case ValuexStore (interp arg-expr env fun-store)
                                             [vxs (arg-value arg-store)
                                                  (local ([define new-loc (next-location arg-store)])
                                                    (interp (closureV-body fun-value)
                                                            (aSub (closureV-param fun-value)
                                                                  new-loc
                                                                  (closureV-env fun-value))
                                                            (aSto new-loc arg-value arg-store)))])])]
             [if0 (test truth falsity)
                  (type-case ValuexStore (interp test env store)
                             [vxs (test-value test-store)
                                  (if (num-zero? test-value)
                                      (interp truth env test-store)
                                      (interp falsity env test-store))])]
             [newbox (value-expr)
                     (type-case ValuexStore (interp value-expr env store)
                                [vxs (expr-value expr-store)
                                     (local ([define new-loc (next-location expr-store)])
                                       (vxs (boxV new-loc)
                                            (aSto new-loc expr-value expr-store)))])]
             [setbox (box-expr value-expr)
                     (type-case ValuexStore (interp box-expr env store)
                                [vxs (box-value box-store)
                                     (type-case ValuexStore (interp value-expr env box-store)
                                                [vxs (value-value value-store)
                                                     (vxs value-value
                                                          (aSto (boxV-location box-value)
                                                                value-value
                                                                value-store))])])]
             [openbox (box-expr)
                      (type-case ValuexStore (interp box-expr env store)
                                 [vxs (box-value box-store)
                                      (vxs (store-lookup (boxV-location box-value)
                                                         box-store)
                                           box-store)])]
             [seqn (e f)
                   (type-case ValuexStore (interp e env store)
                              [vxs (e-value e-store)
                                   (interp f env e-store)])]
             ))
                                                         
;; interp convenience
(define (interp-mt expr)
  (type-case ValuexStore (interp expr (mtSub) (mtSto))
             [vxs (value store) value]))

;; TESTS
(interp-mt (parse '{with {x 5} x}))
(interp-mt (parse '{with {b {newbox 0}}
                         {seqn {setbox b {+ 1 {openbox b}}}
                               {openbox b}}}))
(interp-mt (parse '{with {a {newbox 1}}
                         {with {f {fun {x} {+ x {openbox a}}}}
                               {seqn 
                                {setbox a 2}
                                {f 5}}}}))
{interp-mt (parse '{with {b {newbox 0}}
                         {if0 {seqn {setbox b 5}
                                    {openbox b}}
                              1
                              {openbox b}}})}
(interp-mt (parse '{with {switch {newbox 0}}
                         {with {toggle {fun {dum}
                                            {if0 {openbox switch}
                                                 {seqn
                                                  {setbox switch 1}
                                                  1
                                                  }
                                                 {seqn
                                                  {setbox switch 0}
                                                  0
                                                  }}}}
                               {+ {toggle 41} {toggle 41}}
                               }}))
(interp-mt (parse '{with {b {newbox 4}}
                         {+ {openbox b}
                            {with {dummy {setbox b 5}}
                                  {openbox b}}}}))