;; AST
(define-type FAE
  [num (n number?)]
  [add (l FAE?) (r FAE?)]
  [id (name symbol?)]
  [with (name symbol?) (named-expr FAE?) (body FAE?)]
  [fun (param symbol?) (body FAE?)]
  [app (fun-expr FAE?) (arg-expr FAE?)]
  )

;; Language Values
(define-type FAE-Value
  [numV (n number?)]
  [closureV (param symbol?) (body FAE?) (ds DefrdSub?)]
  )

;; Deferred Substitution
(define-type DefrdSub
  [mtSub]
  [aSub (name symbol?) (value FAE-Value?) (ds DefrdSub?)]
  )

;; lookup : symbol DefrdSub -> FAE-Value
(define (lookup name ds)
  (type-case DefrdSub ds
             [mtSub () (error 'lookup "no binding for identifier")]
             [aSub (bound-name bound-value rest-ds)
                   (if (symbol=? bound-name name)
                       bound-value
                       (lookup name rest-ds))]))

(define (parse sexp)
  (cond
    ((number? sexp) (num sexp))
    ((symbol? sexp) (id sexp))
    ((list? sexp)
     (case (first sexp)
       ((+) (add (parse (second sexp)) (parse (third sexp))))
       ((with) (with (first (second sexp)) (parse (second (second sexp))) (parse (third sexp))))
       ((fun) (fun (first (second sexp)) (parse (third sexp))))
       (else (app (parse (first sexp)) (parse (second sexp))))
       ))))         

;; numV-op : numV numV -> numV
(define (numV-op op n m)
  (numV (op (numV-n n) (numV-n m))))

;; interp : FAE DefrdSub -> FAE-Value
(define (interp expr ds)
  (type-case FAE expr
             [num (n) (numV n)]
             [add (l r) (numV-op + (interp l ds) (interp r ds))]
             [id (v) (lookup v ds)]
             [with (bound-id bound-value bound-body)
                   (interp bound-body (aSub bound-id (interp bound-value ds) ds))]
             [fun (bound-id bound-body)
                  (closureV bound-id bound-body ds)]
             [app (fun-expr arg-expr)
                  (local ([define fun-val (interp fun-expr ds)])
                    (interp (closureV-body fun-val)
                            (aSub (closureV-param fun-val)
                                  (interp arg-expr ds)
                                  (closureV-ds fun-val))))]
             ))

;; interp in empty-environment
(define (interp-mt expr) (interp expr (mtSub)))             
             
;; TEST
(interp-mt (parse '{{fun (x) {+ x x}} 5}))
(interp-mt (parse '{with {double {fun {x} {+ x x}}}
                      {double {double 5}}}))

(interp-mt (parse '{with {sum {fun {x} {fun {y} {+ x y}}}}
                         {{sum 5} 10}}))
                                   
                      