package cl.uchile.cc10a.io.console;

import java.awt.Button;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Event;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.Label;
import java.awt.Panel;
import java.awt.Point;
import java.awt.Toolkit;

/**
 * Message class
 *
 * A class to display a dialog box with a message in it until the user 
 * clicks the OK button.  Do not instantiate a Message from an
 * action or paint method, as it does the constructor does not 
 * return until the OK button is pressed (or the window is closed).
 *
 * Full documentation for the classes in the hsa package available at: 
 *			http://www.holtsoft.com/java/hsa_package.html
 *
 * @author Tom West
 * @version 1.0, Jan 1, 1998
 */
public class Message extends Frame
{
	private Button okButton;

/**
 * Contructor - Message to be displayed on centre of screen.
 *
 * @param message The message to be displayed in the Message dialog box.
 */
public Message (String message)
{
	this (message, "", null);
} // Constructor - Message (String)
/**
 * Contructor - Message to be displayed on centre of a specified Frame.
 *
 * @param message The message to be displayed in the Message dialog box.
 * @param frame The Frame that the dialog box should be centred on.
 */
public Message (String message, Frame frame)
{
	this (message, "", frame);
} // Constructor - Message (String, Frame)
/**
 * Contructor - Message to be displayed on centre of screen with a specified title.
 *
 * @param message The message to be displayed in the Message dialog box.
 * @param title The title of the Message dialog box.
 */
public Message (String message, String title)
{
	this (message, title, null);
} // Constructor - Message (String, String)
/**
 * Contructor - Message to be displayed on centre of a specified Frame.
 *
 * @param message The message to be displayed in the Message dialog box.
 * @param frame The Frame that the dialog box should be centred on.
 * @param title The title of the Message dialog box.
 */
public Message (String message, String title, Frame frame)
{
	// Create the window
	super (title);

	setBackground (Color.lightGray);
	
	// Put the message at the top
	this.add ("Center", new Label ("  " + message + "  ", Label.CENTER));

	// Put OK button at bottom
	okButton = new Button ("OK");
	Panel p = new Panel ();
	p.setLayout (new FlowLayout (FlowLayout.CENTER, 0, 0));
	p.add (okButton);
	add ("South", p);

	pack ();

	// Place the Message Frame in the middle of the screen or Frame, as appropriate.
	Dimension dlg = getSize ();
	Point loc;
	if (frame == null)
	{
		// Place frame in middle of the screen
		Dimension screen = getToolkit ().getScreenSize ();
		loc = new Point ((screen.width - dlg.width) / 2,
			(screen.height - dlg.height) / 3);
	}
	else
	{
		// Place frame in enclosing dialog box.
		Dimension window = frame.getSize ();
		Point windowLoc = frame.getLocation ();
		loc = new Point (windowLoc.x + (window.width - dlg.width) / 2,
			windowLoc.y + (window.height - dlg.height) / 3);
	}	
	setLocation (loc);
			
	// Display the Message Frame.			
	show ();
	toFront ();
	
	// Wait until the OK button is pressed.
	waitUntilDone ();
	hide ();
}  // Constructor - Message (String, String, Frame)
/**
 * Act upon the user hitting the OK button
 */
public synchronized boolean action (Event e, Object arg)
{
	if (e.target == okButton)
	{
	    this.hide ();
	    notifyAll ();
	    return (true);
	}
	return (false);
} // action (Event, Object)
/**
 * Causes the PC speaker to beep.
 */
public static void beep ()
{
	Toolkit.getDefaultToolkit ().beep ();
} // beep (void)
/**
  * Overrides the handleEvent method.  It handles window being closed.
	*/
public synchronized boolean handleEvent (Event evt)
{
	switch (evt.id)
	{
		case Event.WINDOW_DESTROY:
	    this.hide ();
	    notifyAll ();
	    return (true);
	}
	return super.handleEvent (evt);
} // handleEvent (Event)
/**
 * Waits until the user has pressed OK closed the window.
 */
private synchronized void waitUntilDone ()
{
	Message.beep ();
	
	// Pause until button wakes us up */
	try
	{
	    wait ();
	}
	catch (InterruptedException e)
	{
	    ;
	}
	dispose ();
} // waitUntilDone (void)
} /* Message class */
