package cl.uchile.cc10a.io.console;

import java.awt.Canvas;
import java.awt.Color;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Image;
import java.awt.Toolkit;


/**
 * ConsoleCanvas class
 *
 * The ConsoleCanvase class implements the actual drawing surface of
 * a Console object.  It also creates an offscreen image so that the
 * drawing surface can refresh itself when it needs to.
 *	
 * The interface to the console class consists pretty much of methods
 * for drawing on the surface.  The canvas contains no variables to
 * control the general text handling except that it handles the cursor
 * entirely on its own.
 *	 
 * Note that most graphics methods in this class are synchronized.
 * This is because we don't want to worry about messing up the cursor.
 * When an object is drawn to the onscreen Graphics context, the cursor
 * is toggled off if necessary and then when the drawing is finished,
 * it's toggled back on.  If the thread that controls the cursor
 * blinking on and off woke up and tried to toggle the cursor in the
 * middle of the draw operations, the drawing could get messed up.
 *
 * @author Tom West
 * @version 1.0, 1 Jan 1998
 */
class ConsoleCanvas extends Canvas implements Runnable
{
	/**
	 * Constants for drawing the pane.
	 */
	private static final int MARGIN = 2;
	private static final int DEPTH = 3;
	private static final int GREY_MARGIN = 5;
	
	/** 
	 * The name of thread for blinking the cursor.
	 */
	private static final String CONSOLE_CURSOR_THREAD_NAME =
		"Console Cursor Thread";
		
	/**
	 * Variables for controlling the consoleCanvas pane.
	 */
	private Console parentConsole;		
	private int originX, originY;
	private int canvasMaxHeight, canvasMaxWidth; // max size of canvas
	private Image offscreenImage;
	private boolean inXORMode = false;
	private Color xorColor;
	private int numRows, numCols;					// size of drawing surface
	private int numXPixels, numYPixels;		// size of drawing surface
	
	/**
	 * Information about the font used for text drawing.
	 */
	private Font font = null;
	private int fontWidth = 0, fontHeight = 0, fontBase = 0;
	
	/**
	 * Information about the cursor.
	 */
	private Thread cursorThread;
	private int cursorRow = 1, cursorCol = 1;
	private boolean cursorVisible = false;
	private int cursorBlinking = 0;

/**
 * Creates the console canvas given the size of the screen in rows and 
 * columns and font size.
 *
 * @param parent The parent Console class.
 * @param rows The height of the canvas in rows of text.
 * @param columns The width of the canvas in columns of text.
 * @param fontSize The size of the font in the canvas.
 */
ConsoleCanvas (Console parent, int rows, int columns, 
	int fontSize)
{
	parentConsole = parent;
	numRows = rows;
	numCols = columns;

  // Set the appropriate font (Monospaced in 1.1, Courier in 1.0)
  String [] fontNames = Toolkit.getDefaultToolkit ().getFontList ();
  
  for (int fontNum = 0 ; fontNum < fontNames.length ; fontNum++)
  {
  		if (fontNames [fontNum].equalsIgnoreCase ("monospaced"))
  		{
			font = new Font (fontNames [fontNum], Font.PLAIN, fontSize);
			break;
  		}
  }
  
  if (font == null)
  {	
		for (int fontNum = 0 ; fontNum < fontNames.length ; fontNum++)
		{
			if (fontNames [fontNum].equalsIgnoreCase ("Courier"))
			{
				font = new Font (fontNames [fontNum], Font.PLAIN, fontSize);
				break;
			}
		}
	}
	
	if (font == null)
	{			
		font = new Font ("Dialog", Font.PLAIN, fontSize);
	}		

	// Now that we have a real life context, set the canvas size to the
	// appropriate size based on the font size.
	FontMetrics	fm = Toolkit.getDefaultToolkit ().getFontMetrics (font);
	
	fontHeight = fm.getHeight () + fm.getLeading ();
	fontBase = fm.getDescent ();
	// fm.maxAdvance gives a huge number because of never used wide
	// glyphs in Courier.
	// Was: fontWidth = fm.getMaxAdvance ();
	fontWidth = 0;
	for (int ch = 0 ; ch < 256 ; ch++)
	{
		fontWidth = Math.max (fontWidth, fm.charWidth (ch));
	}		
	numXPixels = numCols * fontWidth;
	numYPixels = numRows * fontHeight;
	canvasMaxWidth = numXPixels + 2 * (MARGIN + DEPTH + GREY_MARGIN);
	canvasMaxHeight = numYPixels + 2 * (MARGIN + DEPTH + GREY_MARGIN);
	resize (canvasMaxWidth, canvasMaxHeight);

	cursorThread = new Thread (this);
	cursorThread.setName (CONSOLE_CURSOR_THREAD_NAME);
	cursorThread.start ();
	
	int	w = size ().width;
	int	h = size ().height;

	if (w > canvasMaxWidth)
	{
		originX = (w - canvasMaxWidth) / 2 + MARGIN + DEPTH + GREY_MARGIN;
	}
	else
	{
		originX = MARGIN + DEPTH + GREY_MARGIN;
	}

	if (h > canvasMaxHeight)
	{
		originY = (h - canvasMaxHeight) / 2 + MARGIN + DEPTH + GREY_MARGIN;
	}
	else
	{
		originY = MARGIN + DEPTH + GREY_MARGIN;
	}
} // Constructor - ConsoleCanvas (Console, int, int, int)
/**
 * Overrides addNotify in order to set up the device dependent features
 * once the canvas has actually been created.
 */
public void addNotify ()
{
	super.addNotify ();

	// Create the offscreen bitmap
	offscreenImage = createImage (numXPixels, numYPixels);
	Graphics offscreenGraphics = offscreenImage.getGraphics ();
	offscreenGraphics.setFont (font);

	// Clear the screen to white
	offscreenGraphics.setColor (Color.white);
	offscreenGraphics.fillRect (0, 0, numXPixels, numYPixels);
} // addNotify (void)
/**
 * Blinks the cursor off it was on and on if it was off.
 */
private synchronized void blinkCursor ()
{
	toggleCursor ();
	cursorVisible = !cursorVisible;
} // blinkCursor (void)
/**
 * Clears a rectangle on the console canvas.
 */
synchronized void clearRect (int x, int y, int width, int height)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First clear the rectangle on the offscreen image.
	offscreenGraphics.clearRect (x, y, width, height);
	
	// Then clear the rectangle on the onscreen image.
	if (cursorVisible) toggleCursor ();
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.clearRect (x, y, width, height);
	if (cursorVisible) toggleCursor ();
} // clearRect (int, int, int, int)
/**
 * Clears the entire console canvas to a specified colour
 */
synchronized void clearScreen (Color bgColor)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// Erase the offscreen bitmap.
	offscreenGraphics.setColor (bgColor);
	offscreenGraphics.fillRect (0, 0, numXPixels, numYPixels);
	
	// Erase the onscreen window.
	if (cursorVisible) toggleCursor ();
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (bgColor);
	onscreenGraphics.fillRect (0, 0, numXPixels, numYPixels);
	if (cursorVisible) toggleCursor ();
} // clearScreen (Color)
/**
 * Clears a rectangle on console canvas from the specified row and column 
 * to the end of line 
 */
synchronized void clearToEOL (int row, int col, Color bgColor)
{
	int			x = (col - 1) * fontWidth;
	int			y = (row - 1) * fontHeight;
	int			len = numXPixels - x;
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First clear the rectangle on the offscreen image.
	offscreenGraphics.setColor (bgColor);
	offscreenGraphics.fillRect (x, y, len, fontHeight);
	
	// Then clear the rectangle on the onscreen image.
	if (cursorVisible) toggleCursor ();
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (bgColor);
	onscreenGraphics.fillRect (x, y, len, fontHeight);
	if (cursorVisible) toggleCursor ();
} // clearToEOL (int, int, Color)
/**
 * Copy an area of the screen from (x, y) to (x + width, y + height) onto the 
 * screen with top corner at (x + delta_x, y + delta_y).
 */
synchronized void copyArea (int x, int y, int width, 
	int height, int deltaX, int deltaY)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First copy the area in the offscreen image.
	offscreenGraphics.copyArea (x, y, width, height, deltaX, deltaY);
	
	// Update the onscreen image from the offscreen image
	// We can't use copyArea because there may be windows in front
	// of the console obscuring the screen.
	// Copy offscreen image
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.drawImage (offscreenImage, 0, 0, this);

	if (cursorVisible) toggleCursor ();
} // copyArea (int, int, int, int, int, int)
/**
 * Redraws the entire console canvas including the margins.
 */
private synchronized void doDraw (Graphics onscreenGraphics)
{
	int	cnt;
	int marginSize = MARGIN + DEPTH + GREY_MARGIN;

	if (cursorVisible) toggleCursor ();
	
  // Draw the grey area surrounding
  int[] x = {-marginSize, size ().width,
						size ().width, -marginSize, -marginSize, -MARGIN - DEPTH, 
						-MARGIN - DEPTH, numXPixels + MARGIN + DEPTH,
						numXPixels + MARGIN + DEPTH, - marginSize};
  	int [] y = {-marginSize, -marginSize, size ().height,size ().height,
						-MARGIN - DEPTH, -MARGIN - DEPTH, 
						numYPixels + MARGIN + DEPTH, numYPixels + MARGIN + DEPTH,
						-MARGIN - DEPTH, -MARGIN - DEPTH};
	onscreenGraphics.translate (originX, originY);
  	onscreenGraphics.setColor (Color.lightGray);
  	onscreenGraphics.fillPolygon (x, y, 10);
  
  // Draw the 3D Margin
  for (cnt = MARGIN + 2 ; cnt <= MARGIN + DEPTH ; cnt++)
  {
  		onscreenGraphics.draw3DRect (-cnt, -cnt,
				numXPixels + 2 * cnt - 1, numYPixels + 2 * cnt - 1, false);
  }
  onscreenGraphics.setColor (Color.black);
  onscreenGraphics.drawLine (-MARGIN - 1, -MARGIN - 1, numXPixels + MARGIN,
												-MARGIN - 1);
  onscreenGraphics.drawLine (-MARGIN - 1, -MARGIN - 1, -MARGIN - 1,
												numYPixels + MARGIN);
  	onscreenGraphics.setColor (Color.lightGray);										 
  onscreenGraphics.drawLine (numXPixels + MARGIN, -MARGIN - 1, numXPixels + MARGIN,
												numYPixels + MARGIN);
  onscreenGraphics.drawLine (-MARGIN - 1, numYPixels + MARGIN, numXPixels + MARGIN,
												numYPixels + MARGIN);
  										
	// Draw white margins
	onscreenGraphics.setColor (Color.white);
	for (cnt = 1 ; cnt <= MARGIN ; cnt++)
	{
		onscreenGraphics.drawRect (-cnt, -cnt,
				numXPixels + 2 * cnt - 1, numYPixels + 2 * cnt - 1);
	}

	// Copy offscreen image
	onscreenGraphics.drawImage (offscreenImage, 0, 0, this);

	if (cursorVisible) toggleCursor ();
} // doDraw (Graphics)
/**
 * Draws a 3D rectangle on the screen from (x, y) to 
 * (x + width, y + height) in specified colour.
 */
synchronized void draw3DRect (int x, int y, int width, 
	int height, boolean raised, Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.draw3DRect (x, y, width, height, raised);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.draw3DRect (x, y, width, height, raised);
	if (cursorVisible) toggleCursor ();
} // draw3DRect (int, int, int, int, boolean, Color)
/**
 * Draws an arc on the screen from (x, y) to 
 * (x + width, y + height) from startAngle to startAngle + arcAngle
 * in specified colour.
 */
synchronized void drawArc (int x, int y, int width, 
	int height, int startAngle, int arcAngle, Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.drawArc (x, y, width, height, startAngle, arcAngle);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.drawArc (x, y, width, height, startAngle, arcAngle);
	if (cursorVisible) toggleCursor ();
} // drawArc (int, int, int, int, int, int, Color)
/**
 * Draws a line from (x1, y1) to (x2, y2) in specified colour.
 */
synchronized void drawLine (int x1, int y1, int x2, int y2,
	Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.drawLine (x1, y1, x2, y2);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.drawLine (x1, y1, x2, y2);
	if (cursorVisible) toggleCursor ();
} // drawLine (int, int, int, int, Color)
/**
 * Draws an oval on the screen in the sqaure from (x, y) to 
 * (x + width, y + height) in specified colour.
 */
synchronized void drawOval (int x, int y, int width, int height, 
	Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.drawOval (x, y, width, height);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.drawOval (x, y, width, height);
	if (cursorVisible) toggleCursor ();
} // drawOval (int, int, int, int, Color)
/**
 * Draws a polygon.
 */
synchronized void drawPolygon (int[] xPoints, int[] yPoints, 
	int nPoints, Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the filled polygon to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.drawPolygon (xPoints, yPoints, nPoints);
	
	// Then draw the filled polgon to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.drawPolygon (xPoints, yPoints, nPoints);
	if (cursorVisible) toggleCursor ();
} // drawPolygon (int[], int[], int, Color)
/**
 * Draws a rectangle on the screen from (x, y) to 
 * (x + width, y + height) in specified colour.
 */
synchronized void drawRect (int x, int y, int width, int height, 
	Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.drawRect (x, y, width, height);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.drawRect (x, y, width, height);
	if (cursorVisible) toggleCursor ();
} // drawRect (int, int, int, int, Color)
/**
 * Draws a rounded rectangle on the screen from (x, y) to 
 * (x + width, y + height) in specified colour.
 */
synchronized void drawRoundRect (int x, int y, int width, int height, 
	int arcWidth, int arcHeight, Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.drawRoundRect (x, y, width, height, arcWidth, 
		arcHeight);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.drawRoundRect (x, y, width, height, arcWidth, 
		arcHeight);
	if (cursorVisible) toggleCursor ();
} // drawRoundRect (int, int, int, int, int, int, Color)
/**
 * Draws a line from (x1, y1) to (x2, y2) in specified colour.
 */
synchronized void drawString (String str, int x, int y, Font font, 
	Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.setFont (font);
	offscreenGraphics.drawString (str, x, y);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.setFont (font);
	onscreenGraphics.drawString (str, x, y);
	if (cursorVisible) toggleCursor ();
} // drawLine (int, int, int, int, Color)
/**
 * Draws the specified text to the screen at the specified row and column
 * using the specified foreground and background colours.
 */
synchronized void drawText (int row, int col, String text, 
				       Color fgColor, Color bgColor)
{
	int		x = (col - 1) * fontWidth;
	int		y = (row - 1) * fontHeight;
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw it to the offscreen image.

	// Erase the area that the image will appear on.	
	offscreenGraphics.setColor (bgColor);
	offscreenGraphics.fillRect (x, y, fontWidth * text.length (), fontHeight);

	// Draw the text
	offscreenGraphics.setColor (fgColor);
	offscreenGraphics.setFont (font);
	offscreenGraphics.drawString (text, x, y + fontHeight - fontBase);
	
	// Then draw the string to the onscreen image.
	
	// Erase the area that the image will appear on.	
	if (cursorVisible) toggleCursor ();
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (bgColor);
	onscreenGraphics.fillRect (x, y, fontWidth * text.length (), fontHeight);

	// Draw the text
	onscreenGraphics.setColor (fgColor);
	onscreenGraphics.setFont (font);
	onscreenGraphics.drawString (text, x, y + fontHeight - fontBase);
	if (cursorVisible) toggleCursor ();
} // drawText (int, int, String, Color, Color)
/**
 * Draws a filled 3D rectangle on the screen from (x, y) to 
 * (x + width, y + height) in specified colour.
 */
synchronized void fill3DRect (int x, int y, int width, 
	int height, boolean raised, Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.fill3DRect (x, y, width, height, raised);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.fill3DRect (x, y, width, height, raised);
	if (cursorVisible) toggleCursor ();
} // fill3DRect (int, int, int, int, boolean, Color)
/**
 * Draws a filled arc on the screen from (x, y) to 
 * (x + width, y + height) from startAngle to startAngle + arcAngle
 * in specified colour.
 */
synchronized void fillArc (int x, int y, int width, int height, 
	int startAngle, int arcAngle, Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.fillArc (x, y, width, height, startAngle, arcAngle);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.fillArc (x, y, width, height, startAngle, arcAngle);
	if (cursorVisible) toggleCursor ();
} // fillArc (int, int, int, int, int, int, Color)
/**
 * Draws a filled oval on the screen in the sqaure from (x, y) to 
 * (x + width, y + height) in specified colour.
 */
synchronized void fillOval (int x, int y, int width, int height, 
	Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the filled oval to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.fillOval (x, y, width, height);
	
	// Then draw the filled oval to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.fillOval (x, y, width, height);
	if (cursorVisible) toggleCursor ();
} // fillOval (int, int, int, int, Color)
/**
 * Draws a filled polygon.
 */
synchronized void fillPolygon (int[] xPoints, int[] yPoints, 
	int nPoints, Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the filled polygon to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.fillPolygon (xPoints, yPoints, nPoints);
	
	// Then draw the filled polgon to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.fillPolygon (xPoints, yPoints, nPoints);
	if (cursorVisible) toggleCursor ();
} // fillPolygon (int[], int[], int, Color)
/**
 * Draws a filled rectangle on the screen from (x, y) to 
 * (x + width, y + height) in specified colour.
 */
synchronized void fillRect (int x, int y, int width, int height, 
	Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.fillRect (x, y, width, height);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.fillRect (x, y, width, height);
	if (cursorVisible) toggleCursor ();
} // fillRect (int, int, int, int, Color)
/**
 * Draws a filled rounded rectangle on the screen from (x, y) to 
 * (x + width, y + height) in specified colour.
 */
synchronized void fillRoundRect (int x, int y, int width, int height, 
	int arcWidth, int arcHeight, Color color)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First draw the line to the offscreen image.
	if (inXORMode) offscreenGraphics.setXORMode (xorColor);
	offscreenGraphics.setColor (color);
	offscreenGraphics.fillRoundRect (x, y, width, height, arcWidth, 
		arcHeight);
	
	// Then draw the line to the onscreen image.
	if (cursorVisible) toggleCursor ();
	if (inXORMode) onscreenGraphics.setXORMode (xorColor);
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (color);
	onscreenGraphics.fillRoundRect (x, y, width, height, arcWidth, 
		arcHeight);
	if (cursorVisible) toggleCursor ();
} // fillRoundRect (int, int, int, int, int, int, Color)
/** 
 * Inverts the console canvas for 50 milliseconds.
 */
synchronized void invertScreen ()
{
	int		x = (cursorCol - 1) * fontWidth;
	int		y = (cursorRow - 1) * fontHeight;
	Graphics 	onscreenGraphics = getGraphics ();

	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.setColor (Color.white);
	onscreenGraphics.setXORMode (Color.black);
	onscreenGraphics.fillRect (0, 0, numXPixels, numYPixels);
	try 
	{
		Thread.sleep (50);
	}
	catch (Exception e) {}
	onscreenGraphics.fillRect (0, 0, numXPixels, numYPixels);
	onscreenGraphics.setPaintMode ();
} // invertScreen (void)
/**
 * This returns whether the main method (as generally set up by 
 * VisualAge for Java is still alive.  If it is not, then it means 
 * that the main thread of execution has terminated in the user's
 * program.  This does not work in Sun's JDK, which appears to keep
 * the thread alive even after execution is finished.
 *
 * @return Whether the main thread of execution is alive.
 */
boolean isMainRunning () 
{
	// Go through thread list looking for any alive priority five
	// jobs whose names don't start with AWT
	ThreadGroup rootGrp = Thread.currentThread().getThreadGroup ();
	Thread[] threads;
	while (rootGrp.getParent () != null)
	{
		rootGrp = rootGrp.getParent ();
	}
	threads = new Thread [rootGrp.activeCount()];	
	int count = rootGrp.enumerate (threads, true);
	for (int i = 0 ; i < count ; i++)
	{
		Thread t = threads [i];
		if ((t.getPriority () == 5) && (!t.isDaemon ()) && (t.isAlive ())
			&& (!t.getName ().startsWith ("AWT"))
			&& (!t.getName ().equals (CONSOLE_CURSOR_THREAD_NAME)))
		{
			return (true);
		}
	}
	return (false);	
} // boolean isMainRunning (void)
/**
 * This kills the cursor thread (used when the Console window is closed).
 */
void killCursorThread () 
{
	cursorThread.stop ();
} // killCursorThread (void)
/**
 * Returns the maximum x co-ordinate in the console window.
 */
int maxx ()
{
	return numXPixels - 1;
} // int maxx (void)
/**
 * Returns the maximum y co-ordinate in the console window.
 */
int maxy ()
{
	return numYPixels - 1;
} // int maxy (void)
/**
 * Overrides the paint method to redraw the screen using doDraw
 */
public void paint (Graphics g)
{
	doDraw (g);
} // paint (Graphics)
/**
 * ConsoleCanvas implements the Runnable interface.  When run, this thread
 * blinks the cursor every 300 milliseconds.
 */
public void run ()
{
	boolean	mainIsRunning = true;
	
	for (;;)
	{
		blinkCursor ();
		
		if (mainIsRunning && (!isMainRunning ()))
		{
			mainIsRunning = false;
			parentConsole.mainStopped ();		
			cursorThread.stop ();
		}
			
		try 
		{
			Thread.sleep (300);
		}
		catch (Exception e) {}
	}
} // run (void)
/** 
 * Scrolls up the entire ConsoleCanvas a single line.  The blank space at
 * the bottom is filled in the specified colour.	
 */
synchronized void scrollUpALine (Color bgColor)
{
	Graphics	offscreenGraphics = offscreenImage.getGraphics ();
	Graphics 	onscreenGraphics = getGraphics ();

	// First scroll the offscreen image.
	
	// Scroll the screen up
	offscreenGraphics.copyArea (0, fontHeight, numXPixels, 
					numYPixels - fontHeight, 0, -fontHeight);

	// Erase the last line	
	offscreenGraphics.setColor (bgColor);
	offscreenGraphics.fillRect (0, numYPixels - fontHeight, numXPixels, 
					fontHeight);
	
	// Then scroll the onscreen image.
	
	// Update the onscreen image from the offscreen image
	// We can't use copyArea because there may be windows in front
	// of the console obscuring the screen.
	// Copy offscreen image
	onscreenGraphics.translate (originX, originY);
	onscreenGraphics.drawImage (offscreenImage, 0, 0, this);

	if (cursorVisible) toggleCursor ();
} // scrollUpALine (Color)
/**
 * Set the cursor to the specified row and column.
 */
synchronized void setCursorPos (int row, int col)
{
	if (cursorVisible) toggleCursor ();
	cursorRow = row;
	cursorCol = col;
	if (cursorVisible) toggleCursor ();
} // setCursorPos (int, int)
/**
 * This makes the blinking cursor visible or invisible.
 *
 * @param visible Whether the cursor should be visible or not.
 */
synchronized void setCursorVisible (boolean visible) 
{
	if (visible)
	{
		cursorBlinking++;
		if (cursorBlinking == 1)
		{
			if (cursorVisible)
			{
				toggleCursor ();
			}
		}		
	}
	else
	{
		if (cursorBlinking == 1)
		{
			if (cursorVisible)
			{
				toggleCursor ();
			}
		}		
		cursorBlinking--;
	}		
	return;
} // setCursorVisible (boolean)
/**
 * Sets the drawing mode for any graphics to "Paint".
 */
synchronized void setPaintMode ()
{
	inXORMode = false;
} // setPaintMode (void)
/**
 * Sets the drawing mode for any graphics to "XOR".
 */
synchronized void setXORMode (Color xorColor)
{
	inXORMode = true;
	this.xorColor = xorColor;
} // setXORMode (Color)
/**
 * Toggle the visible cursor on the screen off or on.
 */
private synchronized void toggleCursor ()
{
	if (cursorBlinking > 0)
	{
		int			x = (cursorCol - 1) * fontWidth;
		int			y = (cursorRow - 1) * fontHeight;
		Graphics 	onscreenGraphics = getGraphics ();

		onscreenGraphics.translate (originX, originY);
		onscreenGraphics.setColor (Color.white);
		onscreenGraphics.setXORMode (Color.black);
		onscreenGraphics.fillRect (x, y, fontWidth, fontHeight);
		onscreenGraphics.setPaintMode ();
	}	
} // toggleCursor (void)
/**
 * Overrides the update method to redraw the screen using doDraw
 */
public void update (Graphics g)
{
	doDraw (g);
} // update (Graphics)

public Dimension getMinimumSize()
{
  return new Dimension(canvasMaxWidth, canvasMaxHeight);
}

public Dimension getPreferredSize()
{
  return getMinimumSize();
}

public Dimension getMaximumSize()
{
  return getMinimumSize();
}

} /* ConsoleCanvas class */
