package cl.uchile.cc10a.io.console;

import java.awt.Button;
import java.awt.Color;
import java.awt.Event;
import java.awt.FlowLayout;
import java.awt.Font;
import java.awt.Frame;
import java.awt.Panel;
import java.io.IOException;
import java.text.DecimalFormat;
import java.text.NumberFormat;

/**
 * Console class
 * 
 * A class to implement a standard text window along with graphics. It supports
 * the display of different data types to the screen and the reading in of all
 * the primitive data types as well as string. It also supports most of the
 * standard drawing methods that can used with the Graphics class. It also
 * supports a blinking cursor to indicate where input is taking place.
 * 
 * Full documentation for the classes in the hsa package available at:
 * http://www.holtsoft.com/java/hsa_package.html
 * 
 * @author Tom West
 * @version 1.0, Jan 1, 1998
 */
public class Console extends Frame {

	private static final long serialVersionUID = 1L;

	private static Console firstConsole = null;

	public static Console getFirstConsole() {
		return firstConsole;
	}

	/**
	 * The "Quit" button at the top of the console
	 */
	private Button quitButton = new Button("Quit");

	/**
	 * The actual drawing surface of the canvas
	 */
	private ConsoleCanvas consoleCanvas;

	/**
	 * Default values for Console window
	 */
	private static final int DEFAULT_ROWS = 25;

	private static final int DEFAULT_COLUMNS = 80;

	private static final int DEFAULT_FONT_SIZE = 14;

	private static final String DEFAULT_TITLE = "Console";

	/**
	 * Console text variables
	 */
	private int currentRow = 1, currentCol = 1;

	private int actualRow = 1, actualCol = 1;

	private int maxRow = 0, maxCol = 0;

	private int startCol = 0, startRow = 0;

	private Color textColor = Color.black;

	private Color textBGColor = Color.white;

	private boolean echoOn = true;

	/**
	 * Keyboard buffer variables
	 */
	private static final int BUFFER_SIZE = 2048;

	private static final int EMPTY_BUFFER = -1;

	private char[] kbdBuffer = new char[BUFFER_SIZE];

	private int kbdBufferHead = 0, kbdBufferTail = 0;

	private char[] lineBuffer = new char[BUFFER_SIZE];

	private int lineBufferHead = 0, lineBufferTail = 0;

	private int ungotChar = EMPTY_BUFFER;

	/**
	 * Console graphics variables
	 */
	private Color graphicsColor = Color.black;

	private Font font = new Font("Serif", Font.PLAIN, 12);

	/**
	 * Number of Consoles created
	 */
	private static int numConsoles = 0;

	/**
	 * 
	 */
	private static boolean useStandardIO = false;

	private boolean eofReached = false;

	/**
	 * Constructor - Create a console using the default values of: 25x80 screen,
	 * 14 pt font, "Console" as title.
	 */
	public Console() {
		this(DEFAULT_ROWS, DEFAULT_COLUMNS, DEFAULT_FONT_SIZE, DEFAULT_TITLE);
	} // Constructor - Console (void)

	/**
	 * Constructor - Create a console specifying font size and using the default
	 * values of: 25x80 screen, "Console" as title.
	 * 
	 * @param fontSize
	 *            The size of the font to be used in the Console window.
	 */
	public Console(int fontSize) {
		this(DEFAULT_ROWS, DEFAULT_COLUMNS, fontSize, DEFAULT_TITLE);
	} // Constructor - Console (int)

	/**
	 * Constructor - Create a console specifying screen size and using the
	 * default values of: 14 pt font, "Console" as title.
	 * 
	 * @param rows
	 *            The height of the window in rows.
	 * @param columns
	 *            The width of the window in columns.
	 */
	public Console(int rows, int columns) {
		this(rows, columns, DEFAULT_FONT_SIZE, DEFAULT_TITLE);
	} // Constructor - Console (int, int)

	/**
	 * Constructor - Create a console specifying screen and font size and using
	 * the default values of: "Console" as title.
	 * 
	 * @param rows
	 *            The height of the window in rows.
	 * @param columns
	 *            The width of the window in columns.
	 * @param fontSize
	 *            The size of the font to be used in the Console window.
	 */
	public Console(int rows, int columns, int fontSize) {
		this(rows, columns, fontSize, DEFAULT_TITLE);
	} // Constructor - Console (int, int, int)

	/**
	 * Constructor - Create a console specifying screen, font size and title.
	 * 
	 * @param rows
	 *            The height of the window in rows.
	 * @param columns
	 *            The width of the window in columns.
	 * @param fontSize
	 *            The size of the font to be used in the Console window.
	 */
	public Console(int rows, int columns, int fontSize, String title) {
		super(title);
		if (firstConsole == null)
			firstConsole = this;

		// Increment the number of existing widnows
		numConsoles++;

		// If there's more than one console window, set the title
		if ((title.equals(DEFAULT_TITLE)) && (numConsoles > 1)) {
			title = title + " " + numConsoles;
			setTitle(title);
		}

		// The console canvas
		consoleCanvas = new ConsoleCanvas(this, rows, columns, fontSize);
		maxRow = rows;
		maxCol = columns;

		// The Close button
		Panel quitPanel = new Panel();
		quitPanel.setBackground(Color.lightGray);
		quitPanel.setLayout(new FlowLayout(FlowLayout.CENTER, 0, 0));
		quitPanel.add(quitButton);

		// Set out layout for console
		add("North", quitPanel);
		add("Center", consoleCanvas);

		// Move the console window to the appropriate location
		// move ((numConsoles - 1) * 30, (numConsoles - 1) * 30);

		// Position all and display
		setBackground(Color.lightGray);
		// setResizable (false);
		pack();

		// If we're using Console in a submit command
		// Don't show the window
		if (!useStandardIO) {
			show();
		}

		repaint();
	} // Constructor - Console (int, int, int, String)

	/**
	 * Constructor - Create a console specifying screen size and title and using
	 * the default values of: 14 pt font.
	 * 
	 * @param rows
	 *            The height of the window in rows.
	 * @param columns
	 *            The width of the window in columns.
	 * @param title
	 *            The title of the Console window.
	 */
	public Console(int rows, int columns, String title) {
		this(rows, columns, DEFAULT_FONT_SIZE, title);
	} // Constructor - Console (int, int, String)

	/**
	 * Constructor - Create a console specifying font size and title and using
	 * the default values of: 25x80 screen.
	 * 
	 * @param fontSize
	 *            The size of the font to be used in the Console window.
	 * @param title
	 *            The title of the Console window.
	 */
	public Console(int fontSize, String title) {
		this(DEFAULT_ROWS, DEFAULT_COLUMNS, fontSize, title);
	} // Constructor - Console (int, String)

	/**
	 * Constructor - Create a console specifying the title and using the default
	 * values of: 25x80 screen, 14 pt font.
	 * 
	 * @param title
	 *            The title of the Console window.
	 */
	public Console(String title) {
		this(DEFAULT_ROWS, DEFAULT_COLUMNS, DEFAULT_FONT_SIZE, title);
	} // Constructor - Console (String)

	/**
	 * Clears the screen and moves the cursor to the top left corner.
	 */
	public void clear() {
		consoleCanvas.clearScreen(textBGColor);
		currentRow = 1;
		currentCol = 1;
		actualRow = 1;
		actualCol = 1;
		consoleCanvas.setCursorPos(currentRow, currentCol);
	} // clear (void)

	/**
	 * Clears a rectangle in the console to white.
	 * 
	 * @see java.awt.Graphics.clearRect
	 */
	public void clearRect(int x, int y, int width, int height) {
		consoleCanvas.clearRect(x, y, width, height);
	} // clearRect (int, int, int, int)

	/**
	 * Closes the console window.
	 */
	public void close() {
		consoleCanvas.killCursorThread();
		dispose();
	} // close (void)

	/**
	 * Copy an area of the screen from (x, y) to (x + width, y + height) onto
	 * the screen with top corner at (x + delta_x, y + delta_y).
	 * 
	 * @see java.awt.Graphics.copyArea
	 */
	public void copyArea(int x, int y, int width, int height, int delta_x,
			int delta_y) {
		consoleCanvas.copyArea(x, y, width, height, delta_x, delta_y);
	} // copyArea (int, int, int, int, int, int)

	/**
	 * Draws a 3D rectangle on the screen from (x, y) to (x + width, y + width).
	 * 
	 * @see java.awt.Graphics.draw3DRect
	 */
	public void draw3DRect(int x, int y, int width, int height, boolean raised) {
		consoleCanvas.draw3DRect(x, y, width, height, raised, graphicsColor);
	} // draw3DRect (int, int, int, int, boolean)

	/**
	 * Draws an arc on the screen from (x, y) to (x + width, y + height) from
	 * startAngle to startAngle + arcAngle in specified colour.
	 * 
	 * @see java.awt.Graphics.drawArc
	 */
	public void drawArc(int x, int y, int width, int height, int startAngle,
			int arcAngle) {
		consoleCanvas.drawArc(x, y, width, height, startAngle, arcAngle,
				graphicsColor);
	} // drawArc (int, int, int, int, int, int)

	/**
	 * Draws a line on the screen from (x1, y1) to (x2, y2).
	 * 
	 * @see java.awt.Graphics.drawLine
	 */
	public void drawLine(int x1, int y1, int x2, int y2) {
		consoleCanvas.drawLine(x1, y1, x2, y2, graphicsColor);
	} // drawLine (int, int, int, int)

	/**
	 * Draws a maple leaf on the screen from (x, y) to (x + width, y + width).
	 * 
	 * @param x
	 *            The x coordinate of the top left corner of the rectangle that
	 *            the maple leaf is inscribed in.
	 * @param y
	 *            The y coordinate of the top left corner of the rectangle that
	 *            the maple leaf is inscribed in.
	 * @param width
	 *            The width of the rectangle that the maple leaf is inscribed
	 *            in.
	 * @param height
	 *            The height of the rectangle that the maple leaf is inscribed
	 *            in.
	 */
	public void drawMapleLeaf(int x, int y, int width, int height) {
		int[] xPoints, yPoints;
		float rx, ry, xc, yc;

		rx = width;
		ry = height;
		xc = x + rx / 2;
		yc = y + height;

		xPoints = new int[26];
		yPoints = new int[26];
		xPoints[0] = (int) (xc + rx * 0.021423);
		yPoints[0] = (int) (yc - ry * 0.215686);
		xPoints[1] = (int) (xc + rx * 0.270780);
		yPoints[1] = (int) (yc - ry * 0.203804);
		xPoints[2] = (int) (xc + rx * 0.271820);
		yPoints[2] = (int) (yc - ry * 0.295752);
		xPoints[3] = (int) (xc + rx * 0.482015);
		yPoints[3] = (int) (yc - ry * 0.411765);
		xPoints[4] = (int) (xc + rx * 0.443046);
		yPoints[4] = (int) (yc - ry * 0.483267);
		xPoints[5] = (int) (xc + rx * 0.500000);
		yPoints[5] = (int) (yc - ry * 0.587435);
		xPoints[6] = (int) (xc + rx * 0.363353);
		yPoints[6] = (int) (yc - ry * 0.619576);
		xPoints[7] = (int) (xc + rx * 0.342287);
		yPoints[7] = (int) (yc - ry * 0.693849);
		xPoints[8] = (int) (xc + rx * 0.153596);
		yPoints[8] = (int) (yc - ry * 0.612537);
		xPoints[9] = (int) (xc + rx * 0.201601);
		yPoints[9] = (int) (yc - ry * 0.918462);
		xPoints[10] = (int) (xc + rx * 0.093001);
		yPoints[10] = (int) (yc - ry * 0.894514);
		xPoints[11] = (int) xc;
		yPoints[11] = (int) (yc - ry);
		xPoints[12] = (int) (xc - rx * 0.093001);
		yPoints[12] = yPoints[10];
		xPoints[13] = (int) (xc - rx * 0.201601);
		yPoints[13] = yPoints[9];
		xPoints[14] = (int) (xc - rx * 0.153596);
		yPoints[14] = yPoints[8];
		xPoints[15] = (int) (xc - rx * 0.342287);
		yPoints[15] = yPoints[7];
		xPoints[16] = (int) (xc - rx * 0.363353);
		yPoints[16] = yPoints[6];
		xPoints[17] = (int) (xc - rx * 0.500000);
		yPoints[17] = yPoints[5];
		xPoints[18] = (int) (xc - rx * 0.443046);
		yPoints[18] = yPoints[4];
		xPoints[19] = (int) (xc - rx * 0.482015);
		yPoints[19] = yPoints[3];
		xPoints[20] = (int) (xc - rx * 0.271820);
		yPoints[20] = yPoints[2];
		xPoints[21] = (int) (xc - rx * .2707796);
		yPoints[21] = yPoints[1];
		xPoints[22] = (int) (xc - rx * 0.021423);
		yPoints[22] = yPoints[0];
		xPoints[23] = xPoints[22];
		yPoints[23] = (int) yc;
		xPoints[24] = xPoints[0];
		yPoints[24] = yPoints[23];
		xPoints[25] = xPoints[0];
		yPoints[25] = yPoints[0];
		consoleCanvas.drawPolygon(xPoints, yPoints, 26, graphicsColor);
	} // drawMapleLeaf (int, int, int, int)

	/**
	 * Draws an oval on the screen in the sqaure from (x, y) to (x + width, y +
	 * height).
	 * 
	 * @see java.awt.Graphics.drawOval
	 */
	public void drawOval(int x, int y, int width, int height) {
		consoleCanvas.drawOval(x, y, width, height, graphicsColor);
	} // drawOval (int, int, int, int)

	/**
	 * Draws a polygon specified by the arrays of points.
	 * 
	 * @see java.awt.Graphics.drawPolygon
	 */
	public void drawPolygon(int[] xPoints, int[] yPoints, int nPoints) {
		consoleCanvas.drawPolygon(xPoints, yPoints, nPoints, graphicsColor);
	} // drawPolygon (int[], int[], int)

	/**
	 * Draws a rectangle on the screen from (x, y) to (x + width, y + width).
	 * 
	 * @see java.awt.Graphics.drawRect
	 */
	public void drawRect(int x, int y, int width, int height) {
		consoleCanvas.drawRect(x, y, width, height, graphicsColor);
	} // drawRect (int, int, int, int)

	/**
	 * Draws a rounded rectangle on the screen from (x, y) to (x + width, y +
	 * width).
	 * 
	 * @see java.awt.Graphics.drawRoundRect
	 */
	public void drawRoundRect(int x, int y, int width, int height,
			int arcWidth, int arcHeight) {
		consoleCanvas.drawRoundRect(x, y, width, height, arcWidth, arcHeight,
				graphicsColor);
	} // drawRoundRect (int, int, int, int, int, int)

	/**
	 * Draws a star on the screen from (x, y) to (x + width, y + width).
	 * 
	 * @param x
	 *            The x coordinate of the top left corner of the rectangle that
	 *            the star is inscribed in.
	 * @param y
	 *            The y coordinate of the top left corner of the rectangle that
	 *            the star is inscribed in.
	 * @param width
	 *            The width of the rectangle that the star is inscribed in.
	 * @param height
	 *            The height of the rectangle that the star is inscribed in.
	 */
	public void drawStar(int x, int y, int width, int height) {
		int[] xPoints, yPoints;
		float rx, ry, xc, yc;

		rx = width;
		ry = height;
		xc = x + rx / 2;
		yc = y + height;

		xPoints = new int[11];
		yPoints = new int[11];
		xPoints[0] = (int) xc;
		yPoints[0] = (int) (yc - ry);
		xPoints[1] = (int) (xc + rx * 0.118034);
		yPoints[1] = (int) (yc - ry * 0.618560);
		xPoints[2] = (int) (xc + rx * 0.500000);
		yPoints[2] = yPoints[1];
		xPoints[3] = (int) (xc + rx * 0.190983);
		yPoints[3] = (int) (yc - ry * 0.381759);
		xPoints[4] = (int) (xc + rx * 0.309017);
		yPoints[4] = (int) yc;
		xPoints[5] = (int) xc;
		yPoints[5] = (int) (yc - ry * 0.236068);
		xPoints[6] = (int) (xc - rx * 0.309017);
		yPoints[6] = yPoints[4];
		xPoints[7] = (int) (xc - rx * 0.190983);
		yPoints[7] = yPoints[3];
		xPoints[8] = (int) (xc - rx * 0.500000);
		yPoints[8] = yPoints[2];
		xPoints[9] = (int) (xc - rx * 0.118034);
		yPoints[9] = yPoints[1];
		xPoints[10] = xPoints[0];
		yPoints[10] = yPoints[0];
		consoleCanvas.drawPolygon(xPoints, yPoints, 11, graphicsColor);
	} // drawStar (int, int, int, int)

	/**
	 * Draws a text on the screen at location (x, y).
	 * 
	 * @see java.awt.Graphics.drawString
	 */
	public void drawString(String str, int x, int y) {
		consoleCanvas.drawString(str, x, y, font, graphicsColor);
	} // drawString (String, int, int)

	/**
	 * Erases the entire line of input. Called when the user presses Ctrl+U when
	 * typing.
	 */
	private void eraseLineOfInput() {
		int numChars, cnt;

		numChars = (actualCol - startCol) + maxCol * (actualRow - startRow);
		currentRow = startRow;
		currentCol = startCol;
		actualRow = startRow;
		actualCol = startCol;
		for (cnt = 0; cnt < numChars; cnt++)
			print(" ");
		currentRow = startRow;
		currentCol = startCol;
		actualRow = startRow;
		actualCol = startCol;
		consoleCanvas.setCursorPos(currentRow, currentCol);
	} // eraseLineOfInput (void)

	/**
	 * Erases the previous character in a line of input. Called when the user
	 * presses backspace when typing.
	 */
	private void erasePreviousChar() {
		if (currentCol > 1) {
			currentCol--;
		} else {
			if (currentRow > 1) {
				currentRow--;
				currentCol = maxCol;
			}
		}
		actualRow = currentRow;
		actualCol = currentCol;

		consoleCanvas.drawText(currentRow, currentCol, " ", textColor,
				textBGColor);
		consoleCanvas.setCursorPos(currentRow, currentCol);

		if ((currentCol == 1) && (currentRow != startRow)) {
			currentCol = maxCol + 1;
			currentRow--;
		}
	} // erasePreviousChar (void)

	/**
	 * Draws a filled 3D rectangle on the screen from (x, y) to (x + width, y +
	 * width).
	 * 
	 * @see java.awt.Graphics.fill3DRect
	 */
	public void fill3DRect(int x, int y, int width, int height, boolean raised) {
		consoleCanvas.fill3DRect(x, y, width, height, raised, graphicsColor);
	} // fill3DRect (int, int, int, int, boolean)

	/**
	 * Draws a filled arc on the screen from (x, y) to (x + width, y + height)
	 * from startAngle to startAngle + arcAngle in specified colour.
	 * 
	 * @see java.awt.Graphics.fillArc
	 */
	public void fillArc(int x, int y, int width, int height, int startAngle,
			int arcAngle) {
		consoleCanvas.fillArc(x, y, width, height, startAngle, arcAngle,
				graphicsColor);
	} // fillArc (int, int, int, int, int, int)

	/**
	 * Draws a filled maple leaf on the screen from (x, y) to (x + width, y +
	 * width).
	 * 
	 * @param x
	 *            int The x coordinate of the top left corner of the rectangle
	 *            that the maple leaf is inscribed in.
	 * @param y
	 *            int The y coordinate of the top left corner of the rectangle
	 *            that the maple leaf is inscribed in.
	 * @param width
	 *            int The width of the rectangle that the maple leaf is
	 *            inscribed in.
	 * @param height
	 *            int The height of the rectangle that the maple leaf is
	 *            inscribed in.
	 */
	public void fillMapleLeaf(int x, int y, int width, int height) {
		int[] xPoints, yPoints;
		float rx, ry, xc, yc;

		rx = width;
		ry = height;
		xc = x + rx / 2;
		yc = y + height;

		xPoints = new int[26];
		yPoints = new int[26];
		xPoints[0] = (int) (xc + rx * 0.021423);
		yPoints[0] = (int) (yc - ry * 0.215686);
		xPoints[1] = (int) (xc + rx * 0.270780);
		yPoints[1] = (int) (yc - ry * 0.203804);
		xPoints[2] = (int) (xc + rx * 0.271820);
		yPoints[2] = (int) (yc - ry * 0.295752);
		xPoints[3] = (int) (xc + rx * 0.482015);
		yPoints[3] = (int) (yc - ry * 0.411765);
		xPoints[4] = (int) (xc + rx * 0.443046);
		yPoints[4] = (int) (yc - ry * 0.483267);
		xPoints[5] = (int) (xc + rx * 0.500000);
		yPoints[5] = (int) (yc - ry * 0.587435);
		xPoints[6] = (int) (xc + rx * 0.363353);
		yPoints[6] = (int) (yc - ry * 0.619576);
		xPoints[7] = (int) (xc + rx * 0.342287);
		yPoints[7] = (int) (yc - ry * 0.693849);
		xPoints[8] = (int) (xc + rx * 0.153596);
		yPoints[8] = (int) (yc - ry * 0.612537);
		xPoints[9] = (int) (xc + rx * 0.201601);
		yPoints[9] = (int) (yc - ry * 0.918462);
		xPoints[10] = (int) (xc + rx * 0.093001);
		yPoints[10] = (int) (yc - ry * 0.894514);
		xPoints[11] = (int) xc;
		yPoints[11] = (int) (yc - ry);
		xPoints[12] = (int) (xc - rx * 0.093001);
		yPoints[12] = yPoints[10];
		xPoints[13] = (int) (xc - rx * 0.201601);
		yPoints[13] = yPoints[9];
		xPoints[14] = (int) (xc - rx * 0.153596);
		yPoints[14] = yPoints[8];
		xPoints[15] = (int) (xc - rx * 0.342287);
		yPoints[15] = yPoints[7];
		xPoints[16] = (int) (xc - rx * 0.363353);
		yPoints[16] = yPoints[6];
		xPoints[17] = (int) (xc - rx * 0.500000);
		yPoints[17] = yPoints[5];
		xPoints[18] = (int) (xc - rx * 0.443046);
		yPoints[18] = yPoints[4];
		xPoints[19] = (int) (xc - rx * 0.482015);
		yPoints[19] = yPoints[3];
		xPoints[20] = (int) (xc - rx * 0.271820);
		yPoints[20] = yPoints[2];
		xPoints[21] = (int) (xc - rx * .2707796);
		yPoints[21] = yPoints[1];
		xPoints[22] = (int) (xc - rx * 0.021423);
		yPoints[22] = yPoints[0];
		xPoints[23] = xPoints[22];
		yPoints[23] = (int) yc;
		xPoints[24] = xPoints[0];
		yPoints[24] = yPoints[23];
		xPoints[25] = xPoints[0];
		yPoints[25] = yPoints[0];
		consoleCanvas.fillPolygon(xPoints, yPoints, 26, graphicsColor);
	} // fillMapleLeaf (int, int, int, int)

	/**
	 * Draws a filled oval on the screen in the sqaure from (x, y) to (x +
	 * width, y + height).
	 * 
	 * @see java.awt.Graphics.fillOval
	 */
	public void fillOval(int x, int y, int width, int height) {
		consoleCanvas.fillOval(x, y, width, height, graphicsColor);
	} // fillOval (int, int, int, int)

	/**
	 * Draws a filled polygon specified by the arrays of points..
	 * 
	 * @see java.awt.Graphics.fillPolygon
	 */
	public void fillPolygon(int[] xPoints, int[] yPoints, int nPoints) {
		consoleCanvas.fillPolygon(xPoints, yPoints, nPoints, graphicsColor);
	} // fillPolygon (int[], int[], int)

	/**
	 * Draws a filled rectangle on the screen from (x, y) to (x + width, y +
	 * width).
	 * 
	 * @see java.awt.Graphics.fillRect
	 */
	public void fillRect(int x, int y, int width, int height) {
		consoleCanvas.fillRect(x, y, width, height, graphicsColor);
	} // fillRect (int, int, int, int)

	/**
	 * Draws a filled rounded rectangle on the screen from (x, y) to (x + width,
	 * y + width).
	 * 
	 * @see java.awt.Graphics.fillRoundRect
	 */
	public void fillRoundRect(int x, int y, int width, int height,
			int arcWidth, int arcHeight) {
		consoleCanvas.fillRoundRect(x, y, width, height, arcWidth, arcHeight,
				graphicsColor);
	} // fillRoundRect (int, int, int, int, int, int)

	/**
	 * Draws a filled star on the screen from (x, y) to (x + width, y + width).
	 * 
	 * @param x
	 *            The x coordinate of the top left corner of the rectangle that
	 *            the star is inscribed in.
	 * @param y
	 *            The y coordinate of the top left corner of the rectangle that
	 *            the star is inscribed in.
	 * @param width
	 *            The width of the rectangle that the star is inscribed in.
	 * @param height
	 *            The height of the rectangle that the star is inscribed in.
	 */
	public void fillStar(int x, int y, int width, int height) {
		int[] xPoints, yPoints;
		float rx, ry, xc, yc;

		rx = width;
		ry = height;
		xc = x + rx / 2;
		yc = y + height;

		xPoints = new int[11];
		yPoints = new int[11];
		xPoints[0] = (int) xc;
		yPoints[0] = (int) (yc - ry);
		xPoints[1] = (int) (xc + rx * 0.118034);
		yPoints[1] = (int) (yc - ry * 0.618560);
		xPoints[2] = (int) (xc + rx * 0.500000);
		yPoints[2] = yPoints[1];
		xPoints[3] = (int) (xc + rx * 0.190983);
		yPoints[3] = (int) (yc - ry * 0.381759);
		xPoints[4] = (int) (xc + rx * 0.309017);
		yPoints[4] = (int) yc;
		xPoints[5] = (int) xc;
		yPoints[5] = (int) (yc - ry * 0.236068);
		xPoints[6] = (int) (xc - rx * 0.309017);
		yPoints[6] = yPoints[4];
		xPoints[7] = (int) (xc - rx * 0.190983);
		yPoints[7] = yPoints[3];
		xPoints[8] = (int) (xc - rx * 0.500000);
		yPoints[8] = yPoints[2];
		xPoints[9] = (int) (xc - rx * 0.118034);
		yPoints[9] = yPoints[1];
		xPoints[10] = xPoints[0];
		yPoints[10] = yPoints[0];
		consoleCanvas.fillPolygon(xPoints, yPoints, 11, graphicsColor);
	} // fillStar (int, int, int, int)

	/**
	 * Overrides the handleEvent method. It handles both keystrokes and the Quit
	 * button being pressed.
	 */
	public boolean handleEvent(Event evt) {
		switch (evt.id) {
		case Event.KEY_PRESS:
			handleKeystroke((char) evt.key);
			return (true);
		case Event.ACTION_EVENT:
			if (evt.target == quitButton) {
				System.exit(0);
			}
			break;
		case Event.WINDOW_DESTROY:
			System.exit(0);
			break;
		}
		return super.handleEvent(evt);
	} // handleEvent (Event)

	/**
	 * Places a keystroke in the keyboard buffer. It is synchronized so that
	 * there can't be a problem with input being taken off the keyboard buffer
	 * and placed on the keyboard buffer at the same time.
	 */
	private synchronized void handleKeystroke(char ch) {
		// Place the keystroke into the keyboard buffer.
		kbdBuffer[kbdBufferHead] = ch;
		kbdBufferHead = (kbdBufferHead + 1) % BUFFER_SIZE;

		// The following statements wakes up any processes that are
		// sleeping while waiting for heyboard input.
		notify();
	} // handleKeystroke (char)

	/**
	 * Makes the blinking cursor invisible.
	 */
	public void hideCursor() {
		consoleCanvas.setCursorVisible(false);
	} // hideCursor (void)

	/**
	 * This method is invoked if the "main" method appears to have stopped
	 * executing. This helps the user realize when the program has finished
	 * execution.
	 */
	void mainStopped() {
		quitButton.setLabel("Close");
		quitButton.getParent().invalidate();
		quitButton.getParent().validate();
	} // mainStopped (void)

	/**
	 * Returns the number of columns in the console window.
	 * 
	 * @return The width of the screen in columns.
	 */
	public int maxcol() {
		return (maxCol);
	} // int maxcol (void)

	/**
	 * Returns the number of rows in the console window.
	 * 
	 * @return The height of the screen in rows.
	 */
	public int maxrow() {
		return (maxRow);
	} // int maxrow (void)

	/**
	 * Returns the maximum x co-ordinate in the console window.
	 * 
	 * @return The maximum x co-ordinate in the console window.
	 */
	public int maxx() {
		return consoleCanvas.maxx();
	} // int maxx (void)

	/**
	 * Returns the maximum y co-ordinate in the console window.
	 * 
	 * @return The maximum y co-ordinate in the console window.
	 */
	public int maxy() {
		return consoleCanvas.maxy();
	} // int maxy (void)

	/**
	 * This method alerts Console to send all the output to standard output and
	 * read all input from standard input. This is used when submitting a
	 * program using the Console class.
	 */
	static void prepareForSubmit() {
		useStandardIO = true;
	} // prepareForSubmit (void)

	/**
	 * Write the text representation of an 8-bit integer (a "byte") to the
	 * Console.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void print(byte number) {
		print((int) number);
	} // print (byte)

	/**
	 * Write the text representation of an 8-bit integer (a "byte") to the
	 * Console with a specified field size.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void print(byte number, int fieldSize) {
		print((int) number, fieldSize);
	} // print (byte, int)

	/**
	 * Write a character to the Console.
	 * 
	 * @param ch
	 *            The character to be written to the Console.
	 */
	public void print(char ch) {
		print(String.valueOf(ch));
	} // print (char)

	/**
	 * Write a character to the Console with a specified field size..
	 * 
	 * @param ch
	 *            The character to be written to the Console.
	 * @param fieldSize
	 *            The field width that the character is to be written in.
	 */
	public void print(char ch, int fieldSize) {
		String charStr = String.valueOf(ch);
		StringBuffer padding = new StringBuffer();

		for (int cnt = 0; cnt < fieldSize - charStr.length(); cnt++) {
			padding.append(' ');
		}
		print(charStr + padding);
	} // print (char, int)

	/**
	 * Write a double precision floating point number (a "double") to the
	 * Console.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void print(double number) {
		print(String.valueOf(number));
	} // print (double)

	/**
	 * Write a double precision floating point number (a "double") to the
	 * Console with a specified field size.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void print(double number, int fieldSize) {
		double posValue = Math.abs(number);
		int placesRemaining = fieldSize;
		String format = null, numStr;
		StringBuffer padding = new StringBuffer();

		if (number < 0)
			placesRemaining--; // Space for the minus sign
		if (posValue < 10.0)
			format = "0";
		else if (posValue < 100.0)
			format = "00";
		else if (posValue < 1000.0)
			format = "000";
		else if (posValue < 10000.0)
			format = "0000";
		else if (posValue < 100000.0)
			format = "00000";
		else if (posValue < 1000000.0)
			format = "000000";
		else if (posValue < 10000000.0)
			format = "0000000";
		else if (posValue < 100000000.0)
			format = "00000000";

		if (format == null) {
			// We're using scientific notation
			numStr = String.valueOf(number);
		} else {
			// Add a decimal point, if there's room
			placesRemaining -= format.length();
			if (placesRemaining > 0) {
				format = format + ".";
				placesRemaining--;
			}

			// For any addition room, add decimal places
			for (int cnt = 0; cnt < placesRemaining; cnt++) {
				format = format + "#";
			}

			// Convert the number
			NumberFormat form = new DecimalFormat(format);
			numStr = form.format(number);
		}

		// If the number is not long enough, pad with spaces
		for (int cnt = 0; cnt < fieldSize - numStr.length(); cnt++) {
			padding.append(' ');
		}
		print(padding + numStr);
	} // print (double, int)

	/**
	 * Write a double precision floating point number (a "double") to the
	 * Console with a specified field size and a specified number of decimal
	 * places.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 * @param decimalPlaces
	 *            The number of decimal places of the number to be displayed.
	 */
	public void print(double number, int fieldSize, int decimalPlaces) {
		double posValue = Math.abs(number);
		int placesRemaining = fieldSize;
		String format = null, numStr;
		StringBuffer padding = new StringBuffer();

		if (number < 0)
			placesRemaining--; // Space for the minus sign
		if (posValue < 10.0)
			format = "0";
		else if (posValue < 100.0)
			format = "00";
		else if (posValue < 1000.0)
			format = "000";
		else if (posValue < 10000.0)
			format = "0000";
		else if (posValue < 100000.0)
			format = "00000";
		else if (posValue < 1000000.0)
			format = "000000";
		else if (posValue < 10000000.0)
			format = "0000000";
		else if (posValue < 100000000.0)
			format = "00000000";

		if (Math.abs(number) >= 100000000.0) {
			// We're using scientific notation
			numStr = String.valueOf(number);
		} else {
			format = "0.";

			// For any addition room, add decimal places
			for (int cnt = 0; cnt < decimalPlaces; cnt++) {
				format = format + "0";
			}

			// Convert the number
			NumberFormat form = new DecimalFormat(format);
			numStr = form.format(number);
		}

		// If the number is not long enough, pad with spaces
		for (int cnt = 0; cnt < fieldSize - numStr.length(); cnt++) {
			padding.append(' ');
		}
		print(padding + numStr);
	} // print (double, int, int)

	/**
	 * Write a floating point number (a "float") to the Console.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void print(float number) {
		print(String.valueOf(number));
	} // print (float)

	/**
	 * Write a floating point number (a "float") to the Console with a specified
	 * field size.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void print(float number, int fieldSize) {
		print((double) number, fieldSize);
	} // print (float, int)

	/**
	 * Write a floating point number (a "double") to the Console with a
	 * specified field size and a specified number of decimal places.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 * @param decimalPlaces
	 *            The number of decimal places of the number to be displayed.
	 */
	public void print(float number, int fieldSize, int decimalPlaces) {
		print((double) number, fieldSize, decimalPlaces);
	} // print (float, int, int)

	/**
	 * Write the text representation of an 32-bit integer (an "int") to the
	 * Console.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void print(int number) {
		print(String.valueOf(number));
	} // print (int)

	/**
	 * Write the text representation of an 32-bit integer (an "int") to the
	 * Console with a specified field size.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void print(int number, int fieldSize) {
		String numStr = String.valueOf(number);
		StringBuffer padding = new StringBuffer();

		for (int cnt = 0; cnt < fieldSize - numStr.length(); cnt++) {
			padding.append(' ');
		}
		print(padding + numStr);
	} // print (int, int)

	/**
	 * Write the text representation of an 64-bit integer (a "long") to the
	 * Console.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void print(long number) {
		print(String.valueOf(number));
	} // print (long)

	/**
	 * Write the text representation of an 64-bit integer (a "long") to the
	 * Console with a specified field size.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void print(long number, int fieldSize) {
		String numStr = String.valueOf(number);
		StringBuffer padding = new StringBuffer();

		for (int cnt = 0; cnt < fieldSize - numStr.length(); cnt++) {
			padding.append(' ');
		}
		print(padding + numStr);
	} // print (long, int)

	/**
	 * Write a string to the Console.
	 * 
	 * @param text
	 *            The string to be written to the Console.
	 */
	public void print(String text) {
		int index = 0;
		int len = text.length();
		int start = 0;

		if (useStandardIO) {
			System.out.print(text);
			return;
		}

		for (;;) {
			index = start;
			if (index == len) {
				consoleCanvas.setCursorPos(actualRow, actualCol);
				return;
			}
			while ((index < len) && (text.charAt(index) != '\n')
					&& (index - start < maxCol - currentCol)) {
				index++;
			}
			if (start != index) {
				// Draw what we have so far
				consoleCanvas.drawText(currentRow, currentCol, text.substring(
						start, index), textColor, textBGColor);
				currentCol += index - start;
				actualCol = currentCol;
			}
			if (index == len) {
				consoleCanvas.setCursorPos(actualRow, actualCol);
				return;
			}
			if (text.charAt(index) == '\n') {
				if ((currentRow <= maxRow) && (currentCol <= maxCol)) {
					consoleCanvas.clearToEOL(currentRow, currentCol,
							textBGColor);
				}
				if (currentRow < maxRow) {
					currentCol = 1;
					currentRow++;
					actualCol = currentCol;
					actualRow = currentRow;
				} else {
					consoleCanvas.scrollUpALine(textBGColor);
					startRow--;
					currentCol = 1;
					actualCol = currentCol;
				}
			} else {
				if (currentCol <= maxCol) {
					consoleCanvas.drawText(currentRow, currentCol, text
							.substring(index, index + 1), textColor,
							textBGColor);
					if (currentCol < maxCol) {
						currentCol++;
						actualCol = currentCol;
					} else {
						if (currentRow < maxRow) {
							currentCol++;
							actualCol = 1;
							actualRow++;
						} else {
							currentCol++;
						}
					}
				} else {
					if (currentRow < maxRow) {
						currentRow++;
					} else {
						consoleCanvas.scrollUpALine(textBGColor);
						startRow--;
					}
					consoleCanvas.drawText(currentRow, 1, text.substring(index,
							index + 1), textColor, textBGColor);
					currentCol = 2;
					actualCol = currentCol;
					actualRow = currentRow;
				}
			}
			start = index + 1;
		} // for
	} // print (String)

	/**
	 * Write a string to the Console with a specified field size..
	 * 
	 * @param text
	 *            The string to be written to the Console.
	 * @param fieldSize
	 *            The field width that the string is to be written in.
	 */
	public void print(String text, int fieldSize) {
		StringBuffer padding = new StringBuffer();

		for (int cnt = 0; cnt < fieldSize - text.length(); cnt++) {
			padding.append(' ');
		}
		print(text + padding);
	} // print (String, int)

	/**
	 * Write the text representation of an 16-bit integer (a "short") to the
	 * Console.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void print(short number) {
		print((int) number);
	} // print (short)

	/**
	 * Write the text representation of an 16-bit integer (a "short") to the
	 * Console with a specified field size.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void print(short number, int fieldSize) {
		print((int) number, fieldSize);
	} // print (short, int)

	/**
	 * Write the text representation of a boolean to the Console.
	 * 
	 * @param value
	 *            The boolean to be written to the Console.
	 */
	public void print(boolean value) {
		print(String.valueOf(value));
	} // print (boolean)

	/**
	 * Write the text representation of a boolean to the Console with a
	 * specified field size.
	 * 
	 * @param value
	 *            The boolean to be written to the Console.
	 * @param fieldSize
	 *            The field width that the boolean is to be written in.
	 */
	public void print(boolean value, int fieldSize) {
		String boolStr = String.valueOf(value);
		StringBuffer padding = new StringBuffer();

		for (int cnt = 0; cnt < fieldSize - boolStr.length(); cnt++) {
			padding.append(' ');
		}
		print(boolStr + padding);
	} // print (boolean, int)

	/**
	 * Write a newline to the Console.
	 */
	public void println() {
		print("\n");
	} // println (void)

	/**
	 * Write the text representation of an 8-bit integer (a "byte") to the
	 * Console followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void println(byte number) {
		print(number);
		print("\n");
	} // println (byte)

	/**
	 * Write the text representation of an 8-bit integer (a "byte") to the
	 * Console with a specified field size followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void println(byte number, int fieldSize) {
		print(number, fieldSize);
		print("\n");
	} // println (byte, int)

	/**
	 * Write a character to the Console followed by a newline.
	 * 
	 * @param ch
	 *            The character to be written to the Console.
	 */
	public void println(char ch) {
		print(ch);
		print("\n");
	} // println (char)

	/**
	 * Write a character to the Console with a specified field size..
	 * 
	 * @param ch
	 *            The character to be written to the Console.
	 * @param fieldSize
	 *            The field width that the character is to be written in.
	 */
	public void println(char ch, int fieldSize) {
		print(ch, fieldSize);
		print("\n");
	} // print (char, int)

	/**
	 * Write a double precision floating point number (a "double") to the
	 * Console followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void println(double number) {
		print(number);
		print("\n");
	} // println (double)

	/**
	 * Write a double precision floating point number (a "double") to the
	 * Console with a specified field size followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void println(double number, int fieldSize) {
		print(number, fieldSize);
		print("\n");
	} // println (double, int)

	/**
	 * Write a double precision floating point number (a "double") to the
	 * Console with a specified field size and a specified number of decimal
	 * places followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 * @param decimalPlaces
	 *            The number of decimal places of the number to be displayed.
	 */
	public void println(double number, int fieldSize, int decimalPlaces) {
		print(number, fieldSize, decimalPlaces);
		print("\n");
	} // println (double, int, int)

	/**
	 * Write a floating point number (a "float") to the Console followed by a
	 * newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void println(float number) {
		print(number);
		print("\n");
	} // println (float)

	/**
	 * Write a floating point number (a "float") to the Console with a specified
	 * field size followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void println(float number, int fieldSize) {
		print(number, fieldSize);
		print("\n");
	} // println (float, int)

	/**
	 * Write a floating point number (a "double") to the Console with a
	 * specified field size and a specified number of decimal places followed by
	 * a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 * @param decimalPlaces
	 *            The number of decimal places of the number to be displayed.
	 */
	public void println(float number, int fieldSize, int decimalPlaces) {
		print(number, fieldSize, decimalPlaces);
		print("\n");
	} // println (float, int, int)

	/**
	 * Write the text representation of an 32-bit integer (an "int") to the
	 * Console followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void println(int number) {
		print(number);
		print("\n");
	} // println (int)

	/**
	 * Write the text representation of an 32-bit integer (an "int") to the
	 * Console with a specified field size followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void println(int number, int fieldSize) {
		print(number, fieldSize);
		print("\n");
	} // println (int, int)

	/**
	 * Write the text representation of an 64-bit integer (a "long") to the
	 * Console followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void println(long number) {
		print(number);
		print("\n");
	} // println (long)

	/**
	 * Write the text representation of an 64-bit integer (a "long") to the
	 * Console with a specified field size followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void println(long number, int fieldSize) {
		print(number, fieldSize);
		print("\n");
	} // println (long, int)

	/**
	 * Write a string to the Console followed by a newline.
	 * 
	 * @param text
	 *            The string to be written to the Console.
	 */
	public void println(String text) {
		print(text);
		print("\n");
	} // print (String)

	/**
	 * Write a string to the Console with a specified field size followed by a
	 * newline.
	 * 
	 * @param text
	 *            The string to be written to the Console.
	 * @param fieldSize
	 *            The field width that the string is to be written in.
	 */
	public void println(String text, int fieldSize) {
		print(text, fieldSize);
		print("\n");
	} // println (String, int)

	/**
	 * Write the text representation of an 16-bit integer (a "short") to the
	 * Console followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 */
	public void println(short number) {
		print(number);
		print("\n");
	} // println (short)

	/**
	 * Write the text representation of an 16-bit integer (a "short") to the
	 * Console with a specified field size followed by a newline.
	 * 
	 * @param number
	 *            The number to be written to the Console.
	 * @param fieldSize
	 *            The field width that the number is to be written in.
	 */
	public void println(short number, int fieldSize) {
		print(number, fieldSize);
		print("\n");
	} // println (short, int)

	/**
	 * Write the text representation of a boolean to the Console followed by a
	 * newline.
	 * 
	 * @param value
	 *            The boolean to be written to the Console.
	 */
	public void println(boolean value) {
		print(value);
		print("\n");
	} // println (boolean)

	/**
	 * Write the text representation of a boolean to the Console with a
	 * specified field size followed by a newline.
	 * 
	 * @param value
	 *            The boolean to be written to the Console.
	 * @param fieldSize
	 *            The field width that the boolean is to be written in.
	 */
	public void println(boolean value, int fieldSize) {
		print(value, fieldSize);
		print("\n");
	} // println (boolean, int)

	/**
	 * Read a boolean from the Console. The actual text in the Console must be
	 * either "true" or "false" although case is irrelvant.
	 * 
	 * @return The boolean value read from the Console.
	 */
	public boolean readBoolean() {
		String s;

		s = readToken().toLowerCase();
		if (s.equals("true")) {
			return (true);
		} else if (s.equals("false")) {
			return (false);
		} else {
			new FatalError("Unable to convert \"" + s + "\" to a boolean", this);
			// Never reaches here
		}
		return (false);
	} // boolean readBoolean ()

	/**
	 * Read an 8-bit integer (a "byte") from the Console. The actual text in the
	 * Console must be a number from -128 to 127.
	 * 
	 * @return The byte value read from the Console.
	 */
	public byte readByte() {
		String s = readToken();

		try {
			return (Byte.parseByte(s));
		} catch (NumberFormatException e) {
			new FatalError("Unable to convert \"" + s + "\" to a byte", this);
			// Never reaches here
		}
		return (0);
	} // byte readByte (void)

	/**
	 * Read a single character from the Console. Note that this discards any
	 * whitespace. If you want to get every character on the line, use the
	 * readLine () method.
	 * 
	 * @return The character read from the Console.
	 */
	public synchronized char readChar() {
		char result, ch;

		if (ungotChar != EMPTY_BUFFER) {
			result = (char) ungotChar;
			ungotChar = EMPTY_BUFFER;
			return (result);
		}

		// If we're using standard I/O (for submission) read from
		// System.in instead of the keyboard.
		if (useStandardIO) {
			try {
				// Also, when reading from System.in, skip over '\r'
				do {
					int val = System.in.read();
					if (val != -1) {
						ch = (char) val;
					} else {
						if (eofReached) {
							throw new RuntimeException(
									"Attempt to read past end of input file");
						} else {
							eofReached = true;
							ch = '\n';
						}
					}
				} while (ch == '\r');
				return (ch);
			} catch (IOException e) {
				throw new RuntimeException("I/O Error while reading input file");
			}
		} // if (useStandardIO)

		if (lineBufferHead != lineBufferTail) {
			result = lineBuffer[lineBufferTail];
			lineBufferTail = (lineBufferTail + 1) % BUFFER_SIZE;
			return (result);
		}

		startRow = currentRow;
		startCol = currentCol;
		if (currentRow > maxRow) {
			startRow++;
			currentCol = 1;
		}

		// Turn cursor on if necessary
		consoleCanvas.setCursorVisible(true);

		// Wait for a character to be entered
		for (;;) {
			while (kbdBufferHead == kbdBufferTail) {
				try {
					wait();
				} catch (InterruptedException e) {
				}
			}

			ch = kbdBuffer[kbdBufferTail];
			kbdBufferTail = (kbdBufferTail + 1) % BUFFER_SIZE;

			if (ch == '\n') {
				if (echoOn)
					print("\n");

				lineBuffer[lineBufferHead] = '\n';
				lineBufferHead = (lineBufferHead + 1) % BUFFER_SIZE;
				break;
			}
			if (ch == '\b') {
				if (lineBufferHead == lineBufferTail) {
					consoleCanvas.invertScreen();
				} else {
					int chToErase;

					lineBufferHead = (lineBufferHead + BUFFER_SIZE - 1)
							% BUFFER_SIZE;
					chToErase = lineBuffer[lineBufferHead];
					if (echoOn) {
						if (chToErase != '\t') {
							erasePreviousChar();
						} else {
							int cnt;
							eraseLineOfInput();
							cnt = lineBufferTail;
							while (lineBufferHead != cnt) {
								print(lineBuffer[cnt]);
								cnt = (cnt + 1) % lineBufferHead;
							}
						}
					}
				}
			} /* if backspace */
			else if (ch == '\025') {
				if (echoOn) {
					eraseLineOfInput();
				}
				lineBufferHead = lineBufferTail;
			} else {
				if (echoOn) {
					print(ch);
				}
				lineBuffer[lineBufferHead] = ch;
				lineBufferHead = (lineBufferHead + 1) % BUFFER_SIZE;
			}
		} /* for */

		result = lineBuffer[lineBufferTail];
		lineBufferTail = (lineBufferTail + 1) % BUFFER_SIZE;

		// Turn cursor on if necessary
		consoleCanvas.setCursorVisible(false);

		return (result);
	} // char readChar (void)

	/**
	 * Read a double precision floating point number (a "double") from the
	 * Console.
	 * 
	 * @return The double value read from the Console.
	 */
	public double readDouble() {
		Double d;
		String s;

		s = readToken();
		try {
			d = Double.valueOf(s);
			return (d.doubleValue());
		} catch (NumberFormatException e) {
			new FatalError("Unable to convert \"" + s + "\" to a double", this);
			// Never reaches here
		}
		return (0.0);
	} // double readDouble (void)

	/**
	 * Read a floating point number (a "float") from the Console.
	 * 
	 * @return The float value read from the Console.
	 */
	public float readFloat() {
		Float f;
		String s;

		s = readToken();
		try {
			f = Float.valueOf(s);
			return (f.floatValue());
		} catch (NumberFormatException e) {
			new FatalError("Unable to convert \"" + s + "\" to a float", this);
			// Never reaches here
		}
		return ((float) 0.0);
	} // float readFloat (void)

	/**
	 * Read a 32-bit integer (an "int") from the Console. The actual text in the
	 * Console must be a number from -2147483648 to 2147483647.
	 * 
	 * @return The int value read from the Console.
	 */
	public int readInt() {
		String s = readToken();

		try {
			return (Integer.parseInt(s));
		} catch (NumberFormatException e) {
			new FatalError("Unable to convert \"" + s + "\" to a int", this);
			// Never reaches here
		}
		return (0);
	} // int readInt (void)

	/**
	 * Read a full line of text from the Console.
	 * 
	 * @return The line of text read from the Console.
	 */
	public String readLine() {
		char ch = readChar(); // The character being read in
		String s = ""; // The string typed in

		// Skip whitespace up tio the first newline
		// do
		// {
		// ch = readChar ();
		// }
		// while (ch == ' ');

		// if (ch == '\n')
		// {
		// ch = readChar ();
		// }

		while (ch != '\n') {
			s = s + ch;
			ch = readChar();
		}

		return (s);
	} // String readLine (void)

	/**
	 * Read a 64-bit integer (a "long") from the Console.
	 * 
	 * @return The long value read from the Console.
	 */
	public long readLong() {
		String s = readToken(); // The token read in

		try {
			return (Long.parseLong(s));
		} catch (NumberFormatException e) {
			new FatalError("Unable to convert \"" + s + "\" to a long", this);
			// Never reaches here
		}
		return (0);
	} // long readLong (void)

	/**
	 * Read a 16-bit integer (a "short") from the Console. The actual text in
	 * the Console must be a number from -32768 to 32767.
	 * 
	 * @return The short value read from the Console.
	 */
	public short readShort() {
		String s = readToken();

		try {
			return (Short.parseShort(s));
		} catch (NumberFormatException e) {
			new FatalError("Unable to convert \"" + s + "\" to a short", this);
			// Never reaches here
		}
		return (0);
	} // short readShort (void)

	/**
	 * Read a whitespace delimited token from the Console.
	 * 
	 * @return The token read from the Console.
	 */
	public String readString() {
		return (readToken());
	} // String readString (void)

	/**
	 * Reads in input from the keyboard buffer until it hits a whitespace, which
	 * indicates the end of a token.
	 */
	private String readToken() {
		char ch;

		StringBuffer sb = new StringBuffer();

		// Skip white space
		do {
			ch = readChar();
		} while ((ch == ' ') || (ch == '\n') || (ch == '\t'));

		if (ch == '"') {
			// Read until close quote
			ch = readChar();
			while (ch != '"') {
				sb.append(ch);
				;
				ch = readChar();
				if (ch == '\n') {
					new FatalError("No terminating quote for quoted string");
					// Never reaches here.
				}
			}

			// Read the character following the close quote.
			ch = readChar();
		} else {
			do {
				sb.append(ch);
				;
				ch = readChar();
			} while ((ch != ' ') && (ch != '\n') && (ch != '\t'));
		}

		// Lastly, skip any whitespace until the end of line
		while ((ch == ' ') || (ch == '\t')) {
			ch = readChar();
		}

		if (ch != '\n') {
			ungotChar = (int) ch;
		}

		return (new String(sb));
	} // String readToken (void)

	/**
	 * Sets the foreground colour for any graphics.
	 * 
	 * @see java.awt.Graphics.setColor
	 */
	public void setColor(Color color) {
		graphicsColor = color;
	} // setColor (Color)

	/**
	 * Sets the foreground colour for any graphics.
	 * 
	 * @param colour
	 *            The new foreground color for draw and fill methods.
	 */
	public void setColour(Color colour) {
		setColor(colour);
	} // setColour (Color)

	/**
	 * Moves the cursor to the specified row and column.
	 * 
	 * @param row
	 *            The row to move the cursor to.
	 * @param column
	 *            The column to move the cursor to.
	 */
	public void setCursor(int row, int column) {
		if (useStandardIO) {
			System.out.println("");
		} else {
			currentRow = row;
			currentCol = column;
			actualRow = row;
			actualCol = column;
			consoleCanvas.setCursorPos(currentRow, currentCol);
		}
	} // setCursor (int, int)

	/**
	 * Sets the font for the drawString method.
	 * 
	 * @see java.awt.Graphics.setFont
	 */
	public void setFont(Font font) {
		this.font = font;

		// The following line is for any system calls that happen to
		// use the setFont method on this Frame and expect it to work.
		super.setFont(font);
	} // setFont (Font)

	/**
	 * Sets the drawing mode for any graphics to "Paint".
	 * 
	 * @see java.awt.Graphics.setPaintMode
	 */
	public void setPaintMode() {
		consoleCanvas.setPaintMode();
	} // setPaintMode (void)

	/**
	 * Sets the background color for any text.
	 * 
	 * @param color
	 *            The color that text displayed by print and println methods
	 *            will appear on.
	 */
	public void setTextBackgroundColor(Color color) {
		textBGColor = color;
	} // setTextBackgroundColor (Color)

	/**
	 * Sets the background color for any text.
	 * 
	 * @param colour
	 *            The color that text displayed by print and println methods
	 *            will appear on.
	 */
	public void setTextBackgroundColour(Color colour) {
		setTextBackgroundColor(colour);
	} // setTextBackgroundColour (Color)

	/**
	 * Sets the foreground color for any text.
	 * 
	 * @param color
	 *            The color that text displayed by print and println methods
	 *            will appear in.
	 */
	public void setTextColor(Color color) {
		textColor = color;
	} // setTextColor (Color)

	/**
	 * Sets the foreground colour for any text.
	 * 
	 * @param colour
	 *            The color that text displayed by print and println methods
	 *            will appear in.
	 */
	public void setTextColour(Color colour) {
		setTextColor(colour);
	} // setTextColour (Color)

	/**
	 * Sets the drawing mode for any graphics to "XOR".
	 * 
	 * @see java.awt.Graphics.setXORMode
	 */
	public void setXORMode(Color xorColor) {
		consoleCanvas.setXORMode(xorColor);
	} // setXORMode (Color)

	/**
	 * Makes the blinking cursor visible.
	 */
	public void showCursor() {
		consoleCanvas.setCursorVisible(true);
	} // showCursor (void)
} /* Console class */
