%  
%   HT1DFB2.M   Heat Transfer 1-D Finite Bar  (HT Example 10.2 in Class Notes)
%  
%   Analytical Solution using Separation of Variables to the following problem: 
%     ut(x,t) = alf*uxx(x,t)   with  u(0,t) = ul   u(L,t) = ur   u(x,0) = u0 
%   has solution given by
%     u(x,t) = v(x,t) + w(x)     
%   where v(x,t) is the transient solution and w(x) is the steady state solution                          
%
%   File prepared by J. R. White, UMass-Lowell  (original Nov. 1996)
%     --> modified generation of movie (Aug. 1998)
%

%
%   getting started
      clear all,   close all
%  
%   problem data                                                         
      alf = 0.0001;    % m^2/s  thermal diffusivity                            
      ul = 100;        % C      fixed temp at left end                         
      ur = 400;        % C      fixed temp at right end                        
      u0 = 70;         % C      initial uniform temp of bar                    
      L = 1;           % m      length of bar                                 
      maxt = 50;       % max number of terms in expansion
      tol = 0.001;     % tolerance used to stop series expansion
      nfig = 0;        % figure counter                                                
%
%   the steady state solution w(x)
      Nx = 101;   x = linspace(0,L,Nx)';   w = (ur-ul)*x/L + ul;                                                   
%
%   now v(x,t) is given as an infinite series                                 
%   calc terms for n = 1,2,3,4,5,...,max                                   
      cc1 = 600*L/pi;   cc2 = 60/pi;  
      for n = 1:maxt
        lam(n) = n*pi/L;   c(n) = (1/n)*(cc1*(-1)^n + cc2*((-1)^n-1));
      end      
%
%   define time points and initialize space-time distribution
      opt = menu('Output plot option?','Create profiles at 5 time points', ...
                 'Make a movie with Nt frames');
      switch opt
        case 1     % use for plotting snapshots
          Nt = 5;   tt = [0 50 250 1000 1500];
        case 2     % use for making smooth movie
          Nt = input(' Input number of frames for movie (5-100): ');
          if Nt < 5,    Nt = 5;    end
          if Nt > 100,  Nt = 100;  end
          tt = linspace(0,5000,Nt); 
      end
%
%   start loop over time points
      u = zeros(Nx,Nt);
      for j = 1:Nt                                                     
        t = tt(j);   cc = -alf*t;   mrerr = 1.0;   n = 0;   v = zeros(size(x));
          while mrerr > tol   &   n < maxt
            n = n+1;  vn = c(n)*sin(lam(n)*x).*exp(cc*lam(n)*lam(n));                     
            v = v + vn;   rerr = vn(2:Nx-1)./v(2:Nx-1);   mrerr = max(abs(rerr));
          end
        u(:,j) = v + w;
        disp([' Needed ',num2str(n),' terms for convergence at t = ',num2str(t),' s'])
      end
%
%   plot curves of u for various times (if opt = 1)
      switch opt
        case 1
          nfig = nfig+1;   figure(nfig)                        
          plot(x,u)
          axis([0 1 0 400]);                        
          title('Bar Temperature for Several Times (Example 10.2)')                  
          grid,xlabel('Distance (m)'),ylabel('Temperature (C)')       
          for j = 1:Nt,   gtext(['t = ',num2str(tt(j)),' s']),   end
%
%   let's make a movie of the time-dependent profile (if opt = 2)
        case 2
          nfig = nfig+1;     figure(nfig) 
          plot(x,u(:,1));    axis([0 L 0 400]); 
          M = moviein(Nt); 
          for j = 1:Nt
            plot(x,u(:,j));  axis([0 L 0 400]);    M(:,j) = getframe;
          end
%
%   replicate the frames of the movie Nd times to slow things down a little
%          disp(' ')
%          Nd = input(' Input a delay time to slow down the movie (1-10): ');
%          if Nd < 1  Nd = 1;   end
%          if Nd > 10 Nd = 10;  end
%          index = ones(Nd,Nt);   
%          for i = 2:Nt,   index(:,i) = i*index(:,i);  end
%          M2 = M(:,index);
%
%   play the movie one more time 
          movie(M,0);   hold on;
          plot(x,u(:,1));    grid on;  
          title('Bar Temperature for Initial and Final Times (Example 10.2)');                  
          xlabel('Distance (m)'),ylabel('Temperature (C)');      
          gtext('initial'),gtext('final')
          hold off
      end
%
%   end of problem


