%gseid.m    Programa para resolver una version sencilla del problema del
%        planificador social usando el metodo de Gauss Seidel.
%        El problema a resolver es:
%
%                 inf
%             max sum beta^t ln(ct)
%                 t=0  
%             s.t     ct+it = A*k^alpha
%                      kt+1 = (1-delta)*kt+it        
% 
%        con k0 igual a 1/2 del stock de capital de estado estacionario.
%
%	      Output: archivo gsout.mat con trayectorias para principales variables.

clear

% Parametros del modelo

alpha   = 0.30;
beta    = 0.98;
delta   = 0.06;
A       = 10;

% Otros parametros requeridos (numero maximo de iteraciones, criterio
% de tolerancia, numero de periodos antes de llegar a estado estacionario

maxit   = 1500;
crit    = 1e-2;
T		  = 150;

% Calcula el stock de capital de estado estacionario (kss), el stock de
% capital inicial (k0) y el guess inicial para la secuencia kt (kg)

kss     = ((A*beta*alpha)/(1-(1-delta)*beta))^(1/(1-alpha));
k0		  = (1/2)*kss;
kg      = zeros(T+1,1);
ksol    = zeros(T,1);
kg(1)   = k0;
ksol(1) = k0;
for t=2:T+1
    kg(t) = kg(t-1)+(kss-k0)/T; 
end

% Iteraciones usando Gauss-Seidel

for i=1:maxit
    for t=2:T
        param = [alpha;beta;delta;A;ksol(t-1);kg(t+1)];
        ksol(t) = secant('ft15',kss/2,param,1e-2,20);
	 end
    dist = norm(kg(2:T)-ksol(2:T));
    plot(1:T,ksol(1:T),1:T,kss*ones(T,1))
    title(['Trayectoria del Capital en Iteracion ',int2str(i),...
         ' con Error ',num2str(dist)])
    pause(0.01)
    if dist<crit
       break
    end
    kg(1:T) = ksol(1:T);
end

% Simulacin de las trayectorias optimas para el capital (kt), consumo
% (ct), inversion (it) y producto (yt)

kt	= kg(1:T+1);

figure(2)
yt = A*kt(1:T).^alpha;
plot(1:T,yt)
title('Trayectoria Optima del Producto')

figure(3)
it = kt(2:T+1)-(1-delta)*kt(1:T);
plot(1:T,it)
title('Trayectoria Optima de la Inversin')

figure(4)
ct = yt-it;
plot(1:T,ct)
title('Trayectoria Optima del Consumo')

figure(5)
rt = alpha*A*kt(1:T).^(alpha-1);
plot(1:T,rt)
title('Trayectoria Optima de la Productividad Marginal del Capital')

subplot(2,2,1)
plot(1:T,ksol(1:T),1:T,kss*ones(T,1))
title('Capital')
subplot(2,2,2)
plot(1:T,yt)
title('Producto')
subplot(2,2,3)
plot(1:T,it)
title('Inversin')
subplot(2,2,4)
plot(1:T,ct)
title('Consumo')
%subplot(3,2,5)
%plot(1:T,rt)
%title('Pmg Capital')
%subplot(3,2,6)
%plot(1:T,wt)
%title('Pmg Trabajo')

%save c:/My Documents/rafaelb/lectures/peru/1999/metodos/gsout kt ct it yt
save C:\Mis documentos\rafaelb\lectures\cea\Macro2_MAgcea\2004\matlab\gsout kt ct it yt
