#include <linux/module.h>
#include <linux/kernel.h> /* printk() */
#include <linux/config.h>
#include <linux/init.h>

#include <linux/slab.h>    /* kmalloc(). Antes era: <linux/malloc.h> */
#include <linux/fs.h>     /* Necesario para escribir devices */
#include <asm/uaccess.h> /* copy-to-user y copy-from-user */

#include <asm/string.h> /* manejo de strings */
#include "nDev.h"        /* definiciones locales*/

/*
 *  Variables Globales. Pueden reescribirse al cargar el modulo.
 */

int nDev_major	=   NDEV_MAJOR;
int num_devs 	=   NUM_DEVS;    /* Numero de  devices nDev*/
int bufferSize =	BUFFERSIZE;

MODULE_PARM(nDev_major, "i");
MODULE_PARM(num_devs, "i");
MODULE_PARM(bufferSize, "i");

MODULE_AUTHOR("Mario Leyton");
MODULE_LICENSE("GPL");

Nano_Dev *nano_devices;

/*
 * Abrir y Cerrar el device: /dev/nDev
 */

int nDev_open (struct inode *inode, struct file *filp)
{
    int num = MINOR(inode->i_rdev);
    Nano_Dev *dev; /* Puntero a la estructura del dispositivo*/

	//printk(KERN_DEBUG "nDev: nDev_open\n");

    /* Vemos cual dispositvo se esta abriendo*/
    if (num >= num_devs) return -ENODEV;
    dev = &nano_devices[num];

    /* Guardamos un puntero al nDev en  filp->private_data 
	 * para ser recuperado en read/write */
    filp->private_data = dev;

    //MOD_INC_USE_COUNT;
    return 0;          /* exitoso */
}

int nDev_release (struct inode *inode, struct file *filp)
{
    //MOD_DEC_USE_COUNT;
    return 0;
}

/*
 * Leer y escribir del device: read and write
 */

ssize_t nDev_read (struct file *filp, char *buf, size_t count,
                loff_t *f_pos)
{
    Nano_Dev *dev = filp->private_data; /* recuperamos la estructura del nDev */
	
	//printk(KERN_DEBUG "nDev: nDev_read\n");

    if (down_interruptible (&dev->sem))
		return -ERESTARTSYS;

	/* vemos los casos patologicos */
  	if( *f_pos >= dev->buff_use){
    	up (&dev->sem);
    	return 0;
	}
	if( *f_pos+count > dev->buff_use)
		count = dev->buff_use - *f_pos;
	
	if (copy_to_user (buf, &dev->buffer[*f_pos], count)) {
		up(&dev->sem);
		return -EFAULT;
	}
	
	up (&dev->sem);

    *f_pos += count;
    return count;
}

ssize_t nDev_write (struct file *filp, const char *buf, size_t count,
                loff_t *f_pos)
{
    Nano_Dev *dev = filp->private_data;
	
	//printk(KERN_DEBUG "nDev: nDev_write\n");

    if (down_interruptible (&dev->sem))
            return -ERESTARTSYS;

	if(*f_pos==0)  /* Primera escritura */
		dev->buff_use=0;

	if(dev->buff_use > bufferSize || dev->buff_use+count > bufferSize){
    	up (&dev->sem);
	    return -ENOMEM;
	}
	
	if (copy_from_user (dev->buffer + *f_pos, buf, count)) {
		up (&dev->sem);
		return -EFAULT;
    }

    *f_pos += count;
	dev->buff_use += count;
	dev->buffer[dev->buff_use]='\0';
 
    up (&dev->sem);
    return count;

}

/*
 * El fops: Registramos los metodos
 */

struct file_operations nDev_fops= {
    read: nDev_read,
    write: nDev_write,
    open: nDev_open,
    release: nDev_release,
};

/*
 * CARGAR Y DESCARGAR EL MODULO 
 */
static int __init nDev_init(void)
{
    int result, i;

	//printk(KERN_DEBUG "nDev: nDev_init\n");
	
    //SET_MODULE_OWNER(&nDev_fops);

    /*
     * Registramos el numero Major y aceptamos un numero dinamico.
     */
    result = register_chrdev(nDev_major, "nDev", &nDev_fops);
    if (result < 0) return result;
    if (nDev_major == 0) nDev_major = result; /* dinamico */

    /* 
	 * Pedimos memoria para los devices con kmalloc
     */

	nano_devices=kmalloc(num_devs * sizeof(Nano_Dev), GFP_KERNEL);
    if (!nano_devices) {
    	unregister_chrdev(nDev_major, "nDev");
	    return -ENOMEM;
    }
    memset(nano_devices, 0, num_devs * sizeof (Nano_Dev));
	
    for(i=0; i < num_devs; i++){
		nano_devices[i].buff_use=0;
		sema_init (&nano_devices[i].sem, 1);
	}

    return (0); /* ok */

}

static void __exit nDev_cleanup(void)
{
	//printk(KERN_DEBUG "nDev: nDev_cleanup\n");

    unregister_chrdev(nDev_major, "nDev");

    kfree(nano_devices);
}

/*Registramos los metodos que realizan la carga y descarga del modulo */
module_init(nDev_init);
module_exit(nDev_cleanup);
