// Circuit.java (c) 2005 by Paul Falstad, www.falstad.com
// Traduccin: Juan M Fernandez

import java.io.InputStream;
import java.awt.*;
import java.awt.image.ImageProducer;
import java.applet.Applet;
import java.util.Vector;
import java.io.File;
import java.util.Random;
import java.util.Arrays;
import java.lang.Math;
import java.net.URL;
import java.awt.event.*;
import java.io.FilterInputStream;
import java.io.ByteArrayOutputStream;
import java.util.StringTokenizer;
import java.text.DecimalFormat;
import java.text.NumberFormat;
import java.lang.reflect.Constructor;

class CircuitCanvas extends Canvas {
    CircuitFrame pg;
    CircuitCanvas(CircuitFrame p) {
	pg = p;
    }
    public Dimension getPreferredSize() {
	return new Dimension(300,400);
    }
    public void update(Graphics g) {
	pg.updateCircuit(g);
    }
    public void paint(Graphics g) {
	pg.updateCircuit(g);
    }
};

class CircuitLayout implements LayoutManager {
    public CircuitLayout() {}
    public void addLayoutComponent(String name, Component c) {}
    public void removeLayoutComponent(Component c) {}
    public Dimension preferredLayoutSize(Container target) {
	return new Dimension(500, 500);
    }
    public Dimension minimumLayoutSize(Container target) {
	return new Dimension(100,100);
    }
    public void layoutContainer(Container target) {
	Insets insets = target.insets();
	int targetw = target.size().width - insets.left - insets.right;
	int cw = targetw* 8/10;
	int targeth = target.size().height - (insets.top+insets.bottom);
	target.getComponent(0).move(insets.left, insets.top);
	target.getComponent(0).resize(cw, targeth);
	int barwidth = targetw - cw;
	cw += insets.left;
	int i;
	int h = insets.top;
	for (i = 1; i < target.getComponentCount(); i++) {
	    Component m = target.getComponent(i);
	    if (m.isVisible()) {
		Dimension d = m.getPreferredSize();
		if (m instanceof Scrollbar)
		    d.width = barwidth;
		if (m instanceof Choice && d.width > barwidth)
		    d.width = barwidth;
		if (m instanceof Label) {
		    h += d.height/5;
		    d.width = barwidth;
		}
		m.move(cw, h);
		m.resize(d.width, d.height);
		h += d.height;
	    }
	}
    }
};

public class Circuit extends Applet implements ComponentListener {
    CircuitFrame ogf;
    void destroyFrame() {
	if (ogf != null)
	    ogf.dispose();
	ogf = null;
	repaint();
    }
    boolean started = false;
    public void init() {
	addComponentListener(this);
    }
    
    void showFrame() {
	if (ogf == null) {
	    started = true;
	    ogf = new CircuitFrame(this);
	    ogf.init();
	    repaint();
	}
    }

    public void toggleSwitch(int x) { ogf.toggleSwitch(x); }
    
    public void paint(Graphics g) {
	String s = "El Applet se abre en otra ventana.";
	if (!started)
	    s = "Applet iniciando.";
	else if (ogf == null)
	    s = "Applet terminado.";
	else if (ogf.useFrame)
	    ogf.show();
	g.drawString(s, 10, 30);
    }
    
    public void componentHidden(ComponentEvent e){}
    public void componentMoved(ComponentEvent e){}
    public void componentShown(ComponentEvent e) { showFrame(); }
    public void componentResized(ComponentEvent e) {
	if (ogf != null)
	    ogf.componentResized(e);
    }
    
    public void destroy() {
	if (ogf != null)
	    ogf.dispose();
	ogf = null;
	repaint();
    }
};

class CircuitFrame extends Frame
  implements ComponentListener, ActionListener, AdjustmentListener,
             MouseMotionListener, MouseListener, ItemListener {
    
    Thread engine = null;

    Dimension winSize;
    Image dbimage;
    
    Random random;
    public static final int sourceRadius = 7;
    public static final double freqMult = 3.14159265*2*4;
    
    public String getAppletInfo() {
	return "Circuit by Paul Falstad";
    }

    Container main;
    Label titleLabel;
    Button resetButton;
    //Button dumpMatrixButton;
    MenuItem exportItem, importItem, exitItem;
    Menu optionsMenu;
    Checkbox stoppedCheck;
    CheckboxMenuItem dotsCheckItem;
    CheckboxMenuItem voltsCheckItem;
    CheckboxMenuItem powerCheckItem;
    CheckboxMenuItem smallGridCheckItem;
    CheckboxMenuItem showValuesCheckItem;
    CheckboxMenuItem euroResistorCheckItem;
    Scrollbar speedBar;
    Scrollbar currentBar;
    Label powerLabel;
    Scrollbar powerBar;
    double currentMult, powerMult;
    PopupMenu elmMenu;
    MenuItem elmEditMenuItem;
    MenuItem elmDeleteMenuItem;
    MenuItem elmScopeMenuItem;
    PopupMenu scopeMenu;
    PopupMenu transScopeMenu;
    PopupMenu mainMenu;
    CheckboxMenuItem scopeVMenuItem;
    CheckboxMenuItem scopeIMenuItem;
    CheckboxMenuItem scopeMaxMenuItem;
    CheckboxMenuItem scopeFreqMenuItem;
    CheckboxMenuItem scopePowerMenuItem;
    CheckboxMenuItem scopeIbMenuItem;
    CheckboxMenuItem scopeIcMenuItem;
    CheckboxMenuItem scopeIeMenuItem;
    CheckboxMenuItem scopeVbeMenuItem;
    CheckboxMenuItem scopeVbcMenuItem;
    CheckboxMenuItem scopeVceMenuItem;
    Class addingClass;
    int mouseMode = -1;
    String mouseModeStr = "hola";
    Font unitsFont;
    static final double pi = 3.14159265358979323846;
    static final int MODE_ADD_ELM = 0;
    static final int MODE_DRAG_ALL = 1;
    static final int MODE_DRAG_ROW = 2;
    static final int MODE_DRAG_COLUMN = 3;
    static final int MODE_DRAG_SELECTED = 4;
    static final int MODE_DRAG_POST = 5;
    static final int infoWidth = 120;
    static final int SCOPEVAL_POWER = 1;
    static final int SCOPEVAL_IB = 1;
    static final int SCOPEVAL_IC = 2;
    static final int SCOPEVAL_IE = 3;
    static final int SCOPEVAL_VBE = 4;
    static final int SCOPEVAL_VBC = 5;
    static final int SCOPEVAL_VCE = 6;
    int dragX, dragY;
    int selectedSource;
    int gridSize, gridMask, gridRound;
    boolean dragging;
    boolean analyzeFlag;
    boolean dumpMatrix;
    double t;
    int pause = 10;
    int xpoints[], ypoints[];
    int colorScaleCount = 32;
    Color colorScale[];
    int scopeSelected = -1;
    int menuScope = -1;
    int hintType = -1, hintItem1, hintItem2;
    String stopMessage;
    double timeStep;
    
    static final int HINT_LC = 1;
    static final int HINT_RC = 2;
    static final int HINT_3DB_C = 3;
    static final int HINT_TWINT = 4;
    static final int HINT_3DB_L = 5;
    Vector elmList;
    Vector setupList;
    CircuitElm dragElm, menuElm, mouseElm, stopElm;
    int draggingPost;
    SwitchElm heldSwitchElm;
    double circuitMatrix[][], circuitRightSide[],
	origRightSide[], origMatrix[][];
    RowInfo circuitRowInfo[];
    int circuitPermute[];
    boolean circuitNonLinear;
    int voltageSourceCount;
    double voltageRange = 5;
    int circuitMatrixSize, circuitMatrixFullSize;
    boolean circuitNeedsMap;
    public boolean useFrame;
    int scopeCount;
    Scope scopes[];
    int scopeColCount[];
    EditDialog editDialog;
    ImportDialog impDialog;
    Class dumpTypes[];
    String muString = "u";
    String ohmString = "ohmio(s)";
    Rectangle circuitArea;

    int getrand(int x) {
	int q = random.nextInt();
	if (q < 0) q = -q;
	return q % x;
    }
    CircuitCanvas cv;
    Circuit applet;
    NumberFormat showFormat, shortFormat, noCommaFormat;

    CircuitFrame(Circuit a) {
	super("Simulador de Circuitos v1.1c");
	applet = a;
	useFrame = false;
    }

    public void init() {
	String startCircuit = null;
	String startLabel = null;
	String euroResistor = null;
	String useFrameStr = null;

	try {
	    String param = applet.getParameter("PAUSA");
	    if (param != null)
		pause = Integer.parseInt(param);
	    startCircuit = applet.getParameter("startCircuit");
	    startLabel   = applet.getParameter("startLabel");
	    euroResistor = applet.getParameter("euroResistors");
	    useFrameStr  = applet.getParameter("useFrame");
	} catch (Exception e) { }
	
	if (startLabel == null)
	    startLabel = "Circuito RLC";
	if (startCircuit == null)
	    startCircuit = "lrc.txt";
	boolean euro = (euroResistor != null && euroResistor.equalsIgnoreCase("true"));
	useFrame = (useFrameStr == null || !useFrameStr.equalsIgnoreCase("false"));
	if (useFrame)
	    main = this;
	else
	    main = applet;
	
	unitsFont = new Font("SansSerif", 0, 10);
	String os = System.getProperty("os.name");
	String jv = System.getProperty("java.class.version");
	double jvf = new Double(jv).doubleValue();
	if (jvf >= 48) {
	    muString = "\u03bc";
	    ohmString = "\u03a9";
	}
	
	dumpTypes = new Class[300];
	// these characters are reserved
	dumpTypes[(int)'o'] = Scope.class;
	dumpTypes[(int)'h'] = Scope.class;
	dumpTypes[(int)'$'] = Scope.class;

	main.setLayout(new CircuitLayout());
	cv = new CircuitCanvas(this);
	cv.addComponentListener(this);
	cv.addMouseMotionListener(this);
	cv.addMouseListener(this);
	main.add(cv);

	mainMenu = new PopupMenu();
	MenuBar mb = null;
	if (useFrame)
	    mb = new MenuBar();
	Menu m = new Menu("Archivo");
	if (useFrame)
	    mb.add(m);
	else
	    mainMenu.add(m);
	m.add(importItem = getMenuItem("Importar"));
	m.add(exportItem = getMenuItem("Exportar"));
	m.add(exitItem   = getMenuItem("Salir"));

	m = new Menu("Osciloscopio");
	if (useFrame)
	    mb.add(m);
	else
	    mainMenu.add(m);
	m.add(getMenuItem("Apilar todo", "stackAll"));
	m.add(getMenuItem("Desapilar todo", "unstackAll"));

	optionsMenu = m = new Menu("Opciones");
	if (useFrame)
	    mb.add(m);
	else
	    mainMenu.add(m);
	m.add(dotsCheckItem = getCheckItem("Ver Corriente"));
	dotsCheckItem.setState(true);
	m.add(voltsCheckItem = getCheckItem("Ver Voltaje"));
	voltsCheckItem.setState(true);
	m.add(powerCheckItem = getCheckItem("Ver Potencia"));
	m.add(showValuesCheckItem = getCheckItem("Ver Valores"));
	showValuesCheckItem.setState(true);
	m.add(smallGridCheckItem = getCheckItem("Red pequea"));
	m.add(euroResistorCheckItem = getCheckItem("Resistencias Europeas"));
	euroResistorCheckItem.setState(euro);
	
	Menu circuitsMenu = new Menu("Circuitos");
	if (useFrame)
	    mb.add(circuitsMenu);
	else
	    mainMenu.add(circuitsMenu);
	
	mainMenu.add(getClassCheckItem("Aadir Cable", "WireElm"));
	mainMenu.add(getClassCheckItem("Aadir Resistencia", "ResistorElm"));
	
	Menu passMenu = new Menu("Componentes Pasivos");
	mainMenu.add(passMenu);
	passMenu.add(getClassCheckItem("Aadir Condensador", "CapacitorElm"));
	passMenu.add(getClassCheckItem("Aadir Bobina", "InductorElm"));
	passMenu.add(getClassCheckItem("Aadir Interruptor", "SwitchElm"));
	passMenu.add(getClassCheckItem("Aadir Pulsador", "PushSwitchElm"));
	passMenu.add(getClassCheckItem("Aadir Interruptor Bscula", "Switch2Elm"));
	passMenu.add(getClassCheckItem("Aadir Transformador", "TransformerElm"));
	
	Menu inputMenu = new Menu("Intradas/Salidas");
	mainMenu.add(inputMenu);
	inputMenu.add(getClassCheckItem("Aadir Tierra", "GroundElm"));
	inputMenu.add(getClassCheckItem("Aadir fuente Tensin (2-terminales)", "DCVoltageElm"));
	inputMenu.add(getClassCheckItem("Aadir fuente A/C  (2-terminales)", "ACVoltageElm"));
	inputMenu.add(getClassCheckItem("Aadir fuente Tensin (1-terminal)", "RailElm"));
	inputMenu.add(getClassCheckItem("Aadir fuente   A/C   (1-terminal)", "ACRailElm"));
	inputMenu.add(getClassCheckItem("Aadir Onda Cuadrada  (1-terminal)", "SquareRailElm"));
	inputMenu.add(getClassCheckItem("Aadir Salida Analgica", "OutputElm"));
	inputMenu.add(getClassCheckItem("Aadir Entrada Lgica", "LogicInputElm"));
	inputMenu.add(getClassCheckItem("Aadir Salida Lgica", "LogicOutputElm"));
	inputMenu.add(getClassCheckItem("Aadir Reloj", "ClockElm"));
	inputMenu.add(getClassCheckItem("Aadir Antena", "AntennaElm"));
	inputMenu.add(getClassCheckItem("Aadir fuente de Corriente", "CurrentElm"));
	inputMenu.add(getClassCheckItem("Aadir LED", "LEDElm"));
	
	Menu activeMenu = new Menu("Componentes Activos");
	mainMenu.add(activeMenu);
	activeMenu.add(getClassCheckItem("Aadir  Diodo", "DiodeElm"));
	activeMenu.add(getClassCheckItem("Aadir  Transistor (bipolar, NPN)",
				    "NTransistorElm"));
	activeMenu.add(getClassCheckItem("Aadir  Transistor (bipolar, PNP)",
				    "PTransistorElm"));
	activeMenu.add(getClassCheckItem("Aadir  Op Amp (-  supra)", "OpAmpElm"));
	activeMenu.add(getClassCheckItem("Aadir  Op Amp (+  supra)",
				    "OpAmpSwapElm"));
	activeMenu.add(getClassCheckItem("Aadir  MOSFET (canal-n)",
				    "NMosfetElm"));
	activeMenu.add(getClassCheckItem("Aadir  MOSFET (canal-p)",
				    "PMosfetElm"));
	activeMenu.add(getClassCheckItem("Aadir  JFET (canal-n)",
					 "NJfetElm"));
	activeMenu.add(getClassCheckItem("Aadir  Interruptor Analgico (SPST)", "AnalogSwitchElm"));
	activeMenu.add(getClassCheckItem("Aadir  Interruptor Analgico (SPDT)", "AnalogSwitch2Elm"));

	Menu gateMenu = new Menu("Puertas Lgicas");
	mainMenu.add(gateMenu);
	gateMenu.add(getClassCheckItem("Aadir  Inversor", "InverterElm"));
	gateMenu.add(getClassCheckItem("Aadir  Puerta NAND ", "NandGateElm"));
	gateMenu.add(getClassCheckItem("Aadir  Puerta NOR", "NorGateElm"));
	gateMenu.add(getClassCheckItem("Aadir  Puerta AND", "AndGateElm"));
	gateMenu.add(getClassCheckItem("Aadir  Puerta OR", "OrGateElm"));
	gateMenu.add(getClassCheckItem("Aadir  Puerta XOR", "XorGateElm"));

	Menu chipMenu = new Menu("Chips");
	mainMenu.add(chipMenu);
	chipMenu.add(getClassCheckItem("Aadir  Flip-Flop  D", "DFlipFlopElm"));
	chipMenu.add(getClassCheckItem("Aadir  Flip-Flop JK", "JKFlipFlopElm"));
	chipMenu.add(getClassCheckItem("Aadir  LED 7 Segmentos", "SevenSegElm"));
	chipMenu.add(getClassCheckItem("Aadir  VCO", "VCOElm"));
	chipMenu.add(getClassCheckItem("Aadir  Comparator de Fase", "PhaseCompElm"));
	chipMenu.add(getClassCheckItem("Aadir  Contador", "CounterElm"));
	chipMenu.add(getClassCheckItem("Aadir  Contador Johnson", "JohnsonElm"));
	chipMenu.add(getClassCheckItem("Aadir  Temporizador 555", "TimerElm"));
	
	Menu otherMenu = new Menu("Otros");
	mainMenu.add(otherMenu);
	otherMenu.add(getClassCheckItem("Aadir  Texto", "TextElm"));
	otherMenu.add(getClassCheckItem("Aadir  Sonda Osciloscopio", "ProbeElm"));
	otherMenu.add(getCheckItem("Arrastrar Todo", "DragAll"));
	otherMenu.add(getCheckItem("Arrastrar Fila", "DragRow"));
	otherMenu.add(getCheckItem("Arrastrar Columna", "DragColumn"));
	otherMenu.add(getCheckItem("Arrastrar Seleccionados", "DragSelected"));
	otherMenu.add(getCheckItem("Arrastrar Mensaje", "DragPost"));
	
	main.add(mainMenu);

	main.add(resetButton = new Button("Reiniciar"));
	resetButton.addActionListener(this);
	/*add(dumpMatrixButton = new Button("Dump Matrix"));
	  dumpMatrixButton.addActionListener(this);*/
	stoppedCheck = new Checkbox("Parado");
	stoppedCheck.addItemListener(this);
	main.add(stoppedCheck);
	
	main.add(new Label("Velocidad de simulacin", Label.CENTER));
	
	main.add(speedBar = new Scrollbar(Scrollbar.HORIZONTAL, 3, 1, 1, 140));
	speedBar.addAdjustmentListener(this);

	main.add(new Label("Velocidad Corriente", Label.CENTER));
	main.add(currentBar = new Scrollbar(Scrollbar.HORIZONTAL,50, 1, 1, 100));
	currentBar.addAdjustmentListener(this);

	main.add(powerLabel = new Label("Brillo de Potencia", Label.CENTER));
	main.add(powerBar = new Scrollbar(Scrollbar.HORIZONTAL, 50, 1, 1, 100));
	powerBar.addAdjustmentListener(this);
	powerBar.disable();
	powerLabel.disable();

	main.add(new Label("Autor: Paul Falstad."));
	main.add(new Label("www.falstad.com"));
	main.add(new Label("Trad: Juan M Fernndez"));
	main.add(new Label("www.iesjorgemanrique.com"));
	main.add(new Label(""));
	Font f = new Font("SansSerif", 0, 10);
	Label l;
	main.add(l = new Label("Circuito actual:"));
	l.setFont(f);
	main.add(titleLabel = new Label("Label"));
	titleLabel.setFont(f);

	setGrid();
	elmList = new Vector();
	setupList = new Vector();
	colorScale = new Color[colorScaleCount];
	int i;
	for (i = 0; i != colorScaleCount; i++) {
	    double v = i*2./colorScaleCount - 1;
	    if (v < 0) {
		int n1 = (int) (128*-v)+127;
		int n2 = (int) (127*(1+v));
		colorScale[i] = new Color(n1, n2, n2);
	    } else {
		int n1 = (int) (128*v)+127;
		int n2 = (int) (127*(1-v));
		colorScale[i] = new Color(n2, n1, n2);
	    }
	}

	xpoints = new int[4];
	ypoints = new int[4];
	scopes = new Scope[20];
	scopeColCount = new int[20];
	scopeCount = 0;
	
	random = new Random();
	cv.setBackground(Color.black);
	cv.setForeground(Color.lightGray);
	showFormat = DecimalFormat.getInstance();
	showFormat.setMaximumFractionDigits(2);
	shortFormat = DecimalFormat.getInstance();
	shortFormat.setMaximumFractionDigits(1);
	noCommaFormat = DecimalFormat.getInstance();
	noCommaFormat.setMaximumFractionDigits(10);
	noCommaFormat.setGroupingUsed(false);
	
	elmMenu = new PopupMenu();
	elmMenu.add(elmEditMenuItem = getMenuItem("Editar"));
	elmMenu.add(elmDeleteMenuItem = getMenuItem("Borrar"));
	elmMenu.add(elmScopeMenuItem = getMenuItem("Ver en Osciloscopio"));
	main.add(elmMenu);
	
	scopeMenu = buildScopeMenu(false);
	transScopeMenu = buildScopeMenu(true);

	getSetupList(circuitsMenu);
	if (useFrame)
	    setMenuBar(mb);
	if (stopMessage == null)
	    readSetupFile(startCircuit, startLabel);

	if (useFrame) {
	    Dimension screen = getToolkit().getScreenSize();
	    resize(800, 640);
	    handleResize();
	    Dimension x = getSize();
	    setLocation((screen.width  - x.width)/2,
			(screen.height - x.height)/2);
	    show();
	} else {
	    hide();
	    handleResize();
	    applet.validate();
	}
	main.requestFocus();
    }

    PopupMenu buildScopeMenu(boolean t) {
	PopupMenu m = new PopupMenu();
	m.add(getMenuItem("Eliminar", "remove"));
	m.add(getMenuItem("Velocidad 2x", "speed2"));
	m.add(getMenuItem("Velocidad 1/2x", "speed1/2"));
	m.add(getMenuItem("Escala 2x", "scale"));
	m.add(getMenuItem("Apilar", "stack"));
	m.add(getMenuItem("No apilar", "unstack"));
	if (t) {
	    m.add(scopeIbMenuItem = getCheckItem("Ver Ib"));
	    m.add(scopeIcMenuItem = getCheckItem("Ver Ic"));
	    m.add(scopeIeMenuItem = getCheckItem("Ver Ie"));
	    m.add(scopeVbeMenuItem = getCheckItem("Ver Vbe"));
	    m.add(scopeVbcMenuItem = getCheckItem("Ver Vbc"));
	    m.add(scopeVceMenuItem = getCheckItem("Ver Vce"));
	} else {
	    m.add(scopeVMenuItem = getCheckItem("Ver Voltaje"));
	    m.add(scopeIMenuItem = getCheckItem("Ver Corriente"));
	    m.add(scopePowerMenuItem = getCheckItem("Ver Potencia Consumida"));
	    m.add(scopeMaxMenuItem = getCheckItem("Ver Valor de pico"));
	    m.add(scopeFreqMenuItem = getCheckItem("Ver Frecuencia"));
	}
	main.add(m);
	return m;
    }
    
    MenuItem getMenuItem(String s) {
	MenuItem mi = new MenuItem(s);
	mi.addActionListener(this);
	return mi;
    }

    MenuItem getMenuItem(String s, String ac) {
	MenuItem mi = new MenuItem(s);
	mi.setActionCommand(ac);
	mi.addActionListener(this);
	return mi;
    }

    CheckboxMenuItem getCheckItem(String s) {
	CheckboxMenuItem mi = new CheckboxMenuItem(s);
	mi.addItemListener(this);
	mi.setActionCommand("");
	return mi;
    }

    CheckboxMenuItem getClassCheckItem(String s, String t) {
	try {
	    Class c = Class.forName("CircuitFrame$" + t);
	    CircuitElm elm = constructElement(c, 0, 0);
	    register(c, elm);
	} catch (Exception ee) {
	    ee.printStackTrace();
	}
	return getCheckItem(s, t);
    }
    
    CheckboxMenuItem getCheckItem(String s, String t) {
	CheckboxMenuItem mi = new CheckboxMenuItem(s);
	mi.addItemListener(this);
	mi.setActionCommand(t);
	return mi;
    }

    void register(Class c, CircuitElm elm) {
	int t = elm.getDumpType();
	if (t == 0) {
	    System.out.println("no dump type: " + c);
	    return;
	}
	Class dclass = elm.getDumpClass();
	if (dumpTypes[t] == dclass)
	    return;
	if (dumpTypes[t] != null) {
	    System.out.println("dump type conflict: " + c + " " +
			       dumpTypes[t]);
	    return;
	}
	dumpTypes[t] = dclass;
    }
    
    void handleResize() {
        winSize = cv.getSize();
	if (winSize.width == 0)
	    return;
	dbimage = main.createImage(winSize.width, winSize.height);
	int h = winSize.height / 5;
	circuitArea = new Rectangle(0, 0, winSize.width, winSize.height-h);
	int i;
	int minx = 1000, maxx = 0, miny = 1000, maxy = 0;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    minx = min(ce.x, min(ce.x2, minx));
	    miny = min(ce.y, min(ce.y2, miny));
	    maxx = max(ce.x, max(ce.x2, maxx));
	    maxy = max(ce.y, max(ce.y2, maxy));
	}
	// center circuit; we don't use snapGrid() because that rounds
	int dx = gridMask & ((circuitArea.width -(maxx-minx))/2-minx);
	int dy = gridMask & ((circuitArea.height-(maxy-miny))/2-miny);
	if (dx+minx < 0)
	    dx = gridMask & (-minx);
	if (dy+miny < 0)
	    dy = gridMask & (-miny);
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    ce.move(dx, dy);
	}
    }

    public boolean handleEvent(Event ev) {
        if (ev.id == Event.WINDOW_DESTROY) {
            applet.destroyFrame();
            return true;
        }
        return super.handleEvent(ev);
    }
    
    void centerString(Graphics g, String s, int y) {
	FontMetrics fm = g.getFontMetrics();
        g.drawString(s, (winSize.width-fm.stringWidth(s))/2, y);
    }

    public void paint(Graphics g) {
	cv.repaint();
    }

    static final int resct = 6;
    long lastTime = 0, lastFrameTime, secTime = 0;
    int frames = 0;
    int steps = 0;
    int framerate = 0, steprate = 0;

    public void updateCircuit(Graphics realg) {
	if (winSize == null || winSize.width == 0)
	    return;
	if (analyzeFlag) {
	    analyzeCircuit();
	    analyzeFlag = false;
	}
	if (editDialog != null)
	    mouseElm = editDialog.elm;
	if (mouseElm == null)
	    mouseElm = stopElm;
	setupScopes();
        Graphics g = null;
	g = dbimage.getGraphics();
	g.setColor(cv.getBackground());
	g.fillRect(0, 0, winSize.width, winSize.height);
	if (!stoppedCheck.getState()) {
	    try {
		runCircuit();
	    } catch (Exception e) {
		e.printStackTrace();
		analyzeFlag = true;
		cv.repaint();
		return;
	    }
	}
	if (!stoppedCheck.getState()) {
	    long sysTime = System.currentTimeMillis();
	    if (lastTime != 0) {
		int inc = (int) (sysTime-lastTime);
		double c = currentBar.getValue();
		c = java.lang.Math.exp(c/3.5-14.2);
		currentMult = 1.7 * inc * c;
	    }
	    if (sysTime-secTime >= 1000) {
		framerate = frames; steprate = steps;
		frames = 0; steps = 0;
		secTime = sysTime;
	    }
	    lastTime = sysTime;
	} else
	    lastTime = 0;
	powerMult = Math.exp(powerBar.getValue()/4.762-7);
	
	int i;
	Font oldfont = g.getFont();
	for (i = 0; i != elmList.size(); i++) {
	    if (powerCheckItem.getState())
		g.setColor(Color.gray);
	    getElm(i).draw(g);
	}
	if (mouseMode == MODE_DRAG_ROW || mouseMode == MODE_DRAG_COLUMN)
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		ce.drawPost(g, ce.x , ce.y );
		ce.drawPost(g, ce.x2, ce.y2);
	    }
	/*if (mouseElm != null) {
	    g.setFont(oldfont);
	    g.drawString("+", mouseElm.x+10, mouseElm.y);
	    }*/
	if (dragElm != null &&
	      (dragElm.x != dragElm.x2 || dragElm.y != dragElm.y2))
	    dragElm.draw(g);
	g.setFont(oldfont);
	int ct = scopeCount;
	if (stopMessage != null)
	    ct = 0;
	for (i = 0; i != ct; i++)
	    scopes[i].draw(g);
	g.setColor(Color.white);
	int ybase = circuitArea.height;
	if (stopMessage != null) {
	    g.drawString(stopMessage, 10, ybase+15);
	} else {
	    String info[] = new String[10];
	    if (mouseElm != null) {
		mouseElm.getInfo(info);
		/*for (i = 0; i != mouseElm.getPostCount(); i++)
		  info[0] += " " + mouseElm.nodes[i];*/
	    } else {
		showFormat.setMinimumFractionDigits(2);
		info[0] = "t = " + getUnitText(t, "s");
		showFormat.setMinimumFractionDigits(0);
	    }
	    if (hintType != -1) {
		for (i = 0; info[i] != null; i++)
		    ;
		String s = getHint();
		if (s == null)
		    hintType = -1;
		else
		    info[i] = s;
	    }
	    int x = 0;
	    if (ct != 0)
		x = scopes[ct-1].rightEdge() + 20;
	    x = max(x, winSize.width*2/3);
	    for (i = 0; info[i] != null; i++)
		g.drawString(info[i], x,
			     ybase+15*(i+1));
	}
	if (mouseElm == stopElm)
	    mouseElm = null;
	frames++;
	/*g.setColor(Color.white);
	g.drawString("Framerate: " + framerate, 10, 10);
	g.drawString("Steprate: " + steprate,  10, 30);
	g.drawString("Steprate/iter: " + (steprate/getIterCount()),  10, 50);
	g.drawString("iterc: " + (getIterCount()),  10, 70);*/
	lastFrameTime = lastTime;

	realg.drawImage(dbimage, 0, 0, this);
	if (!stoppedCheck.getState() && circuitMatrix != null)
	    cv.repaint(pause);
    }

    void setupScopes() {
	int i;
	
	// check scopes to make sure the elements still exist, and remove
	// unused scopes/columns
	int pos = -1;
	for (i = 0; i < scopeCount; i++) {
	    if (locateElm(scopes[i].elm) < 0)
		scopes[i].setElm(null);
	    if (scopes[i].elm == null) {
		int j;
		for (j = i; j != scopeCount; j++)
		    scopes[j] = scopes[j+1];
		scopeCount--;
		i--;
		continue;
	    }
	    if (scopes[i].position > pos+1)
		scopes[i].position = pos+1;
	    pos = scopes[i].position;
	}
	while (scopeCount > 0 && scopes[scopeCount-1].elm == null)
	    scopeCount--;
	int h = winSize.height - circuitArea.height;
	pos = 0;
	for (i = 0; i != scopeCount; i++)
	    scopeColCount[i] = 0;
	for (i = 0; i != scopeCount; i++) {
	    pos = max(scopes[i].position, pos);
	    scopeColCount[scopes[i].position]++;
	}
	int colct = pos+1;
	int iw = infoWidth;
	if (colct <= 2)
	    iw = iw*3/2;
	int w = (winSize.width-iw) / colct;
	int marg = 10;
	if (w < marg*2)
	    w = marg*2;
	pos = -1;
	int colh = 0;
	int row = 0;
	int speed = 0;
	for (i = 0; i != scopeCount; i++) {
	    Scope s = scopes[i];
	    if (s.position > pos) {
		pos = s.position;
		colh = h / scopeColCount[pos];
		row = 0;
		speed = s.speed;
	    }
	    if (s.speed != speed) {
		s.speed = speed;
		s.resetGraph();
	    }
	    Rectangle r = new Rectangle(pos*w, winSize.height-h+colh*row,
					w-marg, colh);
	    row++;
	    if (!r.equals(s.rect))
		s.setRect(r);
	}
    }
    
    String getHint() {
	CircuitElm c1 = getElm(hintItem1);
	CircuitElm c2 = getElm(hintItem2);
	if (c1 == null || c2 == null)
	    return null;
	if (hintType == HINT_LC) {
	    if (!(c1 instanceof InductorElm))
		return null;
	    if (!(c2 instanceof CapacitorElm))
		return null;
	    InductorElm ie = (InductorElm) c1;
	    CapacitorElm ce = (CapacitorElm) c2;
	    return "res.f = " + getUnitText(1/(2*pi*Math.sqrt(ie.inductance*
						    ce.capacitance)), "Hz");
	}
	if (hintType == HINT_RC) {
	    if (!(c1 instanceof ResistorElm))
		return null;
	    if (!(c2 instanceof CapacitorElm))
		return null;
	    ResistorElm re = (ResistorElm) c1;
	    CapacitorElm ce = (CapacitorElm) c2;
	    return "RC = " + getUnitText(re.resistance*ce.capacitance,
					 "s");
	}
	if (hintType == HINT_3DB_C) {
	    if (!(c1 instanceof ResistorElm))
		return null;
	    if (!(c2 instanceof CapacitorElm))
		return null;
	    ResistorElm re = (ResistorElm) c1;
	    CapacitorElm ce = (CapacitorElm) c2;
	    return "f.3db = " +
		getUnitText(1/(2*pi*re.resistance*ce.capacitance), "Hz");
	}
	if (hintType == HINT_3DB_L) {
	    if (!(c1 instanceof ResistorElm))
		return null;
	    if (!(c2 instanceof InductorElm))
		return null;
	    ResistorElm re = (ResistorElm) c1;
	    InductorElm ie = (InductorElm) c2;
	    return "f.3db = " +
		getUnitText(re.resistance/(2*pi*ie.inductance), "Hz");
	}
	if (hintType == HINT_TWINT) {
	    if (!(c1 instanceof ResistorElm))
		return null;
	    if (!(c2 instanceof CapacitorElm))
		return null;
	    ResistorElm re = (ResistorElm) c1;
	    CapacitorElm ce = (CapacitorElm) c2;
	    return "fc = " +
		getUnitText(1/(2*pi*re.resistance*ce.capacitance), "Hz");
	}
	return null;
    }

    public void toggleSwitch(int n) {
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce instanceof SwitchElm) {
		n--;
		if (n == 0) {
		    ((SwitchElm) ce).toggle();
		    analyzeFlag = true;
		    cv.repaint();
		    return;
		}
	    }
	}
    }
    
    void needAnalyze() {
	analyzeFlag = true;
	cv.repaint();
    }
    
    void drawDots(Graphics g, int x1, int y1, int x2, int y2,
		  double pos) {
	if (stoppedCheck.getState() || pos == 0 || !dotsCheckItem.getState())
	    return;
	int dx = x2-x1;
	int dy = y2-y1;
	double dn = Math.sqrt(dx*dx+dy*dy);
	g.setColor(Color.yellow);
	int ds = 16;
	pos %= ds;
	if (pos < 0)
	    pos += ds;
	double di = 0;
	for (di = pos; di < dn; di += ds) {
	    int x0 = (int) (x1+di*dx/dn);
	    int y0 = (int) (y1+di*dy/dn);
	    g.fillRect(x0-1, y0-1, 4, 4);
	}
    }

    class CircuitNode {
	int x, y;
	Vector links;
	boolean internal;
	CircuitNode() { links = new Vector(); }
    }
    class CircuitNodeLink {
	int num;
	CircuitElm elm;
    }
    // info about each row/column of the matrix for simplification purposes
    class RowInfo {
	static final int ROW_NORMAL = 0;  // ordinary value
	static final int ROW_CONST  = 1;  // value is constant
	static final int ROW_EQUAL  = 2;  // value is equal to another value
	int nodeEq, type, mapCol, mapRow;
	double value;
	boolean rsChanges; // row's right side changes
	boolean lsChanges; // row's left side changes
	boolean dropRow;   // row is not needed in matrix
	RowInfo() { type = ROW_NORMAL; }
    }

    Vector nodeList;
    CircuitElm voltageSources[];

    CircuitNode getCircuitNode(int n) {
	if (n >= nodeList.size())
	    return null;
	return (CircuitNode) nodeList.elementAt(n);
    }

    CircuitElm getElm(int n) {
	if (n >= elmList.size())
	    return null;
	return (CircuitElm) elmList.elementAt(n);
    }
    
    void analyzeCircuit() {
	if (elmList.isEmpty())
	    return;
	stopMessage = null;
	stopElm = null;
	int i, j;
	int vscount = 0;
	nodeList = new Vector();
	boolean gotGround = false;
	CircuitElm volt = null;
	
	// look for voltage or ground element
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce instanceof GroundElm) {
		gotGround = true;
		break;
	    }
	    if (volt == null && ce instanceof VoltageElm &&
		!(ce instanceof RailElm))
		volt = ce;
	}

	// if no ground, then the voltage elm's first terminal is ground
	if (!gotGround && volt != null) {
	    CircuitNode cn = new CircuitNode();
	    Point pt = volt.getPost(0);
	    cn.x = pt.x;
	    cn.y = pt.y;
	    nodeList.addElement(cn);
	} else {
	    // otherwise allocate extra node for ground
	    CircuitNode cn = new CircuitNode();
	    cn.x = cn.y = -1;
	    nodeList.addElement(cn);
	}

	// allocate nodes and voltage sources
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    int inodes = ce.getInternalNodeCount();
	    int ivs = ce.getVoltageSourceCount();
	    int posts = ce.getPostCount();
	    
	    // allocate a node for each post and match posts to nodes
	    for (j = 0; j != posts; j++) {
		Point pt = ce.getPost(j);
		int k;
		for (k = 0; k != nodeList.size(); k++) {
		    CircuitNode cn = getCircuitNode(k);
		    if (pt.x == cn.x && pt.y == cn.y)
			break;
		}
		if (k == nodeList.size()) {
		    CircuitNode cn = new CircuitNode();
		    cn.x = pt.x;
		    cn.y = pt.y;
		    CircuitNodeLink cnl = new CircuitNodeLink();
		    cnl.num = j;
		    cnl.elm = ce;
		    cn.links.addElement(cnl);
		    ce.setNode(j, nodeList.size());
		    nodeList.addElement(cn);
		} else {
		    CircuitNodeLink cnl = new CircuitNodeLink();
		    cnl.num = j;
		    cnl.elm = ce;
		    getCircuitNode(k).links.addElement(cnl);
		    ce.setNode(j, k);
		}
	    }
	    for (j = 0; j != inodes; j++) {
		CircuitNode cn = new CircuitNode();
		cn.x = cn.y = -1;
		cn.internal = true;
		CircuitNodeLink cnl = new CircuitNodeLink();
		cnl.num = j+posts;
		cnl.elm = ce;
		cn.links.addElement(cnl);
		ce.setNode(cnl.num, nodeList.size());
		nodeList.addElement(cn);
	    }
	    vscount += ivs;
	}
	voltageSources = new CircuitElm[vscount];
	vscount = 0;
	circuitNonLinear = false;

	// determine if circuit is nonlinear
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce.nonLinear())
		circuitNonLinear = true;
	    int ivs = ce.getVoltageSourceCount();
	    for (j = 0; j != ivs; j++) {
		voltageSources[vscount] = ce;
		ce.setVoltageSource(j, vscount++);
	    }
	}
	voltageSourceCount = vscount;

	int matrixSize = nodeList.size()-1 + vscount;
	circuitMatrix = new double[matrixSize][matrixSize];
	circuitRightSide = new double[matrixSize];
	origMatrix = new double[matrixSize][matrixSize];
	origRightSide = new double[matrixSize];
	circuitMatrixSize = circuitMatrixFullSize = matrixSize;
	circuitRowInfo = new RowInfo[matrixSize];
	circuitPermute = new int[matrixSize];
	int vs = 0;
	for (i = 0; i != matrixSize; i++)
	    circuitRowInfo[i] = new RowInfo();
	circuitNeedsMap = false;
	
	// stamp linear circuit elements
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    ce.stamp();
	}

	// determine nodes that are unconnected
	boolean closure[] = new boolean[nodeList.size()];
	boolean tempclosure[] = new boolean[nodeList.size()];
	boolean changed = true;
	closure[0] = true;
	while (changed) {
	    changed = false;
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		// loop through all ce's nodes to see if they are connected
		// to other nodes not in closure
		for (j = 0; j < ce.getPostCount(); j++) {
		    if (!closure[ce.getNode(j)]) {
			if (ce.hasGroundConnection(j))
			    closure[ce.getNode(j)] = changed = true;
			continue;
		    }
		    int k;
		    for (k = 0; k != ce.getPostCount(); k++) {
			if (j == k)
			    continue;
			int kn = ce.getNode(k);
			if (ce.getConnection(j, k) && !closure[kn]) {
			    closure[kn] = true;
			    changed = true;
			}
		    }
		}
	    }
	    if (changed)
		continue;

	    // connect unconnected nodes
	    for (i = 0; i != nodeList.size(); i++)
		if (!closure[i] && !getCircuitNode(i).internal) {
		    System.out.println("nodo " + i + " desconectado");
		    stampResistor(0, i, 1e8);
		    closure[i] = true;
		    changed = true;
		    break;
		}
	}

	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    // look for inductors with no current path
	    if (ce instanceof InductorElm) {
		FindPathInfo fpi = new FindPathInfo(FPI_INDUCT, ce,
						    ce.getNode(1));
		if (!fpi.findPath(ce.getNode(0))) {
		    System.out.println(ce + " no path");
		    ce.reset();
		}
	    }
	    // look for current sources with no current path
	    if (ce instanceof CurrentElm) {
		FindPathInfo fpi = new FindPathInfo(FPI_INDUCT, ce,
						    ce.getNode(1));
		if (!fpi.findPath(ce.getNode(0))) {
		    stop("No hay conexin con la fuente de corriente!", ce);
		    return;
		}
	    }
	    // look for voltage source loops
	    if (ce instanceof VoltageElm && ce.getPostCount() == 2) {
		FindPathInfo fpi = new FindPathInfo(FPI_VOLTAGE, ce,
						    ce.getNode(1));
		if (fpi.findPath(ce.getNode(0))) {
		    stop("Malla con fuente de tensin sin Resistencia!", ce);
		    return;
		}
	    }
	    // look for shorted caps, or caps w/ voltage but no R
	    if (ce instanceof CapacitorElm) {
		FindPathInfo fpi = new FindPathInfo(FPI_SHORT, ce,
						    ce.getNode(1));
		if (fpi.findPath(ce.getNode(0))) {
		    System.out.println(ce + " cortado");
		    ce.reset();
		} else {
		    fpi = new FindPathInfo(FPI_CAP_V, ce, ce.getNode(1));
		    if (fpi.findPath(ce.getNode(0))) {
			stop("Malla con Condensador sin resistencia!", ce);
			return;
		    }
		}
	    }
	}

	// simplify the matrix; this speeds things up quite a bit
	for (i = 0; i != matrixSize; i++) {
	    int qm = -1, qp = -1;
	    double qv = 0;
	    RowInfo re = circuitRowInfo[i];
	    /*System.out.println("row " + i + " " +
	      re.lsChanges + " " + re.rsChanges);*/
	    if (re.lsChanges || re.dropRow || re.rsChanges)
		continue;
	    double rsadd = 0;
	    
	    // look for rows that can be removed
	    for (j = 0; j != matrixSize; j++) {
		double q = circuitMatrix[i][j];
		if (circuitRowInfo[j].type == RowInfo.ROW_CONST) {
		    // keep a running total of const values that have been
		    // removed already
		    rsadd -= circuitRowInfo[j].value*q;
		    continue;
		}
		if (q == 0)
		    continue;
		if (qp == -1) {
		    qp = j;
		    qv = q;
		    continue;
		}
		if (qm == -1 && q == -qv) {
		    qm = j;
		    continue;
		}
		break;
	    }
	    //System.out.println("line " + i + " " + qp + " " + qm);
	    /*if (qp != -1 && circuitRowInfo[qp].lsChanges) {
		System.out.println("lschanges");
		continue;
	    }
	    if (qm != -1 && circuitRowInfo[qm].lsChanges) {
		System.out.println("lschanges");
		continue;
		}*/
	    if (j == matrixSize) {
		RowInfo elt = circuitRowInfo[qp];
		if (qm == -1) {
		    // we found a row with only one nonzero entry; that value
		    // is a constant
		    while (elt.type == RowInfo.ROW_EQUAL) {
			// follow the chain
			/*System.out.println("following equal chain from " +
			  qp + " to " + elt.nodeEq);*/
			qp = elt.nodeEq;
			elt = circuitRowInfo[qp];
		    }
		    if (elt.type != RowInfo.ROW_NORMAL) {
			System.out.println("el tipo ya existe " + elt.type + " para " + qp + "!");
			continue;
		    }
		    elt.type = RowInfo.ROW_CONST;
		    elt.value = (circuitRightSide[i]+rsadd)/qv;
		    circuitRowInfo[i].dropRow = true;
		    //System.out.println(qp + " * " + qv + " = const " + elt.value);
		    i = -1; // start over from scratch
		} else if (circuitRightSide[i]+rsadd == 0) {
		    // we found a row with only two nonzero entries, and one
		    // is the negative of the other; the values are equal
		    if (elt.type != RowInfo.ROW_NORMAL) {
			//System.out.println("swapping");
			int qq = qm;
			qm = qp; qp = qq;
			elt = circuitRowInfo[qp];
			if (elt.type != RowInfo.ROW_NORMAL) {
			    // we should follow the chain here, but this
			    // hardly ever happens so it's not worth worrying
			    // about
			    System.out.println("intercambio fallido");
			    continue;
			}
		    }
		    elt.type = RowInfo.ROW_EQUAL;
		    elt.nodeEq = qm;
		    circuitRowInfo[i].dropRow = true;
		    //System.out.println(qp + " = " + qm);
		}
	    }
	}

	// find size of new matrix
	int nn = 0;
	for (i = 0; i != matrixSize; i++) {
	    RowInfo elt = circuitRowInfo[i];
	    if (elt.type == RowInfo.ROW_NORMAL) {
		elt.mapCol = nn++;
		//System.out.println("col " + i + " maps to " + elt.mapCol);
		continue;
	    }
	    if (elt.type == RowInfo.ROW_EQUAL) {
		RowInfo e2 = null;
		// resolve chains of equality
		while (true) {
		    e2 = circuitRowInfo[elt.nodeEq];
		    if (e2.type != RowInfo.ROW_EQUAL)
			break;
		    elt.nodeEq = e2.nodeEq;
		    //System.out.println("following link");
		}
	    }
	    if (elt.type == RowInfo.ROW_CONST)
		elt.mapCol = -1;
	}
	for (i = 0; i != matrixSize; i++) {
	    RowInfo elt = circuitRowInfo[i];
	    if (elt.type == RowInfo.ROW_EQUAL) {
		RowInfo e2 = circuitRowInfo[elt.nodeEq];
		if (e2.type == RowInfo.ROW_CONST) {
		    // if something is equal to a const, it's a const
		    elt.type = e2.type;
		    elt.value = e2.value;
		    elt.mapCol = -1;
		    //System.out.println(i + " = [late]const " + elt.value);
		} else {
		    elt.mapCol = e2.mapCol;
		    //System.out.println(i + " maps to: " + e2.mapCol);
		}
	    }
	}

	/*System.out.println("matrixSize = " + matrixSize);
	for (j = 0; j != circuitMatrixSize; j++) {
	    for (i = 0; i != circuitMatrixSize; i++)
		System.out.print(circuitMatrix[j][i] + " ");
	    System.out.print("  " + circuitRightSide[j] + "\n");
	}
	System.out.print("\n");*/

	// make the new, simplified matrix
	int newsize = nn;
	double newmatx[][] = new double[newsize][newsize];
	double newrs  []   = new double[newsize];
	int ii = 0;
	for (i = 0; i != matrixSize; i++) {
	    RowInfo rri = circuitRowInfo[i];
	    if (rri.dropRow) {
		rri.mapRow = -1;
		continue;
	    }
	    newrs[ii] = circuitRightSide[i];
	    rri.mapRow = ii;
	    for (j = 0; j != matrixSize; j++) {
		RowInfo ri = circuitRowInfo[j];
		if (ri.type == RowInfo.ROW_CONST)
		    newrs[ii] -= ri.value*circuitMatrix[i][j];
		else
		    newmatx[ii][ri.mapCol] += circuitMatrix[i][j];
	    }
	    ii++;
	}

	circuitMatrix = newmatx;
	circuitRightSide = newrs;
	matrixSize = circuitMatrixSize = newsize;
	for (i = 0; i != matrixSize; i++)
	    origRightSide[i] = circuitRightSide[i];
	for (i = 0; i != matrixSize; i++)
	    for (j = 0; j != matrixSize; j++)
		origMatrix[i][j] = circuitMatrix[i][j];
	circuitNeedsMap = true;

	/*System.out.println("matrixSize = " + matrixSize + " " + circuitNonLinear);
	for (j = 0; j != circuitMatrixSize; j++) {
	    for (i = 0; i != circuitMatrixSize; i++)
		System.out.print(circuitMatrix[j][i] + " ");
	    System.out.print("  " + circuitRightSide[j] + "\n");
	}
	System.out.print("\n");*/

	// if a matrix is linear, we can do the LUdecomp here instead of
	// needing to do it every frame
	if (!circuitNonLinear) {
	    if (!LUdecomp(circuitMatrix, circuitMatrixSize, circuitPermute)) {
		stop("Matriz Singular!", null);
		return;
	    }
	}
    }

    void stop(String s, CircuitElm ce) {
	stopMessage = s;
	circuitMatrix = null;
	stopElm = ce;
	stoppedCheck.setState(true);
	analyzeFlag = false;
	cv.repaint();
    }
    
    static final int FPI_INDUCT  = 1;
    static final int FPI_VOLTAGE = 2;
    static final int FPI_SHORT   = 3;
    static final int FPI_CAP_V   = 4;
    class FindPathInfo {
	boolean used[];
	int dest;
	CircuitElm firstElm;
	int type;
	FindPathInfo(int t, CircuitElm e, int d) {
	    dest = d;
	    type = t;
	    firstElm = e;
	    used = new boolean[nodeList.size()];
	}
	boolean findPath(int n1) {
	    if (n1 == dest)
		return true;
	    if (used[n1]) {
		//System.out.println("used " + n1);
		return false;
	    }
	    used[n1] = true;
	    int i;
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		if (ce == firstElm)
		    continue;
		if (type == FPI_INDUCT) {
		    if (ce instanceof CurrentElm)
			continue;
		}
		if (type == FPI_VOLTAGE) {
		    if (!(ce.isWire() || ce instanceof VoltageElm))
			continue;
		}
		if (type == FPI_SHORT && !ce.isWire())
		    continue;
		if (type == FPI_CAP_V) {
		    if (!(ce.isWire() || ce instanceof CapacitorElm ||
			  ce instanceof VoltageElm))
			continue;
		}
		if (n1 == 0) {
		    // look for posts which have a ground connection;
		    // our path can go through ground
		    int j;
		    for (j = 0; j != ce.getPostCount(); j++)
			if (ce.hasGroundConnection(j) &&
			      findPath(ce.getNode(j))) {
			    used[n1] = false;
			    return true;
			}
		}
		int j;
		for (j = 0; j != ce.getPostCount(); j++) {
		    //System.out.println(ce + " " + ce.getNode(j));
		    if (ce.getNode(j) == n1)
			break;
		}
		if (j == ce.getPostCount())
		    continue;
		if (ce.hasGroundConnection(j) && findPath(0)) {
		    //System.out.println(ce + " has ground");
		    used[n1] = false;
		    return true;
		}
		if (type == FPI_INDUCT && ce instanceof InductorElm) {
		    double c = ce.getCurrent();
		    if (j == 0)
			c = -c;
		    //System.out.println("matching " + c + " to " + firstElm.getCurrent());
		    //System.out.println(ce + " " + firstElm);
		    if (Math.abs(c-firstElm.getCurrent()) > 1e-10)
			continue;
		}
		int k;
		for (k = 0; k != ce.getPostCount(); k++) {
		    if (j == k)
			continue;
		    //System.out.println(ce + " " + ce.getNode(j) + "-" + ce.getNode(k));
		    if (ce.getConnection(j, k) && findPath(ce.getNode(k))) {
			//System.out.println("got findpath " + n1);
			used[n1] = false;
			return true;
		    }
		    //System.out.println("back on findpath " + n1);
		}
	    }
	    used[n1] = false;
	    //System.out.println(n1 + " failed");
	    return false;
	}
    }

    // stamp independent voltage source #vs, from n1 to n2, amount v
    void stampVoltageSource(int n1, int n2, int vs, double v) {
	int vn = nodeList.size()+vs;
	stampMatrix(vn, n1, -1);
	stampMatrix(vn, n2, 1);
	stampRightSide(vn, v);
	stampMatrix(n1, vn, 1);
	stampMatrix(n2, vn, -1);
    }

    // use this if the amount of voltage is going to be updated in doStep()
    void stampVoltageSource(int n1, int n2, int vs) {
	int vn = nodeList.size()+vs;
	stampMatrix(vn, n1, -1);
	stampMatrix(vn, n2, 1);
	stampRightSide(vn);
	stampMatrix(n1, vn, 1);
	stampMatrix(n2, vn, -1);
    }
    
    void updateVoltageSource(int n1, int n2, int vs, double v) {
	int vn = nodeList.size()+vs;
	stampRightSide(vn, v);
    }
    
    void stampResistor(int n1, int n2, double r) {
	double r0 = 1/r;
	if (Double.isNaN(r0) || Double.isInfinite(r0)) {
	    System.out.print("resistencia errnea " + r + " " + r0 + "\n");
	    int a = 0;
	    a /= a;
	}
	stampMatrix(n1, n1, r0);
	stampMatrix(n2, n2, r0);
	stampMatrix(n1, n2, -r0);
	stampMatrix(n2, n1, -r0);
    }

    void stampConductance(int n1, int n2, double r0) {
	stampMatrix(n1, n1, r0);
	stampMatrix(n2, n2, r0);
	stampMatrix(n1, n2, -r0);
	stampMatrix(n2, n1, -r0);
    }

    // current from cn1 to cn2 is equal to voltage from vn1 to 2, divided by g
    void stampVCCurrentSource(int cn1, int cn2, int vn1, int vn2, double g) {
	stampMatrix(cn1, vn1, g);
	stampMatrix(cn2, vn2, g);
	stampMatrix(cn1, vn2, -g);
	stampMatrix(cn2, vn1, -g);
    }

    void stampCurrentSource(int n1, int n2, double i) {
	stampRightSide(n1, -i);
	stampRightSide(n2, i);
    }

    // stamp value x in row i, column j, meaning that a voltage change
    // of dv in node j will increase the current into node i by x dv.
    // (Unless i or j is a voltage source node.)
    void stampMatrix(int i, int j, double x) {
	if (i > 0 && j > 0) {
	    if (circuitNeedsMap) {
		i = circuitRowInfo[i-1].mapRow;
		RowInfo ri = circuitRowInfo[j-1];
		if (ri.type == RowInfo.ROW_CONST) {
		    //System.out.println("Stamping constant " + i + " " + j + " " + x);
		    circuitRightSide[i] -= x*ri.value;
		    return;
		}
		j = ri.mapCol;
		//System.out.println("stamping " + i + " " + j + " " + x);
	    } else {
		i--;
		j--;
	    }
	    circuitMatrix[i][j] += x;
	}
    }

    // stamp value x on the right side of row i, representing an
    // independent current source flowing into node i
    void stampRightSide(int i, double x) {
	if (i > 0) {
	    if (circuitNeedsMap) {
		i = circuitRowInfo[i-1].mapRow;
		//System.out.println("stamping " + i + " " + x);
	    } else
		i--;
	    circuitRightSide[i] += x;
	}
    }

    // indicate that the value on the right side of row i changes in doStep()
    void stampRightSide(int i) {
	if (i > 0)
	    circuitRowInfo[i-1].rsChanges = true;
    }
    
    // indicate that the values on the left side of row i change in doStep()
    void stampNonLinear(int i) {
	if (i > 0)
	    circuitRowInfo[i-1].lsChanges = true;
    }

    double getIterCount() {
	return (Math.exp((speedBar.getValue()-1)/24.) + .5);
    }
    
    boolean converged;
    int subIterations;
    void runCircuit() {
	if (circuitMatrix == null || elmList.size() == 0) {
	    circuitMatrix = null;
	    return;
	}
	int iter;
	//int maxIter = getIterCount();
	boolean debugprint = dumpMatrix;
	dumpMatrix = false;
	long steprate = (long) (160*getIterCount());
	for (iter = 1; ; iter++) {
	    int i, j, k, subiter;
	    for (i = 0; i != elmList.size(); i++) {
		CircuitElm ce = getElm(i);
		ce.startIteration();
	    }
	    steps++;
	    final int subiterCount = 5000;
	    for (subiter = 0; subiter != subiterCount; subiter++) {
		converged = true;
		subIterations = subiter;
		for (i = 0; i != circuitMatrixSize; i++)
		    circuitRightSide[i] = origRightSide[i];
		if (circuitNonLinear) {
		    for (i = 0; i != circuitMatrixSize; i++)
			for (j = 0; j != circuitMatrixSize; j++)
			    circuitMatrix[i][j] = origMatrix[i][j];
		}
		for (i = 0; i != elmList.size(); i++) {
		    CircuitElm ce = getElm(i);
		    ce.doStep();
		}
		if (stopMessage != null)
		    return;
		boolean printit = debugprint;
		debugprint = false;
		for (j = 0; j != circuitMatrixSize; j++) {
		    for (i = 0; i != circuitMatrixSize; i++) {
			double x = circuitMatrix[i][j];
			if (Double.isNaN(x) || Double.isInfinite(x)) {
			    stop("nan/matriz infinita!", null);
			    return;
			}
		    }
		}
		if (printit) {
		    for (j = 0; j != circuitMatrixSize; j++) {
			for (i = 0; i != circuitMatrixSize; i++)
			    System.out.print(circuitMatrix[j][i] + " ");
			System.out.print("  " + circuitRightSide[j] + "\n");
		    }
		    System.out.print("\n");
		}
		if (circuitNonLinear) {
		    if (converged && subiter > 0)
			break;
		    if (!LUdecomp(circuitMatrix, circuitMatrixSize,
				  circuitPermute)) {
			stop("Matriz Singular!", null);
			return;
		    }
		}
		LUsubst(circuitMatrix, circuitMatrixSize, circuitPermute,
		       circuitRightSide);
		
		for (j = 0; j != circuitMatrixFullSize; j++) {
		    RowInfo ri = circuitRowInfo[j];
		    double res = 0;
		    if (ri.type == RowInfo.ROW_CONST)
			res = ri.value;
		    else
			res = circuitRightSide[ri.mapCol];
		    /*System.out.println(j + " " + res + " " +
		      ri.type + " " + ri.mapCol);*/
		    if (Double.isNaN(res)) {
			converged = false;
			//debugprint = true;
			break;
		    }
		    if (j < nodeList.size()-1) {
			CircuitNode cn = getCircuitNode(j+1);
			for (k = 0; k != cn.links.size(); k++) {
			    CircuitNodeLink cnl = (CircuitNodeLink)
				cn.links.elementAt(k);
			    cnl.elm.setNodeVoltage(cnl.num, res);
			}
		    } else {
			int ji = j-(nodeList.size()-1);
			voltageSources[ji].setCurrent(ji, res);
		    }
		}
		if (!circuitNonLinear)
		    break;
	    }
	    if (subiter > 5)
		System.out.print("converge trasr " + subiter + " iteraciones\n");
	    if (subiter == subiterCount) {
		stop("Fallo de Convergencia!", null);
		break;
	    }
	    t += timeStep;
	    for (i = 0; i != scopeCount; i++)
		scopes[i].timeStep();
	    long tm = System.currentTimeMillis();
	    if (iter*1000 >= steprate*(tm-lastFrameTime) ||
		(tm-lastFrameTime > 500))
		break;
	}
    }

    int abs(int x) {
	return x < 0 ? -x : x;
    }

    int sign(int x) {
	return (x < 0) ? -1 : (x == 0) ? 0 : 1;
    }

    int min(int a, int b) { return (a < b) ? a : b; }
    int max(int a, int b) { return (a > b) ? a : b; }

    void editFuncPoint(int x, int y) {
	// XXX
	cv.repaint(pause);
    }

    public void componentHidden(ComponentEvent e){}
    public void componentMoved(ComponentEvent e){}
    public void componentShown(ComponentEvent e) {
	cv.repaint();
    }

    public void componentResized(ComponentEvent e) {
	handleResize();
	cv.repaint(100);
    }
    public void actionPerformed(ActionEvent e) {
	String ac = e.getActionCommand();
	if (e.getSource() == resetButton) {
	    int i;
	    
	    // on IE, drawImage() stops working inexplicably every once in
	    // a while.  Recreating it fixes the problem, so we do that here.
	    handleResize();
	    
	    for (i = 0; i != elmList.size(); i++)
		getElm(i).reset();
	    for (i = 0; i != scopeCount; i++)
		scopes[i].resetGraph();
	    analyzeFlag = true;
	    t = 0;
	    stoppedCheck.setState(false);
	    cv.repaint();
	}
	/*if (e.getSource() == dumpMatrixButton)
	  analyzeFlag = dumpMatrix = true;*/
	if (e.getSource() == exportItem)
	    doImport(false);
	if (e.getSource() == importItem)
	    doImport(true);
	if (e.getSource() == exitItem) {
	    applet.destroyFrame();
	    return;
	}
	if (ac.compareTo("stackAll") == 0)
	    stackAll();
	if (ac.compareTo("unstackAll") == 0)
	    unstackAll();
	if (e.getSource() == elmEditMenuItem)
	    doEdit();
	if (e.getSource() == elmDeleteMenuItem && menuElm != null)
	    deleteElm(menuElm);
	if (e.getSource() == elmScopeMenuItem && menuElm != null) {
	    int i;
	    for (i = 0; i != scopeCount; i++)
		if (scopes[i].elm == null)
		    break;
	    if (i == scopeCount) {
		if (scopeCount == scopes.length)
		    return;
		scopeCount++;
		scopes[i] = new Scope();
		scopes[i].position = i;
		handleResize();
	    }
	    scopes[i].setElm(menuElm);
	}
	if (menuScope != -1) {
	    if (ac.compareTo("remove") == 0)
		scopes[menuScope].setElm(null);
	    if (ac.compareTo("speed2") == 0)
		scopes[menuScope].speedUp();
	    if (ac.compareTo("speed1/2") == 0)
		scopes[menuScope].slowDown();
	    if (ac.compareTo("scale") == 0)
		scopes[menuScope].adjustScale(.5);
	    if (ac.compareTo("stack") == 0)
		stackScope(menuScope);
	    if (ac.compareTo("unstack") == 0)
		unstackScope(menuScope);
	    cv.repaint();
	}
	if (ac.indexOf("setup ") == 0)
	    readSetupFile(ac.substring(6),
			  ((MenuItem) e.getSource()).getLabel());
    }

    void stackScope(int s) {
	if (s == 0) {
	    if (scopeCount < 2)
		return;
	    s = 1;
	}
	if (scopes[s].position == scopes[s-1].position)
	    return;
	scopes[s].position = scopes[s-1].position;
	for (s++; s < scopeCount; s++)
	    scopes[s].position--;
    }
    
    void unstackScope(int s) {
	if (s == 0) {
	    if (scopeCount < 2)
		return;
	    s = 1;
	}
	if (scopes[s].position != scopes[s-1].position)
	    return;
	for (; s < scopeCount; s++)
	    scopes[s].position++;
    }

    void stackAll() {
	int i;
	for (i = 0; i != scopeCount; i++) {
	    scopes[i].position = 0;
	    scopes[i].showMax = false;
	}
    }

    void unstackAll() {
	int i;
	for (i = 0; i != scopeCount; i++) {
	    scopes[i].position = i;
	    scopes[i].showMax = true;
	}
    }
    
    void doEdit() {
	if (editDialog != null) {
	    requestFocus();
	    editDialog.setVisible(false);
	    editDialog = null;
	}
	editDialog = new EditDialog(menuElm, this);
	editDialog.show();
    }

    void doImport(boolean imp) {
	if (impDialog != null) {
	    requestFocus();
	    impDialog.setVisible(false);
	    impDialog = null;
	}
	String dump = "";
	if (!imp) {
	    int i;
	    int f = (dotsCheckItem.getState()) ? 1 : 0;
	    f |= (smallGridCheckItem.getState()) ? 2 : 0;
	    f |= (voltsCheckItem.getState()) ? 0 : 4;
	    f |= (powerCheckItem.getState()) ? 8 : 0;
	    f |= (showValuesCheckItem.getState()) ? 0 : 16;
	    dump = "$ " + f + " " +
		timeStep + " " + getIterCount() + " " +
		currentBar.getValue() + " " + voltageRange + " " +
		powerBar.getValue() + "\n";
	    for (i = 0; i != elmList.size(); i++)
		dump += getElm(i).dump() + "\n";
	    for (i = 0; i != scopeCount; i++) {
		String d = scopes[i].dump();
		if (d != null)
		    dump += d + "\n";
	    }
	    if (hintType != -1)
		dump += "h " + hintType + " " + hintItem1 + " " +
		    hintItem2 + "\n";
	}
	impDialog = new ImportDialog(this, dump);
	impDialog.show();
    }
    
    public void adjustmentValueChanged(AdjustmentEvent e) {
	System.out.print(((Scrollbar) e.getSource()).getValue() + "\n");
    }

    ByteArrayOutputStream readUrlData(URL url) throws java.io.IOException {
	Object o = url.getContent();
	FilterInputStream fis = (FilterInputStream) o;
	ByteArrayOutputStream ba = new ByteArrayOutputStream(fis.available());
	int blen = 1024;
	byte b[] = new byte[blen];
	while (true) {
	    int len = fis.read(b);
	    if (len <= 0)
		break;
	    ba.write(b, 0, len);
	}
	return ba;
    }
    
    void getSetupList(Menu menu) {
	Menu stack[] = new Menu[6];
	int stackptr = 0;
	stack[stackptr++] = menu;
	try {
	    URL url = new URL(applet.getCodeBase() + "setuplist.txt");
	    ByteArrayOutputStream ba = readUrlData(url);
	    byte b[] = ba.toByteArray();
	    int len = ba.size();
	    int p;
	    for (p = 0; p < len; ) {
		int l;
		for (l = 0; l != len-p; l++)
		    if (b[l+p] == '\n') {
			l++;
			break;
		    }
		String line = new String(b, p, l-1);
		if (line.charAt(0) == '+') {
		    Menu n = new Menu(line.substring(1));
		    menu.add(n);
		    menu = stack[stackptr++] = n;
		} else if (line.charAt(0) == '-') {
		    menu = stack[--stackptr-1];
		} else {
		    int i = line.indexOf(' ');
		    if (i > 0)
			menu.add(getMenuItem(line.substring(i+1),
					     "setup " + line.substring(0, i)));
		}
		p += l;
	    }
	} catch (Exception e) {
	    e.printStackTrace();
	    stop("No se puede leer setuplist.txt!", null);
	}
    }

    void readSetup(String text) {
	readSetup(text.getBytes(), text.length());
	titleLabel.setText("sinttulo");
    }
    
    void readSetupFile(String str, String title) {
	t = 0;
	System.out.println(str);
	try {
	    URL url = new URL(applet.getCodeBase() + str);
	    ByteArrayOutputStream ba = readUrlData(url);
	    readSetup(ba.toByteArray(), ba.size());
	} catch (Exception e) {
	    e.printStackTrace();
	    stop("No se puede leer " + str + "!", null);
	}
	titleLabel.setText(title);
    }

    void readSetup(byte b[], int len) {
	elmList.removeAllElements();
	hintType = -1;
	timeStep = 5e-6;
	dotsCheckItem.setState(true);
	smallGridCheckItem.setState(false);
	powerCheckItem.setState(false);
	voltsCheckItem.setState(true);
	showValuesCheckItem.setState(true);
	setGrid();
	speedBar.setValue(57);
	currentBar.setValue(50);
	powerBar.setValue(50);
	voltageRange = 5;
	cv.repaint();
	int p;
	scopeCount = 0;
	for (p = 0; p < len; ) {
	    int l;
	    for (l = 0; l != len-p; l++)
		if (b[l+p] == '\n') {
		    l++;
		    break;
		}
	    String line = new String(b, p, l-1);
	    StringTokenizer st = new StringTokenizer(line);
	    while (st.hasMoreTokens()) {
		String type = st.nextToken();
		int tint = type.charAt(0);
		try {
		    if (tint == 'o') {
			Scope sc = new Scope();
			sc.position = scopeCount;
			sc.undump(st);
			scopes[scopeCount++] = sc;
			break;
		    }
		    if (tint == 'h') {
			readHint(st);
			break;
		    }
		    if (tint == '$') {
			readOptions(st);
			break;
		    }
		    if (tint >= '0' && tint <= '9')
			tint = new Integer(type).intValue();
		    int x1 = new Integer(st.nextToken()).intValue();
		    int y1 = new Integer(st.nextToken()).intValue();
		    int x2 = new Integer(st.nextToken()).intValue();
		    int y2 = new Integer(st.nextToken()).intValue();
		    int f  = new Integer(st.nextToken()).intValue();
		    CircuitElm ce = null;
		    Class cls = dumpTypes[tint];
		    if (cls == null) {
			System.out.println("unrecognized dump type: " + type);
			break;
		    }
		    // find element class
		    Class carr[] = new Class[7];
		    carr[0] = getClass();
		    carr[1] = carr[2] = carr[3] = carr[4] = carr[5] =
			int.class;
		    carr[6] = StringTokenizer.class;
		    Constructor cstr = null;
		    cstr = cls.getConstructor(carr);
		
		    // invoke constructor with starting coordinates
		    Object oarr[] = new Object[7];
		    oarr[0] = this;
		    oarr[1] = new Integer(x1);
		    oarr[2] = new Integer(y1);
		    oarr[3] = new Integer(x2);
		    oarr[4] = new Integer(y2);
		    oarr[5] = new Integer(f );
		    oarr[6] = st;
		    ce = (CircuitElm) cstr.newInstance(oarr);
		    ce.setPoints();
		    elmList.addElement(ce);
		} catch (java.lang.reflect.InvocationTargetException ee) {
		    ee.getTargetException().printStackTrace();
		    break;
		} catch (Exception ee) {
		    ee.printStackTrace();
		    break;
		}
		break;
	    }
	    p += l;
	}
	handleResize(); // for scopes
	analyzeCircuit();
    }

    void readHint(StringTokenizer st) {
	hintType  = new Integer(st.nextToken()).intValue();
	hintItem1 = new Integer(st.nextToken()).intValue();
	hintItem2 = new Integer(st.nextToken()).intValue();
    }

    void readOptions(StringTokenizer st) {
	int flags = new Integer(st.nextToken()).intValue();
	dotsCheckItem.setState((flags & 1) != 0);
	smallGridCheckItem.setState((flags & 2) != 0);
	voltsCheckItem.setState((flags & 4) == 0);
	powerCheckItem.setState((flags & 8) == 8);
	showValuesCheckItem.setState((flags & 16) == 0);
	timeStep = new Double (st.nextToken()).doubleValue();
	double sp = new Double(st.nextToken()).doubleValue();
	int sp2 = (int) (Math.log(sp)*24+1.5);
	speedBar  .setValue(sp2);
	currentBar.setValue(new Integer(st.nextToken()).intValue());
	voltageRange = new Double (st.nextToken()).doubleValue();
	try {
	    powerBar.setValue(new Integer(st.nextToken()).intValue());
	} catch (Exception e) {
	}
	setGrid();
    }
    
    int snapGrid(int x) {
	return (x+gridRound) & gridMask;
    }

    boolean doSwitch(int x, int y) {
	if (mouseElm == null || !(mouseElm instanceof SwitchElm))
	    return false;
	SwitchElm se = (SwitchElm) mouseElm;
	se.toggle();
	if (se.momentary)
	    heldSwitchElm = se;
	analyzeCircuit();
	return true;
    }

    void deleteElm(CircuitElm elm) {
	int e = locateElm(elm);
	if (e >= 0) {
	    elmList.removeElementAt(e);
	    analyzeCircuit();
	}
    }

    int locateElm(CircuitElm elm) {
	int i;
	for (i = 0; i != elmList.size(); i++)
	    if (elm == elmList.elementAt(i))
		return i;
	return -1;
    }
    
    public void mouseDragged(MouseEvent e) {
	if (!circuitArea.contains(e.getX(), e.getY()))
	    return;
	if (dragElm != null)
	    dragElm.drag(e.getX(), e.getY());
	switch (mouseMode) {
	case MODE_DRAG_ALL:
	    dragAll(snapGrid(e.getX()), snapGrid(e.getY()));
	    break;
	case MODE_DRAG_ROW:
	    dragRow(snapGrid(e.getX()), snapGrid(e.getY()));
	    break;
	case MODE_DRAG_COLUMN:
	    dragColumn(snapGrid(e.getX()), snapGrid(e.getY()));
	    break;
	case MODE_DRAG_POST:
	    if (mouseElm != null)
		dragPost(snapGrid(e.getX()), snapGrid(e.getY()));
	    break;
	case MODE_DRAG_SELECTED:
	    if (mouseElm != null)
		dragItem(e.getX(), e.getY());
	    break;
	}
	dragging = true;
	if (mouseMode == MODE_DRAG_SELECTED && mouseElm instanceof TextElm) {
	    dragX = e.getX(); dragY = e.getY();
	} else {
	    dragX = snapGrid(e.getX()); dragY = snapGrid(e.getY());
	}
	cv.repaint(pause);
    }

    void dragAll(int x, int y) {
	int dx = x-dragX;
	int dy = y-dragY;
	if (dx == 0 && dy == 0)
	    return;
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    ce.move(dx, dy);
	}
	removeZeroLengthElements();
    }

    void dragRow(int x, int y) {
	int dy = y-dragY;
	if (dy == 0)
	    return;
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce.y  == dragY)
		ce.movePoint(0, 0, dy);
	    if (ce.y2 == dragY)
		ce.movePoint(1, 0, dy);
	}
	removeZeroLengthElements();
    }
    
    void dragColumn(int x, int y) {
	int dx = x-dragX;
	if (dx == 0)
	    return;
	int i;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce.x  == dragX)
		ce.movePoint(0, dx, 0);
	    if (ce.x2 == dragX)
		ce.movePoint(1, dx, 0);
	}
	removeZeroLengthElements();
    }

    void dragItem(int x, int y) {
	if (!(mouseElm instanceof TextElm)) {
	    x = snapGrid(x);
	    y = snapGrid(y);
	}
	int dx = x-dragX;
	int dy = y-dragY;
	if (dx == 0 && dy == 0)
	    return;
	mouseElm.movePoint(0, dx, dy);
	mouseElm.movePoint(1, dx, dy);
	analyzeCircuit();
    }

    void dragPost(int x, int y) {
	if (draggingPost == -1) {
	    draggingPost =
		(distanceSq(mouseElm.x , mouseElm.y , x, y) >
		 distanceSq(mouseElm.x2, mouseElm.y2, x, y)) ? 1 : 0;
	}
	int dx = x-dragX;
	int dy = y-dragY;
	if (dx == 0 && dy == 0)
	    return;
	mouseElm.movePoint(draggingPost, dx, dy);
	analyzeCircuit();
    }

    void removeZeroLengthElements() {
	int i;
	boolean changed = false;
	for (i = elmList.size()-1; i >= 0; i--) {
	    CircuitElm ce = getElm(i);
	    if (ce.x == ce.x2 && ce.y == ce.y2) {
		elmList.removeElementAt(i);
		changed = true;
	    }
	}
	analyzeCircuit();
    }
    
    public void mouseMoved(MouseEvent e) {
	if ((e.getModifiers() & MouseEvent.BUTTON1_MASK) != 0)
	    return;
	int x = e.getX();
	int y = e.getY();
	dragX = snapGrid(x); dragY = snapGrid(y);
	draggingPost = -1;
	int i;
	CircuitElm origMouse = mouseElm;
	mouseElm = null;
	int bestDist = 100000;
	int bestArea = 100000;
	for (i = 0; i != elmList.size(); i++) {
	    CircuitElm ce = getElm(i);
	    if (ce.boundingBox.contains(x, y)) {
		int j;
		int area = ce.boundingBox.width * ce.boundingBox.height;
		int jn = max(ce.getPostCount(), 2);
		for (j = 0; j != jn; j++) {
		    Point pt = ce.getPost(j);
		    int dist = distanceSq(x, y, pt.x, pt.y);

		    // if multiple elements have overlapping bounding boxes,
		    // we prefer selecting elements that have posts close
		    // to the mouse pointer and that have a small bounding
		    // box area.
		    if (dist <= bestDist && area <= bestArea) {
			bestDist = dist;
			bestArea = area;
			mouseElm = ce;
		    }
		}
		if (ce.getPostCount() == 0)
		    mouseElm = ce;
	    }
	}
	scopeSelected = -1;
	if (mouseElm == null) {
	    for (i = 0; i != scopeCount; i++) {
		Scope s = scopes[i];
		if (s.rect.contains(x, y)) {
		    mouseElm = s.elm;
		    scopeSelected = i;
		}
	    }
	}
	if (mouseElm != origMouse)
	    cv.repaint();
    }

    int distanceSq(int x1, int y1, int x2, int y2) {
	x2 -= x1;
	y2 -= y1;
	return x2*x2+y2*y2;
    }
    
    public void mouseClicked(MouseEvent e) {
    }
    public void mouseEntered(MouseEvent e) {
    }
    public void mouseExited(MouseEvent e) {
    }
    public void mousePressed(MouseEvent e) {
	if (e.isPopupTrigger()) {
	    doPopupMenu(e);
	    return;
	}
	if ((e.getModifiers() & MouseEvent.BUTTON1_MASK) == 0)
	    return;
	if (doSwitch(e.getX(), e.getY()))
	    return;
	dragging = true;
	if (mouseMode != MODE_ADD_ELM || addingClass == null)
	    return;
	
	int x0 = snapGrid(e.getX());
	int y0 = snapGrid(e.getY());
	if (!circuitArea.contains(x0, y0))
	    return;

	dragElm = constructElement(addingClass, x0, y0);
    }

    CircuitElm constructElement(Class c, int x0, int y0) {
	// find element class
	Class carr[] = new Class[3];
	carr[0] = getClass();
	carr[1] = carr[2] = int.class;
	Constructor cstr = null;
	try {
	    cstr = c.getConstructor(carr);
	} catch (Exception ee) {
	    ee.printStackTrace();
	    return null;
	}

	// invoke constructor with starting coordinates
	Object oarr[] = new Object[3];
	oarr[0] = this;
	oarr[1] = new Integer(x0);
	oarr[2] = new Integer(y0);
	try {
	    return (CircuitElm) cstr.newInstance(oarr);
	} catch (Exception ee) { ee.printStackTrace(); }
	return null;
    }
    
    void doPopupMenu(MouseEvent e) {
	menuElm = mouseElm;
	if (scopeSelected != -1) {
	    PopupMenu m = scopes[scopeSelected].getMenu();
	    menuScope = scopeSelected;
	    if (m != null)
		m.show(e.getComponent(), e.getX(), e.getY());
	} else if (mouseElm != null) {
	    elmEditMenuItem .setEnabled(mouseElm.getEditInfo(0) != null);
	    elmScopeMenuItem.setEnabled(mouseElm.canViewInScope());
	    elmMenu.show(e.getComponent(), e.getX(), e.getY());
	} else {
	    doMainMenuChecks(mainMenu);
	    mainMenu.show(e.getComponent(), e.getX(), e.getY());
	}
    }

    void doMainMenuChecks(Menu m) {
	int i;
	if (m == optionsMenu)
	    return;
	for (i = 0; i != m.getItemCount(); i++) {
	    MenuItem mc = m.getItem(i);
	    if (mc instanceof Menu)
		doMainMenuChecks((Menu) mc);
	    if (mc instanceof CheckboxMenuItem) {
		CheckboxMenuItem cmi = (CheckboxMenuItem) mc;
		cmi.setState(
		      mouseModeStr.compareTo(cmi.getActionCommand()) == 0);
	    }
	}
    }
    
    public void mouseReleased(MouseEvent e) {
	if (e.isPopupTrigger()) {
	    doPopupMenu(e);
	    return;
	}
	if ((e.getModifiers() & MouseEvent.BUTTON1_MASK) == 0)
	    return;
	dragging = false;
	boolean circuitChanged = false;
	if (heldSwitchElm != null) {
	    heldSwitchElm.mouseUp();
	    heldSwitchElm = null;
	    circuitChanged = true;
	}
	if (dragElm != null &&
	    !(dragElm.x == dragElm.x2 && dragElm.y == dragElm.y2)) {
	    elmList.addElement(dragElm);
	    circuitChanged = true;
	}
	if (circuitChanged)
	    analyzeCircuit();
	dragElm = null;
	cv.repaint();
    }

    void enableItems() {
	if (powerCheckItem.getState()) {
	    powerBar.enable();
	    powerLabel.enable();
	} else {
	    powerBar.disable();
	    powerLabel.disable();
	}
    }
    
    public void itemStateChanged(ItemEvent e) {
	cv.repaint(pause);
	Object mi = e.getItemSelectable();
	if (mi == stoppedCheck)
	    return;
	if (mi == smallGridCheckItem)
	    setGrid();
	if (mi == powerCheckItem) {
	    if (powerCheckItem.getState())
		voltsCheckItem.setState(false);
	    else
		voltsCheckItem.setState(true);
	}
	if (mi == voltsCheckItem && voltsCheckItem.getState())
	    powerCheckItem.setState(false);
	enableItems();
	if (menuScope != -1) {
	    Scope sc = scopes[menuScope];
	    if (mi == scopeVMenuItem)
		sc.showVoltage(scopeVMenuItem.getState());
	    if (mi == scopeIMenuItem)
		sc.showCurrent(scopeIMenuItem.getState());
	    if (mi == scopeMaxMenuItem)
		sc.showMax(scopeMaxMenuItem.getState());
	    if (mi == scopeFreqMenuItem)
		sc.showFreq(scopeFreqMenuItem.getState());
	    if (mi == scopePowerMenuItem)
		sc.setValue(SCOPEVAL_POWER);
	    if (mi == scopeIbMenuItem)
		sc.setValue(SCOPEVAL_IB);
	    if (mi == scopeIcMenuItem)
		sc.setValue(SCOPEVAL_IC);
	    if (mi == scopeIeMenuItem)
		sc.setValue(SCOPEVAL_IE);
	    if (mi == scopeVbeMenuItem)
		sc.setValue(SCOPEVAL_VBE);
	    if (mi == scopeVbcMenuItem)
		sc.setValue(SCOPEVAL_VBC);
	    if (mi == scopeVceMenuItem)
		sc.setValue(SCOPEVAL_VCE);
	}
	if (mi instanceof CheckboxMenuItem) {
	    MenuItem mmi = (MenuItem) mi;
	    mouseMode = MODE_ADD_ELM;
	    String s = mmi.getActionCommand();
	    if (s.length() > 0)
		mouseModeStr = s;
	    if (s.compareTo("DragAll") == 0)
		mouseMode = MODE_DRAG_ALL;
	    else if (s.compareTo("DragRow") == 0)
		mouseMode = MODE_DRAG_ROW;
	    else if (s.compareTo("DragColumn") == 0)
		mouseMode = MODE_DRAG_COLUMN;
	    else if (s.compareTo("DragSelected") == 0)
		mouseMode = MODE_DRAG_SELECTED;
	    else if (s.compareTo("DragPost") == 0)
		mouseMode = MODE_DRAG_POST;
	    else if (s.length() > 0) {
		try {
		    addingClass = Class.forName("CircuitFrame$" + s);
		} catch (Exception ee) {
		    ee.printStackTrace();
		}
	    }
	}
    }

    void setGrid() {
	gridSize = (smallGridCheckItem.getState()) ? 8 : 16;
	gridMask = ~(gridSize-1);
	gridRound = gridSize/2-1;
    }
    
    boolean comparePair(int x1, int x2, int y1, int y2) {
	return ((x1 == y1 && x2 == y2) || (x1 == y2 && x2 == y1));
    }
    
    String getVoltageDText(double v) {
	return getUnitText(Math.abs(v), "V");
    }
    String getVoltageText(double v) {
	return getUnitText(v, "V");
    }
    String getUnitText(double v, String u) {
	double va = Math.abs(v);
	if (va < 1e-12)
	    return "0 " + u;
	if (va < 1e-9)
	    return showFormat.format(v*1e12) + " p" + u;
	if (va < 1e-6)
	    return showFormat.format(v*1e9) + " n" + u;
	if (va < 1e-3)
	    return showFormat.format(v*1e6) + " " + muString + u;
	if (va < 1)
	    return showFormat.format(v*1e3) + " m" + u;
	if (va < 1e3)
	    return showFormat.format(v) + " " + u;
	if (va < 1e6)
	    return showFormat.format(v*1e-3) + " k" + u;
	return showFormat.format(v*1e-6) + " M" + u;
    }
    String getShortUnitText(double v, String u) {
	double va = Math.abs(v);
	if (va < 1e-12)
	    return null;
	if (va < 1e-9)
	    return shortFormat.format(v*1e12) + "p" + u;
	if (va < 1e-6)
	    return shortFormat.format(v*1e9) + "n" + u;
	if (va < 1e-3)
	    return shortFormat.format(v*1e6) + muString + u;
	if (va < 1)
	    return shortFormat.format(v*1e3) + "m" + u;
	if (va < 1e3)
	    return shortFormat.format(v) + u;
	if (va < 1e6)
	    return shortFormat.format(v*1e-3) + "k" + u;
	return shortFormat.format(v*1e-6) + "M" + u;
    }
    String getCurrentText(double i) {
	return getUnitText(i, "A");
    }
    String getCurrentDText(double i) {
	return getUnitText(Math.abs(i), "A");
    }

    abstract class CircuitElm {
	int x, y, x2, y2, flags, nodes[], voltSource;
	double volts[];
	double current, curcount;
	Rectangle boundingBox;
	boolean noDiagonal;
	int getDumpType() { return 0; }
	Class getDumpClass() { return getClass(); }
	CircuitElm(int xx, int yy) {
	    x = x2 = xx;
	    y = y2 = yy;
	    allocNodes();
	    boundingBox = new Rectangle();
	}
	CircuitElm(int xa, int ya, int xb, int yb, int f) {
	    x = xa; y = ya; x2 = xb; y2 = yb; flags = f;
	    allocNodes();
	    boundingBox = new Rectangle();
	}
	void allocNodes() {
	    nodes = new int[getPostCount()+getInternalNodeCount()];
	    volts = new double[getPostCount()+getInternalNodeCount()];
	}
	String dump() {
	    int t = getDumpType();
	    return (t < 127 ? ((char)t)+" " : t+" ") + x + " " + y + " " +
		x2 + " " + y2 + " " + flags;
	}
	void reset() {
	    int i;
	    for (i = 0; i != getPostCount()+getInternalNodeCount(); i++)
		volts[i] = 0;
	}
	abstract void draw(Graphics g);
	void setCurrent(int x, double c) { current = c; }
	double getCurrent() { return current; }
	void doStep() {}
	void startIteration() {}
	void setNodeVoltage(int n, double c) {
	    volts[n] = c;
	    calculateCurrent();
	}
	void calculateCurrent() {}
	void setPoints() {}
	void drag(int xx, int yy) {
	    xx = snapGrid(xx);
	    yy = snapGrid(yy);
	    if (noDiagonal) {
		if (abs(x-xx) < abs(y-yy))
		    xx = x;
		else
		    yy = y;
	    }
	    x2 = xx; y2 = yy;
	    setPoints();
	}
	void move(int dx, int dy) {
	    x += dx; y += dy; x2 += dx; y2 += dy;
	    setPoints();
	}
	void movePoint(int n, int dx, int dy) {
	    if (n == 0) {
		x += dx; y += dy;
	    } else {
		x2 += dx; y2 += dy;
	    }
	    setPoints();
	}
	void drawPosts(Graphics g) {
	    drawPost(g, x, y, nodes[0]);
	    drawPost(g, x2, y2, nodes[1]);
	}
	void stamp() {}
	int getVoltageSourceCount() { return 0; }
	int getInternalNodeCount() { return 0; }
	void setNode(int p, int n) { nodes[p] = n; }
	void setVoltageSource(int n, int v) { voltSource = v; }
	int getVoltageSource() { return voltSource; }
	double getVoltageDiff() {
	    return volts[0] - volts[1];
	}
	boolean nonLinear() { return false; }
	int getPostCount() { return 2; }
	int getNode(int n) { return nodes[n]; }
	Point getPost(int n) {
	    return (n == 0) ? new Point(x, y) :
		(n == 1) ? new Point(x2, y2) : null;
	}
	void drawPost(Graphics g, int x0, int y0, int n) {
	    if (dragElm == null && mouseElm != this &&
		getCircuitNode(n).links.size() == 2)
		return;
	    if (mouseMode == MODE_DRAG_ROW || mouseMode == MODE_DRAG_COLUMN)
		return;
	    drawPost(g, x0, y0);
	}
	void drawPost(Graphics g, int x0, int y0) {
	    g.setColor(Color.white);
	    g.fillOval(x0-3, y0-3, 7, 7);
	}
	void setBbox(int x1, int y1, int x2, int y2) {
	    if (x1 > x2) { int q = x1; x1 = x2; x2 = q; }
	    if (y1 > y2) { int q = y1; y1 = y2; y2 = q; }
	    boundingBox.setBounds(x1, y1, x2-x1+1, y2-y1+1);
	}
	void adjustBbox(int x1, int y1, int x2, int y2) {
	    if (x1 > x2) { int q = x1; x1 = x2; x2 = q; }
	    if (y1 > y2) { int q = y1; y1 = y2; y2 = q; }
	    x1 = min(boundingBox.x, x1);
	    y1 = min(boundingBox.y, y1);
	    x2 = max(boundingBox.x+boundingBox.width-1,  x2);
	    y2 = max(boundingBox.y+boundingBox.height-1, y2);
	    boundingBox.setBounds(x1, y1, x2-x1, y2-y1);
	}
	void drawValues(Graphics g, String s, int xc, int yc,
			int dpx, int dpy) {
	    if (s == null)
		return;
	    g.setFont(unitsFont);
	    FontMetrics fm = g.getFontMetrics();
	    int w = fm.stringWidth(s);
	    g.setColor(Color.white);
	    int ya = fm.getAscent()/2;
	    if (dpx == 0)
		g.drawString(s, xc-w/2, yc-abs(dpy)-2);
	    else {
		int xx = xc+abs(dpx)+2;
		if (this instanceof VoltageElm || (x < x2 && y > y2))
		    xx = xc-(w+abs(dpx)+2);
		g.drawString(s, xx, yc+dpy+ya);
	    }
	}
	void updateDotCount() {
	    curcount = updateDotCount(current, curcount);
	}
	double updateDotCount(double cur, double cc) {
	    if (stoppedCheck.getState())
		return cc;
	    double cadd = cur*currentMult;
	    /*if (cur != 0 && cadd <= .05 && cadd >= -.05)
	      cadd = (cadd < 0) ? -.05 : .05;*/
	    if (cadd > 8)
		cadd = 8;
	    if (cadd < -8)
		cadd = -8;
	    return cc + cadd;
	}
	void doDots(Graphics g) {
	    updateDotCount();
	    if (dragElm != this)
		drawDots(g, x, y, x2, y2, curcount);
	}
	void doAdjust() {}
	void setupAdjust() {}
	void getInfo(String arr[]) {
	}
	void getBasicInfo(String arr[]) {
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	    arr[2] = "Vd = " + getVoltageDText(getVoltageDiff());
	}
	void setVoltageColor(Graphics g, double volts) {
	    if (this == mouseElm) {
		g.setColor(Color.cyan);
		return;
	    }
	    if (!voltsCheckItem.getState()) {
		if (!powerCheckItem.getState())
		    g.setColor(Color.white);
		return;
	    }
	    int c = (int) ((volts+voltageRange)*(colorScaleCount-1)/
			   (voltageRange*2));
	    if (c < 0)
		c = 0;
	    if (c >= colorScaleCount)
		c = colorScaleCount-1;
	    g.setColor(colorScale[c]);
	}
	void setPowerColor(Graphics g, boolean yellow) {
	    if (!powerCheckItem.getState())
		return;
	    setPowerColor(g, getPower());
	}
	void setPowerColor(Graphics g, double w0) {
	    w0 *= powerMult;
	    //System.out.println(w);
	    double w = (w0 < 0) ? -w0 : w0;
	    if (w > 1)
		w = 1;
	    int rg = 128+(int) (w*127);
	    int b  = (int) (128*(1-w));
	    /*if (yellow)
		g.setColor(new Color(rg, rg, b));
		else */
	    if (w0 > 0)
		g.setColor(new Color(rg, b, b));
	    else
		g.setColor(new Color(b, rg, b));
	}
	double getPower() { return getVoltageDiff()*current; }
	double getScopeValue(int x) {
	    return (x == 1) ? getPower() : getVoltageDiff();
	}
	String getScopeUnits(int x) {
	    return (x == 1) ? "W" : "V";
	}
	EditInfo getEditInfo(int n) { return null; }
	void setEditValue(int n, EditInfo ei) {}
	boolean getConnection(int n1, int n2) { return true; }
	boolean hasGroundConnection(int n1) { return false; }
	boolean isWire() { return false; }
	boolean canViewInScope() { return getPostCount() <= 2; }
    }
    
    class WireElm extends CircuitElm {
	public WireElm(int xx, int yy) { super(xx, yy); }
	public WireElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	/*void setCurrent(double c) {
	    current = c;
	    System.out.print("wire current set to " + c + "\n");
	    }*/
	void draw(Graphics g) {
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x2, y2);
	    doDots(g);
	    if (y == y2)
		setBbox(x, y-3, x2, y+3);
	    else if (x == x2)
		setBbox(x-3, y, x+3, y2);
	    else
		setBbox(x, y, x2, y2);
	    drawPosts(g);
	}
	//static final double resistance = .001;
	/*void calculateCurrent() {
	    current = (volts[0]-volts[1])/resistance;
	    }*/
	void stamp() {
	    stampVoltageSource(nodes[0], nodes[1], voltSource, 0);
	}
	int getVoltageSourceCount() { return 1; }
	void getInfo(String arr[]) {
	    arr[0] = "cable";
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	    arr[2] = "V = " + getVoltageText(volts[0]);
	}
	int getDumpType() { return 'w'; }
	double getPower() { return 0; }
	double getVoltageDiff() { return volts[0]; }
	boolean isWire() { return true; }
    }

    void drawThickLine(Graphics g, int x, int y, int x2, int y2) {
	g.drawLine(x, y, x2, y2);
	g.drawLine(x+1, y, x2+1, y2);
	g.drawLine(x, y+1, x2, y2+1);
	g.drawLine(x+1, y+1, x2+1, y2+1);
    }

    void drawThickPolygon(Graphics g, int xs[], int ys[], int c) {
	int i;
	for (i = 0; i != c-1; i++)
	    drawThickLine(g, xs[i], ys[i], xs[i+1], ys[i+1]);
	drawThickLine(g, xs[i], ys[i], xs[0], ys[0]);
    }
    
    /*void drawDottedLine(Graphics g, int x, int y, int x2, int y2) {
	int dx = x2-x;
	int dy = y2-y;
	double dr = Math.sqrt(dx*dx+dy*dy);
	int i;
	int st = 16;
	for (i = 0; i < dr; i += st) {
	    int x1a = x + (int) (i*dx/dr);
	    int y1a = y + (int) (i*dy/dr);
	    int x2a = x + (int) ((i+4)*dx/dr);
	    int y2a = y + (int) ((i+4)*dy/dr);
	    drawThickLine(g, x1a, y1a, x2a, y2a);
	    }
	    }*/

    void drawThickCircle(Graphics g, int cx, int cy, int ri) {
	int a;
	double m = pi/180;
	double r = ri*.98;
	for (a = 0; a != 360; a += 20) {
	    double ax = Math.cos(a*m)*r + cx;
	    double ay = Math.sin(a*m)*r + cy;
	    double bx = Math.cos((a+20)*m)*r + cx;
	    double by = Math.sin((a+20)*m)*r + cy;
	    drawThickLine(g, (int) ax, (int) ay, (int) bx, (int) by);
	}
    }
    
    class ResistorElm extends CircuitElm {
	double resistance;
	public ResistorElm(int xx, int yy) { super(xx, yy); resistance = 100; }
	public ResistorElm(int xa, int ya, int xb, int yb, int f,
		    StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    resistance = new Double(st.nextToken()).doubleValue();
	}
	int getDumpType() { return 'r'; }
	String dump() {
	    return "r " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + resistance;
	}
	void draw(Graphics g) {
	    drawResistor(g, x, y, x2, y2, volts[0], volts[1]);
	    doDots(g);
	    drawPosts(g);
	}
	
	void drawResistor(Graphics g, int x1, int y1, int x2, int y2,
			  double v1, double v2) {
	    int segments = 16;
	    int i;
	    int ox = 0;
	    int dx = x2-x1;
	    int dy = y2-y1;
	    int hs = euroResistorCheckItem.getState() ? 6 : 8;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    setBbox(x1, y1, x2, y2);
	    if (dn > 40) {
		int x1a = (int) (x1+dx*(dn/2-16)/dn);
		int y1a = (int) (y1+dy*(dn/2-16)/dn);
		int x2a = (int) (x1+dx*(dn/2+16)/dn);
		int y2a = (int) (y1+dy*(dn/2+16)/dn);
		setVoltageColor(g, v1);
		drawThickLine(g, x1, y1, x1a, y1a);
		setVoltageColor(g, v2);
		drawThickLine(g, x2a, y2a, x2, y2);
		dn = 32;
		x1 = x1a; y1 = y1a;
		x2 = x2a; y2 = y2a;
		dx = x2-x1;
		dy = y2-y1;
	    }
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    adjustBbox(x1-dpx, y1-dpy, x2+dpx, y2+dpy);
	    setPowerColor(g, true);
	    if (!euroResistorCheckItem.getState()) {
		for (i = 0; i != segments; i++) {
		    int nx = 0;
		    switch (i & 3) {
		    case 0: nx = 1; break;
		    case 2: nx = -1; break;
		    default: nx = 0; break;
		    }
		    double v = v1+(v2-v1)*i/segments;
		    setVoltageColor(g, v);
		    int xa = x1+dx*i/segments + dpx*ox;
		    int ya = y1+dy*i/segments + dpy*ox;
		    int xb = x1+dx*(i+1)/segments + dpx*nx;
		    int yb = y1+dy*(i+1)/segments + dpy*nx;
		    drawThickLine(g, xa, ya, xb, yb);
		    ox = nx;
		}
	    } else {
		setVoltageColor(g, v1);
		drawThickLine(g, x1+dpx, y1+dpy, x1-dpx, y1-dpy);
		for (i = 0; i != segments; i++) {
		    double v = v1+(v2-v1)*i/segments;
		    setVoltageColor(g, v);
		    int xa = x1+dx*i/segments;
		    int ya = y1+dy*i/segments;
		    int xb = x1+dx*(i+1)/segments;
		    int yb = y1+dy*(i+1)/segments;
		    drawThickLine(g, xa+dpx, ya+dpy, xb+dpx, yb+dpy);
		    drawThickLine(g, xa-dpx, ya-dpy, xb-dpx, yb-dpy);
		}
		drawThickLine(g, x2+dpx, y2+dpy, x2-dpx, y2-dpy);
	    }
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(resistance, "");
		drawValues(g, s, (x2+x1)/2, (y2+y1)/2, dpx, dpy);
	    }
	}
    
	void calculateCurrent() {
	    current = (volts[0]-volts[1])/resistance;
	    //System.out.print(this + " res current set to " + current + "\n");
	}
	void stamp() {
	    stampResistor(nodes[0], nodes[1], resistance);
	}
	void getInfo(String arr[]) {
	    arr[0] = "resistencia";
	    getBasicInfo(arr);
	    arr[3] = "R = " + getUnitText(resistance, ohmString);
	    arr[4] = "P = " + getUnitText(getPower(), "W");
	}
	EditInfo getEditInfo(int n) {
	    // ohmString doesn't work here on linux
	    if (n == 0)
		return new EditInfo("Resistencia (ohmios)", resistance, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    resistance = ei.value;
	}
    }

    class CapacitorElm extends CircuitElm {
	double capacitance;
	double compResistance, voltdiff;
	public CapacitorElm(int xx, int yy) {
	    super(xx, yy);
	    capacitance = 1e-5;
	}
	public CapacitorElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    capacitance = new Double(st.nextToken()).doubleValue();
	    voltdiff = new Double(st.nextToken()).doubleValue();
	}
	void setNodeVoltage(int n, double c) {
	    super.setNodeVoltage(n, c);
	    voltdiff = volts[0]-volts[1];
	}
	void reset() { voltdiff = current = 0; }
	int getDumpType() { return 'c'; }
	String dump() {
	    return "c " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + capacitance + " " + voltdiff;
	}
	void draw(Graphics g) {
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int x1a = (int) (x+dx*(dn/2-4)/dn);
	    int y1a = (int) (y+dy*(dn/2-4)/dn);
	    int x2a = (int) (x+dx*(dn/2+4)/dn);
	    int y2a = (int) (y+dy*(dn/2+4)/dn);
	    int hs = 12;
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    setBbox(x, y, x2, y2);
	    adjustBbox(x-dpx, y-dpy, x2+dpx, y2+dpy);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x1a, y1a);
	    setPowerColor(g, false);
	    drawThickLine(g, x1a-dpx, y1a-dpy, x1a+dpx, y1a+dpy);
	    if (powerCheckItem.getState())
		g.setColor(Color.gray);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, x2a, y2a, x2, y2);
	    setPowerColor(g, false);
	    drawThickLine(g, x2a-dpx, y2a-dpy, x2a+dpx, y2a+dpy);
	    updateDotCount();
	    if (dragElm != this) {
		drawDots(g, x, y, x1a, y1a, curcount);
		drawDots(g, x2, y2, x2a, y2a, -curcount);
	    }
	    drawPosts(g);
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(capacitance, "F");
		drawValues(g, s, (x2+x)/2, (y2+y)/2, dpx, dpy);
	    }
	}
	void stamp() {
	    // capacitor companion model using trapezoidal approximation
	    // (Thevenin equivalent) consists of a voltage source in
	    // series with a resistor
	    stampVoltageSource(nodes[0], nodes[2], voltSource);
	    compResistance = timeStep/(2*capacitance);
	    stampResistor(nodes[2], nodes[1], compResistance);
	}
	void startIteration() {
	    voltSourceValue = -voltdiff-current*compResistance;
	}
	double voltSourceValue;
	void doStep() {
	    updateVoltageSource(nodes[0], nodes[2], voltSource,
				voltSourceValue);
 	}
	int getVoltageSourceCount() { return 1; }
	int getInternalNodeCount() { return 1; }
	void getInfo(String arr[]) {
	    arr[0] = "condensador";
	    getBasicInfo(arr);
	    arr[3] = "C = " + getUnitText(capacitance, "F");
	    arr[4] = "P = " + getUnitText(getPower(), "W");
	    //double v = getVoltageDiff();
	    //arr[4] = "U = " + getUnitText(.5*capacitance*v*v, "J");
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Capacidad (uF)", capacitance*1e6, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    capacitance = ei.value*1e-6;
	}
    }

    class InductorElm extends CircuitElm {
	double inductance;
	double compResistance;
	public InductorElm(int xx, int yy) { super(xx, yy); inductance = 1; }
	public InductorElm(int xa, int ya, int xb, int yb, int f,
		    StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    inductance = new Double(st.nextToken()).doubleValue();
	    current = new Double(st.nextToken()).doubleValue();
	}
	int getDumpType() { return 'l'; }
	String dump() {
	    return "l " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + inductance + " " + current;
	}
	void draw(Graphics g) {
	    drawInductor(g, x, y, x2, y2, volts[0], volts[1]);
	    doDots(g);
	    drawPosts(g);
	}
	void drawInductor(Graphics g, int x1, int y1, int x2, int y2,
			  double v1, double v2) {
	    int segments = 30;
	    int i;
	    int dx = x2-x1;
	    int dy = y2-y1;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    setBbox(x1, y1, x2, y2);
	    if (dn > 40) {
		int x1a = (int) (x1+dx*(dn/2-16)/dn);
		int y1a = (int) (y1+dy*(dn/2-16)/dn);
		int x2a = (int) (x1+dx*(dn/2+16)/dn);
		int y2a = (int) (y1+dy*(dn/2+16)/dn);
		setVoltageColor(g, v1);
		drawThickLine(g, x1, y1, x1a, y1a);
		setVoltageColor(g, v2);
		drawThickLine(g, x2a, y2a, x2, y2);
		dn = 32;
		x1 = x1a; y1 = y1a;
		x2 = x2a; y2 = y2a;
		dx = x2-x1;
		dy = y2-y1;
	    }
	    int xa = x1;
	    int ya = y1;
	    int dpx0 = (int) ( 8*dy/dn);
	    int dpy0 = (int) (-8*dx/dn);
	    adjustBbox(x1-dpx0, y1-dpy0, x2+dpx0, y2+dpy0);
	    setPowerColor(g, false);
	    for (i = 0; i != segments; i++) {
		double cx = (((i+1)*6./segments) % 2)-1;
		double hs = 8*Math.sqrt(1-cx*cx);
		if (hs < 0)
		    hs = -hs;
		int dpx = (int) ( hs*dy/dn);
		int dpy = (int) (-hs*dx/dn);
		double v = v1+(v2-v1)*i/segments;
		setVoltageColor(g, v);
		int xb = x1+dx*(i+1)/segments + dpx;
		int yb = y1+dy*(i+1)/segments + dpy;
		drawThickLine(g, xa, ya, xb, yb);
		xa = xb;
		ya = yb;
	    }
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(inductance, "H");
		drawValues(g, s, (x2+x1)/2, (y2+y1)/2, dpx0, dpy0);
	    }
	}
	void reset() { current = volts[0] = volts[1] = 0; }
	void stamp() {
	    // inductor companion model using trapezoidal approximation
	    // (Norton equivalent) consists of a current source in
	    // parallel with a resistor.
	    compResistance = 2*inductance/timeStep;
	    stampResistor(nodes[0], nodes[1], compResistance);
	    stampRightSide(nodes[0]);
	    stampRightSide(nodes[1]);
	}
	void startIteration() {
	    double voltdiff = volts[0]-volts[1];
	    curSourceValue = voltdiff/compResistance+current;
	    //System.out.println("ind " + this + " " + current + " " + voltdiff);
	}
	void calculateCurrent() {
	    double voltdiff = volts[0]-volts[1];
	    current = voltdiff/compResistance + curSourceValue;
	}
	double curSourceValue;
	void doStep() {
	    stampCurrentSource(nodes[0], nodes[1], curSourceValue);
 	}
	void getInfo(String arr[]) {
	    arr[0] = "inductancia";
	    getBasicInfo(arr);
	    arr[3] = "L = " + getUnitText(inductance, "H");
	    arr[4] = "P = " + getUnitText(getPower(), "W");
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Inductancia (H)", inductance, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    inductance = ei.value;
	}
    }

    class DCVoltageElm extends VoltageElm {
	public DCVoltageElm(int xx, int yy) { super(xx, yy, WF_DC); }
	Class getDumpClass() { return VoltageElm.class; }
    }
    class ACVoltageElm extends VoltageElm {
	public ACVoltageElm(int xx, int yy) { super(xx, yy, WF_AC); }
	Class getDumpClass() { return VoltageElm.class; }
    }
    
    class VoltageElm extends CircuitElm {
	static final int FLAG_COS = 2;
	int waveform;
	static final int WF_DC = 0;
	static final int WF_AC = 1;
	static final int WF_SQUARE = 2;
	static final int WF_TRIANGLE = 3;
	static final int WF_SAWTOOTH = 4;
	static final int WF_PULSE = 5;
	double frequency, maxVoltage, freqTimeZero, bias;
	VoltageElm(int xx, int yy, int wf) {
	    super(xx, yy);
	    waveform = wf;
	    maxVoltage = 5;
	    frequency = 40;
	    reset();
	}
	public VoltageElm(int xa, int ya, int xb, int yb, int f,
		   StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    maxVoltage = 5;
	    frequency = 40;
	    waveform = WF_DC;
	    try {
		waveform = new Integer(st.nextToken()).intValue();
		frequency = new Double(st.nextToken()).doubleValue();
		maxVoltage = new Double(st.nextToken()).doubleValue();
		bias = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
	    }
	    reset();
	}
	int getDumpType() { return 'v'; }
	String dump() {
	    return "v " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + waveform + " " + frequency + " " +
		maxVoltage + " " + bias;
	}
	/*void setCurrent(double c) {
	    current = c;
	    System.out.print("v current set to " + c + "\n");
	    }*/

	void reset() {
	    freqTimeZero = 0;
	    if ((flags & FLAG_COS) != 0)
		freqTimeZero = 1/(frequency*4);
	}
	double triangleFunc(double x) {
	    if (x < pi)
		return x*(2/pi)-1;
	    return 1-(x-pi)*(2/pi);
	}
	void stamp() {
	    if (waveform == WF_DC)
		stampVoltageSource(nodes[0], nodes[1], voltSource,
				   getVoltage());
	    else
		stampVoltageSource(nodes[0], nodes[1], voltSource);
	}
	void doStep() {
	    if (waveform != WF_DC)
		updateVoltageSource(nodes[0], nodes[1], voltSource,
				    getVoltage());
	}
	double getVoltage() {
	    double w = 2*pi*(t-freqTimeZero)*frequency;
	    switch (waveform) {
	    case WF_DC: return maxVoltage+bias;
	    case WF_AC: return Math.sin(w)*maxVoltage+bias;
	    case WF_SQUARE:
		return bias+((w % (2*pi) > pi) ? -maxVoltage : maxVoltage);
	    case WF_TRIANGLE:
		return bias+triangleFunc(w % (2*pi))*maxVoltage;
	    case WF_SAWTOOTH:
		return bias+(w % (2*pi))*(maxVoltage/pi)-maxVoltage;
	    case WF_PULSE:
		return ((w % (2*pi)) < 1) ? maxVoltage+bias : bias;
	    default: return 0;
	    }
	}
	final int circleSize = 17;
	void draw(Graphics g) {
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int centerDist = (waveform == WF_DC) ? 4 : circleSize;
	    int x1a = (int) (x+dx*(dn/2-centerDist)/dn);
	    int y1a = (int) (y+dy*(dn/2-centerDist)/dn);
	    int x2a = (int) (x+dx*(dn/2+centerDist)/dn);
	    int y2a = (int) (y+dy*(dn/2+centerDist)/dn);
	    int xc = (x+x2)/2;
	    int yc = (y+y2)/2;
	    setBbox(x, y, x2, y2);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x1a, y1a);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, x2a, y2a, x2, y2);
	    if (waveform == WF_DC) {
		int hs = 10;
		int dpx = (int) ( hs*dy/dn);
		int dpy = (int) (-hs*dx/dn);
		setPowerColor(g, false);
		setVoltageColor(g, volts[0]);
		drawThickLine(g, x1a-dpx, y1a-dpy, x1a+dpx, y1a+dpy);
		setVoltageColor(g, volts[1]);
		hs = 16;
		dpx = (int) ( hs*dy/dn);
		dpy = (int) (-hs*dx/dn);
		drawThickLine(g, x2a-dpx, y2a-dpy, x2a+dpx, y2a+dpy);
		adjustBbox(x-dpx, y-dpy, x2+dpx, y2+dpy);
	    } else
		drawWaveform(g, xc, yc);
	    updateDotCount();
	    if (dragElm != this) {
		if (waveform == WF_DC)
		    drawDots(g, x, y, x2, y2, curcount);
		else {
		    drawDots(g, x, y, x1a, y1a, curcount);
		    drawDots(g, x2, y2, x2a, y2a, -curcount);
		}
	    }
	    drawPosts(g);
	}
	
	void drawWaveform(Graphics g, int xc, int yc) {
	    g.setColor(this == mouseElm ? Color.cyan : Color.gray);
	    setPowerColor(g, false);
	    drawThickCircle(g, xc, yc, circleSize);
	    adjustBbox(xc-circleSize, yc-circleSize,
		       xc+circleSize, yc+circleSize);
	    int wl = 8;
	    switch (waveform) {
	    case WF_DC:
		{
		    break;
		}
	    case WF_SQUARE:
		drawThickLine(g, xc-wl, yc-wl, xc-wl, yc   );
		drawThickLine(g, xc-wl, yc-wl, xc   , yc-wl);
		drawThickLine(g, xc   , yc-wl, xc   , yc+wl);
		drawThickLine(g, xc+wl, yc+wl, xc   , yc+wl);
		drawThickLine(g, xc+wl, yc   , xc+wl, yc+wl);
		break;
	    case WF_PULSE:
		yc += wl/2;
		drawThickLine(g, xc-wl, yc-wl, xc-wl, yc   );
		drawThickLine(g, xc-wl, yc-wl, xc-wl/2, yc-wl);
		drawThickLine(g, xc-wl/2, yc-wl, xc-wl/2, yc);
		drawThickLine(g, xc-wl/2, yc, xc+wl, yc);
		break;
	    case WF_SAWTOOTH:
		drawThickLine(g, xc   , yc-wl, xc-wl, yc   );
		drawThickLine(g, xc   , yc-wl, xc   , yc+wl);
		drawThickLine(g, xc   , yc+wl, xc+wl, yc   );
		break;
	    case WF_TRIANGLE:
		{
		    int xl = 5;
		    drawThickLine(g, xc-xl*2, yc   , xc-xl, yc-wl);
		    drawThickLine(g, xc-xl, yc-wl, xc, yc);
		    drawThickLine(g, xc   , yc, xc+xl, yc+wl);
		    drawThickLine(g, xc+xl, yc+wl, xc+xl*2, yc);
		    break;
		}
	    case WF_AC:
		{
		    int i;
		    int xl = 10;
		    int ox = -1, oy = -1;
		    for (i = -xl; i <= xl; i++) {
			int yy = yc+(int) (.95*Math.sin(i*pi/xl)*wl);
			if (ox != -1)
			    drawThickLine(g, ox, oy, xc+i, yy);
			ox = xc+i; oy = yy;
		    }
		    break;
		}
	    }
	    if (showValuesCheckItem.getState()) {
		String s = getShortUnitText(frequency, "Hz");
		int dx = x2-x;
		int dy = y2-y;
		if (dx == 0 || dy == 0) {
		    int dpx = (dx == 0) ?  circleSize : 0;
		    int dpy = (dy == 0) ? -circleSize : 0;
		    drawValues(g, s, xc, yc, dpx, dpy);
		}
	    }
	}
	
	int getVoltageSourceCount() {
	    return 1;
	}
	double getPower() { return -getVoltageDiff()*current; }
	double getVoltageDiff() { return volts[1] - volts[0]; }
	void getInfo(String arr[]) {
	    switch (waveform) {
	    case WF_DC:       arr[0] = "fuente tensin"; break;
	    case WF_AC:       arr[0] = "fuente A/C"; break;
	    case WF_SQUARE:   arr[0] = "gen onda cuadrada"; break;
	    case WF_PULSE:    arr[0] = "gen pulsos"; break;
	    case WF_SAWTOOTH: arr[0] = "gen dientesierra"; break;
	    case WF_TRIANGLE: arr[0] = "gen tringulo"; break;
	    }
	    arr[1] = "I = " + getCurrentText(getCurrent());
	    arr[2] = "Vd = " + getVoltageText(getVoltageDiff());
	    if (waveform != WF_DC) {
		arr[3] = "f = " + getUnitText(frequency, "Hz");
		arr[4] = "Vmax = " + getVoltageText(maxVoltage);
		arr[5] = "Vdesp = " + getVoltageText(bias);
		arr[6] = "P = " + getUnitText(getPower(), "W");
	    }
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Voltaje Max", maxVoltage, -20, 20);
	    if (n == 1) {
		EditInfo ei =  new EditInfo("Onda", waveform, -1, -1);
		ei.choice = new Choice();
		ei.choice.add("D/C");
		ei.choice.add("A/C");
		ei.choice.add("Onda Cuadrada");
		ei.choice.add("Tringulo");
		ei.choice.add("Dientesierra");
		ei.choice.add("Pulso");
		ei.choice.select(waveform);
		return ei;
	    }
	    if (n == 2)
		return new EditInfo("Frecuencia (Hz)", frequency, 4, 500);
	    if (n == 3)
		return new EditInfo("Desplazamiento DC (V)", bias, -20, 20);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		maxVoltage = ei.value;
	    if (n == 3)
		bias = ei.value;
	    if (n == 2) {
		// adjust time zero to maintain continuity ind the waveform
		// even though the frequency has changed.
		double oldfreq = frequency;
		frequency = ei.value;
		double maxfreq = 1/(8*timeStep);
		if (frequency > maxfreq)
		    frequency = maxfreq;
		double adj = frequency-oldfreq;
		freqTimeZero = t-oldfreq*(t-freqTimeZero)/frequency;
	    }
	    if (n == 1)
		waveform = ei.choice.getSelectedIndex();
	}
    }

    class CurrentElm extends CircuitElm {
	double currentValue;
	public CurrentElm(int xx, int yy) {
	    super(xx, yy);
	    currentValue = .01;
	}
	public CurrentElm(int xa, int ya, int xb, int yb, int f,
		   StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    try {
		currentValue = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
		currentValue = .01;
	    }
	}
	String dump() {
	    return "i " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + currentValue;
	}
	int getDumpType() { return 'i'; }
	void draw(Graphics g) {
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int cr = 12;
	    int x1a = (int) (x+dx*(dn/2-cr)/dn);
	    int y1a = (int) (y+dy*(dn/2-cr)/dn);
	    int x2a = (int) (x+dx*(dn/2+cr)/dn);
	    int y2a = (int) (y+dy*(dn/2+cr)/dn);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x1a, y1a);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, x2a, y2a, x2, y2);
	    setVoltageColor(g, (volts[0]+volts[1])/2);
	    setPowerColor(g, false);
	    int xm = (x+x2)/2;
	    int ym = (y+y2)/2;
	    drawThickCircle(g, xm, ym, cr);
	    int ar = cr/2;
	    int x1b = (int) (x+dx*(dn/2+ar)/dn);
	    int y1b = (int) (y+dy*(dn/2+ar)/dn);
	    int x2b = (int) (x+dx*(dn/2-ar)/dn);
	    int y2b = (int) (y+dy*(dn/2-ar)/dn);
	    drawThickLine(g, x1b, y1b, x2b, y2b);

	    int arrowPointsX[] = new int[3];
	    int arrowPointsY[] = new int[3];
	    double l = Math.sqrt((xm-x1b)*(xm-x1b)+(ym-y1b)*(ym-y1b));
	    double hatx = (x1b-xm)/l;
	    double haty = (y1b-ym)/l;
	    int ax = x1b;
	    int ay = y1b;
	    final double as = 4;
	    arrowPointsX[0] = ax;   arrowPointsY[0] = ay;
	    arrowPointsX[1] = (int) (ax+haty*as-hatx*as);
	    arrowPointsY[1] = (int) (ay-hatx*as-haty*as);
	    arrowPointsX[2] = (int) (ax-haty*as-hatx*as);
	    arrowPointsY[2] = (int) (ay+hatx*as-haty*as);
	    int i, j;
	    for (i = 0; i != 4; i++) {
		for (j = 0; j != 3; j++) {
		    if (i == 1)
			arrowPointsX[j]++;
		    else if (i == 2)
			arrowPointsY[j]++;
		    else if (i == 3)
			arrowPointsX[j]--;
		}
		g.fillPolygon(arrowPointsX, arrowPointsY, 3);
	    }
	    setBbox(x, y, x2, y2);
	    adjustBbox(xm-cr, ym-cr, xm+cr, ym+cr);
	    doDots(g);
	    drawPosts(g);
	}
	void stamp() {
	    current = currentValue;
	    stampCurrentSource(nodes[0], nodes[1], current);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Corriente (A)", currentValue, 0, .1);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    currentValue = ei.value;
	}
	void getInfo(String arr[]) {
	    arr[0] = "fuente de corriente";
	    getBasicInfo(arr);
	}
	double getVoltageDiff() {
	    return volts[1] - volts[0];
	}
    }

    class PushSwitchElm extends SwitchElm {
	public PushSwitchElm(int xx, int yy) { super(xx, yy, true); }
	Class getDumpClass() { return SwitchElm.class; }
    }
    
    class SwitchElm extends CircuitElm {
	boolean open, momentary;
	int switchx1, switchx2, switchy1, switchy2;
	public SwitchElm(int xx, int yy) {
	    super(xx, yy);
	    open = momentary = false;
	}
	SwitchElm(int xx, int yy, boolean mm) {
	    super(xx, yy);
	    open = mm;
	    momentary = mm;
	}
	public SwitchElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    open = new Boolean(st.nextToken()).booleanValue();
	    momentary = new Boolean(st.nextToken()).booleanValue();
	}
	int getDumpType() { return 's'; }
	String dump() {
	    return "s " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + open + " " + momentary;
	}
	void draw(Graphics g) {
	    drawSwitch(g, x, y, x2, y2, volts[0], volts[1], open);
	    if (!open)
		doDots(g);
	    drawPosts(g);
	}
	void calculateCurrent() {
	    if (open)
		current = 0;
	}
	void stamp() {
	    if (!open)
		stampVoltageSource(nodes[0], nodes[1], voltSource, 0);
	}
	int getVoltageSourceCount() {
	    return (open) ? 0 : 1;
	}
	void drawSwitch(Graphics g, int x1, int y1, int x2, int y2,
			double v1, double v2, boolean open) {
	    int i;
	    int ox = 0;
	    int dx = x2-x1;
	    int dy = y2-y1;
	    int openhs = 16;
	    int hs = (open) ? openhs : 0;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    setBbox(x, y, x2, y2);
	    if (dn > 40) {
		int x1a = (int) (x1+dx*(dn/2-16)/dn);
		int y1a = (int) (y1+dy*(dn/2-16)/dn);
		int x2a = (int) (x1+dx*(dn/2+16)/dn);
		int y2a = (int) (y1+dy*(dn/2+16)/dn);
		setVoltageColor(g, v1);
		drawThickLine(g, x1, y1, x1a, y1a);
		setVoltageColor(g, v2);
		drawThickLine(g, x2a, y2a, x2, y2);
		dn = 32;
		x1 = x1a; y1 = y1a;
		x2 = x2a; y2 = y2a;
		dx = x2-x1;
		dy = y2-y1;
	    }
	    switchx1 = x1;
	    switchx2 = x2;
	    switchy1 = y1;
	    switchy2 = y2;
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    if (mouseElm != this)
		g.setColor(Color.white);
	    drawThickLine(g, x1, y1, x2+dpx, y2+dpy);
	    int dpx0 = (int) ( openhs*dy/dn);
	    int dpy0 = (int) (-openhs*dx/dn);
	    adjustBbox(x1, y1, x2+dpx0, y2+dpy0);
	    adjustBbox(x1, y1, x2-dpx0/2, y2-dpy0/2);
	}
	void mouseUp() {
	    if (momentary)
		toggle();
	}
	void toggle() { open = !open; }
	void getInfo(String arr[]) {
	    arr[0] = (momentary) ? "pulse interruptor (SPST)" : "interruptor (SPST)";
	    if (open) {
		arr[1] = "abierto";
		arr[2] = "Vd = " + getVoltageDText(getVoltageDiff());
	    } else {
		arr[1] = "cerrado";
		arr[2] = "I = " + getCurrentDText(getCurrent());
	    }
	}
	boolean getConnection(int n1, int n2) { return !open; }
	boolean isWire() { return true; }
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Interruptor Momentneo", momentary);
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		momentary = ei.checkbox.getState();
	}
    }

    // DPST switch
    class Switch2Elm extends SwitchElm {
	int link;
	
	public Switch2Elm(int xx, int yy) {
	    super(xx, yy, false);
	    noDiagonal = true;
	}
	Switch2Elm(int xx, int yy, boolean mm) {
	    super(xx, yy, mm);
	    noDiagonal = true;
	}
	public Switch2Elm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    link = new Integer(st.nextToken()).intValue();
	    noDiagonal = true;
	}
	int getDumpType() { return 'S'; }
	String dump() {
	    return "S " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + open + " " + momentary + " " +
		link;
	}
	// posts
	int x3a, y3a, x3b, y3b;
	// switch poles
	int spxa, spya, spxb, spyb;
	void draw(Graphics g) {
	    drawSwitch(g, x, y, x2, y2, volts[0], volts[1], volts[2], open);
	    updateDotCount();
	    drawDots(g, x, y, switchx1, switchy1, curcount);
	    if (!open)
		drawDots(g, spxa, spya, x3a, y3a, curcount);
	    else
		drawDots(g, spxb, spyb, x3b, y3b, curcount);
	    drawPost(g, x, y, nodes[0]);
	    drawPost(g, x3a, y3a, nodes[1]);
	    drawPost(g, x3b, y3b, nodes[2]);
	}
	Point getPost(int n) {
	    return (n == 0) ? new Point(x, y) :
		(n == 1) ? new Point(x3a, y3a) : new Point(x3b, y3b);
	}
	int getPostCount() { return 3; }
	void setPoints() {
	    int hs = 16;
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    x3a = x2+dpx;
	    y3a = y2+dpy;
	    x3b = x2-dpx;
	    y3b = y2-dpy;
	}
	void calculateCurrent() {
	    // XXX
	}
	void stamp() {
	    if (!open)
		stampVoltageSource(nodes[0], nodes[1], voltSource, 0);
	    else
		stampVoltageSource(nodes[0], nodes[2], voltSource, 0);
	}
	int getVoltageSourceCount() {
	    return 1;
	}
	void drawSwitch(Graphics g, int x1, int y1, int x2, int y2,
			double v1, double v2, double v3, boolean open) {
	    int i;
	    int ox = 0;
	    int dx = x2-x1;
	    int dy = y2-y1;
	    int hs = 16;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    setBbox(x1, y1, x2, y2);
	    if (dn > 40) {
		int x1a = (int) (x1+dx*(dn/2-16)/dn);
		int y1a = (int) (y1+dy*(dn/2-16)/dn);
		int x2a = (int) (x1+dx*(dn/2+16)/dn);
		int y2a = (int) (y1+dy*(dn/2+16)/dn);
		setVoltageColor(g, v1);
		drawThickLine(g, x1, y1, x1a, y1a);
		setVoltageColor(g, v2);
		spxa = x2a+dpx;
		spya = y2a+dpy;
		spxb = x2a-dpx;
		spyb = y2a-dpy;
		drawThickLine(g, spxa, spya, x2+dpx, y2+dpy);
		setVoltageColor(g, v3);
		drawThickLine(g, spxb, spyb, x2-dpx, y2-dpy);
		dn = 32;
		x1 = x1a; y1 = y1a;
		x2 = x2a; y2 = y2a;
		dx = x2-x1;
		dy = y2-y1;
	    }
	    switchx1 = x1;
	    switchx2 = x2;
	    switchy1 = y1;
	    switchy2 = y2;
	    g.setColor(Color.white);
	    int dir = (open) ? -1 : 1;
	    drawThickLine(g, x1, y1, x2+dpx*dir, y2+dpy*dir);
	    adjustBbox(x1, y1, x2+dpx, y2+dpy);
	    adjustBbox(x1, y1, x2-dpx, y2-dpy);
	}
	void toggle() {
	    open = !open;
	    if (link != 0) {
		int i;
		for (i = 0; i != elmList.size(); i++) {
		    Object o = elmList.elementAt(i);
		    if (o instanceof Switch2Elm) {
			Switch2Elm s2 = (Switch2Elm) o;
			if (s2.link == link)
			    s2.open = open;
		    }
		}
	    }
	}
	boolean getConnection(int n1, int n2) {
	    if (comparePair(n1, n2, 0, 2) && open)
		return true;
	    if (comparePair(n1, n2, 0, 1) && !open)
		return true;
	    return false;
	}
	void getInfo(String arr[]) {
	    arr[0] = (link == 0) ? "interruptor (DPST)" : "interruptor (DPDT)";
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	}
    }

    class DiodeElm extends CircuitElm {
	public DiodeElm(int xx, int yy) {
	    super(xx, yy);
	    setup();
	}
	public DiodeElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    setup();
	}
	boolean nonLinear() { return true; }
	void setup() {
	    vcrit = vt * Math.log(vt/(Math.sqrt(2)*leakage));
	    //System.out.println("vcrit " + vcrit); // fix
	}
	int getDumpType() { return 'd'; }
	void draw(Graphics g) {
	    drawDiode(g, x, y, x2, y2, volts[0], volts[1]);
	    doDots(g);
	    drawPosts(g);
	}
	void reset() {
	    lastvoltdiff = volts[0] = volts[1] = 0;
	}
	void drawDiode(Graphics g, int x1, int y1, int x2, int y2,
			double v1, double v2) {
	    int i;
	    int ox = 0;
	    int dx = x2-x1;
	    int dy = y2-y1;
	    int hs = 8;
	    setBbox(x1, y1, x2, y2);
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    if (dn > 20) {
		int x1a = (int) (x1+dx*(dn/2-8)/dn);
		int y1a = (int) (y1+dy*(dn/2-8)/dn);
		int x2a = (int) (x1+dx*(dn/2+8)/dn);
		int y2a = (int) (y1+dy*(dn/2+8)/dn);
		setVoltageColor(g, v1);
		drawThickLine(g, x1, y1, x1a, y1a);
		setVoltageColor(g, v2);
		drawThickLine(g, x2a, y2a, x2, y2);
		dn = 16;
		x1 = x1a; y1 = y1a;
		x2 = x2a; y2 = y2a;
		dx = x2-x1;
		dy = y2-y1;
	    }
	    setPowerColor(g, true);
	    setVoltageColor(g, v1);
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    xpoints[0] = x1+dpx; ypoints[0] = y1+dpy;
	    xpoints[1] = x1-dpx; ypoints[1] = y1-dpy;
	    xpoints[2] = x2; ypoints[2] = y2;
	    g.fillPolygon(xpoints, ypoints, 3);
	    setVoltageColor(g, v2);
	    drawThickLine(g, x2+dpx, y2+dpy, x2, y2);
	    drawThickLine(g, x2-dpx, y2-dpy, x2, y2);
	    adjustBbox(x2+dpx, y2+dpy, x2-dpx, y2-dpy);
	}
	double leakage = 1e-14; // was 1e-9;
	static final double vt = .025;
	static final double vdcoef = 1/vt; // *2?
	double lastvoltdiff;
	double vcrit;
	double limitStep(double vnew, double vold) {
	    // this is what spice does for limiting
	    double arg;
	    double oo = vnew;
	    
	    if (vnew > vcrit && Math.abs(vnew - vold) > (vt + vt)) {
		if(vold > 0) {
		    arg = 1 + (vnew - vold) / vt;
		    if(arg > 0) {
			vnew = vold + vt * Math.log(arg);
			double v0 = Math.log(1e-6/leakage)*vt;
			vnew = Math.max(v0, vnew);
			//System.out.println(oo + " " + vnew);
		    } else {
			vnew = vcrit;
		    }
		} else {
		    vnew = vt *Math.log(vnew/vt);
		}
		converged = false;
		//System.out.println(vnew + " " + oo + " " + vold);
	    }
	    return(vnew);
	}
	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	}
	void doStep() {
	    double voltdiff = volts[0] - volts[1];
	    // tried .01 here, didn't work w/ filtered full-wave rect
	    if (Math.abs(voltdiff-lastvoltdiff) > .1)
		converged = false;
	    voltdiff = limitStep(voltdiff, lastvoltdiff);
	    lastvoltdiff = voltdiff;
	    double eval = Math.exp(voltdiff*vdcoef);
	    // make diode linear with negative voltages; aids convergence
	    if (voltdiff < 0)
		eval = 1;
	    double geq = vdcoef*leakage*eval;
	    double nc = (eval-1)*leakage - geq*voltdiff;
	    //System.out.println("D " + voltdiff + " " + geq + " " + current + " " + lastvoltdiff);
	    stampConductance(nodes[0], nodes[1], geq);
	    stampCurrentSource(nodes[0], nodes[1], nc);
	}
	void calculateCurrent() {
	    double voltdiff = volts[0] - volts[1];
	    current = leakage*(Math.exp(voltdiff*vdcoef)-1);
	}
	void getInfo(String arr[]) {
	    arr[0] = "diodo";
	    arr[1] = "I = " + getCurrentText(getCurrent());
	    arr[2] = "Vd = " + getVoltageText(getVoltageDiff());
	    arr[3] = "P = " + getUnitText(getPower(), "W");
	}
    }

    class LEDElm extends DiodeElm {
	double colorR, colorG, colorB;
	public LEDElm(int xx, int yy) {
	    super(xx, yy);
	    leakage = 3e-37;
	    setup();
	    colorR = 1; colorG = colorB = 0;
	}
	public LEDElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    leakage = 3e-37;
	    setup();
	    colorR = new Double(st.nextToken()).doubleValue();
	    colorG = new Double(st.nextToken()).doubleValue();
	    colorB = new Double(st.nextToken()).doubleValue();
	}
	int getDumpType() { return 162; }
	String dump() {
	    return "162 " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + colorR + " " + colorG + " " + colorB;
	}
	
	void draw(Graphics g) {
	    if (this == mouseElm || this == dragElm) {
		drawDiode(g, x, y, x2, y2, volts[0], volts[1]);
		drawPosts(g);
		return;
	    }
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int cr = 12;
	    int x1a = (int) (x+dx*(dn/2-cr)/dn);
	    int y1a = (int) (y+dy*(dn/2-cr)/dn);
	    int x2a = (int) (x+dx*(dn/2+cr)/dn);
	    int y2a = (int) (y+dy*(dn/2+cr)/dn);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x1a, y1a);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, x2a, y2a, x2, y2);
	    g.setColor(Color.gray);
	    int xm = (x+x2)/2;
	    int ym = (y+y2)/2;
	    drawThickCircle(g, xm, ym, cr);
	    cr -= 3;
	    double w = 255*current/.01;
	    if (w > 255)
		w = 255;
	    Color cc = new Color((int) (colorR*w), (int) (colorG*w),
				 (int) (colorB*w));
	    g.setColor(cc);
	    g.fillOval(xm-cr, ym-cr, cr*2, cr*2);
	    setBbox(x, y, x2, y2);
	    adjustBbox(xm-cr, ym-cr, xm+cr, ym+cr);
	    updateDotCount();
	    drawDots(g, x, y, x1a, y1a, curcount);
	    drawDots(g, x2, y2, x2a, y2a, -curcount);
	    drawPosts(g);
	}
    
	void getInfo(String arr[]) {
	    super.getInfo(arr);
	    arr[0] = "LED";
	}
    }

    class NTransistorElm extends TransistorElm {
	public NTransistorElm(int xx, int yy) { super(xx, yy, false); }
	Class getDumpClass() { return TransistorElm.class; }
    }
    class PTransistorElm extends TransistorElm {
	public PTransistorElm(int xx, int yy) { super(xx, yy, true); }
	Class getDumpClass() { return TransistorElm.class; }
    }
    
    class TransistorElm extends CircuitElm {
	int pnp;
	TransistorElm(int xx, int yy, boolean pnpflag) {
	    super(xx, yy);
	    pnp = (pnpflag) ? -1 : 1;
	    setup();
	}
	public TransistorElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    pnp = new Integer(st.nextToken()).intValue();
	    try {
		lastvbe = new Double(st.nextToken()).doubleValue();
		lastvbc = new Double(st.nextToken()).doubleValue();
		volts[0] = 0;
		volts[1] = -lastvbe;
		volts[2] = -lastvbc;
	    } catch (Exception e) {
	    }
	    setup();
	}
	void setup() {
	    vcrit = vt * Math.log(vt/(Math.sqrt(2)*leakage));
	    noDiagonal = true;
	}
	boolean nonLinear() { return true; }
	void reset() {
	    volts[0] = volts[1] = volts[2] = 0;
	    lastvbc = lastvbe = 0;
	}
	int getDumpType() { return 't'; }
	String dump() {
	    return "t " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + pnp + " " + (volts[0]-volts[1]) + " " +
		(volts[0]-volts[2]);
	}
	int collx, colly, emitx, emity, midpx, midpy,
	    col2x, col2y, emi2x, emi2y;
	int arrowPointsX[], arrowPointsY[];
	int rectX, rectY, rectW, rectH;
	double ic, ie, ib, curcount_c, curcount_e, curcount_b;
	void draw(Graphics g) {
	    if (arrowPointsX == null)
		return;
	    setBbox(x, y, collx, colly);
	    adjustBbox(x, y, emitx, emity);
	    setPowerColor(g, true);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, col2x, col2y, collx, colly);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, emi2x, emi2y, emitx, emity);
	    g.setColor(Color.lightGray);
	    g.fillPolygon(arrowPointsX, arrowPointsY, 3);
	    setVoltageColor(g, volts[0]);
	    if (powerCheckItem.getState())
		g.setColor(Color.gray);
	    drawThickLine(g, midpx, midpy, x , y );
	    curcount_b = updateDotCount(-ib, curcount_b);
	    drawDots(g, midpx, midpy, x , y , curcount_b);
	    curcount_c = updateDotCount(-ic, curcount_c);
	    drawDots(g, col2x, col2y, collx, colly, curcount_c);
	    curcount_e = updateDotCount(-ie, curcount_e);
	    drawDots(g, emi2x, emi2y, emitx, emity, curcount_e);
	    setVoltageColor(g, volts[0]);
	    setPowerColor(g, true);
	    g.fillRect(rectX, rectY, rectW, rectH);
	    drawPost(g, x, y, nodes[0]);
	    drawPost(g, collx, colly, nodes[1]);
	    drawPost(g, emitx, emity, nodes[2]);
	}
	Point getPost(int n) {
	    return (n == 0) ? new Point(x, y) :
		(n == 1) ? new Point(collx, colly) : new Point(emitx, emity);
	}
	int getPostCount() { return 3; }
	double getPower() {
	    return (volts[0]-volts[2])*ib + (volts[1]-volts[2])*ic;
	}
	void setPoints() {
	    int hs = 16;
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( Math.abs(hs*dy/dn));
	    int dpy = pnp*(int) (-Math.abs(hs*dx/dn));
	    collx = x2+dpx;
	    colly = y2+dpy;
	    emitx = x2-dpx;
	    emity = y2-dpy;

	    midpx = x;
	    midpy = y;
	    if (dn > 16) {
		midpx = (int) (x+dx*(dn-16)/dn);
		midpy = (int) (y+dy*(dn-16)/dn);
	    }
	    int bar1x = midpx+dpx;
	    int bar1y = midpy+dpy;
	    int bar2x = midpx-dpx;
	    int bar2y = midpy-dpy;
	    if (dy == 0) {
		rectX = bar1x-2;
		rectY = min(bar1y, bar2y);
		rectW = bar2x-bar1x+5;
		rectH = pnp*(bar2y-bar1y);
	    } else {
		rectX = bar2x;
		rectY = bar1y-2;
		rectW = bar1x-bar2x;
		rectH = bar2y-bar1y+5;
	    }
	    col2x = (bar1x*2+bar2x)/3;
	    col2y = (bar1y*2+bar2y)/3;
	    emi2x = (bar1x+bar2x*2)/3;
	    emi2y = (bar1y+bar2y*2)/3;
	    arrowPointsX = new int[3];
	    arrowPointsY = new int[3];
	    double l = Math.sqrt((emi2x-emitx)*(emi2x-emitx)+
					   (emi2y-emity)*(emi2y-emity));
	    double hatx = pnp*(emitx-emi2x)/l;
	    double haty = pnp*(emity-emi2y)/l;
	    int ax = (pnp == -1) ? (emi2x*4+emitx)/5 : emitx;
	    int ay = (pnp == -1) ? (emi2y*4+emity)/5 : emity;
	    final double as = 7;
	    arrowPointsX[0] = ax; arrowPointsY[0] = ay;
	    arrowPointsX[1] = (int) (ax+haty*as-hatx*as);
	    arrowPointsY[1] = (int) (ay-hatx*as-haty*as);
	    arrowPointsX[2] = (int) (ax-haty*as-hatx*as);
	    arrowPointsY[2] = (int) (ay+hatx*as-haty*as);
	}
	static final double leakage = 1e-13; // 1e-6;
	static final double vt = .025;
	static final double vdcoef = 1/vt;
	static final double fgain = .99;
	static final double rgain = .5;
	double vcrit;
	double lastvbc, lastvbe;
	double limitStep(double vnew, double vold) {
	    // this is what spice does for limiting
	    double arg;
	    double oo = vnew;
	    
	    if (vnew > vcrit && Math.abs(vnew - vold) > (vt + vt)) {
		if(vold > 0) {
		    arg = 1 + (vnew - vold) / vt;
		    if(arg > 0) {
			vnew = vold + vt * Math.log(arg);
		    } else {
			vnew = vcrit;
		    }
		} else {
		    vnew = vt *Math.log(vnew/vt);
		}
		converged = false;
		//System.out.println(vnew + " " + oo + " " + vold);
	    }
	    return(vnew);
	}
	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	    stampNonLinear(nodes[2]);
	}
	void doStep() {
	    double vbc = volts[0]-volts[1]; // typically negative
	    double vbe = volts[0]-volts[2]; // typically positive
	    if (Math.abs(vbc-lastvbc) > .01 || // .01
		Math.abs(vbe-lastvbe) > .01)
		converged = false;
	    //if (subIterations > 300 && subIterations < 330)
	    //System.out.print("T " + vbc + " " + vbe + "\n");
	    vbc = pnp*limitStep(pnp*vbc, pnp*lastvbc);
	    vbe = pnp*limitStep(pnp*vbe, pnp*lastvbe);
	    lastvbc = vbc;
	    lastvbe = vbe;
	    double pcoef = vdcoef*pnp;
	    double expbc = Math.exp(vbc*pcoef);
	    /*if (expbc > 1e13 || Double.isInfinite(expbc))
	      expbc = 1e13;*/
	    double expbe = Math.exp(vbe*pcoef);
	    if (expbe < 1)
		expbe = 1;
	    /*if (expbe > 1e13 || Double.isInfinite(expbe))
	      expbe = 1e13;*/
	    ie = pnp*leakage*(-(expbe-1)+rgain*(expbc-1));
	    ic = pnp*leakage*(fgain*(expbe-1)-(expbc-1));
	    ib = -(ie+ic);
	    //System.out.println("gain " + ic/ib);
	    //System.out.print("T " + vbc + " " + vbe + " " + ie + " " + ic + "\n");
	    double gee = -leakage*vdcoef*expbe;
	    double gec = rgain*leakage*vdcoef*expbc;
	    double gce = -gee*fgain;
	    double gcc = -gec*(1/rgain);
	    /*System.out.print("gee = " + gee + "\n");
	    System.out.print("gec = " + gec + "\n");
	    System.out.print("gce = " + gce + "\n");
	    System.out.print("gcc = " + gcc + "\n");
	    System.out.print("gce+gcc = " + (gce+gcc) + "\n");
	    System.out.print("gee+gec = " + (gee+gec) + "\n");*/
	    
	    // stamps from page 302 of Pillage.  Node 0 is the base,
	    // node 1 the collector, node 2 the emitter
	    stampMatrix(nodes[0], nodes[0], -gee-gec-gce-gcc);
	    stampMatrix(nodes[0], nodes[1], gec+gcc);
	    stampMatrix(nodes[0], nodes[2], gee+gce);
	    stampMatrix(nodes[1], nodes[0], gce+gcc);
	    stampMatrix(nodes[1], nodes[1], -gcc);
	    stampMatrix(nodes[1], nodes[2], -gce);
	    stampMatrix(nodes[2], nodes[0], gee+gec);
	    stampMatrix(nodes[2], nodes[1], -gec);
	    stampMatrix(nodes[2], nodes[2], -gee);

	    // we are solving for v(k+1), not delta v, so we use formula
	    // 10.5.13, multiplying J by v(k)
	    stampRightSide(nodes[0], -ib - (gec+gcc)*vbc - (gee+gce)*vbe);
	    stampRightSide(nodes[1], -ic + gce*vbe + gcc*vbc);
	    stampRightSide(nodes[2], -ie + gee*vbe + gec*vbc);
	}
	void getInfo(String arr[]) {
	    arr[0] = "transistor (" + ((pnp == -1) ? "PNP)" : "NPN)");
	    double vbc = volts[0]-volts[1];
	    double vbe = volts[0]-volts[2];
	    double vce = volts[1]-volts[2];
	    arr[1] = "Vbe = " + getVoltageText(vbe);
	    arr[2] = "Vbc = " + getVoltageText(vbc);
	    arr[3] = "Vce = " + getVoltageText(vce);
	    arr[4] = "Ic = " + getCurrentText(ic);
	    arr[5] = "Ib = " + getCurrentText(ib);
	    if (vbc*pnp > .2)
		arr[6] = vbe*pnp > .2 ? "saturacin" : "inversin activa";
	    else
		arr[6] = vbe*pnp > .2 ? "fwd activo" : "corte";
	}
	double getScopeValue(int x) {
	    switch (x) {
	    case SCOPEVAL_IB: return ib;
	    case SCOPEVAL_IC: return ic;
	    case SCOPEVAL_IE: return ie;
	    case SCOPEVAL_VBE: return volts[0]-volts[2];
	    case SCOPEVAL_VBC: return volts[0]-volts[1];
	    case SCOPEVAL_VCE: return volts[1]-volts[2];
	    }
	    return 0;
	}
	String getScopeUnits(int x) {
	    switch (x) {
	    case SCOPEVAL_IB: case SCOPEVAL_IC:
	    case SCOPEVAL_IE: return "A";
	    default: return "V";
	    }
	}
	boolean canViewInScope() { return true; }
    }

    class Scope {
	double minV[], maxV[], minMaxV;
	double minI[], maxI[], minMaxI;
	int scopePointCount = 128;
	int ptr, ctr, speed, position;
	int value;
	String text;
	Rectangle rect;
	boolean showI, showV, showMax, showFreq, lockScale;
	CircuitElm elm;
	Scope() {
	    rect = new Rectangle();
	    reset();
	}
	void showCurrent(boolean b) { showI = b; value = 0; }
	void showVoltage(boolean b) { showV = b; value = 0; }
	void showMax    (boolean b) { showMax = b; }
	void showFreq   (boolean b) { showFreq = b; }
	void setLockScale  (boolean b) { lockScale = b; }
	void resetGraph() {
	    scopePointCount = 1;
	    while (scopePointCount < rect.width)
		scopePointCount *= 2;
	    minV = new double[scopePointCount];
	    maxV = new double[scopePointCount];
	    minI = new double[scopePointCount];
	    maxI = new double[scopePointCount];
	    ptr = ctr = 0;
	}
	boolean active() { return elm != null; }
	void reset() {
	    resetGraph();
	    minMaxV = 5;
	    minMaxI = .1;
	    speed = 64;
	    showI = showV = showMax = true;
	    showFreq = lockScale = false;
	    // no showI for Output
	    if (elm != null && (elm instanceof OutputElm ||
				elm instanceof LogicOutputElm ||
				elm instanceof ProbeElm))
		showI = false;
	    value = 0;
	    if (elm instanceof TransistorElm)
		value = SCOPEVAL_VCE;
	}
	void setRect(Rectangle r) {
	    rect = r;
	    resetGraph();
	}
	int getWidth() { return rect.width; }
	int rightEdge() { return rect.x+rect.width; }
	
	void setElm(CircuitElm ce) {
	    elm = ce;
	    reset();
	}
	
	void timeStep() {
	    if (elm == null)
		return;
	    double v = elm.getScopeValue(value);
	    if (v < minV[ptr])
		minV[ptr] = v;
	    if (v > maxV[ptr])
		maxV[ptr] = v;
	    double i = 0;
	    if (value == 0) {
		i = elm.getCurrent();
		if (i < minI[ptr])
		    minI[ptr] = i;
		if (i > maxI[ptr])
		    maxI[ptr] = i;
	    }
	    ctr++;
	    if (ctr >= speed) {
		ptr = (ptr+1) & (scopePointCount-1);
		minV[ptr] = maxV[ptr] = v;
		minI[ptr] = maxI[ptr] = i;
		ctr = 0;
	    }
	}

	void adjustScale(double x) {
	    minMaxV *= x;
	    minMaxI *= x;
	}
	
	void draw(Graphics g) {
	    //System.out.println(this + " " + g);
	    if (elm == null)
		return;
	    int i;
	    int x = rect.x;
	    int y = rect.y;
	    int maxy = (rect.height-1)/2;
	    y += maxy;
	    g.setColor(Color.gray);
	    g.drawLine(x, y, x+rect.width-1, y);
	    boolean gotI = false;
	    boolean gotV = false;
	    int minRange = 4;
	    double realMaxV = -1e8;
	    double realMaxI = -1e8;
	    Color curColor = Color.yellow;
	    Color voltColor = (value > 0) ? Color.white :
		colorScale[colorScaleCount-1];
	    if (scopeSelected == -1 && elm == mouseElm)
		if (value == 0 && !showV)
		    curColor = Color.cyan;
		else
		    voltColor = Color.cyan;
	    int ipa = ptr+scopePointCount-rect.width;
	    for (i = 0; i != rect.width; i++) {
		int ip = (i+ipa) & (scopePointCount-1);
		while (maxV[ip] > minMaxV)
		    minMaxV *= 2;
		while (minV[ip] < -minMaxV)
		    minMaxV *= 2;
		while (maxI[ip] > minMaxI)
		    minMaxI *= 2;
		while (minI[ip] < -minMaxI)
		    minMaxI *= 2;
	    }
	    
	    // these two loops are pretty much the same, and should be
	    // combined!
	    if (value == 0 && showI) {
		g.setColor(curColor);
		int ox = -1, oy = -1;
		for (i = 0; i != rect.width; i++) {
		    int ip = (i+ipa) & (scopePointCount-1);
		    int miniy = (int) ((maxy/minMaxI)*minI[ip]);
		    int maxiy = (int) ((maxy/minMaxI)*maxI[ip]);
		    if (maxI[ip] > realMaxI)
			realMaxI = maxI[ip];
		    if (miniy <= maxy) {
			if (miniy < -minRange || maxiy > minRange)
			    gotI = true;
			if (ox != -1) {
			    if (miniy == oy && maxiy == oy)
				continue;
			    g.drawLine(ox, y-oy, x+i-1, y-oy);
			    ox = oy = -1;
			}
			if (miniy == maxiy) {
			    ox = x+i;
			    oy = miniy;
			    continue;
			}
			g.drawLine(x+i, y-miniy, x+i, y-maxiy);
		    }
		}
		if (ox != -1)
		    g.drawLine(ox, y-oy, x+i-1, y-oy);
	    }
	    if (value != 0 || showV) {
		g.setColor(voltColor);
		int ox = -1, oy = -1;
		for (i = 0; i != rect.width; i++) {
		    int ip = (i+ipa) & (scopePointCount-1);
		    int minvy = (int) ((maxy/minMaxV)*minV[ip]);
		    int maxvy = (int) ((maxy/minMaxV)*maxV[ip]);
		    if (maxV[ip] > realMaxV)
			realMaxV = maxV[ip];
		    if ((value != 0 || showV) && minvy <= maxy) {
			if (minvy < -minRange || maxvy > minRange)
			    gotV = true;
			if (ox != -1) {
			    if (minvy == oy && maxvy == oy)
				continue;
			    g.drawLine(ox, y-oy, x+i-1, y-oy);
			    ox = oy = -1;
			}
			if (minvy == maxvy) {
			    ox = x+i;
			    oy = minvy;
			    continue;
			}
			g.drawLine(x+i, y-minvy, x+i, y-maxvy);
		    }
		}
		if (ox != -1)
		    g.drawLine(ox, y-oy, x+i-1, y-oy);
	    }
	    double freq = 0;
	    if (showFreq) {
		// try to get frequency
		// get average
		double avg = 0;
		for (i = 0; i != rect.width; i++) {
		    int ip = (i+ipa) & (scopePointCount-1);
		    avg += minV[ip]+maxV[ip];
		}
		avg /= i*2;
		int state = 0;
		double thresh = avg*.05;
		int oi = 0;
		double avperiod = 0;
		int periodct = -1;
		double avperiod2 = 0;
		// count period lengths
		for (i = 0; i != rect.width; i++) {
		    int ip = (i+ipa) & (scopePointCount-1);
		    double q = maxV[ip]-avg;
		    int os = state;
		    if (q < thresh)
			state = 1;
		    else if (q > -thresh)
			state = 2;
		    if (state == 2 && os == 1) {
			int pd = i-oi;
			oi = i;
			// short periods can't be counted properly
			if (pd < 12)
			    continue;
			// skip first period, it might be too short
			if (periodct >= 0) {
			    avperiod += pd;
			    avperiod2 += pd*pd;
			}
			periodct++;
		    }
		}
		avperiod /= periodct;
		avperiod2 /= periodct;
		double periodstd = Math.sqrt(avperiod2-avperiod*avperiod);
		freq = 1/(avperiod*timeStep*speed);
		// don't show freq if standard deviation is too great
		if (periodct < 1 || periodstd > 2)
		    freq = 0;
		// System.out.println(freq + " " + periodstd + " " + periodct);
	    }
	    g.setColor(Color.white);
	    int yt = rect.y+10;
	    if (showMax) {
		if (value != 0)
		    g.drawString(getUnitText(realMaxV,
					     elm.getScopeUnits(value)),
				 x, yt);
		else if (showV)
		    g.drawString(getVoltageText(realMaxV), x, yt);
		else if (showI)
		    g.drawString(getCurrentText(realMaxI), x, yt);
		yt += 15;
	    }
	    if (text != null && rect.y + rect.height > yt+5) {
		g.drawString(text, x, yt);
		yt += 15;
	    }
	    if (showFreq && freq != 0 && rect.y + rect.height > yt+5)
		g.drawString(getUnitText(freq, "Hz"), x, yt);
	    if (ptr > 5 && !lockScale) {
		if (!gotI && minMaxI > 1e-4)
		    minMaxI /= 2;
		if (!gotV && minMaxV > 1e-4)
		    minMaxV /= 2;
	    }
	}
	
	void speedUp() {
	    if (speed > 1) {
		speed /= 2;
		resetGraph();
	    }
	}
	void slowDown() {
	    speed *= 2;
	    resetGraph();
	}
	
	PopupMenu getMenu() {
	    if (elm == null)
		return null;
	    if (elm instanceof TransistorElm) {
		scopeIbMenuItem.setState(value == SCOPEVAL_IB);
		scopeIcMenuItem.setState(value == SCOPEVAL_IC);
		scopeIeMenuItem.setState(value == SCOPEVAL_IE);
		scopeVbeMenuItem.setState(value == SCOPEVAL_VBE);
		scopeVbcMenuItem.setState(value == SCOPEVAL_VBC);
		scopeVceMenuItem.setState(value == SCOPEVAL_VCE);
		return transScopeMenu;
	    } else {
		scopeVMenuItem    .setState(showV && value == 0);
		scopeIMenuItem    .setState(showI && value == 0);
		scopeMaxMenuItem  .setState(showMax);
		scopeFreqMenuItem .setState(showFreq);
		scopePowerMenuItem.setState(value == SCOPEVAL_POWER);
		return scopeMenu;
	    }
	}
	void setValue(int x) { reset(); value = x; }
	String dump() {
	    if (elm == null)
		return null;
	    int flags = (showI ? 1 : 0) | (showV ? 2 : 0) |
		(showMax ? 0 : 4) | (showFreq ? 8 : 0) |
		(lockScale ? 16 : 0);
	    int eno = locateElm(elm);
	    if (eno < 0)
		return null;
	    String x = "o " + eno + " " +
		speed + " " + value + " " + flags + " " +
		minMaxV + " " + minMaxI + " " + position;
	    if (text != null)
		x += " " + text;
	    return x;
	}
	void undump(StringTokenizer st) {
	    reset();
	    int e = new Integer(st.nextToken()).intValue();
	    if (e == -1)
		return;
	    elm = getElm(e);
	    speed = new Integer(st.nextToken()).intValue();
	    value = new Integer(st.nextToken()).intValue();
	    int flags = new Integer(st.nextToken()).intValue();
	    minMaxV = new Double(st.nextToken()).doubleValue();
	    minMaxI = new Double(st.nextToken()).doubleValue();
	    if (minMaxV == 0)
		minMaxV = .5;
	    if (minMaxI == 0)
		minMaxI = 1;
	    text = null;
	    try {
		position = new Integer(st.nextToken()).intValue();
		while (st.hasMoreTokens()) {
		    if (text == null)
			text = st.nextToken();
		    else
			text += " " + st.nextToken();
		}
	    } catch (Exception ee) {
	    }
	    showI = (flags & 1) != 0;
	    showV = (flags & 2) != 0;
	    showMax = (flags & 4) == 0;
	    showFreq = (flags & 8) != 0;
	    lockScale = (flags & 16) != 0;
	}
    }

    class NMosfetElm extends MosfetElm {
	public NMosfetElm(int xx, int yy) { super(xx, yy, false); }
	Class getDumpClass() { return MosfetElm.class; }
    }
    class PMosfetElm extends MosfetElm {
	public PMosfetElm(int xx, int yy) { super(xx, yy, true); }
	Class getDumpClass() { return MosfetElm.class; }
    }
    class MosfetElm extends CircuitElm {
	int pnp;
	int FLAG_PNP = 1;
	MosfetElm(int xx, int yy, boolean pnpflag) {
	    super(xx, yy);
	    pnp = (pnpflag) ? -1 : 1;
	    flags = (pnpflag) ? FLAG_PNP : 0;
	    noDiagonal = true;
	}
	public MosfetElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    pnp = ((f & FLAG_PNP) != 0) ? -1 : 1;
	    noDiagonal = true;
	}
	double getThreshold() { return 1.5; }
	double getBeta() { return .02; }
	boolean nonLinear() { return true; }
	void reset() {
	    lastv1 = lastv2 = volts[0] = volts[1] = volts[2] = 0;
	}
	int getDumpType() { return 'f'; }
	// post coordinates
	int src_px, src_py, drn_px, drn_py;
	// midpoint coordinates
	int src_mx, src_my, drn_mx, drn_my;
	// gate coordinates
	int gate1x, gate1y, gate2x, gate2y, gate3x, gate3y;
	int pcirclex, pcircley, pcircler;
	void draw(Graphics g) {
	    setBbox   (x, y, drn_px, drn_py);
	    adjustBbox(x, y, src_px, src_py);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, src_px, src_py, src_mx, src_my);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, drn_mx, drn_my, drn_px, drn_py);
	    int segments = 6;
	    int i;
	    setPowerColor(g, true);
	    for (i = 0; i != segments; i++) {
		double v = volts[1]+(volts[2]-volts[1])*i/segments;
		setVoltageColor(g, v);
		int xa = src_mx + (drn_mx-src_mx)*i/segments;
		int ya = src_my + (drn_my-src_my)*i/segments;
		int xb = src_mx + (drn_mx-src_mx)*(i+1)/segments;
		int yb = src_my + (drn_my-src_my)*(i+1)/segments;
		drawThickLine(g, xa, ya, xb, yb);
	    }
	    if (powerCheckItem.getState())
		g.setColor(Color.gray);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, gate2x, gate2y);
	    drawThickLine(g, gate1x, gate1y, gate3x, gate3y);
	    if (pnp == -1)
		drawThickCircle(g, pcirclex, pcircley, pcircler);
	    curcount = updateDotCount(-ids, curcount);
	    drawDots(g, src_px, src_py, src_mx, src_my, curcount);
	    drawDots(g, src_mx, src_my, drn_mx, drn_my, curcount);
	    drawDots(g, drn_mx, drn_my, drn_px, drn_py, curcount);
	    drawPost(g, x, y, nodes[0]);
	    drawPost(g, src_px, src_py, nodes[1]);
	    drawPost(g, drn_px, drn_py, nodes[2]);
	}
	Point getPost(int n) {
	    return (n == 0) ? new Point(x, y) :
		(n == 1) ? new Point(src_px, src_py) :
		new Point(drn_px, drn_py);
	}
	double getCurrent() { return ids; }
	double getPower() { return ids*(volts[2]-volts[1]); }
	int getPostCount() { return 3; }
	void setPoints() {
	    // find the coordinates of the various points we need to draw
	    // the MOSFET.
	    int hs = 16;
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( Math.abs(hs*dy/dn));
	    int dpy = (int) (-Math.abs(hs*dx/dn));
	    src_px = x2-dpx;
	    src_py = y2-dpy;
	    drn_px = x2+dpx;
	    drn_py = y2+dpy;

	    int dax = (int) (12*dx/dn);
	    int day = (int) (12*dy/dn);
	    src_mx = src_px-dax;
	    src_my = src_py-day;
	    drn_mx = drn_px-dax;
	    drn_my = drn_py-day;
	    dax = (int) (8*dx/dn);
	    day = (int) (8*dy/dn);
	    gate1x = src_mx-dax;
	    gate1y = src_my-day;
	    gate3x = drn_mx-dax;
	    gate3y = drn_my-day;
	    gate2x = (gate1x+gate3x)/2;
	    gate2y = (gate1y+gate3y)/2;
	    if (pnp == -1) {
		gate2x -= dax;
		gate2y -= day;
		pcirclex = gate2x+dax/2+day/8;
		pcircley = gate2y+day/2+dax/8;
		pcircler = (int) Math.abs(dax/2 + day/2) - 1;
		if (dx < 0 || dy < 0) {
		    pcirclex++;
		    pcircley++;
		}
	    }
	}

	double lastv1, lastv2;
	double ids;
	int mode = 0;
	double gm = 0;
	
	void stamp() {
	    stampNonLinear(nodes[1]);
	    stampNonLinear(nodes[2]);
	}
	void doStep() {
	    double vs[] = new double[3];
	    vs[0] = volts[0];
	    vs[1] = volts[1];
	    vs[2] = volts[2];
	    if (vs[1] > lastv1 + .5)
		vs[1] = lastv1 + .5;
	    if (vs[1] < lastv1 - .5)
		vs[1] = lastv1 - .5;
	    if (vs[2] > lastv2 + .5)
		vs[2] = lastv2 + .5;
	    if (vs[2] < lastv2 - .5)
		vs[2] = lastv2 - .5;
	    int source = 1;
	    int drain = 2;
	    if (pnp*vs[1] > pnp*vs[2]) {
		source = 2;
		drain = 1;
	    }
	    int gate = 0;
	    double vgs = vs[gate ]-vs[source];
	    double vds = vs[drain]-vs[source];
	    if (Math.abs(lastv1-vs[1]) > .01 ||
		Math.abs(lastv2-vs[2]) > .01)
		converged = false;
	    lastv1 = vs[1];
	    lastv2 = vs[2];
	    double realvgs = vgs;
	    double realvds = vds;
	    vgs *= pnp;
	    vds *= pnp;
	    ids = 0;
	    gm = 0;
	    double Gds = 0;
	    double vt = getThreshold();
	    double beta = getBeta();
	    if (vgs > .5 && this instanceof JfetElm) {
		stop("JFET est en oposicin!", this);
		return;
	    }
	    if (vgs < vt) {
		// should be all zero, but that causes a singular matrix,
		// so instead we treat it as a large resistor
		Gds = 1e-8;
		ids = vds*Gds;
		mode = 0;
	    } else if (vds < vgs-vt) {
		// linear
		ids = beta*((vgs-vt)*vds - vds*vds*.5);
		gm  = beta*vds;
		Gds = beta*(vgs-vds-vt);
		mode = 1;
	    } else {
		// saturation; Gds = 0
		gm  = beta*(vgs-vt);
		// use very small Gds to avoid nonconvergence
		Gds = 1e-8;
		ids = .5*beta*(vgs-vt)*(vgs-vt) + (vds-(vgs-vt))*Gds;
		mode = 2;
	    }
	    double rs = -pnp*ids + Gds*realvds + gm*realvgs;
	    //System.out.println("M " + vds + " " + vgs + " " + ids + " " + gm + " "+ Gds + " " + volts[0] + " " + volts[1] + " " + volts[2] + " " + source + " " + rs + " " + this);
	    stampMatrix(nodes[drain],  nodes[drain],  Gds);
	    stampMatrix(nodes[drain],  nodes[source], -Gds-gm); 
	    stampMatrix(nodes[drain],  nodes[gate],   gm);
	    
	    stampMatrix(nodes[source], nodes[drain],  -Gds);
	    stampMatrix(nodes[source], nodes[source], Gds+gm); 
	    stampMatrix(nodes[source], nodes[gate],  -gm);
	    
	    stampRightSide(nodes[drain],  rs);
	    stampRightSide(nodes[source], -rs);
	    if (source == 2 && pnp == 1 ||
		source == 1 && pnp == -1)
		ids = -ids;
	}
	void getFetInfo(String arr[], String n) {
	    arr[0] = ((pnp == -1) ? "p-" : "n-") + n;
	    arr[1] = "Ids = " + getCurrentText(ids);
	    arr[2] = "Vgs = " + getVoltageText(volts[0]-volts[1]);
	    arr[3] = "Vds = " + getVoltageText(volts[2]-volts[1]);
	    arr[4] = (mode == 0) ? "desc" :
		(mode == 1) ? "lineal" : "saturacin";
	    arr[5] = "gm = " + getUnitText(gm, "mho");
	}
	void getInfo(String arr[]) {
	    getFetInfo(arr, "MOSFET");
	}
	boolean canViewInScope() { return true; }
	double getVoltageDiff() { return volts[2] - volts[1]; }
	boolean getConnection(int n1, int n2) {
	    return !(n1 == 0 || n2 == 0);
	}
    }
    class NJfetElm extends JfetElm {
	public NJfetElm(int xx, int yy) { super(xx, yy, false); }
	Class getDumpClass() { return JfetElm.class; }
    }
    class JfetElm extends MosfetElm {
	JfetElm(int xx, int yy, boolean pnpflag) {
	    super(xx, yy, pnpflag);
	    noDiagonal = true;
	}
	public JfetElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    noDiagonal = true;
	}
	// post coordinates (gate post is x, y)
	// int src_px, src_py, drn_px, drn_py; (inherited)
	// midpoint coordinates
	// int src_mx, src_my, drn_mx, drn_my; (inherited)
	// gate coordinates
	int src_gx, src_gy, drn_gx, drn_gy, gategx, gategy;
	// gate rectangle
	int gaterx, gatery, gaterw, gaterh;
	int arrowPointsX[], arrowPointsY[];
	
	void draw(Graphics g) {
	    setBbox   (x, y, drn_px, drn_py);
	    adjustBbox(x, y, src_px, src_py);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, src_px, src_py, src_mx, src_my);
	    drawThickLine(g, src_mx, src_my, src_gx, src_gy);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, drn_mx, drn_my, drn_px, drn_py);
	    drawThickLine(g, drn_mx, drn_my, drn_gx, drn_gy);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, gategx, gategy);
	    g.fillPolygon(arrowPointsX, arrowPointsY, 3);
	    setPowerColor(g, true);
	    g.fillRect(gaterx, gatery, gaterw, gaterh);
	    curcount = updateDotCount(-ids, curcount);
	    if (curcount != 0) {
		drawDots(g, src_px, src_py, src_mx, src_my, curcount);
		drawDots(g, src_mx, src_my, src_gx, src_gy, curcount+8);
		drawDots(g, drn_px, drn_py, drn_mx, drn_my, -curcount);
		drawDots(g, drn_mx, drn_my, drn_gx, drn_gy, -(curcount+8));
	    }
	    
	    drawPost(g, x, y, nodes[0]);
	    drawPost(g, src_px, src_py, nodes[1]);
	    drawPost(g, drn_px, drn_py, nodes[2]);
	}
	void setPoints() {
	    // find the coordinates of the various points we need to draw
	    // the JFET.
	    int hs = 16;
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( Math.abs(hs*dy/dn));
	    int dpy = (int) (-Math.abs(hs*dx/dn));
	    src_px = x2-dpx;
	    src_py = y2-dpy;
	    drn_px = x2+dpx;
	    drn_py = y2+dpy;
	    int dpx2 = dpx/2;
	    int dpy2 = dpy/2;
	    src_mx = x2-dpx2;
	    src_my = y2-dpy2;
	    drn_mx = x2+dpx2;
	    drn_my = y2+dpy2;

	    /*int dax1 = (int) ((dn/2-8)*dx/dn);
	    int day1 = (int) ((dn/2-8)*dy/dn);
	    int dax2 = (int) ((dn/2+3)*dx/dn);
	    int day2 = (int) ((dn/2+3)*dy/dn);*/
	    int dax1 = (int) (10*dx/dn);
	    int day1 = (int) (10*dy/dn);
	    int dax2 = (int) ((dn-15)*dx/dn);
	    int day2 = (int) ((dn-15)*dy/dn);
	    src_gx = src_mx-dax1;
	    src_gy = src_my-day1;
	    drn_gx = drn_mx-dax1;
	    drn_gy = drn_my-day1;
	    gategx = x+dax2;
	    gategy = y+day2;

	    if (dy == 0) {
		gaterx = min(gategx, src_gx);
		gaterw = max(gategx, src_gx)-gaterx;
		gatery = min(src_py, drn_py);
		gaterh = max(src_py, drn_py)-gatery;
	    } else {
		gaterx = min(src_px, drn_px);
		gaterw = max(src_px, drn_px)-gaterx;
		gatery = min(gategy, src_gy);
		gaterh = max(gategy, src_gy)-gatery;
	    }

	    arrowPointsX = new int[3];
	    arrowPointsY = new int[3];
	    double l = Math.sqrt((gategx-x)*(gategx-x)+
				 (gategy-y)*(gategy-y));
	    double hatx = (gategx-x)/l;
	    double haty = (gategy-y)/l;
	    int ax = gategx;
	    int ay = gategy;
	    final double as = 7;
	    arrowPointsX[0] = ax; arrowPointsY[0] = ay;
	    arrowPointsX[1] = (int) (ax+haty*as-hatx*as);
	    arrowPointsY[1] = (int) (ay-hatx*as-haty*as);
	    arrowPointsX[2] = (int) (ax-haty*as-hatx*as);
	    arrowPointsY[2] = (int) (ay+hatx*as-haty*as);
	}
	int getDumpType() { return 'j'; }
	// these values are taken from Hayes+Horowitz p155
	double getThreshold() { return -4; }
	double getBeta() { return .00125; }
	void getInfo(String arr[]) {
	    getFetInfo(arr, "JFET");
	}
	/*EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("threshold", th, .01, 5);
	    if (n == 1)
		return new EditInfo("beta", bb, .01, 5);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		th = ei.value;
	    else
		bb = ei.value;
		}*/
    }

    class OpAmpSwapElm extends OpAmpElm {
	public OpAmpSwapElm(int xx, int yy) {
	    super(xx, yy);
	    flags |= FLAG_SWAP;
	}
	Class getDumpClass() { return OpAmpElm.class; }
    }
    
    class OpAmpElm extends CircuitElm {
	int opsize, opheight, opwidth, opaddtext;
	double maxOut, minOut;
	final int FLAG_SWAP = 1;
	final int FLAG_SMALL = 2;
	public OpAmpElm(int xx, int yy) {
	    super(xx, yy);
	    noDiagonal = true;
	    maxOut = 15;
	    minOut = -15;
	    setSize(smallGridCheckItem.getState() ? 1 : 2);
	}
	public OpAmpElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    try {
		maxOut = new Double(st.nextToken()).doubleValue();
		minOut = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
		maxOut = 15; minOut = -15;
	    }
	    noDiagonal = true;
	    setSize((f & FLAG_SMALL) != 0 ? 1 : 2);
	}
	String dump() {
	    return "a " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + maxOut + " " + minOut;
	}
	boolean nonLinear() { return true; }
	void draw(Graphics g) {
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, in1px, in1py, in1nx, in1ny);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, in2px, in2py, in2nx, in2ny);
	    g.setColor(this == mouseElm ? Color.cyan : Color.lightGray);
	    setPowerColor(g, true);
	    drawThickPolygon(g, triPointsX, triPointsY, 3);
	    Font f = new Font("SansSerif", 0, 10);
	    g.setFont(f);
	    g.setColor(this == mouseElm ? Color.cyan : Color.lightGray);
	    g.drawString("-", in1nx+5, in1ny+3-opaddtext);
	    g.drawString("+", in2nx+5, in2ny+5+opaddtext);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, outnx, outny, outpx, outpy);
	    curcount = updateDotCount(current, curcount);
	    drawDots(g, outpx, outpy, outnx, outny, curcount);
	    drawPost(g, in1px, in1py, nodes[0]);
	    drawPost(g, in2px, in2py, nodes[1]);
	    drawPost(g, outpx, outpy, nodes[2]);
	}
	double getPower() { return volts[2]*current; }
	int in1px, in1py, in2px, in2py, outpx, outpy;
	int in1nx, in1ny, in2nx, in2ny, outnx, outny;
	int triPointsX[], triPointsY[];
	void setSize(int s) {
	    opsize = s;
	    opheight = 8*s;
	    opwidth = 13*s;
	    flags = (flags & ~FLAG_SMALL) | ((s == 1) ? FLAG_SMALL : 0);
	}
	void setPoints() {
	    if (x2 == x)
		y2 = y;
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    if (dn > 150)
		setSize(2);
	    int hs = opheight;
	    int dpx = (int) ( Math.abs(hs*dy/dn));
	    int dpy = (int) (-Math.abs(hs*dx/dn));
	    if ((flags & FLAG_SWAP) != 0) {
		dpx = -dpx;
		dpy = -dpy;
	    }
	    opaddtext = (opsize == 1) ? dpy/2 : 0;
	    in1px = x + dpx;
	    in1py = y + dpy;
	    in2px = x - dpx;
	    in2py = y - dpy;
	    outpx = x2; outpy = y2;

	    int ww = opwidth; // 32;
	    if (ww > dn/2)
		ww = (int) (dn/2);
	    int x1a = (int) (x+dx*(dn/2-ww)/dn);
	    int y1a = (int) (y+dy*(dn/2-ww)/dn);
	    int x2a = (int) (x+dx*(dn/2+ww)/dn);
	    int y2a = (int) (y+dy*(dn/2+ww)/dn);
	    in1nx = x1a + dpx;
	    in1ny = y1a + dpy;
	    in2nx = x1a - dpx;
	    in2ny = y1a - dpy;
	    outnx = x2a;
	    outny = y2a;
	    
	    triPointsX = new int[3];
	    triPointsY = new int[3];
	    triPointsX[0] = x1a + dpx*2;
	    triPointsY[0] = y1a + dpy*2;
	    triPointsX[1] = x1a - dpx*2;
	    triPointsY[1] = y1a - dpy*2;
	    triPointsX[2] = x2a;
	    triPointsY[2] = y2a;
	    setBbox(triPointsX[0], triPointsY[0],
		    triPointsX[1], triPointsY[1]);
	    adjustBbox(x, y, x2, y2);
	}
	int getPostCount() { return 3; }
	Point getPost(int n) {
	    return (n == 0) ? new Point(in1px, in1py) :
		(n == 1) ? new Point(in2px, in2py) :
		new Point(outpx, outpy);
	}
	int getVoltageSourceCount() { return 1; }
	void getInfo(String arr[]) {
	    arr[0] = "op-amp";
	    arr[1] = "V+ = " + getVoltageText(volts[1]);
	    arr[2] = "V- = " + getVoltageText(volts[0]);
	    // sometimes the voltage goes slightly outside range, to make
	    // convergence easier.  so we hide that here.
	    double vo = Math.max(Math.min(volts[2], maxOut), minOut);
	    arr[3] = "Vsal = " + getVoltageText(vo);
	    arr[4] = "Isal = " + getCurrentText(getCurrent());
	    arr[5] = "rango = " + getVoltageText(minOut) + " to " +
		getVoltageText(maxOut);
	}
	
	final double gain = 1000;
	double lastvd;

	void stamp() {
	    int vn = nodeList.size()+voltSource;
	    stampNonLinear(vn);
	    stampMatrix(nodes[2], vn, 1);
	}
	void doStep() {
	    double vd = volts[1] - volts[0];
	    if (Math.abs(lastvd-vd) > .1)
		converged = false;
	    else if (volts[2] > maxOut+.1 || volts[2] < minOut-.1)
		converged = false;
	    double x = 0;
	    int vn = nodeList.size()+voltSource;
	    double dx = 0;
	    if (vd >= maxOut/gain && (lastvd >= 0 || getrand(4) == 1)) {
		dx = 1e-4;
		x = maxOut - dx*maxOut/gain;
	    } else if (vd <= minOut/gain && (lastvd <= 0 || getrand(4) == 1)) {
		dx = 1e-4;
		x = minOut - dx*minOut/gain;
	    } else
		dx = gain;
	    //System.out.println("opamp " + vd + " " + volts[2] + " " + dx + " "  + x + " " + lastvd + " " + converged);
	    
	    // newton-raphson
	    stampMatrix(vn, nodes[0], dx);
	    stampMatrix(vn, nodes[1], -dx);
	    stampMatrix(vn, nodes[2], 1);
	    stampRightSide(vn, x);
	    
	    lastvd = vd;
	    /*if (converged)
	      System.out.println((volts[1]-volts[0]) + " " + volts[2] + " " + initvd);*/
	}
	// there is no current path through the op-amp inputs, but there
	// is an indirect path through the output to ground.
	boolean getConnection(int n1, int n2) { return false; }
	boolean hasGroundConnection(int n1) {
	    return (n1 == 2);
	}
	double getVoltageDiff() { return volts[2] - volts[1]; }
	int getDumpType() { return 'a'; }
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Salida Max (V)", maxOut, 1, 20);
	    if (n == 1)
		return new EditInfo("Salida Min (V)", minOut, -20, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		maxOut = ei.value;
	    if (n == 1)
		minOut = ei.value;
	}
    }
    
    abstract class GateElm extends CircuitElm {
	final int FLAG_SMALL = 1;
	int inputCount = 2;
	boolean lastOutput;
	
	public GateElm(int xx, int yy) {
	    super(xx, yy);
	    noDiagonal = true;
	    inputCount = 2;
	    setSize(smallGridCheckItem.getState() ? 1 : 2);
	}
	public GateElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    inputCount = new Integer(st.nextToken()).intValue();
	    lastOutput = new Double (st.nextToken()).doubleValue() > 2.5;
	    noDiagonal = true;
	    setSize((f & FLAG_SMALL) != 0 ? 1 : 2);
	}
	boolean isInverting() { return false; }
	int gsize, gwidth, gwidth2, gheight;
	void setSize(int s) {
	    gsize = s;
	    gwidth = 7*s;
	    gwidth2 = 14*s;
	    gheight = 8*s;
	    flags = (s == 1) ? FLAG_SMALL : 0;
	}
	String dump() {
	    return getDumpType() + " " + x + " " + y + " " +
		x2 + " " + y2 + " " + flags + " " + inputCount + " " +
		volts[inputCount];
	}
	Point inPosts[], inGates[];
	void draw(Graphics g) {
	    int i;
	    for (i = 0; i != inputCount; i++) {
		setVoltageColor(g, volts[i]);
		drawThickLine(g, inPosts[i].x, inPosts[i].y,
			         inGates[i].x, inGates[i].y);
	    }
	    setVoltageColor(g, volts[inputCount]);
	    drawThickLine(g, outnx, outny, outpx, outpy);
	    g.setColor(this == mouseElm ? Color.cyan : Color.lightGray);
	    drawThickPolygon(g, triPointsX, triPointsY, triPointsX.length);
	    if (linePointsX != null) {
		for (i = 0; i != linePointsX.length-1; i++)
		    drawThickLine(g, linePointsX[i],   linePointsY[i],
				     linePointsX[i+1], linePointsY[i+1]);
	    }
	    if (isInverting())
		drawThickCircle(g, pcirclex, pcircley, 3);
	    curcount = updateDotCount(current, curcount);
	    drawDots(g, outnx, outny, outpx, outpy, curcount);
	    for (i = 0; i != inputCount; i++)
		drawPost(g, inPosts[i].x, inPosts[i].y, nodes[i]);
	    drawPost(g, outpx, outpy, nodes[inputCount]);
	}
	int outpx, outpy, outnx, outny;
	int triPointsX [], triPointsY [];
	int linePointsX[], linePointsY[];
	int pcirclex, pcircley;
	int getPostCount() { return inputCount+1; }
	Point getPost(int n) {
	    if (n == inputCount)
		return new Point(outpx, outpy);
	    return inPosts[n];
	}
	int getVoltageSourceCount() { return 1; }
	abstract String getGateName();
	void getInfo(String arr[]) {
	    arr[0] = getGateName();
	    arr[1] = "Vsal = " + getVoltageText(volts[inputCount]);
	    arr[2] = "Isal = " + getCurrentText(getCurrent());
	}
	void stamp() {
	    stampVoltageSource(0, nodes[inputCount], voltSource);
	}
	boolean getInput(int x) {
	    return volts[x] > 2.5;
	}
	abstract boolean calcFunction();
	void doStep() {
	    int i;
	    boolean f = calcFunction();
	    if (isInverting())
		f = !f;
	    lastOutput = f;
	    double res = f ? 5 : 0;
	    updateVoltageSource(0, nodes[inputCount], voltSource, res);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Intradas", inputCount, 1, 8);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    inputCount = (int) ei.value;
	    setPoints();
	}
	// there is no current path through the gate inputs, but there
	// is an indirect path through the output to ground.
	boolean getConnection(int n1, int n2) { return false; }
	boolean hasGroundConnection(int n1) {
	    return (n1 == inputCount);
	}
    }

    class AndGateElm extends GateElm {
	public AndGateElm(int xx, int yy) { super(xx, yy); }
	public AndGateElm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	void setPoints() {
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    if (dn > 150)
		setSize(2);
	    int hs = gheight;
	    int dpx = (int) ( Math.abs(hs*dy/dn));
	    int dpy = (int) (-Math.abs(hs*dx/dn));
	    int i;
	    outpx = x2; outpy = y2;

	    int ww = gwidth2; // was 24
	    if (ww > dn/2)
		ww = (int) (dn/2);
	    if (isInverting() && ww+8 > dn/2)
		ww = (int) (dn/2-8);
	    int x1a = (int) (x+dx*(dn/2-ww)/dn);
	    int y1a = (int) (y+dy*(dn/2-ww)/dn);
	    int x2a = (int) (x+dx*(dn/2+ww)/dn);
	    int y2a = (int) (y+dy*(dn/2+ww)/dn);
	    int i0 = -inputCount/2;
	    inPosts = new Point[inputCount];
	    inGates = new Point[inputCount];
	    allocNodes();
	    for (i = 0; i != inputCount; i++, i0++) {
		if (i0 == 0 && (inputCount & 1) == 0)
		    i0++;
		inPosts[i] = new Point(x  +dpx*i0, y  +dpy*i0);
		inGates[i] = new Point(x1a+dpx*i0, y1a+dpy*i0);
		volts[i] = (lastOutput ^ isInverting()) ? 5 : 0;
	    }

	    if (isInverting()) {
		pcirclex = (int) (x+dx*(dn/2+ww+4)/dn);
		pcircley = (int) (y+dy*(dn/2+ww+4)/dn);
		outnx = (int) (x+dx*(dn/2+ww+6)/dn);
		outny = (int) (y+dy*(dn/2+ww+6)/dn);
	    } else {
		outnx = x2a;
		outny = y2a;
	    }

	    // 0=topleft, 1-10 = top curve, 11 = right, 12-21=bottom curve,
	    // 22 = bottom left
	    int hs2 = gwidth*(inputCount/2+1);
	    dpx = (int) ( Math.abs(hs2*dy/dn));
	    dpy = (int) (-Math.abs(hs2*dx/dn));
	    triPointsX = new int[23];
	    triPointsY = new int[23];
	    triPointsX[0] = x1a + dpx;
	    triPointsY[0] = y1a + dpy;
	    triPointsX[22] = x1a - dpx;
	    triPointsY[22] = y1a - dpy;
	    for (i = 0; i != 10; i++) {
		double a = i*.1;
		double b = Math.sqrt(1-a*a);
		int xa = (int) (x+dx*(dn/2+ww*a)/dn+dpx*b);
		int ya = (int) (y+dy*(dn/2+ww*a)/dn+dpy*b);
		triPointsX[i+1] = xa;
		triPointsY[i+1] = ya;
		xa = (int) (x+dx*(dn/2+ww*a)/dn-dpx*b);
		ya = (int) (y+dy*(dn/2+ww*a)/dn-dpy*b);
		triPointsX[21-i] = xa;
		triPointsY[21-i] = ya;
	    }
	    triPointsX[11] = x2a; triPointsY[11] = y2a;
	    setBbox(triPointsX[0], triPointsY[0],
		    triPointsX[11], triPointsY[11]);
	    adjustBbox(triPointsX[22], triPointsY[22],
		       triPointsX[11], triPointsY[11]);
	}
	String getGateName() { return "puerta AND"; }
	boolean calcFunction() {
	    int i;
	    boolean f = true;
	    for (i = 0; i != inputCount; i++)
		f &= getInput(i);
	    return f;
	}
	int getDumpType() { return 150; }
    }
    
    class NandGateElm extends AndGateElm {
	public NandGateElm(int xx, int yy) { super(xx, yy); }
	public NandGateElm(int xa, int ya, int xb, int yb, int f,
			   StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	boolean isInverting() { return true; }
	String getGateName() { return "puerta NAND"; }
	int getDumpType() { return 151; }
    }
    
    class OrGateElm extends GateElm {
	public OrGateElm(int xx, int yy) { super(xx, yy); }
	public OrGateElm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getGateName() { return "puerta OR"; }
	void setPoints() {
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    if (dn > 150)
		setSize(2);
	    int hs = gheight;
	    int dpx = (int) ( Math.abs(hs*dy/dn));
	    int dpy = (int) (-Math.abs(hs*dx/dn));
	    int i;
	    outpx = x2; outpy = y2;

	    int ww = gwidth2; // was 24
	    if (ww > dn/2)
		ww = (int) (dn/2);
	    if (isInverting() && ww+8 > dn/2)
		ww = (int) (dn/2-8);
	    int x1a = (int) (x+dx*(dn/2-ww)/dn);
	    int y1a = (int) (y+dy*(dn/2-ww)/dn);
	    int x2a = (int) (x+dx*(dn/2+ww)/dn);
	    int y2a = (int) (y+dy*(dn/2+ww)/dn);
	    int i0 = -inputCount/2;
	    inPosts = new Point[inputCount];
	    inGates = new Point[inputCount];
	    nodes = new int   [getPostCount()];
	    volts = new double[getPostCount()];
	    for (i = 0; i != inputCount; i++, i0++) {
		if (i0 == 0 && (inputCount & 1) == 0)
		    i0++;
		inPosts[i] = new Point(x  +dpx*i0, y  +dpy*i0);
		inGates[i] = new Point(x1a+dpx*i0, y1a+dpy*i0);
		volts[i] = (lastOutput ^ isInverting()) ? 5 : 0;
	    }

	    if (isInverting()) {
		pcirclex = (int) (x+dx*(dn/2+ww+4)/dn);
		pcircley = (int) (y+dy*(dn/2+ww+4)/dn);
		outnx = (int) (x+dx*(dn/2+ww+6)/dn);
		outny = (int) (y+dy*(dn/2+ww+6)/dn);
	    } else {
		outnx = x2a;
		outny = y2a;
	    }

	    // 0-15 = top curve, 16 = right, 17-32=bottom curve,
	    // 33-37 = left curve
	    int hs2 = gwidth*(inputCount/2+1);
	    dpx = (int) ( Math.abs(hs2*dy/dn));
	    dpy = (int) (-Math.abs(hs2*dx/dn));
	    triPointsX = new int[38];
	    triPointsY = new int[38];
	    if (this instanceof XorGateElm) {
		linePointsX = new int[5];
		linePointsY = new int[5];
	    }
	    for (i = 0; i != 16; i++) {
		double a = i/16.;
		double b = 1-a*a;
		int xa = (int) (x+dx*(dn/2+ww*a)/dn+dpx*b);
		int ya = (int) (y+dy*(dn/2+ww*a)/dn+dpy*b);
		triPointsX[i] = xa;
		triPointsY[i] = ya;
		xa = (int) (x+dx*(dn/2+ww*a)/dn-dpx*b);
		ya = (int) (y+dy*(dn/2+ww*a)/dn-dpy*b);
		triPointsX[32-i] = xa;
		triPointsY[32-i] = ya;
	    }
	    for (i = 0; i != 5; i++) {
		double a = (i-2)/2.;
		double b = 4*(1-a*a)-2;
		int xa = x1a + (int) (dx*b/dn + dpx*a);
		int ya = y1a + (int) (dy*b/dn + dpy*a);
		triPointsX[33+i] = xa;
		triPointsY[33+i] = ya;
		if (this instanceof XorGateElm) {
		    linePointsX[i] = xa - (int) (dx*5/dn);
		    linePointsY[i] = ya - (int) (dy*5/dn);
		}
	    }
	    triPointsX[16] = x2a; triPointsY[16] = y2a;
	    setBbox(triPointsX[37], triPointsY[37],
		    triPointsX[16], triPointsY[16]);
	    adjustBbox(triPointsX[33], triPointsY[33],
		       triPointsX[16], triPointsY[16]);
	}
	boolean calcFunction() {
	    int i;
	    boolean f = false;
	    for (i = 0; i != inputCount; i++)
		f |= getInput(i);
	    return f;
	}
	int getDumpType() { return 152; }
    }
    
    class NorGateElm extends OrGateElm {
	public NorGateElm(int xx, int yy) { super(xx, yy); }
	public NorGateElm(int xa, int ya, int xb, int yb, int f,
			   StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getGateName() { return "puerta NOR"; }
	boolean isInverting() { return true; }
	int getDumpType() { return 153; }
    }
    
    class XorGateElm extends OrGateElm {
	public XorGateElm(int xx, int yy) { super(xx, yy); }
	public XorGateElm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getGateName() { return "puerta XOR"; }
	boolean calcFunction() {
	    int i;
	    boolean f = false;
	    for (i = 0; i != inputCount; i++)
		f ^= getInput(i);
	    return f;
	}
	int getDumpType() { return 154; }
    }
    
    class TransformerElm extends CircuitElm {
	double inductance, ratio;
	int x3, y3, x4, y4;
	int coil1ax, coil1ay, coil1bx, coil1by;
	int coil2ax, coil2ay, coil2bx, coil2by;
	int core1ax, core1ay, core1bx, core1by;
	int core2ax, core2ay, core2bx, core2by;
	double current1, current2;
	double curcount1, curcount2;
	public TransformerElm(int xx, int yy) {
	    super(xx, yy);
	    inductance = 4;
	    ratio = 1;
	    noDiagonal = true;
	}
	public TransformerElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    inductance = new Double(st.nextToken()).doubleValue();
	    ratio = new Double(st.nextToken()).doubleValue();
	    current1 = new Double(st.nextToken()).doubleValue();
	    current2 = new Double(st.nextToken()).doubleValue();
	    noDiagonal = true;
	}
	int getDumpType() { return 'T'; }
	String dump() {
	    return "T " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " +
		inductance + " " + ratio + " " + current1 + " " +
		current2;
	}
	void draw(Graphics g) {
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x , y , coil1ax, coil1ay);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, x3, y3, coil1bx, coil1by);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, x2, y2, coil2ax, coil2ay);
	    setVoltageColor(g, volts[3]);
	    drawThickLine(g, x4, y4, coil2bx, coil2by);
	    setPowerColor(g, current1*(volts[0]-volts[2]));
	    drawCoil(g, coil1ax, coil1ay, coil1bx, coil1by,
		     core1ax, core1ay, volts[0], volts[2]);
	    setPowerColor(g, current2*(volts[1]-volts[3]));
	    drawCoil(g, coil2ax, coil2ay, coil2bx, coil2by,
		     core2ax, core2ay, volts[1], volts[3]);
	    g.setColor(this == mouseElm ? Color.cyan : Color.lightGray);
	    drawThickLine(g, core1ax, core1ay, core1bx, core1by);
	    drawThickLine(g, core2ax, core2ay, core2bx, core2by);
	    curcount1 = updateDotCount(current1, curcount1);
	    curcount2 = updateDotCount(current2, curcount2);
	    setBbox(x, y, x4, y4);
	    drawDots(g, x,       y,       coil1ax, coil1ay, curcount1);
	    drawDots(g, coil1ax, coil1ay, coil1bx, coil1by, curcount1);
	    drawDots(g, coil1bx, coil1by, x3,      y3,      curcount1);
	    drawDots(g, x2,      y2,      coil2ax, coil2ay, curcount2);
	    drawDots(g, coil2ax, coil2ay, coil2bx, coil2by, curcount2);
	    drawDots(g, coil2bx, coil2by, x4,      y4,      curcount2);
	    drawPosts(g);
	    drawPost(g, x3, y3, nodes[2]);
	    drawPost(g, x4, y4, nodes[3]);
	}
	void drawCoil(Graphics g, int x1, int y1, int x2, int y2,
		      int cox, int coy, double v1, double v2) {
	    int segments = 30;
	    int xa = x1;
	    int ya = y1;
	    int i;
	    int dx = x2-x1;
	    int dy = y2-y1;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    for (i = 0; i != segments; i++) {
		double cx = (((i+1)*6./segments) % 2)-1;
		double hs = .5*Math.sqrt(1-cx*cx);
		if (hs < 0)
		    hs = -hs;
		int dpx = (int) ((cox-x1)*hs);
		int dpy = (int) ((coy-y1)*hs);
		double v = v1+(v2-v1)*i/segments;
		setVoltageColor(g, v);
		int xb = x1+dx*(i+1)/segments + dpx;
		int yb = y1+dy*(i+1)/segments + dpy;
		drawThickLine(g, xa, ya, xb, yb);
		xa = xb;
		ya = yb;
	    }
	}
	
	void setPoints() {
	    int hs = 32;
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( Math.abs(hs*dy/dn));
	    int dpy = (int) ( Math.abs(hs*dx/dn));
	    x3 = x +dpx;
	    y3 = y +dpy;
	    x4 = x2+dpx;
	    y4 = y2+dpy;
	    int ce = 12;
	    int cd = 2;
	    coil1ax = (int) (x +dx*(dn/2-ce)/dn);
	    coil1ay = (int) (y +dy*(dn/2-ce)/dn);
	    coil1bx = (int) (x3+dx*(dn/2-ce)/dn);
	    coil1by = (int) (y3+dy*(dn/2-ce)/dn);
	    coil2ax = (int) (x +dx*(dn/2+ce)/dn);
	    coil2ay = (int) (y +dy*(dn/2+ce)/dn);
	    coil2bx = (int) (x3+dx*(dn/2+ce)/dn);
	    coil2by = (int) (y3+dy*(dn/2+ce)/dn);
	    core1ax = (int) (x +dx*(dn/2-cd)/dn);
	    core1ay = (int) (y +dy*(dn/2-cd)/dn);
	    core1bx = (int) (x3+dx*(dn/2-cd)/dn);
	    core1by = (int) (y3+dy*(dn/2-cd)/dn);
	    core2ax = (int) (x +dx*(dn/2+cd)/dn);
	    core2ay = (int) (y +dy*(dn/2+cd)/dn);
	    core2bx = (int) (x3+dx*(dn/2+cd)/dn);
	    core2by = (int) (y3+dy*(dn/2+cd)/dn);
	    setBbox(x, y, x2, y2);
	    adjustBbox(x3, y3, x4, y4);
	}
	Point getPost(int n) {
	    return (n == 0) ? new Point(x, y) :
		(n == 1) ? new Point(x2, y2) :
		(n == 2) ? new Point(x3, y3) : new Point(x4, y4);
	}
	int getPostCount() { return 4; }
	void reset() {
	    current1 = current2 = volts[0] = volts[1] = volts[2] =
		volts[3] = 0;
	}
	double a1, a2, a3, a4;
	void stamp() {
	    // equations for transformer:
	    //   v1 = L1 di1/dt + M  di2/dt
	    //   v2 = M  di1/dt + L2 di2/dt
	    // we invert that to get:
	    //   di1/dt = a1 v1 + a2 v2
	    //   di2/dt = a3 v1 + a4 v2
	    // integrate di1/dt using trapezoidal approx and we get:
	    //   i1(t2) = i1(t1) + dt/2 (i1(t1) + i1(t2))
	    //          = i1(t1) + a1 dt/2 v1(t1) + a2 dt/2 v2(t1) +
	    //                     a1 dt/2 v1(t2) + a2 dt/2 v2(t2)
	    // the norton equivalent of this for i1 is:
	    //  a. current source, I = i1(t1) + a1 dt/2 v1(t1) + a2 dt/2 v2(t1)
	    //  b. resistor, G = a1 dt/2
	    //  c. current source controlled by voltage v2, G = a2 dt/2
	    // and for i2:
	    //  a. current source, I = i2(t1) + a3 dt/2 v1(t1) + a4 dt/2 v2(t1)
	    //  b. resistor, G = a3 dt/2
	    //  c. current source controlled by voltage v2, G = a4 dt/2
	    // 
	    // first winding goes from node 0 to 2, second is from 1 to 3
	    double l1 = inductance;
	    double l2 = inductance*ratio*ratio;
	    double m = .999*Math.sqrt(l1*l2);
	    double deti = 1/(l1*l2-m*m);
	    a1 = l2*deti*timeStep/2; // we multiply dt/2 into a1..a4 here
	    a2 = -m*deti*timeStep/2;
	    a3 = -m*deti*timeStep/2;
	    a4 = l1*deti*timeStep/2;
	    stampConductance(nodes[0], nodes[2], a1);
	    stampVCCurrentSource(nodes[0], nodes[2], nodes[1], nodes[3], a2);
	    stampVCCurrentSource(nodes[1], nodes[3], nodes[0], nodes[2], a3);
	    stampConductance(nodes[1], nodes[3], a4);
	    stampRightSide(nodes[0]);
	    stampRightSide(nodes[1]);
	    stampRightSide(nodes[2]);
	    stampRightSide(nodes[3]);
	}
	void startIteration() {
	    double voltdiff1 = volts[0]-volts[2];
	    double voltdiff2 = volts[1]-volts[3];
	    curSourceValue1 = voltdiff1*a1+voltdiff2*a2+current1;
	    curSourceValue2 = voltdiff1*a3+voltdiff2*a4+current2;
	}
	double curSourceValue1, curSourceValue2;
	void doStep() {
	    stampCurrentSource(nodes[0], nodes[2], curSourceValue1);
	    stampCurrentSource(nodes[1], nodes[3], curSourceValue2);
 	}
	void calculateCurrent() {
	    double voltdiff1 = volts[0]-volts[2];
	    double voltdiff2 = volts[1]-volts[3];
	    current1 = voltdiff1*a1 + voltdiff2*a2 + curSourceValue1;
	    current2 = voltdiff1*a3 + voltdiff2*a4 + curSourceValue2;
	}
	void getInfo(String arr[]) {
	    arr[0] = "transformador";
	    arr[1] = "L = " + getUnitText(inductance, "H");
	    arr[2] = "Ratio = " + ratio;
	    //arr[3] = "I1 = " + getCurrentText(current1);
	    arr[3] = "Vd1 = " + getVoltageText(volts[0]-volts[2]);
	    //arr[5] = "I2 = " + getCurrentText(current2);
	    arr[4] = "Vd2 = " + getVoltageText(volts[1]-volts[3]);
	}
	boolean getConnection(int n1, int n2) {
	    if (comparePair(n1, n2, 0, 2))
		return true;
	    if (comparePair(n1, n2, 1, 3))
		return true;
	    return false;
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Inductancia (H)", inductance, .01, 5);
	    if (n == 1)
		return new EditInfo("Ratio", ratio, 1, 10);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		inductance = ei.value;
	    if (n == 1)
		ratio = ei.value;
	}
    }

    class ACRailElm extends RailElm {
	public ACRailElm(int xx, int yy) { super(xx, yy, WF_AC); }
	Class getDumpClass() { return RailElm.class; }
    }
    class SquareRailElm extends RailElm {
	public SquareRailElm(int xx, int yy) { super(xx, yy, WF_SQUARE); }
	Class getDumpClass() { return RailElm.class; }
    }
    class ClockElm extends RailElm {
	public ClockElm(int xx, int yy) {
	    super(xx, yy, WF_SQUARE);
	    maxVoltage = 2.5;
	    bias = 2.5;
	    frequency = 100;
	    flags |= FLAG_CLOCK;
	}
	Class getDumpClass() { return RailElm.class; }
    }
    
    class RailElm extends VoltageElm {
	public RailElm(int xx, int yy) { super(xx, yy, WF_DC); }
	RailElm(int xx, int yy, int wf) { super(xx, yy, wf); }
	public RailElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	final int FLAG_CLOCK = 1;
	int getDumpType() { return 'R'; }
	String dump() {
	    return "R " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + waveform + " " + frequency + " " +
		maxVoltage + " " + bias;
	}
	int getPostCount() { return 1; }
	void draw(Graphics g) {
	    double dx = x2-x;
	    double dy = y2-y;
	    double dr = Math.sqrt(dx*dx+dy*dy);
	    dx /= dr;
	    dy /= dr;
	    int x3 = x2 - (int)(dx*circleSize);
	    int y3 = y2 - (int)(dy*circleSize);
	    setBbox(x, y, x2, y2);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x3, y3);
	    boolean clock = waveform == WF_SQUARE && (flags & FLAG_CLOCK) != 0;
	    if (waveform == WF_DC || clock) {
		Font f = new Font("SansSerif", 0, 12);
		g.setFont(f);
		FontMetrics fm = g.getFontMetrics();
		g.setColor(this == mouseElm ? Color.cyan : Color.white);
		setPowerColor(g, false);
		String s = showFormat.format(maxVoltage) + "V";
		if (maxVoltage > 0)
		    s = "+" + s;
		if (this instanceof AntennaElm)
		    s = "Ant";
		if (clock)
		    s = "CLK";
		int w = fm.stringWidth(s)/2;
		g.drawString(s, x2 - w, y2 + fm.getAscent()/2-1);
		adjustBbox(x2-w, y2-w, x2+w, y2+w);
	    } else
		drawWaveform(g, x2, y2);
	    drawPost(g, x, y, nodes[0]);
	    curcount = updateDotCount(-current, curcount);
	    if (dragElm != this)
		drawDots(g, x, y, x3, y3, curcount);
	}
	double getVoltageDiff() { return volts[0]; }
	void stamp() {
	    if (waveform == WF_DC)
		stampVoltageSource(0, nodes[0], voltSource, getVoltage());
	    else
		stampVoltageSource(0, nodes[0], voltSource);
	}
	void doStep() {
	    if (waveform != WF_DC)
		updateVoltageSource(0, nodes[0], voltSource, getVoltage());
	}
	boolean hasGroundConnection(int n1) { return true; }
    }

    class GroundElm extends CircuitElm {
	public GroundElm(int xx, int yy) { super(xx, yy); }
	public GroundElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	int getDumpType() { return 'g'; }
	int getPostCount() { return 1; }
	void draw(Graphics g) {
	    setVoltageColor(g, 0);
	    drawThickLine(g, x, y, x2, y2);
	    double dx = x2-x;
	    double dy = y2-y;
	    double dr = Math.sqrt(dx*dx+dy*dy);
	    dx /= dr;
	    dy /= dr;
	    //drawThickLine(g, x, y, (int) (x2-dx*10), (int) (y2-dy*10));
	    int i;
	    for (i = 0; i != 3; i++) {
		int a = 10-i*4;
		int b = i*5; // -10;
		drawThickLine(g,
			      (int) (x2+dx*b+dy*a), (int) (y2+dy*b-dx*a),
			      (int) (x2+dx*b-dy*a), (int) (y2+dy*b+dx*a));
	    }
	    doDots(g);
	    setBbox(x, y, x2, y2);
	    adjustBbox(x2+(int) (dx*10+dy*10), y2+(int) (dy*10-dx*10),
		       x2+(int) (dx*10-dy*10), y2+(int) (dy*10+dx*10));
	    drawPost(g, x, y, nodes[0]);
	}
	void setCurrent(int x, double c) { current = -c; }
	void stamp() {
	    stampVoltageSource(0, nodes[0], voltSource, 0);
	}
	double getVoltageDiff() { return 0; }
	int getVoltageSourceCount() { return 1; }
	void getInfo(String arr[]) {
	    arr[0] = "tierra";
	    arr[1] = "I = " + getCurrentText(getCurrent());
	}
	boolean hasGroundConnection(int n1) { return true; }
    }

    class AntennaElm extends RailElm {
	public AntennaElm(int xx, int yy) { super(xx, yy, WF_DC); }
	public AntennaElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    waveform = WF_DC;
	}
	String dump() {
	    return "A " + x + " " + y + " " + x2 + " " + y2 + " " + flags;
	}
	double fmphase;
	void stamp() {
	    stampVoltageSource(0, nodes[0], voltSource);
	}
	void doStep() {
	    updateVoltageSource(0, nodes[0], voltSource, getVoltage());
	}
	double getVoltage() {
	    fmphase += 2*pi*(2200+Math.sin(2*pi*t*13)*100)*timeStep;
	    double fm = 3*Math.sin(fmphase);
	    return Math.sin(2*pi*t*3000)*(1.3+Math.sin(2*pi*t*12))*3 +
	           Math.sin(2*pi*t*2710)*(1.3+Math.sin(2*pi*t*13))*3 +
		   Math.sin(2*pi*t*2433)*(1.3+Math.sin(2*pi*t*14))*3 + fm;
	}
	int getDumpType() { return 'A'; }
    }

    class LogicInputElm extends SwitchElm {
	double hiV, loV;
	public LogicInputElm(int xx, int yy) {
	    super(xx, yy, false);
	    hiV = 5;
	    loV = 0;
	}
	public LogicInputElm(int xa, int ya, int xb, int yb, int f,
			     StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    try {
		hiV = new Double(st.nextToken()).doubleValue();
		loV = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
		hiV = 5;
		loV = 0;
	    }
	    
	}
	int getDumpType() { return 'L'; }
	String dump() {
	    return "L " + x + " " + y + " " + x2 + " " + y2 + " " +
		flags + " " + open + " " + momentary + " " +
		hiV + " " + loV;
	}
	int getPostCount() { return 1; }
	void draw(Graphics g) {
	    Font f = new Font("SansSerif", Font.BOLD, 20);
	    g.setFont(f);
	    FontMetrics fm = g.getFontMetrics();
	    g.setColor(this == mouseElm ? Color.cyan : Color.white);
	    String s = open ? "L" : "H";
	    double dx = x2-x;
	    double dy = y2-y;
	    double dr = Math.sqrt(dx*dx+dy*dy);
	    dx /= dr;
	    dy /= dr;
	    int x3 = x2 - (int)(dx*12);
	    int y3 = y2 - (int)(dy*12);
	    setBbox(x, y, x2, y2);
	    int w = fm.stringWidth(s)/2;
	    int yy = y2 + fm.getAscent()/2-1;
	    g.drawString(s, x2 - w, yy);
	    adjustBbox(x2-w, yy-fm.getAscent(), x2+w, yy);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x3, y3);
	    updateDotCount();
	    drawDots(g, x, y, x3, y3, curcount);
	    drawPost(g, x, y, nodes[0]);
	}
	void setCurrent(int vs, double c) { current = -c; }
	void stamp() {
	    stampVoltageSource(0, nodes[0], voltSource, (open) ? loV : hiV);
	}
	int getVoltageSourceCount() { return 1; }
	double getVoltageDiff() { return volts[0]; }
	void getInfo(String arr[]) {
	    arr[0] = "entrada lgica";
	    arr[1] = (open) ? "baja" : "alta";
	    arr[1] += " (" + getVoltageText(volts[0]) + ")";
	    arr[2] = "I = " + getCurrentText(getCurrent());
	}
	boolean hasGroundConnection(int n1) { return true; }
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("", 0, 0, 0);
		ei.checkbox = new Checkbox("Interruptor Momentneo", momentary);
		return ei;
	    }
	    if (n == 1)
		return new EditInfo("Alto Voltaje", hiV, 10, -10);
	    if (n == 2)
		return new EditInfo("Bajo Voltaje", loV, 10, -10);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		momentary = ei.checkbox.getState();
	    if (n == 1)
		hiV = ei.value;
	    if (n == 2)
		loV = ei.value;
	}
    }

    class LogicOutputElm extends CircuitElm {
	double threshold;
	public LogicOutputElm(int xx, int yy) {
	    super(xx, yy);
	    threshold = 2.5;
	}
	public LogicOutputElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    try {
		threshold = new Double(st.nextToken()).doubleValue();
	    } catch (Exception e) {
		threshold = 2.5;
	    }
	}
	String dump() {
	    return "M " + x + " " + y + " " +
		x2 + " " + y2 + " " + flags + " " + threshold;
	}
	int getDumpType() { return 'M'; }
	int getPostCount() { return 1; }
	void draw(Graphics g) {
	    Font f = new Font("SansSerif", Font.BOLD, 20);
	    g.setFont(f);
	    FontMetrics fm = g.getFontMetrics();
	    //g.setColor(this == mouseElm ? Color.cyan : Color.lightGray);
	    g.setColor(Color.lightGray);
	    String s = (volts[0] < threshold) ? "L" : "H";
	    double dx = x2-x;
	    double dy = y2-y;
	    double dr = Math.sqrt(dx*dx+dy*dy);
	    dx /= dr;
	    dy /= dr;
	    int x3 = x2 - (int)(dx*12);
	    int y3 = y2 - (int)(dy*12);
	    setBbox(x, y, x2, y2);
	    int w = fm.stringWidth(s)/2;
	    int yy = y2 + fm.getAscent()/2-1;
	    g.drawString(s, x2 - w, yy);
	    adjustBbox(x2-w, yy-fm.getAscent(), x2+w, yy);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x3, y3);
	    doDots(g);
	    drawPost(g, x, y, nodes[0]);
	}
	double getVoltageDiff() { return volts[0]; }
	void getInfo(String arr[]) {
	    arr[0] = "salida lgica";
	    arr[1] = (volts[0] < threshold) ? "baja" : "alta";
	    arr[2] = "V = " + getVoltageText(volts[0]);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Umbral", threshold, 10, -10);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		threshold = ei.value;
	}
    }

    class OutputElm extends CircuitElm {
	final int FLAG_VALUE = 1;
	public OutputElm(int xx, int yy) { super(xx, yy); }
	public OutputElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	int getDumpType() { return 'O'; }
	int getPostCount() { return 1; }
	void draw(Graphics g) {
	    Font f = new Font("SansSerif",
			      this == mouseElm ? Font.BOLD : 0, 14);
	    g.setFont(f);
	    FontMetrics fm = g.getFontMetrics();
	    g.setColor(this == mouseElm ? Color.cyan : Color.white);
	    String s = (flags & FLAG_VALUE) != 0 ?
		getVoltageText(volts[0]) : "sal";
	    double dx = x2-x;
	    double dy = y2-y;
	    double dr = Math.sqrt(dx*dx+dy*dy);
	    dx /= dr;
	    dy /= dr;
	    setBbox(x, y, x2, y2);
	    int w = fm.stringWidth(s)/2;
	    int wa = w+8;
	    int x3 = x2 - (int)(dx*wa);
	    int y3 = y2 - (int)(dy*wa);
	    int yy = y2 + fm.getAscent()/2-1;
	    g.drawString(s, x2 - w, yy);
	    adjustBbox(x2-w, yy-fm.getAscent(), x2+w, yy);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x3, y3);
	    drawPost(g, x, y, nodes[0]);
	}
	double getVoltageDiff() { return volts[0]; }
	void getInfo(String arr[]) {
	    arr[0] = "salida";
	    arr[1] = "V = " + getVoltageText(volts[0]);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox = new Checkbox("Ver Voltaje",
					   (flags & FLAG_VALUE) != 0);
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		flags = (ei.checkbox.getState()) ?
		    (flags | FLAG_VALUE) :
		    (flags & ~FLAG_VALUE);
	}
    }

    class InverterElm extends CircuitElm {
	double slewRate; // V/ns
	public InverterElm(int xx, int yy) {
	    super(xx, yy);
	    noDiagonal = true;
	    slewRate = .5;
	}
	public InverterElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    noDiagonal = true;
	    try {
		slewRate = new Double (st.nextToken()).doubleValue();
	    } catch (Exception e) {
		slewRate = .5;
	    }
	}
	String dump() {
	    return "I " + x + " " + y + " " +
		x2 + " " + y2 + " " + flags + " " + slewRate;
	}
	
	int getDumpType() { return 'I'; }
	void draw(Graphics g) {
	    drawPosts(g);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, in_nx, in_ny);
	    g.setColor(this == mouseElm ? Color.cyan : Color.lightGray);
	    drawThickPolygon(g, triPointsX, triPointsY, 3);
	    drawThickCircle(g, pcirclex, pcircley, 3);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, outnx, outny, x2, y2);
	    curcount = updateDotCount(current, curcount);
	    drawDots(g, outnx, outny, x2, y2, curcount);
	}
	int in_nx, in_ny, outnx, outny, pcirclex, pcircley;
	int triPointsX[], triPointsY[];
	void setPoints() {
	    int hs = 8;
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( Math.abs(hs*dy/dn));
	    int dpy = (int) (-Math.abs(hs*dx/dn));
	    
	    int ww = 16;
	    if (ww > dn/2)
		ww = (int) (dn/2);
	    int x1a = (int) (x+dx*(dn/2-ww)/dn);
	    int y1a = (int) (y+dy*(dn/2-ww)/dn);
	    int x2a = (int) (x+dx*(dn/2+ww-5)/dn);
	    int y2a = (int) (y+dy*(dn/2+ww-5)/dn);
	    in_nx = x1a;
	    in_ny = y1a;
	    outnx = (int) (x+dx*(dn/2+ww+2)/dn);
	    outny = (int) (y+dy*(dn/2+ww+2)/dn);

	    pcirclex = (int) (x+dx*(dn/2+ww-2)/dn);
	    pcircley = (int) (y+dy*(dn/2+ww-2)/dn);
	    
	    triPointsX = new int[3];
	    triPointsY = new int[3];
	    triPointsX[0] = x1a + dpx*2;
	    triPointsY[0] = y1a + dpy*2;
	    triPointsX[1] = x1a - dpx*2;
	    triPointsY[1] = y1a - dpy*2;
	    triPointsX[2] = x2a;
	    triPointsY[2] = y2a;
	    setBbox(triPointsX[0], triPointsY[0],
		    triPointsX[1], triPointsY[1]);
	    adjustBbox(x, y, x2, y2);
	}
	int getVoltageSourceCount() { return 1; }
	void stamp() {
	    stampVoltageSource(0, nodes[1], voltSource);
	}
	void doStep() {
	    double v0 = volts[1];
	    double out = volts[0] > 2.5 ? 0 : 5;
	    double maxStep = slewRate * timeStep * 1e9;
	    out = Math.max(Math.min(v0+maxStep, out), v0-maxStep);
	    updateVoltageSource(0, nodes[1], voltSource, out);
	}
	double getVoltageDiff() { return volts[0]; }
	void getInfo(String arr[]) {
	    arr[0] = "inversor";
	    arr[1] = "Ven = " + getVoltageText(volts[0]);
	    arr[2] = "Vsal= " + getVoltageText(volts[1]);
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0)
		return new EditInfo("Tasa acomodacin (V/ns)", slewRate, 0, 0);
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    slewRate = ei.value;
	}
	// there is no current path through the inverter input, but there
	// is an indirect path through the output to ground.
	boolean getConnection(int n1, int n2) { return false; }
	boolean hasGroundConnection(int n1) {
	    return (n1 == 1);
	}
    }

    abstract class ChipElm extends CircuitElm {
	int csize, cspc, cspc2;
	int bits;
	final int FLAG_SMALL = 1;
	public ChipElm(int xx, int yy) {
	    super(xx, yy);
	    if (needsBits())
		bits = (this instanceof JohnsonElm) ? 10 : 4;
	    noDiagonal = true;
	    setupPins();
	    setSize(smallGridCheckItem.getState() ? 1 : 2);
	}
	public ChipElm(int xa, int ya, int xb, int yb, int f,
		       StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    if (needsBits())
		bits = new Integer(st.nextToken()).intValue();
	    noDiagonal = true;
	    setupPins();
	    setSize((f & FLAG_SMALL) != 0 ? 1 : 2);
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		if (pins[i].state) {
		    volts[i] = new Double(st.nextToken()).doubleValue();
		    pins[i].value = volts[i] > 2.5;
		}
	    }
	}
	boolean needsBits() { return false; }
	void setSize(int s) {
	    csize = s;
	    cspc = 8*s;
	    cspc2 = cspc*2;
	    flags &= ~FLAG_SMALL;
	    flags |= (s == 1) ? FLAG_SMALL : 0;
	}
	abstract void setupPins();
	void draw(Graphics g) {
	    drawChip(g);
	}
	void drawChip(Graphics g) {
	    int i;
	    Font f = new Font("SansSerif", 0, 10*csize);
	    g.setFont(f);
	    FontMetrics fm = g.getFontMetrics();
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		setVoltageColor(g, volts[i]);
		Point a = p.post;
		Point b = p.stub;
		drawThickLine(g, a.x, a.y, b.x, b.y);
		p.curcount = updateDotCount(p.current, p.curcount);
		drawDots(g, b.x, b.y, a.x, a.y, p.curcount);
		if (p.bubble) {
		    g.setColor(Color.black);
		    drawThickCircle(g, p.bubbleX, p.bubbleY, 1);
		    g.setColor(Color.lightGray);
		    drawThickCircle(g, p.bubbleX, p.bubbleY, 3);
		}
		g.setColor(Color.white);
		int sw = fm.stringWidth(p.text);
		g.drawString(p.text, p.textloc.x-sw/2,
			     p.textloc.y+fm.getAscent()/2);
		if (p.lineOver) {
		    int ya = p.textloc.y-fm.getAscent()/2;
		    g.drawLine(p.textloc.x-sw/2, ya, p.textloc.x+sw/2, ya);
		}
	    }
	    g.setColor(this == mouseElm ? Color.cyan : Color.lightGray);
	    drawThickPolygon(g, rectPointsX, rectPointsY, 4);
	    if (clockPointsX != null)
		g.drawPolyline(clockPointsX, clockPointsY, 3);
	    for (i = 0; i != getPostCount(); i++)
		drawPost(g, pins[i].post.x, pins[i].post.y, nodes[i]);
	}
	int rectPointsX[], rectPointsY[];
	int clockPointsX[], clockPointsY[];
	Pin pins[];
	int sizeX, sizeY;
	boolean lastClock;
	void drag(int xx, int yy) {
	    yy = snapGrid(yy);
	    if (xx < x) {
		xx = x; yy = y;
	    } else {
		y = y2 = yy;
		x2 = snapGrid(xx);
	    }
	    setPoints();
	}
	void setPoints() {
	    if (x2-x > sizeX*cspc2)
		setSize(2);
	    int hs = cspc;
	    int dx = 1; int dy = 0;
	    int dpx = -dy;       int dpy = dx;
	    rectPointsX = new int[4];
	    rectPointsY = new int[4];
	    int x0 = x+cspc2; int y0 = y;
	    int xr = x0-cspc;
	    int yr = y0-cspc;
	    int xs = sizeX*cspc2;
	    int ys = sizeY*cspc2;
	    rectPointsX[0] = xr;
	    rectPointsY[0] = yr;
	    rectPointsX[1] = xr+dx*xs;
	    rectPointsY[1] = yr+dy*xs;
	    rectPointsX[2] = xr+dx*xs+dpx*ys;
	    rectPointsY[2] = yr+dy*xs+dpy*ys;
	    rectPointsX[3] = xr+dpx*ys;
	    rectPointsY[3] = yr+dpy*ys;
	    setBbox(xr, yr, rectPointsX[2], rectPointsY[2]);
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		switch (p.side) {
		case SIDE_N: p.setPoint(x0,y0, 1, 0, 0, -1, 0, 0); break;
		case SIDE_S: p.setPoint(x0,y0, 1, 0, 0,  1, 0, ys-cspc2);break;
		case SIDE_W: p.setPoint(x0,y0, 0, 1, -1, 0, 0, 0); break;
		case SIDE_E: p.setPoint(x0,y0, 0, 1,  1, 0, xs-cspc2, 0);break;
		}
	    }
	}
	Point getPost(int n) {
	    return pins[n].post;
	}
	abstract int getVoltageSourceCount(); // output count
	void setVoltageSource(int j, int vs) {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (p.output && j-- == 0) {
		    p.voltSource = vs;
		    return;
		}
	    }
	    System.out.println("setVoltageSource fall por " + this);
	}
	void stamp() {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (p.output)
		    stampVoltageSource(0, nodes[i], p.voltSource);
	    }
	}
	void execute() {}
	void doStep() {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (!p.output)
		    p.value = volts[i] > 2.5;
	    }
	    execute();
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (p.output)
		    updateVoltageSource(0, nodes[i], p.voltSource,
					p.value ? 5 : 0);
	    }
	}
	void reset() {
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		pins[i].value = false;
		volts[i] = 0;
	    }
	    lastClock = false;
	}
	
	String dump() {
	    int t = getDumpType();
	    String s = t + " " + x + " " + y + " " +
		x2 + " " + y2 + " " + flags;
	    if (needsBits())
		s += " " + bits;
	    int i;
	    for (i = 0; i != getPostCount(); i++) {
		if (pins[i].state)
		    s += " " + volts[i];
	    }
	    return s;
	}
	
	void getInfo(String arr[]) {
	    arr[0] = getChipName();
	    int i, a = 1;
	    for (i = 0; i != getPostCount(); i++) {
		Pin p = pins[i];
		if (arr[a] != null)
		    arr[a] += "; ";
		else
		    arr[a] = "";
		String t = p.text;
		if (p.lineOver)
		    t += '\'';
		if (p.clock)
		    t = "Rel";
		arr[a] += t + " = " + getVoltageText(volts[i]);
		if (i % 2 == 1)
		    a++;
	    }
	}
	void setCurrent(int x, double c) {
	    int i;
	    for (i = 0; i != getPostCount(); i++)
		if (pins[i].output && pins[i].voltSource == x)
		    pins[i].current = c;
	}
	String getChipName() { return "chip"; }
	boolean getConnection(int n1, int n2) { return false; }
	boolean hasGroundConnection(int n1) {
	    return pins[n1].output;
	}

	final int SIDE_N = 0;
	final int SIDE_S = 1;
	final int SIDE_W = 2;
	final int SIDE_E = 3;
	class Pin {
	    Pin(int p, int s, String t) {
		pos = p; side = s; text = t;
	    }
	    Point post, stub;
	    Point textloc;
	    int pos, side, voltSource, bubbleX, bubbleY;
	    String text;
	    boolean lineOver, bubble, clock, output, value, state;
	    double curcount, current;
	    void setPoint(int x, int y, int dx, int dy, int dax, int day,
			  int sx, int sy) {
		int xa = x+cspc2*dx*pos+sx;
		int ya = y+cspc2*dy*pos+sy;
		post    = new Point(xa+dax*cspc2, ya+day*cspc2);
		stub    = new Point(xa+dax*cspc , ya+day*cspc );
		textloc = new Point(xa       , ya       );
		if (bubble) {
		    bubbleX = xa+dax*10*csize;
		    bubbleY = ya+day*10*csize;
		}
		if (clock) {
		    clockPointsX = new int[3];
		    clockPointsY = new int[3];
		    clockPointsX[0] = xa+dax*cspc-dx*cspc/2;
		    clockPointsY[0] = ya+day*cspc-dy*cspc/2;
		    clockPointsX[1] = xa;
		    clockPointsY[1] = ya;
		    clockPointsX[2] = xa+dax*cspc+dx*cspc/2;
		    clockPointsY[2] = ya+day*cspc+dy*cspc/2;
		}
	    }
	}
    }

    class DFlipFlopElm extends ChipElm {
	final int FLAG_RESET = 2;
	public DFlipFlopElm(int xx, int yy) { super(xx, yy); }
	public DFlipFlopElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    pins[2].value = !pins[1].value;
	}
	String getChipName() { return "flip-flop D"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = 3;
	    pins = new Pin[getPostCount()];
	    pins[0] = new Pin(0, SIDE_W, "D");
	    pins[1] = new Pin(0, SIDE_E, "Q");
	    pins[1].output = pins[1].state = true;
	    pins[2] = new Pin(2, SIDE_E, "Q");
	    pins[2].output = true;
	    pins[2].lineOver = true;
	    pins[3] = new Pin(1, SIDE_W, "");
	    pins[3].clock = true;
	    if ((flags & FLAG_RESET) != 0)
		pins[4] = new Pin(2, SIDE_W, "R");
	}
	int getPostCount() {
	    return ((flags & FLAG_RESET) != 0) ? 5 : 4;
	}
	int getVoltageSourceCount() { return 2; }
	void execute() {
	    if (pins[3].value && !lastClock) {
		pins[1].value =  pins[0].value;
		pins[2].value = !pins[0].value;
	    }
	    if (pins.length > 4 && pins[4].value) {
		pins[1].value = false;
		pins[2].value = true;
	    }
	    lastClock = pins[3].value;
	}
	int getDumpType() { return 155; }
    }
    
    class JKFlipFlopElm extends ChipElm {
	public JKFlipFlopElm(int xx, int yy) { super(xx, yy); }
	public JKFlipFlopElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	    pins[4].value = !pins[3].value;
	}
	String getChipName() { return "flip-flop JK"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = 3;
	    pins = new Pin[5];
	    pins[0] = new Pin(0, SIDE_W, "J");
	    pins[1] = new Pin(1, SIDE_W, "");
	    pins[1].clock = true;
	    pins[1].bubble = true;
	    pins[2] = new Pin(2, SIDE_W, "K");
	    pins[3] = new Pin(0, SIDE_E, "Q");
	    pins[3].output = pins[3].state = true;
	    pins[4] = new Pin(2, SIDE_E, "Q");
	    pins[4].output = true;
	    pins[4].lineOver = true;
	}
	int getPostCount() { return 5; }
	int getVoltageSourceCount() { return 2; }
	void execute() {
	    if (!pins[1].value && lastClock) {
		boolean q = pins[3].value;
		if (pins[0].value) {
		    if (pins[2].value)
			q = !q;
		    else
			q = true;
		} else if (pins[2].value)
		    q = false;
		pins[3].value = q;
		pins[4].value = !q;
	    }
	    lastClock = pins[1].value;
	}
	int getDumpType() { return 156; }
    }
    
    class JohnsonElm extends ChipElm {
	public JohnsonElm(int xx, int yy) { super(xx, yy); }
	public JohnsonElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "contador Johnson"; }
	boolean needsBits() { return true; }
	void setupPins() {
	    sizeX = bits > 2 ? bits : 2;
	    sizeY = 2;
	    pins = new Pin[getPostCount()];
	    pins[0] = new Pin(1, SIDE_W, "");
	    pins[0].clock = true;
	    pins[1] = new Pin(sizeX-1, SIDE_S, "R");
	    pins[1].bubble = true;
	    int i;
	    for (i = 0; i != bits; i++) {
		int ii = i+2;
		pins[ii] = new Pin(i, SIDE_N, "Q" + i);
		pins[ii].output = pins[ii].state = true;
	    }
	    allocNodes();
	}
	int getPostCount() { return bits+2; }
	int getVoltageSourceCount() { return bits; }
	void execute() {
	    int i;
	    if (pins[0].value && !lastClock) {
		for (i = 0; i != bits; i++)
		    if (pins[i+2].value)
			break;
		if (i < bits)
		    pins[i++ +2].value = false;
		i %= bits;
		pins[i+2].value = true;
	    }
	    if (!pins[1].value) {
		for (i = 1; i != bits; i++)
		    pins[i+2].value = false;
		pins[2].value = true;
	    }
	    lastClock = pins[0].value;
	}
	int getDumpType() { return 163; }
    }
    
    class CounterElm extends ChipElm {
	public CounterElm(int xx, int yy) { super(xx, yy); }
	public CounterElm(int xa, int ya, int xb, int yb, int f,
			    StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	boolean needsBits() { return true; }
	String getChipName() { return "Contador"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = bits > 2 ? bits : 2;
	    pins = new Pin[getPostCount()];
	    pins[0] = new Pin(0, SIDE_W, "");
	    pins[0].clock = true;
	    pins[1] = new Pin(sizeY-1, SIDE_W, "R");
	    pins[1].bubble = true;
	    int i;
	    for (i = 0; i != bits; i++) {
		int ii = i+2;
		pins[ii] = new Pin(i, SIDE_E, "Q" + (bits-i-1));
		pins[ii].output = pins[ii].state = true;
	    }
	    allocNodes();
	}
	int getPostCount() { return bits+2; }
	int getVoltageSourceCount() { return bits; }
	void execute() {
	    if (pins[0].value && !lastClock) {
		int i;
		for (i = bits-1; i >= 0; i--) {
		    int ii = i+2;
		    if (!pins[ii].value) {
			pins[ii].value = true;
			break;
		    }
		    pins[ii].value = false;
		}
	    }
	    if (!pins[1].value) {
		int i;
		for (i = 0; i != bits; i++)
		    pins[i+2].value = false;
	    }
	    lastClock = pins[0].value;
	}
	int getDumpType() { return 164; }
    }
    
    class SevenSegElm extends ChipElm {
	public SevenSegElm(int xx, int yy) { super(xx, yy); }
	public SevenSegElm(int xa, int ya, int xb, int yb, int f,
			   StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "visor 7-segmentos"; }
	Color darkred;
	void setupPins() {
	    darkred = new Color(60, 0, 0);
	    sizeX = 4;
	    sizeY = 4;
	    pins = new Pin[7];
	    pins[0] = new Pin(0, SIDE_W, "a");
	    pins[1] = new Pin(1, SIDE_W, "b");
	    pins[2] = new Pin(2, SIDE_W, "c");
	    pins[3] = new Pin(3, SIDE_W, "d");
	    pins[4] = new Pin(1, SIDE_S, "e");
	    pins[5] = new Pin(2, SIDE_S, "f");
	    pins[6] = new Pin(3, SIDE_S, "g");
	}
	void draw(Graphics g) {
	    drawChip(g);
	    g.setColor(Color.red);
	    int xl = x+cspc*5;
	    int yl = y+cspc;
	    setColor(g, 0);
	    drawThickLine(g, xl, yl, xl+cspc, yl);
	    setColor(g, 1);
	    drawThickLine(g, xl+cspc, yl, xl+cspc, yl+cspc);
	    setColor(g, 2);
	    drawThickLine(g, xl+cspc, yl+cspc, xl+cspc, yl+cspc2);
	    setColor(g, 3);
	    drawThickLine(g, xl, yl+cspc2, xl+cspc, yl+cspc2);
	    setColor(g, 4);
	    drawThickLine(g, xl, yl+cspc, xl, yl+cspc2);
	    setColor(g, 5);
	    drawThickLine(g, xl, yl, xl, yl+cspc);
	    setColor(g, 6);
	    drawThickLine(g, xl, yl+cspc, xl+cspc, yl+cspc);
	}
	void setColor(Graphics g, int p) {
	    g.setColor(pins[p].value ? Color.red : darkred);
	}
	int getPostCount() { return 7; }
	int getVoltageSourceCount() { return 0; }
	int getDumpType() { return 157; }
    }
    
    class VCOElm extends ChipElm {
	public VCOElm(int xx, int yy) { super(xx, yy); }
	public VCOElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "VCO"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = 4;
	    pins = new Pin[6];
	    pins[0] = new Pin(0, SIDE_W, "Ve");
	    pins[1] = new Pin(3, SIDE_W, "Vs");
	    pins[1].output = true;
	    pins[2] = new Pin(0, SIDE_E, "C");
	    pins[3] = new Pin(1, SIDE_E, "C");
	    pins[4] = new Pin(2, SIDE_E, "R1");
	    pins[4].output = true;
	    pins[5] = new Pin(3, SIDE_E, "R2");
	    pins[5].output = true;
	}
	boolean nonLinear() { return true; }
	void stamp() {
	    // output pin
	    stampVoltageSource(0, nodes[1], pins[1].voltSource);
	    // attach Vi to R1 pin so its current is proportional to Vi
	    stampVoltageSource(nodes[0], nodes[4], pins[4].voltSource, 0);
	    // attach 5V to R2 pin so we get a current going
	    stampVoltageSource(0, nodes[5], pins[5].voltSource, 5);
	    // put resistor across cap pins to give current somewhere to go
	    // in case cap is not connected
	    stampResistor(nodes[2], nodes[3], cResistance);
	    stampNonLinear(nodes[2]);
	    stampNonLinear(nodes[3]);
	}
	final double cResistance = 1e6;
	double cCurrent;
	int cDir;
	void doStep() {
	    double vc = volts[3]-volts[2];
	    double vo = volts[1];
	    int dir = (vo < 2.5) ? 1 : -1;
	    // switch direction of current through cap as we oscillate
	    if (vo < 2.5 && vc > 4.5) {
		vo = 5;
		dir = -1;
	    }
	    if (vo > 2.5 && vc < .5) {
		vo = 0;
		dir = 1;
	    }

	    // generate output voltage
	    updateVoltageSource(0, nodes[1], pins[1].voltSource, vo);
	    // now we set the current through the cap to be equal to the
	    // current through R1 and R2, so we can measure the voltage
	    // across the cap
	    int cur1 = nodeList.size() + pins[4].voltSource;
	    int cur2 = nodeList.size() + pins[5].voltSource;
	    stampMatrix(nodes[2], cur1, dir);
	    stampMatrix(nodes[2], cur2, dir);
	    stampMatrix(nodes[3], cur1, -dir);
	    stampMatrix(nodes[3], cur2, -dir);
	    cDir = dir;
	}
	// can't do this in calculateCurrent() because it's called before
        // we get pins[4].current and pins[5].current, which we need
	void computeCurrent() {
	    double c = cDir*(pins[4].current + pins[5].current) +
		(volts[3]-volts[2])/cResistance;
	    pins[2].current = -c;
	    pins[3].current = c;
	    pins[0].current = -pins[4].current;
	}
	void draw(Graphics g) {
	    computeCurrent();
	    drawChip(g);
	}
	int getPostCount() { return 6; }
	int getVoltageSourceCount() { return 3; }
	int getDumpType() { return 158; }
    }
    
    class TimerElm extends ChipElm {
	public TimerElm(int xx, int yy) { super(xx, yy); }
	public TimerElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "Temporizador 555"; }
	void setupPins() {
	    sizeX = 3;
	    sizeY = 5;
	    pins = new Pin[6];
	    pins[0] = new Pin(1, SIDE_W, "dis");
	    pins[1] = new Pin(3, SIDE_W, "tr");
	    pins[1].lineOver = true;
	    pins[2] = new Pin(4, SIDE_W, "th");
	    pins[3] = new Pin(1, SIDE_N, "V");
	    pins[4] = new Pin(1, SIDE_S, "ctl");
	    pins[5] = new Pin(2, SIDE_E, "sal");
	    pins[5].output = pins[5].state = true;
	}
	boolean nonLinear() { return true; }
	void stamp() {
	    // stamp voltage divider to put ctl pin at 2/3 V
	    stampResistor(nodes[3], nodes[4],  5000);
	    stampResistor(nodes[4], 0,        10000);
	    // output pin
	    stampVoltageSource(0, nodes[5], pins[5].voltSource);
	    // discharge pin
	    stampNonLinear(0);
	    stampNonLinear(nodes[0]);
	}
	void calculateCurrent() {
	    // need current for V, discharge, control; output current is
	    // calculated for us, and other pins have no current
	    pins[3].current = (volts[4]-volts[3])/5000;
	    pins[4].current = -volts[4]/10000 - pins[3].current;
	    pins[0].current = out ? 0 : -volts[0]/10;
	}
	boolean out;
	void startIteration() {
	    out = volts[5] > volts[3]/2;
	    // check comparators
	    if (volts[4]/2 > volts[1])
		out = true;
	    if (volts[2] > volts[4])
		out = false;
	}
	void doStep() {
	    // if output is low, discharge pin 0.  we use a small
	    // resistor because it's easier, and sometimes people tie
	    // the discharge pin to the trigger and threshold pins.
	    if (!out)
		stampResistor(nodes[0], 0, 10);
	    // output
	    updateVoltageSource(0, nodes[5], pins[5].voltSource,
				out ? volts[3] : 0);
	}
	int getPostCount() { return 6; }
	int getVoltageSourceCount() { return 1; }
	int getDumpType() { return 165; }
    }
    
    class PhaseCompElm extends ChipElm {
	public PhaseCompElm(int xx, int yy) { super(xx, yy); }
	public PhaseCompElm(int xa, int ya, int xb, int yb, int f,
		      StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	String getChipName() { return "comparador de fase"; }
	void setupPins() {
	    sizeX = 2;
	    sizeY = 2;
	    pins = new Pin[3];
	    pins[0] = new Pin(0, SIDE_W, "I1");
	    pins[1] = new Pin(1, SIDE_W, "I2");
	    pins[2] = new Pin(0, SIDE_E, "O");
	    pins[2].output = true;
	}
	boolean nonLinear() { return true; }
	void stamp() {
	    int vn = nodeList.size()+pins[2].voltSource;
	    stampNonLinear(vn);
	    stampNonLinear(0);
	    stampNonLinear(nodes[2]);
	}
	boolean ff1, ff2;
	void doStep() {
	    boolean v1 = volts[0] > 2.5;
	    boolean v2 = volts[1] > 2.5;
	    if (v1 && !pins[0].value)
		ff1 = true;
	    if (v2 && !pins[1].value)
		ff2 = true;
	    if (ff1 && ff2)
		ff1 = ff2 = false;
	    double out = (ff1) ? 5 : (ff2) ? 0 : -1;
	    //System.out.println(out + " " + v1 + " " + v2);
	    if (out != -1)
		stampVoltageSource(0, nodes[2], pins[2].voltSource, out);
	    else {
		// tie current through output pin to 0
		int vn = nodeList.size()+pins[2].voltSource;
		stampMatrix(vn, vn, 1);
	    }
	    pins[0].value = v1;
	    pins[1].value = v2;
	}
	int getPostCount() { return 3; }
	int getVoltageSourceCount() { return 1; }
	int getDumpType() { return 161; }
    }
    
    class TextElm extends CircuitElm {
	String text;
	int size;
	final int FLAG_CENTER = 1;
	final int FLAG_BAR = 2;
	public TextElm(int xx, int yy) {
	    super(xx, yy);
	    text = "hola";
	    size = 24;
	}
	public TextElm(int xa, int ya, int xb, int yb, int f,
			      StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	    size = new Integer(st.nextToken()).intValue();
	    text = st.nextToken();
	    while (st.hasMoreTokens())
		text += " " + st.nextToken();
	}
	String dump() {
	    return ((char) getDumpType()) + " " + x + " " + y + " " +
		x2 + " " + y2 + " " + flags + " " + size + " " + text;
	}
	int getDumpType() { return 'x'; }
	void drag(int xx, int yy) {
	    x = xx;
	    y = yy;
	    x2 = xx+16;
	    y2 = yy;
	}
	void draw(Graphics g) {
	    g.setColor(this == mouseElm ? Color.cyan : Color.lightGray);
	    Font f = new Font("SansSerif", 0, size);
	    g.setFont(f);
	    FontMetrics fm = g.getFontMetrics();
	    int w = fm.stringWidth(text);
	    if ((flags & FLAG_CENTER) != 0)
		x = (winSize.width-w)/2;
	    g.drawString(text, x, y);
	    if ((flags & FLAG_BAR) != 0) {
		int by = y-fm.getAscent();
		g.drawLine(x, by, x+w-1, by);
	    }
	    setBbox(x, y-fm.getAscent(),
		    x+w, y+fm.getDescent());
	}
	EditInfo getEditInfo(int n) {
	    if (n == 0) {
		EditInfo ei = new EditInfo("Texto", 0, -1, -1);
		ei.text = text;
		return ei;
	    }
	    if (n == 1)
		return new EditInfo("Tamao", size, 5, 100);
	    if (n == 2) {
		EditInfo ei = new EditInfo("", 0, -1, -1);
		ei.checkbox =
		    new Checkbox("Invertido", (flags & FLAG_BAR) != 0);
		return ei;
	    }
	    return null;
	}
	void setEditValue(int n, EditInfo ei) {
	    if (n == 0)
		text = ei.textf.getText();
	    if (n == 1)
		size = (int) ei.value;
	    if (n == 2) {
		if (ei.checkbox.getState())
		    flags |= FLAG_BAR;
		else
		    flags &= ~FLAG_BAR;
	    }
	}
	void getInfo(String arr[]) {
	    arr[0] = text;
	}
	int getPostCount() { return 0; }
    }

    class ProbeElm extends CircuitElm {
	public ProbeElm(int xx, int yy) { super(xx, yy); }
	public ProbeElm(int xa, int ya, int xb, int yb, int f,
			StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	int getDumpType() { return 'p'; }
	void draw(Graphics g) {
	    int segments = 16;
	    int i;
	    int ox = 0;
	    int dx = x2-x;
	    int dy = y2-y;
	    int hs = 8;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    setBbox(x, y, x2, y2);
	    double len = (mouseElm == this || dragElm == this) ? dn/2-8 : 16;
	    if (len > dn/2)
		len = dn/2;
	    int x1a = (int) (x+dx*(len)/dn);
	    int y1a = (int) (y+dy*(len)/dn);
	    int x2a = (int) (x+dx*(dn-len)/dn);
	    int y2a = (int) (y+dy*(dn-len)/dn);
	    setVoltageColor(g, volts[0]);
	    drawThickLine(g, x, y, x1a, y1a);
	    setVoltageColor(g, volts[1]);
	    drawThickLine(g, x2a, y2a, x2, y2);
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    adjustBbox(x-dpx, y-dpy, x2+dpx, y2+dpy);
	    drawPosts(g);
	}
	
	void getInfo(String arr[]) {
	    arr[0] = "sonda osciloscopio";
	    arr[1] = "Vd = " + getVoltageText(getVoltageDiff());
	}
	boolean getConnection(int n1, int n2) { return false; }
    }
    
    class AnalogSwitchElm extends CircuitElm {
	int x3, y3, x3a, y3a;
	public AnalogSwitchElm(int xx, int yy) {
	    super(xx, yy);
	}
	public AnalogSwitchElm(int xa, int ya, int xb, int yb, int f,
			 StringTokenizer st) {
	    super(xa, ya, xb, yb, f);
	}
	int getDumpType() { return 159; }
	boolean open;
	void draw(Graphics g) {
	    drawSwitch(g, x, y, x2, y2, volts[0], volts[1], open);
	    if (!open)
		doDots(g);
	    drawPosts(g);
	    drawPost(g, x3, y3, nodes[2]);
	}
	final double resistance = 20;
	void calculateCurrent() {
	    if (open)
		current = 0;
	    else
		current = (volts[0]-volts[1])/resistance;
	}
	
	// we need this to be able to change the matrix for each step
	boolean nonLinear() { return true; }

	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	}
	void doStep() {
	    open = (volts[2] < 2.5);
	    if (!open)
		stampResistor(nodes[0], nodes[1], resistance);
	}
	void drawSwitch(Graphics g, int x1, int y1, int x2, int y2,
			double v1, double v2, boolean open) {
	    int i;
	    int ox = 0;
	    int dx = x2-x1;
	    int dy = y2-y1;
	    int openhs = 16;
	    int hs = (open) ? openhs : 0;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    setBbox(x, y, x2, y2);
	    if (dn > 40) {
		int x1a = (int) (x1+dx*(dn/2-16)/dn);
		int y1a = (int) (y1+dy*(dn/2-16)/dn);
		int x2a = (int) (x1+dx*(dn/2+16)/dn);
		int y2a = (int) (y1+dy*(dn/2+16)/dn);
		setVoltageColor(g, v1);
		drawThickLine(g, x1, y1, x1a, y1a);
		setVoltageColor(g, v2);
		drawThickLine(g, x2a, y2a, x2, y2);
		dn = 32;
		x1 = x1a; y1 = y1a;
		x2 = x2a; y2 = y2a;
		dx = x2-x1;
		dy = y2-y1;
	    }
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    g.setColor(Color.lightGray);
	    drawThickLine(g, x1, y1, x2+dpx, y2+dpy);
	    setVoltageColor(g, volts[2]);
	    drawThickLine(g, x3, y3, x3a, y3a);
	    adjustBbox(x3, y3, x2, y2);
	}
	void setPoints() {
	    int dx = x2-x;
	    int dy = y2-y;
	    int openhs = 16;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx0 = (int) ( openhs*dy/dn);
	    int dpy0 = (int) (-openhs*dx/dn);
	    x3 = x+dx/2-dpx0;
	    y3 = y+dy/2-dpy0;
	    x3a = x+dx/2-dpx0/2;
	    y3a = y+dy/2-dpy0/2;
	}
	void drag(int xx, int yy) {
	    xx = snapGrid(xx);
	    yy = snapGrid(yy);
	    if (abs(x-xx) < abs(y-yy))
		xx = x;
	    else
		yy = y;
	    int q1 = abs(x-xx)+abs(y-yy);
	    int q2 = (q1/2) % gridSize;
	    if (q2 != 0)
		return;
	    x2 = xx; y2 = yy;
	    setPoints();
	}
	int getPostCount() { return 3; }
	Point getPost(int n) {
	    return (n == 0) ? new Point(x, y) :
		(n == 1) ? new Point(x2, y2) : new Point(x3, y3);
	}
	void getInfo(String arr[]) {
	    arr[0] = "interruptor analgico";
	    arr[1] = open ? "abierto" : "cerrado";
	    arr[2] = "Vd = " + getVoltageDText(getVoltageDiff());
	    arr[3] = "I = " + getCurrentDText(getCurrent());
	    arr[4] = "Vc = " + getVoltageText(volts[2]);
	}
	boolean getConnection(int n1, int n2) { return false; }
    }

    class AnalogSwitch2Elm extends AnalogSwitchElm {
	public AnalogSwitch2Elm(int xx, int yy) {
	    super(xx, yy);
	}
	public AnalogSwitch2Elm(int xa, int ya, int xb, int yb, int f,
			  StringTokenizer st) {
	    super(xa, ya, xb, yb, f, st);
	}
	// posts
	int x3a, y3a, x3b, y3b, xsw, ysw;
	// switch poles
	int spxa, spya, spxb, spyb;
	int spx1, spy1;
	void draw(Graphics g) {
	    drawSwitch(g, x, y, x2, y2, volts[0], volts[1], volts[2], open);
	    updateDotCount();
	    drawDots(g, x, y, spx1, spy1, curcount);
	    if (!open)
		drawDots(g, spxa, spya, x3a, y3a, curcount);
	    else
		drawDots(g, spxb, spyb, x3b, y3b, curcount);
	    drawPost(g, x,   y,   nodes[0]);
	    drawPost(g, x3a, y3a, nodes[1]);
	    drawPost(g, x3b, y3b, nodes[2]);
	    drawPost(g, xsw, ysw, nodes[3]);
	}
	Point getPost(int n) {
	    return (n == 0) ? new Point(x, y) :
		(n == 1) ? new Point(x3a, y3a) :
		(n == 2) ? new Point(x3b, y3b) : new Point(xsw, ysw);
	}
	int getPostCount() { return 4; }
	void setPoints() {
	    int hs = 16;
	    int dx = x2-x;
	    int dy = y2-y;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    x3a = x2+dpx;
	    y3a = y2+dpy;
	    x3b = x2-dpx;
	    y3b = y2-dpy;
	    xsw = (x+x2)/2 + dpx;
	    ysw = (y+y2)/2 + dpy;
	}
	void drawSwitch(Graphics g, int x1, int y1, int x2, int y2,
			double v1, double v2, double v3, boolean open) {
	    int i;
	    int ox = 0;
	    int dx = x2-x1;
	    int dy = y2-y1;
	    int hs = 16;
	    double dn = Math.sqrt(dx*dx+dy*dy);
	    int dpx = (int) ( hs*dy/dn);
	    int dpy = (int) (-hs*dx/dn);
	    setBbox(x1, y1, x2, y2);
	    if (dn > 40) {
		int x1a = (int) (x1+dx*(dn/2-16)/dn);
		int y1a = (int) (y1+dy*(dn/2-16)/dn);
		int x2a = (int) (x1+dx*(dn/2+16)/dn);
		int y2a = (int) (y1+dy*(dn/2+16)/dn);
		setVoltageColor(g, v1);
		drawThickLine(g, x1, y1, x1a, y1a);
		setVoltageColor(g, v2);
		spxa = x2a+dpx;
		spya = y2a+dpy;
		spxb = x2a-dpx;
		spyb = y2a-dpy;
		drawThickLine(g, spxa, spya, x2+dpx, y2+dpy);
		setVoltageColor(g, v3);
		drawThickLine(g, spxb, spyb, x2-dpx, y2-dpy);
		dn = 32;
		x1 = x1a; y1 = y1a;
		x2 = x2a; y2 = y2a;
		dx = x2-x1;
		dy = y2-y1;
	    }
	    spx1 = x1; spy1 = y1;
	    g.setColor(Color.lightGray);
	    int dir = (open) ? -1 : 1;
	    drawThickLine(g, x1, y1, x2+dpx*dir, y2+dpy*dir);
	    adjustBbox(x1, y1, x2+dpx, y2+dpy);
	    adjustBbox(x1, y1, x2-dpx, y2-dpy);
	}
	int getDumpType() { return 160; }

	void calculateCurrent() {
	    if (open)
		current = (volts[0]-volts[2])/resistance;
	    else
		current = (volts[0]-volts[1])/resistance;
	}
	
	void stamp() {
	    stampNonLinear(nodes[0]);
	    stampNonLinear(nodes[1]);
	    stampNonLinear(nodes[2]);
	}
	void doStep() {
	    open = (volts[3] < 2.5);
	    if (open)
		stampResistor(nodes[0], nodes[2], resistance);
	    else
		stampResistor(nodes[0], nodes[1], resistance);
	}
	
	boolean getConnection(int n1, int n2) { return false; }
	void getInfo(String arr[]) {
	    arr[0] = "interruptor analgico (DPST)";
	    arr[1] = "I = " + getCurrentDText(getCurrent());
	}
    }

    // Given a matrix a[0..n-1][0..n-1] this routine replaces it by the LU
    // decomposition of rowwise permutation of itself. a and n are
    // input. The output matrix a is rearranged, indx[0..n-1] records
    // the permutation, it will be used in the LUsubst() routine.
    // This routine is used in combination with LUsubst()
    // to solve linear equations or invert matrices.
    boolean LUdecomp(double a[][], int n, int indx[]) {
	int i,imax = -1,j,k;
	double big,dum,sum,temp;
	double vv[];  /* stores the scaling of each row */

	vv=new double[n];
        /* get the scaling factors */
	for (i=0;i<n;i++) { 
	    big=0.0;
	    for (j=0;j<n;j++)
		if ((temp=Math.abs(a[i][j])) > big) big=temp;
	    if (big == 0.0)
		return false;
	    vv[i]=1.0/big;
	}
        /* Crout's method */
	for (j=0;j<n;j++) {  
	    for (i=0;i<j;i++) {
		sum=a[i][j];
		for (k=0;k<i;k++) sum -= a[i][k]*a[k][j];
		a[i][j]=sum;
	    }
	    big=0.0;
	    for (i=j;i<n;i++) {
		sum=a[i][j];
		for (k=0;k<j;k++)
		    sum -= a[i][k]*a[k][j];
		a[i][j]=sum;
		if ( (dum=vv[i]*Math.abs(sum)) >= big) {
		    big=dum;
		    imax=i;
		}
	    }
	    /* interchange rows if neccessary */
	    if (j != imax) {
		for (k=0;k<n;k++) {
		    dum=a[imax][k];
		    a[imax][k]=a[j][k];
		    a[j][k]=dum;
		}
		vv[imax]=vv[j];
	    }
	    indx[j]=imax;
	    if (a[j][j] == 0.0) {
		//System.out.println("used tiny " + j);
		a[j][j]=1e-20;
	    }
	    if (j != n-1) {
		dum=1.0/(a[j][j]);
		for (i=j+1;i<n;i++) a[i][j] *= dum;
	    }
	}
	return true;
    }

    // Solves the set of n linear equations. a is the LU decomposition
    // determined by LUdecomp(). indx is the permutation vector also
    // from LUdecomp(). b[0..n-1] is the right hand side of the equations
    // as an input and returns with the solution of the equations. a,
    // n and indx is not modifed by this routine and can be left in
    // space for succesive calls with different right hand side
    // vectors.
    void LUsubst(double a[][], int n, int indx[], double b[]) {
	int i,ii=-1,ip,j;
	double sum;

	for (i=0;i<n;i++) {
	    ip=indx[i];
	    sum=b[ip];
	    b[ip]=b[i];
	    if (ii != -1)
		for (j=ii;j<=i-1;j++) sum -= a[i][j]*b[j];
	    else if (sum != 0) ii=i;
	    b[i]=sum;
	}
	for (i=n-1;i>=0;i--) {
	    sum=b[i];
	    for (j=i+1;j<n;j++) sum -= a[i][j]*b[j];
	    b[i]=sum/a[i][i];
	}
    }

    class EditDialogLayout implements LayoutManager {
	public EditDialogLayout() {}
	public void addLayoutComponent(String name, Component c) {}
	public void removeLayoutComponent(Component c) {}
	public Dimension preferredLayoutSize(Container target) {
	    return new Dimension(500, 500);
	}
	public Dimension minimumLayoutSize(Container target) {
	    return new Dimension(100,100);
	}
	public void layoutContainer(Container target) {
	    Insets insets = target.insets();
	    int targetw = target.size().width - insets.left - insets.right;
	    int targeth = target.size().height - (insets.top+insets.bottom);
	    int i;
	    int h = insets.top;
	    int pw = 300;
	    int x = 0;
	    for (i = 0; i < target.getComponentCount(); i++) {
		Component m = target.getComponent(i);
		boolean newline = true;
		if (m.isVisible()) {
		    Dimension d = m.getPreferredSize();
		    if (pw < d.width)
			pw = d.width;
		    if (m instanceof Scrollbar) {
			h += 10;
			d.width = targetw-x;
		    }
		    if (m instanceof Choice && d.width > targetw)
			d.width = targetw-x;
		    if (m instanceof Label) {
			Dimension d2 =
			    target.getComponent(i+1).getPreferredSize();
			if (d.height < d2.height)
			    d.height = d2.height;
			h += d.height/5;
			newline = false;
		    }
		    if (m instanceof Button) {
			if (x == 0)
			    h += 20;
			if (i != target.getComponentCount()-1)
			    newline = false;
		    }
		    m.move(insets.left+x, h);
		    m.resize(d.width, d.height);
		    if (newline) {
			h += d.height;
			x = 0;
		    } else
			x += d.width;
		}
	    }
	    if (target.size().height < h)
		target.resize(pw + insets.right, h + insets.bottom);
	}
    };

    class EditInfo {
	EditInfo(String n, double val, double mn, double mx) {
	    name = n;
	    value = val;
	    if (minval == 0 && maxval == 0 && val > 0) {
		minval = 1e10;
		while (minval > val/100)
		    minval /= 10.;
		maxval = minval * 1000;
	    } else {
		minval = mn;
		maxval = mx;
	    }
	}
	String name, text;
	double value, minval, maxval;
	TextField textf;
	Scrollbar bar;
	Choice choice;
	Checkbox checkbox;
    }
    
    class EditDialog extends Dialog
	             implements AdjustmentListener, ActionListener,
                                ItemListener {
	CircuitElm elm;
	CircuitFrame cframe;
	Button applyButton, okButton;
	EditInfo einfos[];
	int einfocount;
	final int barmax = 1000;
	
	EditDialog(CircuitElm ce, CircuitFrame f) {
	    super(f, "Editar Componente", false);
	    cframe = f;
	    elm = ce;
	    setLayout(new EditDialogLayout());
	    einfos = new EditInfo[10];
	    int i;
	    for (i = 0; ; i++) {
		einfos[i] = elm.getEditInfo(i);
		if (einfos[i] == null)
		    break;
		EditInfo ei = einfos[i];
		add(new Label(ei.name));
		if (ei.choice != null) {
		    add(ei.choice);
		    ei.choice.addItemListener(this);
		} else if (ei.checkbox != null) {
		    add(ei.checkbox);
		    ei.checkbox.addItemListener(this);
		} else {
		    add(ei.textf =
			new TextField(noCommaFormat.format(ei.value), 10));
		    if (ei.text != null)
			ei.textf.setText(ei.text);
		    ei.textf.addActionListener(this);
		    if (ei.text == null) {
			add(ei.bar = new Scrollbar(Scrollbar.HORIZONTAL,
						   50, 10, 0, barmax+2));
			setBar(ei);
			ei.bar.addAdjustmentListener(this);
		    }
		}
	    }
	    einfocount = i;
	    add(applyButton = new Button("Aplicar"));
	    applyButton.addActionListener(this);
	    add(okButton = new Button("OK"));
	    okButton.addActionListener(this);
	    Point x = main.getLocationOnScreen();
	    Dimension d = getSize();
	    setLocation(x.x + (winSize.width-d.width)/2,
			x.y + (winSize.height-d.height)/2);
	}

	void apply() {
	    int i;
	    for (i = 0; i != einfocount; i++) {
		EditInfo ei = einfos[i];
		if (ei.textf == null)
		    continue;
		if (ei.text == null) {
		    double d = new Double(ei.textf.getText()).doubleValue();
		    ei.value = d;
		}
		elm.setEditValue(i, ei);
		if (ei.text == null)
		    setBar(ei);
	    }
	    cframe.needAnalyze();
	}
	
	public void actionPerformed(ActionEvent e) {
	    int i;
	    Object src = e.getSource();
	    for (i = 0; i != einfocount; i++) {
		EditInfo ei = einfos[i];
		if (src == ei.textf) {
		    if (ei.text == null) {
			double d =
			    new Double(ei.textf.getText()).doubleValue();
			ei.value = d;
		    }
		    elm.setEditValue(i, ei);
		    if (ei.text == null)
			setBar(ei);
		    cframe.needAnalyze();
		}
	    }
	    if (e.getSource() == okButton) {
		apply();
		main.requestFocus();
		setVisible(false);
		editDialog = null;
	    }
	    if (e.getSource() == applyButton)
		apply();
	}
	
	public void adjustmentValueChanged(AdjustmentEvent e) {
	    Object src = e.getSource();
	    int i;
	    for (i = 0; i != einfocount; i++) {
		EditInfo ei = einfos[i];
		if (ei.bar == src) {
		    double v = ei.bar.getValue() / 1000.;
		    if (v < 0)
			v = 0;
		    if (v > 1)
			v = 1;
		    ei.value = (ei.maxval-ei.minval)*v + ei.minval;
		    if (ei.maxval-ei.minval > 100)
			ei.value = Math.round(ei.value);
		    else
			ei.value = Math.round(ei.value*100)/100.;
 		    elm.setEditValue(i, ei);
		    ei.textf.setText(noCommaFormat.format(ei.value));
		    cframe.needAnalyze();
		}
	    }
	}

	public void itemStateChanged(ItemEvent e) {
	    Object src = e.getItemSelectable();
	    int i;
	    for (i = 0; i != einfocount; i++) {
		EditInfo ei = einfos[i];
		if (ei.choice == src || ei.checkbox == src) {
		    elm.setEditValue(i, ei);
		    cframe.needAnalyze();
		}
	    }
	}
	
	public boolean handleEvent(Event ev) {
	    if (ev.id == Event.WINDOW_DESTROY) {
		main.requestFocus();
		setVisible(false);
		editDialog = null;
		return true;
	    }
	    return super.handleEvent(ev);
	}

	void setBar(EditInfo ei) {
	    int x = (int) (barmax*(ei.value-ei.minval)/(ei.maxval-ei.minval));
	    ei.bar.setValue(x);
	}
    }

    class ImportDialogLayout implements LayoutManager {
	public ImportDialogLayout() {}
	public void addLayoutComponent(String name, Component c) {}
	public void removeLayoutComponent(Component c) {}
	public Dimension preferredLayoutSize(Container target) {
	    return new Dimension(500, 500);
	}
	public Dimension minimumLayoutSize(Container target) {
	    return new Dimension(100,100);
	}
	public void layoutContainer(Container target) {
	    Insets insets = target.insets();
	    int targetw = target.size().width - insets.left - insets.right;
	    int targeth = target.size().height - (insets.top+insets.bottom);
	    int i;
	    int pw = 300;
	    if (target.getComponentCount() == 0)
		return;
	    Component cl = target.getComponent(target.getComponentCount()-1);
	    Dimension dl = cl.getPreferredSize();
	    target.getComponent(0).move(insets.left, insets.top);
	    int cw = target.size().width - insets.left - insets.right;
	    int ch = target.size().height - insets.top - insets.bottom -
		dl.height;
	    target.getComponent(0).resize(cw, ch);
	    int h = ch + insets.top;
	    int x = 0;
	    for (i = 1; i < target.getComponentCount(); i++) {
		Component m = target.getComponent(i);
		if (m.isVisible()) {
		    Dimension d = m.getPreferredSize();
		    m.move(insets.left+x, h);
		    m.resize(d.width, d.height);
		    x += d.width;
		}
	    }
	}
    };

    class ImportDialog extends Dialog implements ActionListener {
	CircuitFrame cframe;
	Button importButton, closeButton;
	TextArea text;
	
	ImportDialog(CircuitFrame f, String str) {
	    super(f, (str.length() > 0) ? "Exportar" : "Importar", false);
	    cframe = f;
	    setLayout(new ImportDialogLayout());
	    add(text = new TextArea(str, 10, 60, TextArea.SCROLLBARS_BOTH));
	    add(importButton = new Button("Importar"));
	    importButton.addActionListener(this);
	    add(closeButton = new Button("Cerrar"));
	    closeButton.addActionListener(this);
	    Point x = main.getLocationOnScreen();
	    resize(400, 300);
	    Dimension d = getSize();
	    setLocation(x.x + (winSize.width-d.width)/2,
			x.y + (winSize.height-d.height)/2);
	    show();
	    if (str.length() > 0)
		text.selectAll();
	}

	public void actionPerformed(ActionEvent e) {
	    int i;
	    Object src = e.getSource();
	    if (src == importButton) {
		cframe.readSetup(text.getText());
		setVisible(false);
	    }
	    if (src == closeButton)
		setVisible(false);
	}
	
	public boolean handleEvent(Event ev) {
	    if (ev.id == Event.WINDOW_DESTROY) {
		main.requestFocus();
		setVisible(false);
		editDialog = null;
		return true;
	    }
	    return super.handleEvent(ev);
	}
    }
    
}
