# Modelling soil fertility for precision forestry in New Zealand
# Horacio E. Bown, PhD
# December 11, 2018

# Section 1: Preliminary Step ----
# 1. preliminar steps /clean everything
ls()
rm(list=ls())
ls()


# Section 2: Set working directory ----
# 2. Set working directory 
dir()
getwd() 
setwd("~/Desktop/GESTREN/STATII/EJERCICIO2")
dir()

# Section 3: Read libraries ----
# 3. Read libraries


library(car)
library(raster)
library(rgdal)
library(mapdata)
library(proj4)
library(maps)
library(mapproj)
library(mapplots)
library(rgdal)
library(rgeos)
library(sp)
library(prettymapr)     # add scale bar and arrow


# Section 3a: Tidyverse ----
library(XML)
library(shapefiles)
library(data.table)
library(tiff)
library(tcltk)
library(rgl)
library(RColorBrewer)
library(Hmisc)
library(tidyverse)

# Section 4: Read shape files ----
# 4. Read shape files


RollTrial<-readOGR( dsn="~/Desktop/GESTREN/STATII/EJERCICIO2/RollestonTrial.kml")
RollTrial<-spTransform(RollTrial,CRS("+init=epsg:2193"))
plot(RollTrial)

RollPlots<-readOGR( dsn="~/Desktop/GESTREN/STATII/EJERCICIO2/PLOT_ROLL.kml")
RollPlots<-spTransform(RollPlots,CRS("+init=epsg:2193"))
RollPlots$Plot<-substring(RollPlots$Name, 2)
plot(RollPlots)


ext2<-extent(bbox(RollTrial))

# 5. read sentinel-2

trans<-stack("trans.tif")
plot(trans)

# 6. function to overlay boundaries of trials

fun <- function() {
  plot(RollTrial, border="Blue", add=TRUE)
  plot(RollPlots, border="Red", lwd=0.5, add=TRUE)
}

# 7. Sentinel bands

B2<-trans[[1]]
B3<-trans[[2]]
B4<-trans[[3]]
B5<-trans[[4]]
B6<-trans[[5]]
B7<-trans[[6]]
B8<-trans[[7]]
B8a<-trans[[8]]
B11<-trans[[9]]
B12<-trans[[10]]

# 8. Calculate Sentinel Indices ----

# 11.1 Soil Adjusted Vegetation Index  (abbrv. SAVI)

savi<-(B8-B4)/(B8+B4+0.428)*(1+0.428)
#savi<-raster:mask(savi,TLL)
writeRaster(savi,"savi.tif", format="GTiff", overwrite=TRUE)
rm(savi)
savi<-raster("savi.tif")
plot(savi,main="savi", addfun=fun)


# 11.2 Simple Ratio 800/650 Pigment specific simple ratio B1 (abbrv. PSSRb1) B08 / B04

pssrb1<-B8/B4
writeRaster(pssrb1,"pssrb1.tif", format="GTiff", overwrite=TRUE)
rm(pssrb1)
pssrb1<-raster("pssrb1.tif")
plot(pssrb1, addfun=fun)


# 11.3 NDWI (Normalized Difference Water Index) /There is an option 1 that needs band 11 which not available
#var ndwi = (B08 - B11) / (B08 + B11);
ndwi<-(B8-B11)/(B8+B11)
writeRaster(ndwi,"ndwi.tif", format="GTiff", overwrite=TRUE)
rm(ndwi)
ndwi<-raster("ndwi.tif")

# 11.4 NDVI
ndvi<-(B8-B4)/(B8+B4)

writeRaster(ndvi,"ndvi.tif", format="GTiff", overwrite=TRUE)
rm(ndvi)
ndvi<-raster("ndvi.tif")

# 11.5 NDII
ndii<-(B8-B11)/(B8+B11)

writeRaster(ndii,"ndii.tif", format="GTiff", overwrite=TRUE)
rm(ndii)
ndii<-raster("ndii.tif")

# 11.6 NBR
nbr<-(B8-B12)/(B8+B12)

writeRaster(nbr,"nbr.tif", format="GTiff", overwrite=TRUE)
rm(nbr)
nbr<-raster("nbr.tif")

# 11.7 MSI
msi<-B11/B8

writeRaster(msi,"msi.tif", format="GTiff", overwrite=TRUE)
rm(msi)
msi<-raster("msi.tif")

# 11.8 Modified Chlorophyll Absorption in Reflectance Index   (abbrv. MCARI)
# General formula: ((700nm - 670nm) - 0.2 * (700nm - 550nm)) * (700nm /670nm)
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=41
# ((B05 - B04) - 0.2 * (B05 - B03)) * (B05 / B04);
mcari<-((B5 - B4)-0.2*(B5 - B3))*(B5/B4)/10^3

writeRaster(mcari,"mcari.tif", format="GTiff", overwrite=TRUE)
rm(mcari)
mcari<-raster("mcari.tif")

# 11.9 Green Normalized Difference Vegetation Index   (abbrv. GNDVI)
# General formula: (NIR - [540:570]) / (NIR + [540:570])
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=28
# let index = (B08 - B03) / (B08 + B03);
gndvi<-(B8-B3)/(B8+B3)

writeRaster(gndvi,"gndvi.tif", format="GTiff", overwrite=TRUE)
rm(gndvi)
gndvi<-raster("gndvi.tif")


# 11.10 Enhanced Vegetation Index  (abbrv. EVI)
# General formula: 2.5 * (NIR - RED) / ((NIR + 6*RED - 7.5*BLUE) + 1)
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=16
# let index = 2.5 * (B08 - B04) / ((B08 + 6.0 * B04 - 7.5 * B02) + 1.0);
evi<-2.5*(B8-B4)/((B8+6.0*B4-7.5*B2)+1.0)

#evi<-mask(evi, TLL)
writeRaster(evi,"evi.tif", format="GTiff", overwrite=TRUE)
rm(evi)
evi<-raster("evi.tif")

# 11.11 Enhanced Vegetation Index 2  (abbrv. EVI2)
# General formula: 2.4 * (NIR - RED) / (NIR + RED + 1)
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=237
# let index = 2.4 * (B08 - B04) / (B08 + B04 + 1.0);
evi2<-2.4*(B8-B4)/(B8+B4+1.0)

writeRaster(evi2,"evi2.tif", format="GTiff", overwrite=TRUE)
rm(evi2)
evi2<-raster("evi2.tif")

# 11.12 Chlorophyll Red-Edge  (abbrv. Chlred-edge)
# General formula: ([760:800]/[690:720])^(-1)
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=252
# let index = Math.pow((B07 / B05), (-1.0));
Chlred<-(B7/B5)^(-1.0)

writeRaster(Chlred,"Chlred.tif", format="GTiff", overwrite=TRUE)
rm(Chlred)
Chlred<-raster("Chlred.tif")

# 11.13 // Anthocyanin reflectance index  (abbrv. ARI)
# General formula: 1/550nm-1/700nm
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=214
# let index = 1.0 / B03 - 1.0 / B05;
ari<-(1.0/B3-1/B5)*10^3

writeRaster(ari,"ari.tif", format="GTiff", overwrite=TRUE)
rm(ari)
ari<-raster("ari.tif")

# 11.14 (Table 4) IRECI / Frampton et al. (2013)

IRECI<-(B7-B4)/(B5/B6)/10^3
CCC_14<-0.1795+0.9004*IRECI       # Canopy Chlorophyll Content (g m-2)
LAI_14<-0.2 + 3.2*IRECI           # Leaf area index (m2 m-2)

writeRaster(IRECI,"IRECI.tif", format="GTiff", overwrite=TRUE)
rm(IRECI)
IRECI<-raster("IRECI.tif")

writeRaster(CCC_14,"CCC_14.tif", format="GTiff", overwrite=TRUE)
rm(CCC_14)
CCC_14<-raster("CCC_14.tif")

writeRaster(LAI_14,"LAI_14.tif", format="GTiff", overwrite=TRUE)
rm(LAI_14)
LAI_14<-raster("LAI_14.tif")



# 11.15 (Table 4) MTCI / Frampton et al. (2013)

MTCI<-(B6-B5)/(B5-B4)

writeRaster(MTCI,"MTCI.tif", format="GTiff", overwrite=TRUE)
rm(MTCI)
MTCI<-raster("MTCI.tif")

# 11.16 (Table 4) S2REP / Frampton et al. (2013)

S2REP <- 705 + 35*((((B7+B4)/2)-B5)/(B6-B5))
LCC_16<--10.292 + 0.0147*S2REP

writeRaster(S2REP,"S2REP.tif", format="GTiff", overwrite=TRUE)
rm(S2REP)
S2REP<-raster("S2REP.tif")

writeRaster(LCC_16,"LCC_16.tif", format="GTiff", overwrite=TRUE)
rm(LCC_16)
LCC_16<-raster("LCC_16.tif")

# 11.17 (Table 4) NDI45 / Frampton et al. (2013)

NDI45<-(B5-B4)/(B5+B4)
LAI_17<--1.5481+14.074*NDI45 

writeRaster(NDI45,"NDI45.tif", format="GTiff", overwrite=TRUE)
rm(NDI45)
NDI45<-raster("NDI45.tif")

writeRaster(LAI_17,"LAI_17.tif", format="GTiff", overwrite=TRUE)
rm(LAI_17)
LAI_17<-raster("LAI_17.tif")


# 11.18  (Table 4) CIREDEDGE # Clevers et al. (2017)

CIREDEDGE<-B7/B5-1
CCC_18<-0.8013*CIREDEDGE-0.4704

writeRaster(CIREDEDGE,"CIREDEDGE.tif", format="GTiff", overwrite=TRUE)
rm(CIREDEDGE)
CIREDEDGE<-raster("CIREDEDGE.tif")

writeRaster(CCC_18,"CCC_18.tif", format="GTiff", overwrite=TRUE)
rm(CCC_18)
CCC_18<-raster("CCC_18.tif")

# 11.19 (Table 4) LAI2     # Delegido et al. (2011)

LAI_19<-8.452*NDI45 

writeRaster(LAI_19,"LAI_19.tif", format="GTiff", overwrite=TRUE)
rm(LAI_19)
LAI_19<-raster("LAI_19.tif")



# 11.20 (Table 4) NAOC    # Delegido et al. (2011)

NAOC = 1 - (B4*30+B5*15+B6*15+B7*20)/(152*B7)
Ch_20= -3.8868+101.94*NAOC  
ChTot_20<-Ch_20*10^(-2)*LAI_19

writeRaster(NAOC,"NAOC.tif", format="GTiff", overwrite=TRUE)
rm(NAOC)
NAOC<-raster("NAOC.tif")

writeRaster(Ch_20,"Ch_20.tif", format="GTiff", overwrite=TRUE)
rm(Ch_20)
Ch_20<-raster("Ch_20.tif")

writeRaster(ChTot_20,"ChTot_20.tif", format="GTiff", overwrite=TRUE)
rm(ChTot_20)
ChTot_20<-raster("ChTot_20.tif")

# 11.21 (Table 4) CVI

CVI<-(B8/B3)*(B4/B3)

writeRaster(CVI,"CVI.tif", format="GTiff", overwrite=TRUE)
rm(CVI)
CVI<-raster("CVI.tif")

# 11.22 (Table 4) TCI/OSAVI

OSAVI<-(B7/10^3-B4/10^3)/(B7/10^3+B4/10^3+0.16)
TCIOSAVI<-(1.2*(B5-B3)/10^3-1.5*(B4-B3)/10^3*sqrt(B5/B4))/OSAVI

writeRaster(OSAVI,"OSAVI.tif", format="GTiff", overwrite=TRUE)
rm(OSAVI)
OSAVI<-raster("OSAVI.tif")

writeRaster(TCIOSAVI,"TCIOSAVI.tif", format="GTiff", overwrite=TRUE)
rm(TCIOSAVI)
TCIOSAVI<-raster("TCIOSAVI.tif")




# Section 12: Plot Sentinel bands and indices ----
# 12. Plot Sentinel bands and indices
    
    # 12.1 Band 2



cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B2,0.01), quantile(B2,0.99), length=10),quantile(B2,1))

#cols<-c("#000000","#a50026", "#d73027", "#f46d43", "#fdae61", "#fee08b", "#ffffbf", "#d9ef8b", "#a6d96a", "#66bd63", "#1a9850", "#006837")
#quiebre<-c(seq(quantile(B2,0.05), quantile(B2,0.95), length=11),quantile(B2,1))


#pdf("~/Desktop/NZHybrid/B2TLL.pdf")
plot(B2, main="B2 TLL", addfun=fun,  legend=FALSE)

plot(B2, main="B2 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)



    # 12.2 Band 3

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B3,0.01), quantile(B3,0.99), length=10),quantile(B3,1))

#pdf("~/Desktop/NZHybrid/B3TLL.pdf")
plot(B3, main="B3 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

      # 12.3 Band 4

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B4,0.01), quantile(B4,0.99), length=10),quantile(B4,1))

#pdf("~/Desktop/NZHybrid/B3TLL.pdf")
plot(B4, main="B4 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()


      # 12.4 Band 5

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B5,0.01), quantile(B5,0.99), length=10),quantile(B5,1))

#pdf("~/Desktop/NZHybrid/B3TLL.pdf")
plot(B5, main="B5 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

      # 12.5 Band 6

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B6,0.01), quantile(B6,0.99), length=10),quantile(B6,1))

#pdf("~/Desktop/NZHybrid/B3TLL.pdf")
plot(B6, main="B6 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

      # 12.6 Band 7

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B7,0.01), quantile(B7,0.99), length=10),quantile(B7,1))

#pdf("~/Desktop/NZHybrid/B3TLL.pdf")
plot(B7, main="B7 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

      # 12.7 Band 8

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B8,0.01), quantile(B8,0.99), length=10),quantile(B8,1))

#pdf("~/Desktop/NZHybrid/B3TLL.pdf")
plot(B8, main="B8 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

      # 12.8 Band 8a

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B8a,0.01), quantile(B8a,0.99), length=10),quantile(B8a,1))

#pdf("~/Desktop/NZHybrid/B8aTLL.pdf")
plot(B8a, main="B8a TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

      # 12.9 Band 11

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B11,0.01), quantile(B11,0.99), length=10),quantile(B11,1))

#pdf("~/Desktop/NZHybrid/B11TLL.pdf")
plot(B11, main="B11 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

      # 12.10 Band 12

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(B12,0.01), quantile(B12,0.99), length=10),quantile(B12,1))

#pdf("~/Desktop/NZHybrid/B12TLL.pdf")
plot(B12, main="B12 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,0),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()


# 12.11 Plot RGB composites

plot(NA, xlim=c(-5,5), ylim=c(-5,5), bty='n',
     xaxt='n', yaxt='n', xlab='', ylab='')
text(0,0,"Table 2 \n Sentinel 2 Color Composites", cex=2)



#pdf("~/Desktop/NZHybrid/truergbTLL.pdf")
plotRGB(trans, r = 3, g = 2, b = 1, axes = TRUE, stretch = "lin", main = "Sentinel True Color Composite")
plot(TLL, border="Blue", add=TRUE)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

#pdf("~/Desktop/NZHybrid/falsergbTLL.pdf")
plotRGB(trans, r = 7, g = 3, b = 2, axes = TRUE, stretch = "lin", main = "Sentinel False Color Composite")
plot(TLL, border="Blue", add=TRUE)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

#pdf("~/Desktop/NZHybrid/geolrgbTLL.pdf")
plotRGB(trans, r = 10, g = 9, b = 1, axes = TRUE, stretch = "lin", main = "RGB(12,11,2) Composite")
plot(RollPlots, border="Blue", add=TRUE)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.12 Plot Sentinel Indices



# 12.1 Soil Adjusted Vegetation Index  (abbrv. SAVI)

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(savi,0.01), quantile(savi,0.99), length=10), quantile(savi,1))

#pdf("~/Desktop/NZHybrid/saviTLL.pdf")
plot(savi, main="SAVI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.2 Simple Ratio 800/650 Pigment specific simple ratio B1 (abbrv. PSSRb1) B08 / B04

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(pssrb1,0.05), quantile(pssrb1,0.95), length=10), quantile(pssrb1,1))

#pdf("~/Desktop/NZHybrid/PSSRb1TLL.pdf")
plot(pssrb1, main="PSSRb1 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,1),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.3 NDWI (Normalized Difference Water Index) /There is an option 1 that needs band 11 which not available
#var ndwi = (B08 - B11) / (B08 + B11);

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(ndwi,0.05), quantile(ndwi,0.95), length=10), quantile(ndwi,1))

#pdf("~/Desktop/NZHybrid/nwdiTLL.pdf")
plot(ndwi, main="NDWI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,1),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.4
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.4 NDVI

cols<-c("#000000","#a50026", "#d73027", "#f46d43", "#fdae61", "#fee08b", "#ffffbf", "#d9ef8b", "#a6d96a", "#66bd63", "#1a9850", "#006837")

#pdf("~/Desktop/NZHybrid/ndviTLL.pdf")
plot(ndvi, main="NDVI TLL", addfun=fun, breaks=c(-0.2,0, 0.1, 0.2,0.3, 0.4,0.5, 0.6,0.7, 0.8,0.9,1), col=cols, legend=FALSE)
legend("topright", 
       legend = c("0.2", "0", "0.1", "0.2", "0.3", "0.4", "0.5", "0.6","0.7", "0.8", "0.9", "1.0"),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.4
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()


# 12.5 NDII

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(ndii,0.05), quantile(ndii,0.95), length=10),quantile(ndii,1))

#pdf("~/Desktop/NZHybrid/ndiiTLL.pdf")
plot(ndii, main="NDII TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,1),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.6 NBR

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(nbr,0.05), quantile(nbr,0.95), length=10),quantile(nbr,1))

#pdf("~/Desktop/NZHybrid/nbrTLL.pdf")
plot(nbr, main="NBR TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,1),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.7 MSI

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(msi,0.05), quantile(msi,0.95), length=10),quantile(msi,1))

#pdf("~/Desktop/NZHybrid/msiTLL.pdf")
plot(msi, main="MSI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,1),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.8 Modified Chlorophyll Absorption in Reflectance Index   (abbrv. MCARI)
# General formula: ((700nm - 670nm) - 0.2 * (700nm - 550nm)) * (700nm /670nm)
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=41
# ((B05 - B04) - 0.2 * (B05 - B03)) * (B05 / B04);

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(mcari,0.05), quantile(mcari,0.95), length=10), quantile(mcari,1))

#pdf("~/Desktop/NZHybrid/mcariTLL.pdf")
plot(mcari, main="MCARI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,1),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.9 Green Normalized Difference Vegetation Index   (abbrv. GNDVI)
# General formula: (NIR - [540:570]) / (NIR + [540:570])
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=28
# let index = (B08 - B03) / (B08 + B03);

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(gndvi,0.05), quantile(gndvi,0.95), length=10), quantile(gndvi,1))

#pdf("~/Desktop/NZHybrid/gndviTLL.pdf")
plot(gndvi, main="GNDVI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,1),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()


# 12.10 Enhanced Vegetation Index  (abbrv. EVI)
# General formula: 2.5 * (NIR - RED) / ((NIR + 6*RED - 7.5*BLUE) + 1)
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=16
# let index = 2.5 * (B08 - B04) / ((B08 + 6.0 * B04 - 7.5 * B02) + 1.0);

cols<- gray.colors(11, start=1, end=0)
#cols<- terrain.colors(n=11)
#cols<-heat.colors(n=11)
#cols<-topo.colors(n=11)
#cols<-rainbow(n=11)
#cols<-cm.colors(n=11)
#cols<-c("#000000","#a50026", "#d73027", "#f46d43", "#fdae61", "#fee08b", "#ffffbf", "#d9ef8b", "#a6d96a", "#66bd63", "#1a9850", "#006837")
#quiebre<-c(seq(quantile(evi,0.05), quantile(evi,0.95), length=11), quantile(evi,1))


#quiebre<-c(seq(0.4, 2.8, length=10),5)
quiebre<-c(seq(quantile(evi,0.05), quantile(evi,0.95), length=10), quantile(evi,1))

pdf("~/Desktop/NZHybrid/eviTLL.pdf")
plot(evi, main="EVI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,1),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
dev.off()



# 12.11 Enhanced Vegetation Index 2  (abbrv. EVI2)
# General formula: 2.4 * (NIR - RED) / (NIR + RED + 1)
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=237
# let index = 2.4 * (B08 - B04) / (B08 + B04 + 1.0);

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(evi2,0.01), quantile(evi2,0.99), length=10), quantile(evi2,1))

#pdf("~/Desktop/NZHybrid/evi2TLL.pdf")
plot(evi2, main="EVI2 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()

# 12.12 Chlorophyll Red-Edge  (abbrv. Chlred-edge)
# General formula: ([760:800]/[690:720])^(-1)
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=252
# let index = Math.pow((B07 / B05), (-1.0));

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(Chlred,0.01), quantile(Chlred,0.99), length=10),quantile(Chlred,1))

#pdf("~/Desktop/NZHybrid/ChlredTLL.pdf")
plot(Chlred, main="Chlred TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()


# 12.13 // Anthocyanin reflectance index  (abbrv. ARI)
# General formula: 1/550nm-1/700nm
# URL https://www.indexdatabase.de/db/si-single.php?sensor_id=96&rsindex_id=214
# let index = 1.0 / B03 - 1.0 / B05;

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(ari,0.01), quantile(ari,0.99), length=10), quantile(ari,1))

#pdf("~/Desktop/NZHybrid/ariTLL.pdf")
plot(ari, main="ARI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()    # ru up to here for a pdf with Tables 1,2,3


plot(NA, xlim=c(-5,5), ylim=c(-5,5), bty='n',
     xaxt='n', yaxt='n', xlab='', ylab='')
text(0,0,"Table 4 \n Other Sentinel 2 Indices", cex=2)

# 12.14 (Table 4) IRECI / Frampton et al. (2013)



cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(IRECI,0.01), quantile(IRECI,0.99), length=10), quantile(IRECI,1))


#pdf("~/Desktop/NZHybrid/IRECITLL.pdf")
plot(IRECI, main="IRECI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()    




cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(CCC_14,0.01), quantile(CCC_14,0.99), length=10), quantile(CCC_14,1))


#pdf("~/Desktop/NZHybrid/CCCTLL.pdf")
plot(CCC_14, main="CCC_14 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  


cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(LAI_14,0.01), quantile(LAI_14,0.99), length=10), quantile(LAI_14,1))

#pdf("~/Desktop/NZHybrid/LAI_14TLL.pdf")
plot(LAI_14, main="LAI_14 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  

# 12.15 (Table 4) MTCI / Frampton et al. (2013)

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(MTCI,0.01), quantile(MTCI,0.99), length=10), quantile(MTCI,1))


#pdf("~/Desktop/NZHybrid/MTCITLL.pdf")
plot(MTCI, main="MTCI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  

# 12.16 (Table 4) S2REP / Frampton et al. (2013)

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(S2REP ,0.01), quantile(S2REP ,0.99), length=10), quantile(S2REP ,1))


#pdf("~/Desktop/NZHybrid/S2REPTLL.pdf")
plot(S2REP, main="S2REP TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(LCC_16,0.01), quantile(LCC_16,0.99), length=10), quantile(LCC_16,1))

#pdf("~/Desktop/NZHybrid/LCCTLL.pdf")
plot(LCC_16, main="LCC_16 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  


# 12.17 (Table 4) NDI45 / Frampton et al. (2013)

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(NDI45,0.01), quantile(NDI45,0.99), length=10), quantile(NDI45,1))

#pdf("~/Desktop/NZHybrid/NDI45TLL.pdf")
plot(NDI45, main="NDI45 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  


cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(LAI_17,0.01), quantile(LAI_17,0.99), length=10), quantile(LAI_17,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(LAI_17, main="LAI_17 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  



# 12.18 (Table 4) CIREDEDGE # Clevers et al. (2017)

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(CIREDEDGE,0.01), quantile(CIREDEDGE,0.99), length=10), quantile(CIREDEDGE,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(CIREDEDGE, main="CIREDEDGE TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(CCC_18,0.01), quantile(CCC_18,0.99), length=10), quantile(CCC_18,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(CCC_18, main="CCC_18 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  


# 12.19 (Table 4) LAI19     # Delegido et al. (2011)

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(LAI_19,0.01), quantile(LAI_19,0.99), length=10), quantile(LAI_19,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(LAI_19, main="LAI_19 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  



# 12.20 (Table 4) NAOC    # Delegido et al. (2011)

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(NAOC,0.01), quantile(NAOC,0.99), length=10), quantile(NAOC,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(NAOC, main="NAOC TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  


cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(Ch_20,0.01), quantile(Ch_20,0.99), length=10), quantile(Ch_20,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(Ch_20, main="Ch_20 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  

cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(ChTot_20,0.01), quantile(ChTot_20,0.99), length=10), quantile(ChTot_20,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(ChTot_20, main="ChTot_20 TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  


# 12.21 (Table 4) CVI


cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(CVI,0.01), quantile(CVI,0.99), length=10), quantile(CVI,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(CVI, main="CVI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  


# 12.22 (Table 4) TCI/OSAVI


cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(OSAVI,0.01), quantile(OSAVI,0.99), length=10), quantile(OSAVI,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(OSAVI, main="OSAVI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
#dev.off()  


cols<- gray.colors(11, start=1, end=0)
quiebre<-c(seq(quantile(TCIOSAVI,0.01), quantile(TCIOSAVI,0.99), length=10), quantile(TCIOSAVI,1))

#pdf("~/Desktop/NZHybrid/LAI_17TLL.pdf")
plot(TCIOSAVI, main="TCIOSAVI TLL", addfun=fun, breaks=quiebre, col=cols, legend=FALSE)
legend("topright", 
       legend = round(quiebre,2),
       fill = cols,
       horiz = FALSE, inset=.02,
       cex=0.7
)
addscalebar(plotepsg=2193)
addnortharrow(pos = "topleft", scale=0.5)
dev.off()  

